import argparse
import logging
import os
import sys

logging.basicConfig(level=logging.INFO, format="[%(levelname)s] %(message)s")


def find_toolchain_file(directory):
    """
    Searches for gcc-arm-none-eabi.cmake in the specified directory.
    :param directory: Root directory containing the CMake configuration.
    :return: The path to the found file, or None if not found.
    """
    if not os.path.isdir(directory):
        print(
            f"ERROR: The specified directory '{directory}' does not exist or is invalid.",
            file=sys.stderr,
        )
        return None

    target_filename = "gcc-arm-none-eabi.cmake"
    cmake_path = os.path.join(directory, "cmake", target_filename)

    return cmake_path if os.path.exists(cmake_path) else None


def parse_cmake_file(file_path):
    """
    Parses the CMake configuration file and extracts necessary data.
    :param file_path: Path to the CMake configuration file.
    :return: A dictionary containing parsed data.
    """
    data = {
        "linker_script": None,
        "target_flags": None,
        "raw_linker_script_line": None,
        "raw_target_flags_line": None,
    }

    try:
        with open(file_path, "r", encoding="utf-8") as file:
            for line in file:
                line = line.strip()
                if ("flash.ld" in line or "FLASH.ld" in line or "flash.LD" in line or "FLASH.LD" in line) and data[
                    "linker_script"] is None:
                    data["raw_linker_script_line"] = line.replace("# ", "")
                    data["linker_script"] = line.split("/")[-1].strip(' \\")')
                elif "-mcpu=" in line and data["target_flags"] is None:
                    data["raw_target_flags_line"] = line.replace("# ", "")
                    data["target_flags"] = (
                        line.replace('set(TARGET_FLAGS "', "")
                        .replace(' ")', "")
                        .strip()
                        .replace("# ", "")
                    )

    except FileNotFoundError:
        print(f"ERROR: CMake file '{file_path}' not found.", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"ERROR: Failed to parse the CMake file - {e}", file=sys.stderr)
        sys.exit(1)

    return data


def gen_toolchain_cmake(data):
    """
    Generates the new CMake toolchain configuration based on extracted data.
    :param data: Parsed CMake data.
    :return: Generated CMake file content.
    """
    cmake_file_content = f"""# Generated by libxr

# Old CMakeLists.txt info
# {data["raw_linker_script_line"]}
# {data["raw_target_flags_line"]}

# Target definition
set(CMAKE_SYSTEM_NAME Generic)
set(CMAKE_SYSTEM_PROCESSOR ARM)

# Toolchain settings
set(CMAKE_LINKER arm-none-eabi-ld)
set(CMAKE_AR arm-none-eabi-ar)
set(CMAKE_SIZE arm-none-eabi-size)
set(CMAKE_OBJCOPY arm-none-eabi-objcopy)
set(CMAKE_OBJDUMP arm-none-eabi-objdump)

set(CMAKE_FIND_ROOT_PATH_MODE_PROGRAM NEVER)
set(CMAKE_FIND_ROOT_PATH_MODE_LIBRARY ONLY)
set(CMAKE_FIND_ROOT_PATH_MODE_INCLUDE ONLY)
set(CMAKE_FIND_ROOT_PATH_MODE_PACKAGE ONLY)

# this makes the test compiles use static library option so that we don't need
# to pre-set linker flags and scripts
set(CMAKE_TRY_COMPILE_TARGET_TYPE STATIC_LIBRARY)

set(CLANG_TARGET arm-none-eabi)
set(GNU_COMPILER arm-none-eabi)

set(CMAKE_C_COMPILER clang)
set(CMAKE_CXX_COMPILER clang++)
set(CMAKE_ASM_COMPILER clang)

execute_process(
    COMMAND arm-none-eabi-gcc -print-sysroot
    OUTPUT_VARIABLE GCC_ARM_NONE_EABI_ROOT
    OUTPUT_STRIP_TRAILING_WHITESPACE
)

if(NOT EXISTS ${{GCC_ARM_NONE_EABI_ROOT}})
    if(NOT EXISTS /usr/lib/arm-none-eabi OR NOT EXISTS /usr/include/newlib)
    message(FATAL_ERROR "Could not find arm-none-eabi toolchain.")
    endif()
    file(GLOB GCC_ARM_NONE_EABI_INCLUDE
    "/usr/include/newlib/c++/*/cstddef")

    get_filename_component(GCC_ARM_NONE_EABI_INCLUDE
    "${{GCC_ARM_NONE_EABI_INCLUDE}}" DIRECTORY)
    add_compile_options(
        --sysroot=/usr/lib/arm-none-eabi
        -isystem${{GCC_ARM_NONE_EABI_INCLUDE}}
        -isystem${{GCC_ARM_NONE_EABI_INCLUDE}}/arm-none-eabi
        -isystem${{GCC_ARM_NONE_EABI_INCLUDE}}/../../
    )
else()
file(GLOB_RECURSE GCC_ARM_NONE_EABI_INCLUDE
    "${{GCC_ARM_NONE_EABI_ROOT}}/include/c++/*/cstddef")

get_filename_component(GCC_ARM_NONE_EABI_INCLUDE
    "${{GCC_ARM_NONE_EABI_INCLUDE}}" DIRECTORY)

add_compile_options(
    -isystem${{GCC_ARM_NONE_EABI_INCLUDE}}
    -isystem${{GCC_ARM_NONE_EABI_INCLUDE}}/arm-none-eabi
    -isystem${{GCC_ARM_NONE_EABI_INCLUDE}}/arm-none-eabi/include
    -isystem${{GCC_ARM_NONE_EABI_ROOT}}/include
)
endif()

add_compile_options(
    --target=${{CLANG_TARGET}}
)

# Use GUN linker. Because this project use nano and nosys lib, but lld.ld do not
# support specs files.
set(CMAKE_C_LINK_EXECUTABLE
    "${{GNU_COMPILER}}-gcc <FLAGS> <CMAKE_C_LINK_FLAGS> <LINK_FLAGS> <OBJECTS> -o <TARGET> <LINK_LIBRARIES> -lc"
)
set(CMAKE_CXX_LINK_EXECUTABLE
    "${{GNU_COMPILER}}-g++ <FLAGS> <CMAKE_C_LINK_FLAGS> <LINK_FLAGS> <OBJECTS> -o <TARGET> <LINK_LIBRARIES> -lc"
)

set(CPU_FLAGS
    "{data["target_flags"]}"
    CACHE INTERNAL "" FORCE)

set(GENERAL_FLAGS
    "-Wall -Wextra -fno-builtin -fno-exceptions -ffunction-sections -fdata-sections"
    CACHE INTERNAL "" FORCE)

set(CMAKE_C_FLAGS
    "${{CPU_FLAGS}} ${{GENERAL_FLAGS}} -fshort-enums -fdiagnostics-color=auto"
    CACHE INTERNAL "" FORCE)

set(CMAKE_CXX_FLAGS
    "${{CPU_FLAGS}} ${{GENERAL_FLAGS}} -fno-threadsafe-statics -fno-rtti -fshort-enums -fdiagnostics-color=auto"
    CACHE INTERNAL "" FORCE)

set(CMAKE_ASM_FLAGS
    "${{CPU_FLAGS}} -x assembler-with-cpp"
    CACHE INTERNAL "" FORCE)

# Linker Flag
set(LINKER_SCRIPT "${{CMAKE_CURRENT_SOURCE_DIR}}/{data["linker_script"]}")

set(CMAKE_EXE_LINKER_FLAGS
    "-T${{LINKER_SCRIPT}} --specs=nano.specs --specs=nosys.specs -Wl,--cref,--gc-sections,--print-memory-usage,-Map=${{CMAKE_PROJECT_NAME}}.map,--no-warn-execstack,--no-warn-rwx-segments"
    CACHE INTERNAL "" FORCE)

add_compile_options(-gdwarf-4)

set(CMAKE_EXPORT_COMPILE_COMMANDS ON)

file(MAKE_DIRECTORY ${{CMAKE_SOURCE_DIR}}/build)

if(NOT TARGET copy_compile_commands_to_build)
    add_custom_target(copy_compile_commands_to_build ALL
        COMMAND ${{CMAKE_COMMAND}} -E copy_if_different
                ${{CMAKE_BINARY_DIR}}/compile_commands.json
                ${{CMAKE_SOURCE_DIR}}/build/compile_commands.json
        DEPENDS ${{CMAKE_BINARY_DIR}}/compile_commands.json
    )
endif()
"""
    return cmake_file_content


def main():
    parser = argparse.ArgumentParser(
        description="Parse the CMake configuration file and extract key data. Overwrites the original file by default."
    )
    parser.add_argument(
        "input_dir", help="Directory containing the CMake configuration file"
    )
    parser.add_argument(
        "output_file",
        nargs="?",
        default=None,
        help="Output file path (default: overwrite the original file)",
    )

    args = parser.parse_args()

    # Locate the CMake configuration file
    toolchain_file = find_toolchain_file(args.input_dir)
    if not toolchain_file:
        print(
            f"ERROR: Could not find 'gcc-arm-none-eabi.cmake' in '{args.input_dir}/cmake'.",
            file=sys.stderr,
        )
        sys.exit(1)

    cmake_file_path = os.path.join(os.path.abspath(args.input_dir), "CMakeLists.txt")

    if not os.path.exists(cmake_file_path):
        logging.error(f"CMake file not found: {cmake_file_path}")
        sys.exit(1)

    print(f"Found CMake file: {toolchain_file}")

    # Parse the CMake configuration file
    data = parse_cmake_file(toolchain_file)

    if data["linker_script"] is None or data["target_flags"] is None:
        logging.error("Failed to parse CMake file. Required fields are missing.")
        sys.exit(1)

    # Default to overwriting the original file
    output_file = args.output_file if args.output_file else toolchain_file

    print(f"Writing output to: {output_file}")

    # Check if the linker script exists
    linker_script_path = os.path.join(args.input_dir, data["linker_script"])
    if not os.path.exists(linker_script_path):
        print(
            f"ERROR: Linker script '{data['linker_script']}' not found.",
            file=sys.stderr,
        )
        sys.exit(-1)

    # Overwrite the original CMake configuration file
    try:
        with open(output_file, "w", encoding="utf-8") as out_file:
            out_file.write(gen_toolchain_cmake(data))
            out_file.close()

        logging.info(f"Toolchain CMake configuration updated: {output_file}")
        logging.info(f"Using linker script: {data['linker_script']}")
        logging.info(f"Target CPU flags: {data['target_flags']}")

    except Exception as e:
        logging.error(f"Failed to write output file '{output_file}': {repr(e)}")
        sys.exit(1)

    try:
        with open(cmake_file_path, "r", encoding="utf-8") as out_file:
            cmake_file_lines = out_file.readlines()
    except Exception as e:
        print(
            f"ERROR: Failed to write to file '{cmake_file_path}' - {e}", file=sys.stderr
        )
        sys.exit(1)

    include_line = 'include("cmake/gcc-arm-none-eabi.cmake")\n'
    found_in_first_five = any(
        include_line.strip() in line.strip() for line in cmake_file_lines[:5]
    )
    if found_in_first_five:
        logging.info(
            "Include line already present in first 5 lines of CMakeLists.txt; no modification needed."
        )
        sys.exit(0)

    cmake_file_lines = [
        line for line in cmake_file_lines if line.strip() != include_line.strip()
    ]

    if len(cmake_file_lines) >= 2:
        cmake_file_lines.insert(2, "\n")
        cmake_file_lines.insert(2, include_line)
    else:
        cmake_file_lines.append(include_line)

    with open(cmake_file_path, "w") as f:
        f.writelines(cmake_file_lines)

    logging.info("CMakeLists.txt modified successfully with include directive.")
