"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensureLambda = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("cdk-iam-floyd");
const path = require("path");
const lambdaTimeout = cdk.Duration.seconds(10);
function ensureLambda(scope) {
    const stack = cdk.Stack.of(scope);
    const lambdaName = 'AthenaManager';
    const createdByTag = 'CreatedByCfnCustomResource';
    const ID = 'CFN::Resource::Custom::Athena';
    const existing = stack.node.tryFindChild(lambdaName);
    if (existing) {
        return existing;
    }
    const policy = new iam.ManagedPolicy(stack, 'Athena-Manager-Policy', {
        managedPolicyName: `${stack.stackName}-${lambdaName}`,
        description: `Used by Lambda ${lambdaName}, which is a custom CFN resource, managing Athena resources`,
        statements: [
            new statement.Athena() //
                .allow()
                .toGetWorkGroup(),
            new statement.Athena()
                .allow()
                .toCreateWorkGroup()
                .toTagResource()
                .ifAwsRequestTag(createdByTag, `${ID}-WorkGroup`),
            new statement.Athena()
                .allow()
                .toDeleteWorkGroup()
                .toUpdateWorkGroup()
                .toTagResource()
                .toUntagResource()
                .ifAwsResourceTag(createdByTag, `${ID}-WorkGroup`),
            new statement.Athena()
                .allow()
                .toGetNamedQuery()
                .toListNamedQueries()
                .toCreateNamedQuery()
                .toDeleteNamedQuery(),
        ],
    });
    const role = new iam.Role(stack, 'Athena-Manager-Role', {
        roleName: `${stack.stackName}-${lambdaName}`,
        description: `Used by Lambda ${lambdaName}, which is a custom CFN resource, managing Athena resources`,
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        managedPolicies: [
            policy,
            iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
        ],
    });
    const fn = new lambda.Function(stack, lambdaName, {
        functionName: `${stack.stackName}-${lambdaName}`,
        role: role,
        description: 'Custom CFN resource: Manage Athena resources',
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
        timeout: lambdaTimeout,
    });
    return fn;
}
exports.ensureLambda = ensureLambda;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGFtYmRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF5QztBQUN6Qyw4Q0FBK0M7QUFDL0MscUNBQXNDO0FBQ3RDLDJDQUEyQztBQUMzQyw2QkFBOEI7QUFFOUIsTUFBTSxhQUFhLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7QUFFL0MsU0FBZ0IsWUFBWSxDQUFDLEtBQW9CO0lBQy9DLE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2xDLE1BQU0sVUFBVSxHQUFHLGVBQWUsQ0FBQztJQUNuQyxNQUFNLFlBQVksR0FBRyw0QkFBNEIsQ0FBQztJQUNsRCxNQUFNLEVBQUUsR0FBRywrQkFBK0IsQ0FBQztJQUMzQyxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNyRCxJQUFJLFFBQVEsRUFBRTtRQUNaLE9BQU8sUUFBMkIsQ0FBQztLQUNwQztJQUVELE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsdUJBQXVCLEVBQUU7UUFDbkUsaUJBQWlCLEVBQUUsR0FBRyxLQUFLLENBQUMsU0FBUyxJQUFJLFVBQVUsRUFBRTtRQUNyRCxXQUFXLEVBQUUsa0JBQWtCLFVBQVUsNkRBQTZEO1FBQ3RHLFVBQVUsRUFBRTtZQUNWLElBQUksU0FBUyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUU7aUJBQ3RCLEtBQUssRUFBRTtpQkFDUCxjQUFjLEVBQUU7WUFDbkIsSUFBSSxTQUFTLENBQUMsTUFBTSxFQUFFO2lCQUNuQixLQUFLLEVBQUU7aUJBQ1AsaUJBQWlCLEVBQUU7aUJBQ25CLGFBQWEsRUFBRTtpQkFDZixlQUFlLENBQUMsWUFBWSxFQUFFLEdBQUcsRUFBRSxZQUFZLENBQUM7WUFDbkQsSUFBSSxTQUFTLENBQUMsTUFBTSxFQUFFO2lCQUNuQixLQUFLLEVBQUU7aUJBQ1AsaUJBQWlCLEVBQUU7aUJBQ25CLGlCQUFpQixFQUFFO2lCQUNuQixhQUFhLEVBQUU7aUJBQ2YsZUFBZSxFQUFFO2lCQUNqQixnQkFBZ0IsQ0FBQyxZQUFZLEVBQUUsR0FBRyxFQUFFLFlBQVksQ0FBQztZQUNwRCxJQUFJLFNBQVMsQ0FBQyxNQUFNLEVBQUU7aUJBQ25CLEtBQUssRUFBRTtpQkFDUCxlQUFlLEVBQUU7aUJBQ2pCLGtCQUFrQixFQUFFO2lCQUNwQixrQkFBa0IsRUFBRTtpQkFDcEIsa0JBQWtCLEVBQUU7U0FDeEI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3RELFFBQVEsRUFBRSxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksVUFBVSxFQUFFO1FBQzVDLFdBQVcsRUFBRSxrQkFBa0IsVUFBVSw2REFBNkQ7UUFDdEcsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1FBQzNELGVBQWUsRUFBRTtZQUNmLE1BQU07WUFDTixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUN4QywwQ0FBMEMsQ0FDM0M7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sRUFBRSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQ2hELFlBQVksRUFBRSxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksVUFBVSxFQUFFO1FBQ2hELElBQUksRUFBRSxJQUFJO1FBQ1YsV0FBVyxFQUFFLDhDQUE4QztRQUMzRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO1FBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDO1FBQ3ZFLE9BQU8sRUFBRSxhQUFhO0tBQ3ZCLENBQUMsQ0FBQztJQUVILE9BQU8sRUFBRSxDQUFDO0FBQ1osQ0FBQztBQTdERCxvQ0E2REMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgaWFtID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWlhbScpO1xuaW1wb3J0IGxhbWJkYSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnKTtcbmltcG9ydCBjZGsgPSByZXF1aXJlKCdAYXdzLWNkay9jb3JlJyk7XG5pbXBvcnQgKiBhcyBzdGF0ZW1lbnQgZnJvbSAnY2RrLWlhbS1mbG95ZCc7XG5pbXBvcnQgcGF0aCA9IHJlcXVpcmUoJ3BhdGgnKTtcblxuY29uc3QgbGFtYmRhVGltZW91dCA9IGNkay5EdXJhdGlvbi5zZWNvbmRzKDEwKTtcblxuZXhwb3J0IGZ1bmN0aW9uIGVuc3VyZUxhbWJkYShzY29wZTogY2RrLkNvbnN0cnVjdCk6IGxhbWJkYS5GdW5jdGlvbiB7XG4gIGNvbnN0IHN0YWNrID0gY2RrLlN0YWNrLm9mKHNjb3BlKTtcbiAgY29uc3QgbGFtYmRhTmFtZSA9ICdBdGhlbmFNYW5hZ2VyJztcbiAgY29uc3QgY3JlYXRlZEJ5VGFnID0gJ0NyZWF0ZWRCeUNmbkN1c3RvbVJlc291cmNlJztcbiAgY29uc3QgSUQgPSAnQ0ZOOjpSZXNvdXJjZTo6Q3VzdG9tOjpBdGhlbmEnO1xuICBjb25zdCBleGlzdGluZyA9IHN0YWNrLm5vZGUudHJ5RmluZENoaWxkKGxhbWJkYU5hbWUpO1xuICBpZiAoZXhpc3RpbmcpIHtcbiAgICByZXR1cm4gZXhpc3RpbmcgYXMgbGFtYmRhLkZ1bmN0aW9uO1xuICB9XG5cbiAgY29uc3QgcG9saWN5ID0gbmV3IGlhbS5NYW5hZ2VkUG9saWN5KHN0YWNrLCAnQXRoZW5hLU1hbmFnZXItUG9saWN5Jywge1xuICAgIG1hbmFnZWRQb2xpY3lOYW1lOiBgJHtzdGFjay5zdGFja05hbWV9LSR7bGFtYmRhTmFtZX1gLFxuICAgIGRlc2NyaXB0aW9uOiBgVXNlZCBieSBMYW1iZGEgJHtsYW1iZGFOYW1lfSwgd2hpY2ggaXMgYSBjdXN0b20gQ0ZOIHJlc291cmNlLCBtYW5hZ2luZyBBdGhlbmEgcmVzb3VyY2VzYCxcbiAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICBuZXcgc3RhdGVtZW50LkF0aGVuYSgpIC8vXG4gICAgICAgIC5hbGxvdygpXG4gICAgICAgIC50b0dldFdvcmtHcm91cCgpLFxuICAgICAgbmV3IHN0YXRlbWVudC5BdGhlbmEoKVxuICAgICAgICAuYWxsb3coKVxuICAgICAgICAudG9DcmVhdGVXb3JrR3JvdXAoKVxuICAgICAgICAudG9UYWdSZXNvdXJjZSgpXG4gICAgICAgIC5pZkF3c1JlcXVlc3RUYWcoY3JlYXRlZEJ5VGFnLCBgJHtJRH0tV29ya0dyb3VwYCksXG4gICAgICBuZXcgc3RhdGVtZW50LkF0aGVuYSgpXG4gICAgICAgIC5hbGxvdygpXG4gICAgICAgIC50b0RlbGV0ZVdvcmtHcm91cCgpXG4gICAgICAgIC50b1VwZGF0ZVdvcmtHcm91cCgpXG4gICAgICAgIC50b1RhZ1Jlc291cmNlKClcbiAgICAgICAgLnRvVW50YWdSZXNvdXJjZSgpXG4gICAgICAgIC5pZkF3c1Jlc291cmNlVGFnKGNyZWF0ZWRCeVRhZywgYCR7SUR9LVdvcmtHcm91cGApLFxuICAgICAgbmV3IHN0YXRlbWVudC5BdGhlbmEoKVxuICAgICAgICAuYWxsb3coKVxuICAgICAgICAudG9HZXROYW1lZFF1ZXJ5KClcbiAgICAgICAgLnRvTGlzdE5hbWVkUXVlcmllcygpXG4gICAgICAgIC50b0NyZWF0ZU5hbWVkUXVlcnkoKVxuICAgICAgICAudG9EZWxldGVOYW1lZFF1ZXJ5KCksXG4gICAgXSxcbiAgfSk7XG5cbiAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ0F0aGVuYS1NYW5hZ2VyLVJvbGUnLCB7XG4gICAgcm9sZU5hbWU6IGAke3N0YWNrLnN0YWNrTmFtZX0tJHtsYW1iZGFOYW1lfWAsXG4gICAgZGVzY3JpcHRpb246IGBVc2VkIGJ5IExhbWJkYSAke2xhbWJkYU5hbWV9LCB3aGljaCBpcyBhIGN1c3RvbSBDRk4gcmVzb3VyY2UsIG1hbmFnaW5nIEF0aGVuYSByZXNvdXJjZXNgLFxuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgcG9saWN5LFxuICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKFxuICAgICAgICAnc2VydmljZS1yb2xlL0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZSdcbiAgICAgICksXG4gICAgXSxcbiAgfSk7XG5cbiAgY29uc3QgZm4gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCBsYW1iZGFOYW1lLCB7XG4gICAgZnVuY3Rpb25OYW1lOiBgJHtzdGFjay5zdGFja05hbWV9LSR7bGFtYmRhTmFtZX1gLFxuICAgIHJvbGU6IHJvbGUsXG4gICAgZGVzY3JpcHRpb246ICdDdXN0b20gQ0ZOIHJlc291cmNlOiBNYW5hZ2UgQXRoZW5hIHJlc291cmNlcycsXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE0X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGFtYmRhL2NvZGUuemlwJykpLFxuICAgIHRpbWVvdXQ6IGxhbWJkYVRpbWVvdXQsXG4gIH0pO1xuXG4gIHJldHVybiBmbjtcbn1cbiJdfQ==