"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
// import * as ecs from '@aws-cdk/aws-ecs';
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const cdk = require("aws-cdk-lib");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
test('Test new vpc, load balancer, service', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '10.0.0.0/16',
    });
});
test('Test new load balancer, service, existing vpc', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testProps = {
        existingVpc: defaults.getTestVpc(stack),
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '172.168.0.0/16'
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
});
test('Test new service, existing load balancer, vpc', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testName = 'test-value';
    const existingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: existingVpc,
        internetFacing: true,
        loadBalancerName: testName,
    });
    const testProps = {
        existingVpc,
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingLoadBalancerObj: existingAlb,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::Listener', 1);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: testName
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::LoadBalancer', 1);
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '172.168.0.0/16'
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
});
test('Test existing load balancer, vpc, service', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testName = 'test-value';
    const existingVpc = defaults.getTestVpc(stack);
    const [testService, testContainer] = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: existingVpc,
        internetFacing: true,
        loadBalancerName: testName,
    });
    const testProps = {
        existingVpc,
        publicApi: true,
        existingFargateServiceObject: testService,
        existingContainerDefinitionObject: testContainer,
        existingLoadBalancerObj: existingAlb,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::Listener', 1);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: testName
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::LoadBalancer', 1);
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '172.168.0.0/16'
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
});
test('Test add a second target with rules', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testProps = {
        existingVpc: defaults.getTestVpc(stack),
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    const firstConstruct = new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const testPropsTwo = {
        existingVpc: firstConstruct.vpc,
        existingContainerDefinitionObject: firstConstruct.container,
        existingFargateServiceObject: firstConstruct.service,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true,
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
    };
    new lib_1.AlbToFargate(stack, 'test-two-construct', testPropsTwo);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::ListenerRule', 1);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ],
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '172.168.0.0/16'
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
});
test('Test new vpc, load balancer, service - custom Service Props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const serviceName = 'test-value';
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
        fargateServiceProps: {
            serviceName
        }
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        ServiceName: serviceName,
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true
    });
});
test('Test new vpc, load balancer, service - custom VPC Props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testCidr = '192.0.0.0/19';
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
        vpcProps: { cidr: testCidr },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: testCidr,
    });
});
test('Test new vpc, load balancer, service - custom LoadBalancer and targetGroup Props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testLoadBalancerName = "test-lb";
    const testTargetGroupName = 'test-tg';
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
        loadBalancerProps: {
            loadBalancerName: testLoadBalancerName,
        },
        targetGroupProps: {
            targetGroupName: testTargetGroupName,
        }
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: testLoadBalancerName
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        Name: testTargetGroupName
    });
});
test('Test HTTPS API with new vpc, load balancer, service', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const fakeCert = defaults.getFakeCertificate(stack, 'fake-cert');
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTPS',
            certificates: [fakeCert]
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
        DefaultActions: [
            {
                RedirectConfig: {
                    Port: "443",
                    Protocol: "HTTPS",
                    StatusCode: "HTTP_302"
                },
                Type: "redirect"
            }
        ],
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
        Port: 443,
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true
    });
});
test('Test HTTPS API with new vpc, load balancer, service and private API', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const fakeCert = defaults.getFakeCertificate(stack, 'fake-cert');
    const testProps = {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTPS',
            certificates: [fakeCert]
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
        DefaultActions: [
            {
                RedirectConfig: {
                    Port: "443",
                    Protocol: "HTTPS",
                    StatusCode: "HTTP_302"
                },
                Type: "redirect"
            }
        ],
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
        Port: 443,
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 3);
    expect(stack).toHaveResource("AWS::EC2::Subnet", {
        Tags: [
            {
                Key: "aws-cdk:subnet-name",
                Value: "isolated"
            },
            {
                Key: "aws-cdk:subnet-type",
                Value: "Isolated"
            },
            {
                Key: "Name",
                Value: "Default/Vpc/isolatedSubnet1"
            }
        ]
    });
    expect(stack).not.toHaveResource("AWS::EC2::Subnet", {
        Tags: [
            {
                Key: "aws-cdk:subnet-name",
                Value: "Public"
            },
            {
                Key: "aws-cdk:subnet-type",
                Value: "Public"
            },
            {
                Key: "Name",
                Value: "Default/Vpc/PublicSubnet1"
            }
        ]
    });
});
//# sourceMappingURL=data:application/json;base64,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