"""Code Interpreter integration for ostruct CLI.

This module provides support for uploading files to OpenAI's Code Interpreter
and integrating code execution capabilities with the OpenAI Responses API.
"""

import logging
import os
from pathlib import Path
from typing import Any, Dict, List

from openai import AsyncOpenAI

logger = logging.getLogger(__name__)


class CodeInterpreterManager:
    """Manager for Code Interpreter file uploads and tool integration."""

    def __init__(self, client: AsyncOpenAI):
        """Initialize Code Interpreter manager.

        Args:
            client: AsyncOpenAI client instance
        """
        self.client = client
        self.uploaded_file_ids: List[str] = []

    async def upload_files_for_code_interpreter(
        self, files: List[str]
    ) -> List[str]:
        """Upload files for Code Interpreter (validated working pattern).

        This method uploads files to OpenAI's file storage with the correct
        purpose for Code Interpreter usage.

        Args:
            files: List of file paths to upload

        Returns:
            List of file IDs from successful uploads

        Raises:
            FileNotFoundError: If a file doesn't exist
            Exception: If upload fails
        """
        file_ids = []

        for file_path in files:
            try:
                # Validate file exists
                if not os.path.exists(file_path):
                    raise FileNotFoundError(f"File not found: {file_path}")

                # Get file info
                file_size = os.path.getsize(file_path)
                logger.debug(
                    f"Uploading file: {file_path} ({file_size} bytes)"
                )

                # Upload with correct purpose for Code Interpreter
                with open(file_path, "rb") as f:
                    file_obj = await self.client.files.create(
                        file=f,
                        purpose="assistants",  # Validated correct purpose
                    )
                    file_ids.append(file_obj.id)
                    logger.debug(
                        f"Successfully uploaded {file_path} with ID: {file_obj.id}"
                    )

            except Exception as e:
                logger.error(f"Failed to upload file {file_path}: {e}")
                # Clean up any successfully uploaded files on error
                await self._cleanup_uploaded_files(file_ids)
                raise

        # Store for potential cleanup
        self.uploaded_file_ids.extend(file_ids)
        return file_ids

    def build_tool_config(self, file_ids: List[str]) -> dict:
        """Build Code Interpreter tool configuration.

        Creates a tool configuration compatible with the OpenAI Responses API
        for Code Interpreter functionality.

        Args:
            file_ids: List of uploaded file IDs

        Returns:
            Tool configuration dict for Responses API
        """
        return {
            "type": "code_interpreter",
            "container": {"type": "auto", "file_ids": file_ids},
        }

    async def download_generated_files(
        self, response_file_ids: List[str], output_dir: str = "."
    ) -> List[str]:
        """Download files generated by Code Interpreter.

        Args:
            response_file_ids: List of file IDs from Code Interpreter response
            output_dir: Directory to save downloaded files

        Returns:
            List of local file paths where files were saved

        Raises:
            Exception: If download fails
        """
        downloaded_paths = []
        output_path = Path(output_dir)
        output_path.mkdir(exist_ok=True)

        for file_id in response_file_ids:
            try:
                # Get file info
                file_info = await self.client.files.retrieve(file_id)
                filename = (
                    file_info.filename or f"generated_file_{file_id[:8]}.dat"
                )

                # Download file content
                file_content = await self.client.files.content(file_id)

                # Save to local file
                local_path = output_path / filename
                with open(local_path, "wb") as f:
                    f.write(file_content.read())

                downloaded_paths.append(str(local_path))
                logger.debug(f"Downloaded generated file: {local_path}")

            except Exception as e:
                logger.error(f"Failed to download file {file_id}: {e}")
                raise

        return downloaded_paths

    async def cleanup_uploaded_files(self) -> None:
        """Clean up uploaded files from OpenAI storage.

        This method removes files that were uploaded during the session
        to avoid accumulating files in the user's OpenAI storage.
        """
        await self._cleanup_uploaded_files(self.uploaded_file_ids)
        self.uploaded_file_ids.clear()

    async def _cleanup_uploaded_files(self, file_ids: List[str]) -> None:
        """Internal method to clean up specific file IDs.

        Args:
            file_ids: List of file IDs to delete
        """
        for file_id in file_ids:
            try:
                await self.client.files.delete(file_id)
                logger.debug(f"Cleaned up uploaded file: {file_id}")
            except Exception as e:
                logger.warning(f"Failed to clean up file {file_id}: {e}")

    def validate_files_for_upload(self, files: List[str]) -> List[str]:
        """Validate files are suitable for Code Interpreter upload.

        Args:
            files: List of file paths to validate

        Returns:
            List of validation error messages, empty if all files are valid
        """
        errors = []

        # Common file types supported by Code Interpreter
        supported_extensions = {
            ".py",
            ".txt",
            ".csv",
            ".json",
            ".xlsx",
            ".xls",
            ".pdf",
            ".docx",
            ".md",
            ".xml",
            ".html",
            ".js",
            ".sql",
            ".log",
            ".yaml",
            ".yml",
            ".toml",
            ".ini",
        }

        # Size limits (approximate - OpenAI has file size limits)
        max_file_size = 100 * 1024 * 1024  # 100MB

        for file_path in files:
            try:
                if not os.path.exists(file_path):
                    errors.append(f"File not found: {file_path}")
                    continue

                # Check file size
                file_size = os.path.getsize(file_path)
                if file_size > max_file_size:
                    errors.append(
                        f"File too large: {file_path} ({file_size / 1024 / 1024:.1f}MB > 100MB)"
                    )

                # Check file extension
                file_ext = Path(file_path).suffix.lower()
                if file_ext not in supported_extensions:
                    logger.warning(
                        f"File extension {file_ext} may not be supported by Code Interpreter: {file_path}"
                    )

            except Exception as e:
                errors.append(f"Error validating file {file_path}: {e}")

        return errors

    def get_container_limits_info(self) -> Dict[str, Any]:
        """Get information about Code Interpreter container limits.

        Returns:
            Dictionary with container limit information
        """
        return {
            "max_runtime_minutes": 20,
            "idle_timeout_minutes": 2,
            "max_file_size_mb": 100,
            "supported_languages": ["python"],
            "note": "Container expires after 20 minutes of runtime or 2 minutes of inactivity",
        }
