"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlarmRule = exports.AlarmState = void 0;
/**
 * Enumeration indicates state of Alarm used in building Alarm Rule.
 */
var AlarmState;
(function (AlarmState) {
    /**
     * State indicates resource is in ALARM
     */
    AlarmState["ALARM"] = "ALARM";
    /**
     * State indicates resource is not in ALARM
     */
    AlarmState["OK"] = "OK";
    /**
     * State indicates there is not enough data to determine is resource is in ALARM
     */
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
/**
 * Enumeration of supported Composite Alarms operators.
 */
var Operator;
(function (Operator) {
    Operator["AND"] = "AND";
    Operator["OR"] = "OR";
    Operator["NOT"] = "NOT";
})(Operator || (Operator = {}));
/**
 * Class with static functions to build AlarmRule for Composite Alarms.
 */
class AlarmRule {
    /**
     * function to join all provided AlarmRules with AND operator.
     *
     * @param operands IAlarmRules to be joined with AND operator.
     */
    static allOf(...operands) {
        return this.concat(Operator.AND, ...operands);
    }
    /**
     * function to join all provided AlarmRules with OR operator.
     *
     * @param operands IAlarmRules to be joined with OR operator.
     */
    static anyOf(...operands) {
        return this.concat(Operator.OR, ...operands);
    }
    /**
     * function to wrap provided AlarmRule in NOT operator.
     *
     * @param operand IAlarmRule to be wrapped in NOT operator.
     */
    static not(operand) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                return `(NOT (${operand.renderAlarmRule()}))`;
            }
        };
    }
    /**
     * function to build TRUE/FALSE intent for Rule Expression.
     *
     * @param value boolean value to be used in rule expression.
     */
    static fromBoolean(value) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                return `${String(value).toUpperCase()}`;
            }
        };
    }
    /**
     * function to build Rule Expression for given IAlarm and AlarmState.
     *
     * @param alarm IAlarm to be used in Rule Expression.
     * @param alarmState AlarmState to be used in Rule Expression.
     */
    static fromAlarm(alarm, alarmState) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                return `${alarmState}(${alarm.alarmArn})`;
            }
        };
    }
    /**
     * function to build Rule Expression for given Alarm Rule string.
     *
     * @param alarmRule string to be used in Rule Expression.
     */
    static fromString(alarmRule) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                return alarmRule;
            }
        };
    }
    static concat(operator, ...operands) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                const expression = operands
                    .map(operand => `${operand.renderAlarmRule()}`)
                    .join(` ${operator} `);
                return `(${expression})`;
            }
        };
    }
}
exports.AlarmRule = AlarmRule;
//# sourceMappingURL=data:application/json;base64,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