# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['perceiver',
 'perceiver.data',
 'perceiver.data.image',
 'perceiver.data.text',
 'perceiver.model',
 'perceiver.model.core',
 'perceiver.model.image',
 'perceiver.model.text',
 'perceiver.scripts',
 'perceiver.scripts.image',
 'perceiver.scripts.text',
 'perceiver.scripts.utils']

package_data = \
{'': ['*']}

install_requires = \
['einops>=0.4.0,<0.5.0',
 'fairscale>=0.4.0,<0.5.0',
 'fsspec[s3]==2022.5.0',
 'jsonargparse[signatures]>=4.7.0,<4.8.0',
 'pytorch-lightning>=1.6.0,<1.7.0',
 'torch-optimizer>=0.3.0,<0.4.0',
 'torch==1.11.0',
 'torchmetrics>=0.8.0,<0.9.0']

extras_require = \
{'image': ['datasets>=2.3.0,<2.4.0', 'torchvision==0.12.0'],
 'text': ['datasets>=2.3.0,<2.4.0',
          'tokenizers>=0.12.0,<0.13.0',
          'transformers>=4.19.0,<4.20.0']}

setup_kwargs = {
    'name': 'perceiver-io',
    'version': '0.5.0',
    'description': 'Perceiver IO',
    'long_description': '# Perceiver IO\n\nThis library is a PyTorch and PyTorch Lightning implementation of\n\n- [Perceiver IO: A General Architecture for Structured Inputs & Outputs](https://arxiv.org/abs/2107.14795) and\n- [Perceiver: General Perception with Iterative Attention](https://arxiv.org/abs/2103.03206)\n\nAn introduction to the model interfaces provided by the library is given in [Interfaces](docs/interfaces.md). Further\nimplementation details are described in [Architecture](docs/architecture.md). The codebase was designed for easy\nextension to new tasks and datasets. The integration with [PyTorch Lightning](https://www.pytorchlightning.ai/)\nsupports model training at any scale. The command line interface is implemented with the [Lightning CLI](https://pytorch-lightning.readthedocs.io/en/1.6.5/common/lightning_cli.html).\n\nDatasets used for model training are 🤗 [Datasets](https://huggingface.co/docs/datasets) wrapped into PyTorch Lightning\ndata modules (see [data](perceiver/data) package). Datasets are automatically downloaded, preprocessed and cached\nwhen their corresponding Lightning data module is loaded during training. For larger datasets, however, it is\nrecommended to do this prior to training as described [here](docs/datasets.md).\n\nFor NLP tasks, this library also supports 🤗 [fast tokenizers](https://huggingface.co/docs/transformers/fast_tokenizers)\n(see [this list](https://huggingface.co/docs/transformers/index#supported-frameworks) for an overview which 🤗 tokenizers\nare fast tokenizers). It additionally provides special support for the 🤗 UTF-8 bytes Perceiver tokenizer so that it can\nbe used here for masked language modeling with whole word masking.\n\n## Overview\n\n- [Installation](#installation)\n- [Pretrained models](#pretrained-models)\n- [Training examples](#training-examples)\n- [Inference examples](notebooks/inference_examples.ipynb) [![Open In Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/krasserm/perceiver-io/blob/main/notebooks/inference_examples.ipynb)\n- [Architecture](docs/architecture.md)\n- [Interfaces](docs/interfaces.md)\n- [Docker](docs/docker.md)\n\n## Installation\n\n### Via pip\n\n```shell\npip install perceiver-io[image,text]\n```\n\n### From sources\n\nInstallation from sources requires a [Miniconda](https://docs.conda.io/en/latest/miniconda.html) and a\n[Poetry](https://python-poetry.org/docs/master/#installation) (1.2.0b2 or higher) installation.\n\n```shell\nconda env create -f environment.yml\nconda activate perceiver-io\npoetry install --all-extras\n```\n\n## Pretrained models\n\nParameters of pretrained models can be imported from the 🤗 [Hub](https://huggingface.co/models) as described in the\nfollowing subsections.\n\n### Language model\n\nPerceiver IO language model (UTF-8 bytes tokenization, vocabulary size of 262, 201M parameters) specified in Section 4\n(Table 1) and Appendix F (Table 11) of the [Perceiver IO paper](https://arxiv.org/abs/2107.14795). See [Interfaces](docs/interfaces.md)\nfor further details.\n\n```python\nfrom transformers import AutoConfig\nfrom perceiver.model.text.language import convert_config, LanguageModel, LitLanguageModel\n\n# Import and convert language model configuration from Huggingface Hub  \nconfig = convert_config(AutoConfig.from_pretrained("deepmind/language-perceiver"))\n\n# Construct a PyTorch model and load pretrained parameters\nmodel = LanguageModel(config)\n\n# Alternatively, construct a PyTorch Lightning module and load pretrained parameters  \nlit_model = LitLanguageModel.create(config)\n```\n\nOn the command line, the pretrained model can be loaded with the `--model.params=deepmind/language-perceiver` option.\n\n```shell\npython -m perceiver.scripts.text.lm fit \\\n  --model.params=deepmind/language-perceiver \\\n  ...\n```\n\n### Image classifier\n\nThe Perceiver IO image classifier (config A, 2D Fourier features, 48.8M parameters) specified in Appendix A of the\n[Perceiver IO paper](https://arxiv.org/abs/2107.14795).\n\n```python\nfrom transformers import AutoConfig\nfrom perceiver.model.image.classifier import convert_config, ImageClassifier, LitImageClassifier\n\n# Import and convert language model configuration from Huggingface Hub  \nconfig = convert_config(AutoConfig.from_pretrained("deepmind/vision-perceiver-fourier"))\n\n# Construct a PyTorch model and load pretrained parameters\nmodel = ImageClassifier(config)\n\n# Alternatively, construct a PyTorch Lightning module and load pretrained parameters  \nlit_model = LitImageClassifier.create(config)\n```\n\nOn the command line, the pretrained model can be loaded with the `--model.params=deepmind/vision-perceiver-fourier`\noption.\n\n```shell\npython -m perceiver.scripts.image.classifier fit \\\n  --model.params=deepmind/vision-perceiver-fourier \\\n  ...\n```\n\n## Training examples\n\nHere are some command line examples how to train Perceiver IO models with this library. If a model must be initialized\nwith parameters from a previous run, it references a checkpoint from that run with the `--model.params` option. You can\ndownload these checkpoints [here](https://martin-krasser.com/perceiver/logs-update-6.zip) (2.3 GB) or create your own\ncheckpoints by running the examples yourself. Training results are used in [Inference examples](notebooks/inference_examples.ipynb)\n[![Open In Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/krasserm/perceiver-io/blob/main/notebooks/inference_examples.ipynb)\n\nThese examples were tested on a machine with 4x RTX 3080ti GPUs (12 GB memory each). You\'ll need to adjust some\nsettings (batch size, ...) for running them on a different hardware configuration. Furthermore, I didn\'t really\ntune these examples, so you\'ll likely get better results with a bit of experimentation.\n\n### Dataset preprocessing\n\nAlthough data modules automatically download and preprocess datasets if needed, it is recommended to preprocess at\nleast IMDb and WikiText prior to training:\n\n```shell\npython -m perceiver.scripts.text.preproc imdb \\\n  --tokenizer=deepmind/language-perceiver \\\n  --max_seq_len=2048 \\\n  --add_special_tokens=true\n\npython -m perceiver.scripts.text.preproc wikitext \\\n  --tokenizer=bert-base-uncased \\\n  --max_seq_len=128 \\\n  --add_special_tokens=true \\\n  --filter_empty=true \\\n  --filter_headers=true\n```\n\n### Language model fine-tuning\n\nFine-tune a pretrained `deepmind/language-perceiver` model with masked language modeling and whole word masking on\nthe IMDb dataset (*unsupervised* split). It prepares the language model for a better performance on IMDb [sentiment\nclassification](#sentiment-classification). The tokenizer is a UTF-8 bytes tokenizer and the model attends to the\nraw bytes of the input. Word masking is done dynamically at data loading time i.e. each epoch has a different set\nof words masked.\n\n```shell\npython -m perceiver.scripts.text.lm fit \\\n  --model.params=deepmind/language-perceiver \\\n  --model.activation_checkpointing=true \\\n  --data=ImdbDataModule \\\n  --data.target_task=mlm \\\n  --data.tokenizer=deepmind/language-perceiver \\\n  --data.add_special_tokens=true \\\n  --data.max_seq_len=2048 \\\n  --data.batch_size=32 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=2e-5 \\\n  --optimizer.weight_decay=0.01 \\\n  --lr_scheduler.warmup_steps=1000 \\\n  --trainer.max_steps=5200 \\\n  --trainer.accelerator=gpu \\\n  --trainer.precision=16 \\\n  --trainer.devices=2 \\\n  --trainer.strategy=ddp_sharded \\\n  --trainer.log_every_n_steps=20 \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.name=mlm\n```\n\n### Sentiment classification\n\nTrain a text classification model on the IMDb dataset (*train* split). The encoder of the classifier is the fine-tuned\nlanguage model encoder from the [previous run](#language-model-fine-tuning) (`--model.encoder.params=...`), the decoder\nis a randomly initialized classification decoder (see `TextClassifier` and `LitTextClassifier` in [classifier.py](perceiver/model/text/classifier.py)).\nFirst, only the decoder is trained, the encoder is frozen (`--model.encoder.freeze=true`)\n\n```shell\npython -m perceiver.scripts.text.classifier fit \\\n  --model.encoder.params="logs/mlm/version_0/checkpoints/epoch=009-val_loss=1.174.ckpt" \\\n  --model.encoder.freeze=true \\\n  --model.encoder.dropout=0.0 \\\n  --model.decoder.dropout=0.1 \\\n  --data=ImdbDataModule \\\n  --data.target_task=clf \\\n  --data.tokenizer=deepmind/language-perceiver \\\n  --data.add_special_tokens=true \\\n  --data.max_seq_len=2048 \\\n  --data.batch_size=64 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=1e-3 \\\n  --optimizer.weight_decay=0.01 \\\n  --trainer.accelerator=gpu \\\n  --trainer.precision=16 \\\n  --trainer.devices=4 \\\n  --trainer.max_epochs=12 \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger.name=txt_clf_dec\n```\n\nThen, we unfreeze the encoder, initialize the classifier parameters with a checkpoint from the first classifier training\n(`--model.params=...`) and fine-tune the encoder and decoder together on the IMDb training set for further 4 epochs.\n\n```shell\npython -m perceiver.scripts.text.classifier fit \\\n  --model.params="logs/txt_clf_dec/version_1/checkpoints/epoch=010-val_loss=0.212.ckpt" \\\n  --model.activation_checkpointing=true \\\n  --model.encoder.freeze=false \\\n  --model.encoder.dropout=0.1 \\\n  --model.decoder.dropout=0.1 \\\n  --data=ImdbDataModule \\\n  --data.target_task=clf \\\n  --data.tokenizer=deepmind/language-perceiver \\\n  --data.add_special_tokens=true \\\n  --data.max_seq_len=2048 \\\n  --data.batch_size=16 \\\n  --data.num_workers=3 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=5e-6 \\\n  --optimizer.weight_decay=0.01 \\\n  --trainer.accelerator=gpu \\\n  --trainer.precision=16 \\\n  --trainer.devices=4 \\\n  --trainer.max_epochs=4 \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger.name=txt_clf_all\n```\n\nThe validation accuracy of these two runs can be obtained with\n\n```shell\npython -m perceiver.scripts.text.classifier validate \\\n  --config=logs/txt_clf_dec/version_1/config.yaml \\\n  --model.encoder.params=null \\\n  --ckpt_path="logs/txt_clf_dec/version_1/checkpoints/epoch=010-val_loss=0.212.ckpt"\n```\n\n```\n──────────────────────────────────────────────────\n     Validate metric           DataLoader 0\n──────────────────────────────────────────────────\n         val_acc            0.9162399768829346\n        val_loss            0.21216852962970734\n──────────────────────────────────────────────────\n```\n\nand\n\n```shell\npython -m perceiver.scripts.text.classifier validate \\\n  --config=logs/txt_clf_all/version_0/config.yaml \\\n  --model.params=null \\\n  --ckpt_path="logs/txt_clf_all/version_0/checkpoints/epoch=002-val_loss=0.156.ckpt"\n```\n\n```\n──────────────────────────────────────────────────\n     Validate metric           DataLoader 0\n──────────────────────────────────────────────────\n         val_acc            0.9444400072097778\n        val_loss            0.15595446527004242\n──────────────────────────────────────────────────\n```\n\nWhen training only the classification decoder, the validation accuracy is 91.6%. Fine-tuning encoder and decoder on the\nclassification task further increases validation accuracy to 94.4%.\n\n### Language model pretraining\n\nPretrain a smaller language model (45.2M parameters) with masked language modeling and whole word masking on the\nWikitext-103 dataset. This is a toy example for demonstrating how to use a custom model configuration/architecture\nand another 🤗 tokenizer (`bert-base-uncased`, a SentencePiece tokenizer with a vocabulary of size of 30,522). To\nspeed up training, `--data.max_seq_len=128` and `--model.num_latents=64` is used (a quarter of the default values).\n\n```shell\npython -m perceiver.scripts.text.lm fit \\\n  --model.activation_checkpointing=true \\\n  --model.num_latents=64 \\\n  --model.num_latent_channels=768 \\\n  --model.encoder.num_input_channels=512 \\\n  --model.encoder.num_cross_attention_v_channels=768 \\\n  --model.encoder.num_self_attention_v_channels=768 \\\n  --model.encoder.num_self_attention_layers_per_block=6 \\\n  --model.encoder.cross_attention_widening_factor=2 \\\n  --model.encoder.self_attention_widening_factor=2 \\\n  --model.encoder.dropout=0.0 \\\n  --model.decoder.num_cross_attention_v_channels=512 \\\n  --model.decoder.cross_attention_widening_factor=2 \\\n  --model.decoder.dropout=0.0 \\\n  --data=WikiTextDataModule \\\n  --data.tokenizer=bert-base-uncased \\\n  --data.add_special_tokens=true \\\n  --data.filter_empty=true \\\n  --data.filter_headers=true \\\n  --data.max_seq_len=128 \\\n  --data.batch_size=128 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=1e-4 \\\n  --optimizer.weight_decay=0.01 \\\n  --lr_scheduler.warmup_steps=1000 \\\n  --trainer.max_steps=25000 \\\n  --trainer.accelerator=gpu \\\n  --trainer.precision=16 \\\n  --trainer.devices=4 \\\n  --trainer.strategy=ddp_sharded \\\n  --trainer.accumulate_grad_batches=2 \\\n  --trainer.val_check_interval=0.5 \\\n  --trainer.log_every_n_steps=20 \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.name=mlm_pre\n```\n\n### Image classification\n\nTrain a tiny image classifier (805K parameters) on the MNIST dataset. The model attends to individual pixels of the\ninput image and uses Fourier position encodings. This is another toy example that demonstrates how to use a custom\nmodel configuration compared to the defaults in [classifier.py](perceiver/scripts/image/classifier.py).\n\n```shell\npython -m perceiver.scripts.image.classifier fit \\\n  --model.num_latents=32 \\\n  --model.num_latent_channels=128 \\\n  --model.encoder.num_frequency_bands=32 \\\n  --model.encoder.num_self_attention_layers_per_block=3 \\\n  --model.encoder.num_self_attention_blocks=3 \\\n  --model.encoder.first_self_attention_block_shared=false \\\n  --model.encoder.dropout=0.0 \\\n  --model.encoder.init_scale=0.1 \\\n  --model.decoder.num_output_query_channels=128 \\\n  --model.decoder.dropout=0.0 \\\n  --model.decoder.init_scale=0.1 \\\n  --data=MNISTDataModule \\\n  --data.batch_size=128 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=1e-3 \\\n  --optimizer.weight_decay=0.01 \\\n  --trainer.accelerator=gpu \\\n  --trainer.devices=2 \\\n  --trainer.max_epochs=20 \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger.name=img_clf\n```\n\nThe validation accuracy is 98.1%:\n\n```shell\npython -m perceiver.scripts.image.classifier validate \\\n  --config=logs/img_clf/version_0/config.yaml \\\n  --ckpt_path="logs/img_clf/version_0/checkpoints/epoch=015-val_loss=0.068.ckpt"\n```\n\n```\n──────────────────────────────────────────────────\n     Validate metric           DataLoader 0\n──────────────────────────────────────────────────\n         val_acc            0.9807000160217285\n        val_loss            0.06775263696908951\n──────────────────────────────────────────────────\n```\n',
    'author': 'Martin Krasser',
    'author_email': 'krasserm@googlemail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/krasserm/perceiver-io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
