"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryNotificationEvents = exports.RepositoryEventTrigger = exports.Repository = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const notifications = require("@aws-cdk/aws-codestarnotifications");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const codecommit_generated_1 = require("./codecommit.generated");
/**
 * Represents a reference to a CodeCommit Repository.
 *
 * If you want to create a new Repository managed alongside your CDK code,
 * use the {@link Repository} class.
 *
 * If you want to reference an already existing Repository,
 * use the {@link Repository.import} method.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.codecommit'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit
     * Repository State Change" event occurs.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeCommit Repository State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * created (i.e. a new branch/tag is created) to the repository.
     */
    onReferenceCreated(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceCreated'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     */
    onReferenceUpdated(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceCreated', 'referenceUpdated'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * delete (i.e. a branch/tag is deleted) from the repository.
     */
    onReferenceDeleted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceDeleted'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     */
    onPullRequestStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Pull Request State Change'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     */
    onCommentOnPullRequest(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Comment on Pull Request'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     */
    onCommentOnCommit(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Comment on Commit'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     */
    onCommit(id, options = {}) {
        const rule = this.onReferenceUpdated(id, options);
        if (options.branches) {
            rule.addEventPattern({ detail: { referenceName: options.branches } });
        }
        return rule;
    }
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
        });
    }
    grantPull(grantee) {
        return this.grant(grantee, 'codecommit:GitPull');
    }
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'codecommit:GitPush');
    }
    grantRead(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'codecommit:EvaluatePullRequestApprovalRules', 'codecommit:Get*', 'codecommit:Describe*');
    }
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    notifyOnPullRequestComment(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.PULL_REQUEST_COMMENT],
        });
    }
    notifyOnApprovalStatusChanged(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.APPROVAL_STATUS_CHANGED],
        });
    }
    notifyOnApprovalRuleOverridden(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.APPROVAL_RULE_OVERRIDDEN],
        });
    }
    notifyOnPullRequestCreated(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.PULL_REQUEST_CREATED],
        });
    }
    notifiyOnPullRequestMerged(id, target, options) {
        return this.notifyOnPullRequestMerged(id, target, options);
    }
    notifyOnPullRequestMerged(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.PULL_REQUEST_MERGED],
        });
    }
    notifyOnBranchOrTagCreated(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.BRANCH_OR_TAG_CREATED],
        });
    }
    notifyOnBranchOrTagDeleted(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [RepositoryNotificationEvents.BRANCH_OR_TAG_DELETED],
        });
    }
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.repositoryArn,
        };
    }
}
/**
 * Provides a CodeCommit Repository.
 */
class Repository extends RepositoryBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.triggers = new Array();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codecommit_RepositoryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const repository = new codecommit_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: props.repositoryName,
            repositoryDescription: props.description,
            triggers: core_1.Lazy.any({ produce: () => this.triggers }, { omitEmptyArray: true }),
            code: (_c = ((_b = props.code) === null || _b === void 0 ? void 0 : _b.bind(this))) === null || _c === void 0 ? void 0 : _c.code,
        });
        this.repositoryName = this.getResourceNameAttribute(repository.attrName);
        this.repositoryArn = this.getResourceArnAttribute(repository.attrArn, {
            service: 'codecommit',
            resource: this.physicalName,
        });
        this.repositoryCloneUrlHttp = repository.attrCloneUrlHttp;
        this.repositoryCloneUrlSsh = repository.attrCloneUrlSsh;
        this.repositoryCloneUrlGrc = makeCloneUrl(core_1.Stack.of(this), this.repositoryName, 'grc');
    }
    /**
     * Imports a codecommit repository.
     * @param repositoryArn (e.g. `arn:aws:codecommit:us-east-1:123456789012:MyDemoRepo`)
     */
    static fromRepositoryArn(scope, id, repositoryArn) {
        const stack = core_1.Stack.of(scope);
        const arn = stack.splitArn(repositoryArn, core_1.ArnFormat.NO_RESOURCE_NAME);
        const repositoryName = arn.resource;
        const region = arn.region;
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryArn = repositoryArn;
                this.repositoryName = repositoryName;
                this.repositoryCloneUrlHttp = makeCloneUrl(stack, repositoryName, 'https', region);
                this.repositoryCloneUrlSsh = makeCloneUrl(stack, repositoryName, 'ssh', region);
                this.repositoryCloneUrlGrc = makeCloneUrl(stack, repositoryName, 'grc', region);
            }
        }
        return new Import(scope, id, {
            account: arn.account,
            region,
        });
    }
    static fromRepositoryName(scope, id, repositoryName) {
        const stack = core_1.Stack.of(scope);
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = core_1.Stack.of(scope).formatArn({
                    service: 'codecommit',
                    resource: repositoryName,
                });
                this.repositoryCloneUrlHttp = makeCloneUrl(stack, repositoryName, 'https');
                this.repositoryCloneUrlSsh = makeCloneUrl(stack, repositoryName, 'ssh');
                this.repositoryCloneUrlGrc = makeCloneUrl(stack, repositoryName, 'grc');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Create a trigger to notify another service to run actions on repository events.
     * @param arn   Arn of the resource that repository events will notify
     * @param options Trigger options to run actions
     */
    notify(arn, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codecommit_RepositoryTriggerOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.notify);
            }
            throw error;
        }
        let evt = options && options.events;
        if (evt && evt.length > 1 && evt.indexOf(RepositoryEventTrigger.ALL) > -1) {
            evt = [RepositoryEventTrigger.ALL];
        }
        const customData = options && options.customData;
        const branches = options && options.branches;
        let name = options && options.name;
        if (!name) {
            name = this.node.path + '/' + arn;
        }
        if (this.triggers.find(prop => prop.name === name)) {
            throw new Error(`Unable to set repository trigger named ${name} because trigger names must be unique`);
        }
        this.triggers.push({
            destinationArn: arn,
            name,
            customData,
            branches,
            events: evt || [RepositoryEventTrigger.ALL],
        });
        return this;
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "@aws-cdk/aws-codecommit.Repository", version: "1.153.0" };
/**
 * Repository events that will cause the trigger to run actions in another service.
 */
var RepositoryEventTrigger;
(function (RepositoryEventTrigger) {
    RepositoryEventTrigger["ALL"] = "all";
    RepositoryEventTrigger["UPDATE_REF"] = "updateReference";
    RepositoryEventTrigger["CREATE_REF"] = "createReference";
    RepositoryEventTrigger["DELETE_REF"] = "deleteReference";
})(RepositoryEventTrigger = exports.RepositoryEventTrigger || (exports.RepositoryEventTrigger = {}));
/**
 * Returns the clone URL for a protocol.
 */
function makeCloneUrl(stack, repositoryName, protocol, region) {
    switch (protocol) {
        case 'https':
        case 'ssh':
            return `${protocol}://git-codecommit.${region !== null && region !== void 0 ? region : stack.region}.${stack.urlSuffix}/v1/repos/${repositoryName}`;
        case 'grc':
            return `codecommit::${region !== null && region !== void 0 ? region : stack.region}://${repositoryName}`;
    }
}
/**
 * List of event types for AWS CodeCommit
 * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-repositories
 */
var RepositoryNotificationEvents;
(function (RepositoryNotificationEvents) {
    /**
     * Trigger notication when comment made on commit.
     */
    RepositoryNotificationEvents["COMMIT_COMMENT"] = "codecommit-repository-comments-on-commits";
    /**
     * Trigger notification when comment made on pull request.
     */
    RepositoryNotificationEvents["PULL_REQUEST_COMMENT"] = "codecommit-repository-comments-on-pull-requests";
    /**
     * Trigger notification when approval status changed.
     */
    RepositoryNotificationEvents["APPROVAL_STATUS_CHANGED"] = "codecommit-repository-approvals-status-changed";
    /**
     * Trigger notifications when approval rule is overridden.
     */
    RepositoryNotificationEvents["APPROVAL_RULE_OVERRIDDEN"] = "codecommit-repository-approvals-rule-override";
    /**
     * Trigger notification when pull request created.
     */
    RepositoryNotificationEvents["PULL_REQUEST_CREATED"] = "codecommit-repository-pull-request-created";
    /**
     * Trigger notification when pull request source updated.
     */
    RepositoryNotificationEvents["PULL_REQUEST_SOURCE_UPDATED"] = "codecommit-repository-pull-request-source-updated";
    /**
     * Trigger notification when pull request status is changed.
     */
    RepositoryNotificationEvents["PULL_REQUEST_STATUS_CHANGED"] = "codecommit-repository-pull-request-status-changed";
    /**
     * Trigger notification when pull requset is merged.
     */
    RepositoryNotificationEvents["PULL_REQUEST_MERGED"] = "codecommit-repository-pull-request-merged";
    /**
     * Trigger notification when a branch or tag is created.
     */
    RepositoryNotificationEvents["BRANCH_OR_TAG_CREATED"] = "codecommit-repository-branches-and-tags-created";
    /**
     * Trigger notification when a branch or tag is deleted.
     */
    RepositoryNotificationEvents["BRANCH_OR_TAG_DELETED"] = "codecommit-repository-branches-and-tags-deleted";
    /**
     * Trigger notification when a branch or tag is updated.
     */
    RepositoryNotificationEvents["BRANCH_OR_TAG_UPDATED"] = "codecommit-repository-branches-and-tags-updated";
})(RepositoryNotificationEvents = exports.RepositoryNotificationEvents || (exports.RepositoryNotificationEvents = {}));
//# sourceMappingURL=data:application/json;base64,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