import os
import shutil
import pytest

from mirdata import jams_utils, download_utils, core
from mirdata.datasets import acousticbrainz_genre
from tests.test_utils import run_track_tests


def test_track(httpserver):
    default_trackid = "tagtraum#validation#be9e01e5-8f93-494d-bbaa-ddcc5a52f629#2b6bfcfd-46a5-3f98-a58f-2c51d7c9e960#trance########"
    data_home = "tests/resources/mir_datasets/acousticbrainz_genre"

    httpserver.serve_content(
        open(
            "tests/resources/download/acousticbrainz_genre_dataset_little_test.json.zip",
            "rb",
        ).read()
    )
    remote_index = {
        "index": download_utils.RemoteFileMetadata(
            filename="acousticbrainz_genre_dataset_little_test.json.zip",
            url=httpserver.url,
            checksum="c5fbdd4f8b7de383796a34143cb44c4f",
            destination_dir="",
        )
    }
    track = acousticbrainz_genre.Track(
        default_trackid,
        data_home=data_home,
        remote_index=remote_index,
        remote_index_name="acousticbrainz_genre_dataset_little_test.json",
    )

    expected_attributes = {
        "path": "tests/resources/mir_datasets/acousticbrainz_genre/acousticbrainz-mediaeval-validation/be/be9e01e5-8f93-494d-bbaa-ddcc5a52f629.json",
        "track_id": "tagtraum#validation#be9e01e5-8f93-494d-bbaa-ddcc5a52f629#2b6bfcfd-46a5-3f98-a58f-2c51d7c9e960#trance########",
    }

    expected_property_types = {
        "genre": list,
        "mbid": str,
        "mbid_group": str,
        "artist": str,
        "title": str,
        "date": str,
        "file_name": str,
        "album": str,
        "tracknumber": str,
        "tonal": dict,
        "low_level": dict,
        "rhythm": dict,
    }

    run_track_tests(track, expected_attributes, expected_property_types)
    os.remove(
        os.path.join(
            "mirdata",
            "datasets/indexes",
            "acousticbrainz_genre_dataset_little_test.json",
        )
    )


features = {
    "tonal": {
        "thpcp": [
            1,
            0.579320728779,
            0.214836657047,
            0.410239934921,
            0.421927720308,
            0.225528225303,
            0.183076187968,
            0.202793732285,
            0.191496774554,
            0.195926904678,
            0.19538384676,
            0.17725071311,
            0.19343534112,
            0.195846363902,
            0.193740859628,
            0.209312275052,
            0.166110798717,
            0.12996301055,
            0.183770611882,
            0.209641098976,
            0.162735670805,
            0.137884125113,
            0.142696648836,
            0.222093731165,
            0.687183618546,
            0.865788459778,
            0.439368784428,
            0.291335314512,
            0.285592496395,
            0.168208643794,
            0.112465105951,
            0.0973277166486,
            0.106767326593,
            0.139660894871,
            0.141137599945,
            0.563603103161,
        ],
        "hpcp": {
            "dmean2": [
                0.261957257986,
                0.248936057091,
                0.227226093411,
                0.214822933078,
                0.185719549656,
                0.166620016098,
                0.19218352437,
                0.213925108314,
                0.201654553413,
                0.195705741644,
                0.195796221495,
                0.232935741544,
                0.340740889311,
                0.380624711514,
                0.310503959656,
                0.275244802237,
                0.25047698617,
                0.184646636248,
                0.150202214718,
                0.127092003822,
                0.123402029276,
                0.145707964897,
                0.156906038523,
                0.343307852745,
                0.575405955315,
                0.357600450516,
                0.212107673287,
                0.278686583042,
                0.2768920362,
                0.204572796822,
                0.232681691647,
                0.274391710758,
                0.267259836197,
                0.257940381765,
                0.254153877497,
                0.250754475594,
            ],
            "median": [
                0.00702382624149,
                0.0104925427586,
                0.0265087448061,
                0.0359445922077,
                0.025654386729,
                0.00958184618503,
                0.0113075301051,
                0.013004174456,
                0.0103413462639,
                0.00578989461064,
                0.00591916590929,
                0.0359903424978,
                0.216057181358,
                0.279158771038,
                0.127309978008,
                0.0242714583874,
                0.0206926688552,
                0.01287034899,
                0.00521051790565,
                0.00470173824579,
                0.00891952775419,
                0.0112975630909,
                0.00772325787693,
                0.171776384115,
                0.294544279575,
                0.182794481516,
                0.0341288149357,
                0.0937796682119,
                0.0891414806247,
                0.0357652790844,
                0.00981951318681,
                0.00645823031664,
                0.00640677567571,
                0.0101658720523,
                0.0109609831125,
                0.00720301363617,
            ],
            "min": [
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
            ],
            "dvar2": [
                0.20182004571,
                0.181256204844,
                0.160531163216,
                0.138031870127,
                0.116254262626,
                0.105536788702,
                0.123104974627,
                0.135154604912,
                0.136645048857,
                0.14191840589,
                0.134744182229,
                0.113522216678,
                0.123939432204,
                0.141078904271,
                0.126689687371,
                0.146191120148,
                0.134346649051,
                0.113142922521,
                0.111554063857,
                0.0928051173687,
                0.0830625072122,
                0.0921071469784,
                0.102565199137,
                0.122344501317,
                0.268501013517,
                0.134693875909,
                0.115629725158,
                0.120229043067,
                0.124573647976,
                0.0989440754056,
                0.149035617709,
                0.198961064219,
                0.195493474603,
                0.181768119335,
                0.186205849051,
                0.188829809427,
            ],
            "dvar": [
                0.0709016770124,
                0.0642212927341,
                0.0540215522051,
                0.0459277741611,
                0.039158269763,
                0.0358573682606,
                0.0441283173859,
                0.0500697679818,
                0.048032399267,
                0.049877922982,
                0.0475668683648,
                0.0409390516579,
                0.0483189336956,
                0.059373728931,
                0.046393956989,
                0.057118140161,
                0.0552255362272,
                0.040049944073,
                0.0385572277009,
                0.0317769236863,
                0.02851219289,
                0.0318522453308,
                0.0354646109045,
                0.047548353672,
                0.122033506632,
                0.0509799085557,
                0.0392313636839,
                0.0442327298224,
                0.0466478951275,
                0.034126713872,
                0.052664835006,
                0.0715441480279,
                0.0702358782291,
                0.0655381903052,
                0.0668850839138,
                0.0668021589518,
            ],
            "dmean": [
                0.137185156345,
                0.131976485252,
                0.119955100119,
                0.114594981074,
                0.0980019420385,
                0.0876002237201,
                0.105321630836,
                0.117977328598,
                0.106846518815,
                0.101396635175,
                0.102731078863,
                0.125963360071,
                0.199974015355,
                0.221728652716,
                0.173448696733,
                0.153004571795,
                0.140776693821,
                0.0996030718088,
                0.0784875378013,
                0.0666015073657,
                0.066075257957,
                0.0793278291821,
                0.0844658911228,
                0.192652150989,
                0.319886952639,
                0.201404705644,
                0.116125285625,
                0.160826355219,
                0.159044191241,
                0.11338737607,
                0.123394109309,
                0.144251897931,
                0.139449134469,
                0.135212510824,
                0.133663699031,
                0.130060389638,
            ],
            "max": [
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
                1,
            ],
            "var": [
                0.0449322424829,
                0.0427629947662,
                0.0326486118138,
                0.0284581966698,
                0.023814085871,
                0.0215451624244,
                0.0332439541817,
                0.0404991842806,
                0.0301142297685,
                0.0290783978999,
                0.0284893140197,
                0.0286722127348,
                0.0851084291935,
                0.124972507358,
                0.0465349033475,
                0.0546792373061,
                0.059368070215,
                0.0267473664135,
                0.0226037502289,
                0.0189583078027,
                0.0179162640125,
                0.0223223939538,
                0.0250757019967,
                0.0585578717291,
                0.175973117352,
                0.0614937394857,
                0.0289196409285,
                0.0518814213574,
                0.0589718073606,
                0.0273308288306,
                0.0349330119789,
                0.0468821898103,
                0.043950650841,
                0.0417246446013,
                0.0426235720515,
                0.0404398441315,
            ],
            "mean": [
                0.0862462967634,
                0.0873212888837,
                0.0863825157285,
                0.0933252871037,
                0.0740632042289,
                0.0579461269081,
                0.0819371193647,
                0.0934718996286,
                0.0725583508611,
                0.0614778809249,
                0.0636236220598,
                0.0990241095424,
                0.306392014027,
                0.386025875807,
                0.19589972496,
                0.129896596074,
                0.127336069942,
                0.0749985650182,
                0.0501444004476,
                0.0433951467276,
                0.0476039499044,
                0.0622700825334,
                0.0629284977913,
                0.251291632652,
                0.445866286755,
                0.258299589157,
                0.0957884192467,
                0.182912155986,
                0.188123345375,
                0.100555434823,
                0.0816275030375,
                0.0904188901186,
                0.0853819549084,
                0.0873572006822,
                0.0871150717139,
                0.0790301188827,
            ],
        },
        "tuning_equal_tempered_deviation": 0.165096893907,
        "chords_changes_rate": 0.0786039158702,
        "key_key": "F#",
        "tuning_diatonic_strength": 0.534535348415,
        "key_scale": "minor",
        "chords_number_rate": 0.000889855669811,
        "tuning_frequency": 434.193115234,
        "tuning_nontempered_energy_ratio": 0.88192075491,
        "chords_histogram": [
            24.7083244324,
            1.89835870266,
            2.62013053894,
            0.108760133386,
            0,
            0,
            0,
            10.2135658264,
            8.15701007843,
            36.1281394958,
            0,
            0,
            0,
            0,
            0,
            0.701997220516,
            5.45778131485,
            0,
            0,
            0,
            0,
            0.425153255463,
            7.3067035675,
            2.27407550812,
        ],
        "key_strength": 0.546354293823,
        "chords_key": "C#",
        "chords_strength": {
            "dmean2": 0.0105424607173,
            "median": 0.444946020842,
            "min": -1,
            "dvar2": 0.000453131913673,
            "dvar": 0.000237715838011,
            "dmean": 0.00846278760582,
            "max": 0.725530564785,
            "var": 0.0112007251009,
            "mean": 0.450856178999,
        },
        "chords_scale": "major",
        "hpcp_entropy": {
            "dmean2": 1.08373880386,
            "median": 2.00362324715,
            "min": 0,
            "dvar2": 0.735475599766,
            "dvar": 0.2642352283,
            "dmean": 0.642614841461,
            "max": 4.56000328064,
            "var": 0.522367954254,
            "mean": 2.03967022896,
        },
    },
    "metadata": {
        "audio_properties": {
            "equal_loudness": 0,
            "codec": "mp3",
            "downmix": "mix",
            "sample_rate": 44100,
            "analysis_sample_rate": 44100,
            "bit_rate": 320000,
            "md5_encoded": "2bf9caa8bea8a46924db3280f8e8dab9",
            "length": 469.629394531,
            "replay_gain": -12.2262840271,
            "lossless": False,
        },
        "version": {
            "essentia_build_sha": "cead25079874084f62182a551b7393616cd33d87",
            "essentia": "2.1-beta2",
            "extractor": "music 1.0",
            "essentia_git_sha": "v2.1_beta2-1-ge3940c0",
        },
        "tags": {
            "title": ["Still Dreaming (Anything Can Happen)"],
            "barcode": ["7296134204121"],
            "media": ["CD"],
            "albumartist": ["Astral Projection"],
            "musicbrainz_recordingid": ["be9e01e5-8f93-494d-bbaa-ddcc5a52f629"],
            "musicbrainz_workid": ["7295f0a8-e75d-36a0-bbeb-1dba0eea9916"],
            "composer": ["Lior Perlmutter/Avi Nissim"],
            "musicbrainz_albumartistid": ["2ba31cc4-df9e-4c21-9dbc-bb8fa9af424f"],
            "discnumber": ["1/1"],
            "album": ["Trust in Trance"],
            "catalognumber": ["2041-2"],
            "musicbrainz album release country": ["IL"],
            "musicbrainz album type": ["album"],
            "musicbrainz album status": ["official"],
            "work": ["Still Dreaming (Anything Can Happen)"],
            "date": ["1996"],
            "file_name": "09 Still Dreaming (Anything Can Happen).mp3",
            "musicbrainz_releasegroupid": ["2b6bfcfd-46a5-3f98-a58f-2c51d7c9e960"],
            "script": ["Latn"],
            "originaldate": ["1996"],
            "musicbrainz_albumid": ["07118f6d-9643-4315-bd84-b248bd68c610"],
            "artist": ["Astral Projection"],
            "artistsort": ["Astral Projection"],
            "tracknumber": ["9/9"],
            "label": ["Phokol"],
            "albumartistsort": ["Astral Projection"],
            "musicbrainz_artistid": ["2ba31cc4-df9e-4c21-9dbc-bb8fa9af424f"],
        },
    },
    "rhythm": {
        "bpm_histogram_first_peak_bpm": {
            "dmean2": 0,
            "median": 99,
            "min": 99,
            "dvar2": 0,
            "dvar": 0,
            "dmean": 0,
            "max": 99,
            "var": 0,
            "mean": 99,
        },
        "bpm": 99.8474197388,
        "bpm_histogram_second_peak_spread": {
            "dmean2": 0,
            "median": 0.285714268684,
            "min": 0.285714268684,
            "dvar2": 0,
            "dvar": 0,
            "dmean": 0,
            "max": 0.285714268684,
            "var": 0,
            "mean": 0.285714268684,
        },
        "bpm_histogram_first_peak_weight": {
            "dmean2": 0,
            "median": 0.716302931309,
            "min": 0.716302931309,
            "dvar2": 0,
            "dvar": 0,
            "dmean": 0,
            "max": 0.716302931309,
            "var": 0,
            "mean": 0.716302931309,
        },
        "onset_rate": 5.58291912079,
        "beats_position": [
            0.557278871536,
            1.13777780533,
            1.74149656296,
            2.34521532059,
            2.93732428551,
            3.52943301201,
            4.13315200806,
            4.73687076569,
            5.34058952332,
            5.94430828094,
            6.54802703857,
            7.1517457962,
            7.74385452271,
            8.33596324921,
            8.93968200684,
            9.54340076447,
            10.1471195221,
            10.7508392334,
            11.3429479599,
            11.9466667175,
            12.538775444,
            13.1424942017,
            13.7462129593,
            14.3499317169,
            14.9536504745,
            15.5573692322,
            16.1610889435,
            16.7648067474,
            17.3569164276,
            17.9490242004,
            18.5527439117,
            19.1564617157,
            19.760181427,
            20.363899231,
            20.9676189423,
            21.5713367462,
            22.1634464264,
            22.7555541992,
            23.3592739105,
            23.9629936218,
            24.5667114258,
            25.1704311371,
            25.774148941,
            26.3778686523,
            26.9699764252,
            27.5620861053,
            28.1658039093,
            28.7695236206,
            29.3732414246,
            29.9769611359,
            30.5690689087,
            31.17278862,
            31.7648983002,
            32.3686180115,
            32.9723358154,
            33.5760536194,
            34.1797714233,
            34.783493042,
            35.3755989075,
            35.9793205261,
            36.5714263916,
            37.1751480103,
            37.7788658142,
            38.3825836182,
            38.9863014221,
            39.5900230408,
            40.1937408447,
            40.7858505249,
            41.3779602051,
            41.981678009,
            42.585395813,
            43.1891136169,
            43.7928352356,
            44.3849411011,
            44.9770507812,
            45.5807685852,
            46.1844902039,
            46.7882080078,
            47.3919258118,
            47.9956436157,
            48.5877532959,
            49.1798629761,
            49.78358078,
            50.3873023987,
            50.9910202026,
            51.5947380066,
            52.1984558105,
            52.8021774292,
            53.3942832947,
            53.9980049133,
            54.5901107788,
            55.1938323975,
            55.7975502014,
            56.4012680054,
            57.0049858093,
            57.608707428,
            58.2008171082,
            58.7929229736,
            59.3966445923,
            60.0003623962,
            60.6040802002,
            61.2077980042,
            61.8115196228,
            62.4036254883,
            62.9957351685,
            63.5994529724,
            64.2031707764,
            64.806892395,
            65.4106140137,
            66.0143280029,
            66.6180496216,
            67.2101593018,
            67.8022689819,
            68.4059829712,
            69.0097045898,
            69.6134262085,
            70.2171401978,
            70.8208618164,
            71.4129714966,
            72.0166854858,
            72.608795166,
            73.2125167847,
            73.8162307739,
            74.4199523926,
            75.0120620728,
            75.6041717529,
            76.2078933716,
            76.8116073608,
            77.4153289795,
            78.0190429688,
            78.6227645874,
            79.2264862061,
            79.8418121338,
            80.433921814,
            81.0260314941,
            81.6181411743,
            82.2218551636,
            82.8255767822,
            83.4292984009,
            84.0214080811,
            84.6135101318,
            85.2172317505,
            85.8209533691,
            86.4246673584,
            87.0283889771,
            87.6321105957,
            88.235824585,
            88.8279342651,
            89.4200439453,
            90.023765564,
            90.6158676147,
            91.2195892334,
            91.8233108521,
            92.4270248413,
            93.0191345215,
            93.6228561401,
            94.2265777588,
            94.830291748,
            95.4340133667,
            96.037727356,
            96.6414489746,
            97.2451705933,
            97.8488845825,
            98.4409942627,
            99.0331039429,
            99.6368255615,
            100.240539551,
            100.844261169,
            101.447982788,
            102.051696777,
            102.655418396,
            103.247528076,
            103.839637756,
            104.443351746,
            105.047073364,
            105.650794983,
            106.254508972,
            106.858230591,
            107.450340271,
            108.05405426,
            108.64616394,
            109.249885559,
            109.853607178,
            110.457321167,
            111.061042786,
            111.653152466,
            112.245262146,
            112.848976135,
            113.452697754,
            114.056411743,
            114.660133362,
            115.26385498,
            115.855964661,
            116.448066711,
            117.05178833,
            117.655509949,
            118.259223938,
            118.862945557,
            119.466667175,
            120.070381165,
            120.662490845,
            121.254600525,
            121.858322144,
            122.462036133,
            123.065757751,
            123.66947937,
            124.273193359,
            124.876914978,
            125.469024658,
            126.061134338,
            126.664848328,
            127.268569946,
            127.872291565,
            128.476013184,
            129.079727173,
            129.671829224,
            130.263946533,
            130.867660522,
            131.471374512,
            132.07510376,
            132.678817749,
            133.282531738,
            133.886260986,
            134.478363037,
            135.070480347,
            135.674194336,
            136.277908325,
            136.881637573,
            137.485351562,
            138.089065552,
            138.681182861,
            139.284896851,
            139.876998901,
            140.480728149,
            141.084442139,
            141.688156128,
            142.291885376,
            142.895599365,
            143.487701416,
            144.079818726,
            144.683532715,
            145.287246704,
            145.890975952,
            146.494689941,
            147.098403931,
            147.69052124,
            148.282623291,
            148.886352539,
            149.490066528,
            150.093780518,
            150.697509766,
            151.301223755,
            151.893325806,
            152.497055054,
            153.089157104,
            153.692871094,
            154.296600342,
            154.900314331,
            155.50402832,
            156.107757568,
            156.699859619,
            157.291976929,
            157.895690918,
            158.499404907,
            159.103118896,
            159.706848145,
            160.310562134,
            160.914276123,
            161.506393433,
            162.098495483,
            162.702224731,
            163.305938721,
            163.90965271,
            164.513381958,
            165.117095947,
            165.720809937,
            166.312927246,
            166.905029297,
            167.508743286,
            168.112472534,
            168.716186523,
            169.319900513,
            169.923629761,
            170.515731812,
            171.107849121,
            171.71156311,
            172.3152771,
            172.919006348,
            173.522720337,
            174.126434326,
            174.730148315,
            175.322265625,
            175.914367676,
            176.518096924,
            177.121810913,
            177.725524902,
            178.32925415,
            178.93296814,
            179.536682129,
            180.128799438,
            180.720901489,
            181.324615479,
            181.928344727,
            182.532058716,
            183.135772705,
            183.739501953,
            184.331604004,
            184.923721313,
            185.527435303,
            186.131149292,
            186.73487854,
            187.338592529,
            187.942306519,
            188.534423828,
            189.126525879,
            189.730239868,
            190.333969116,
            190.937683105,
            191.541397095,
            192.145126343,
            192.737228394,
            193.329345703,
            193.933059692,
            194.536773682,
            195.140487671,
            195.744216919,
            196.347930908,
            196.951644897,
            197.543762207,
            198.135864258,
            198.739593506,
            199.343307495,
            199.947021484,
            200.550750732,
            201.142852783,
            201.746566772,
            202.338684082,
            202.942398071,
            203.546112061,
            204.149841309,
            204.753555298,
            205.357269287,
            205.949386597,
            206.541488647,
            207.145217896,
            207.748931885,
            208.352645874,
            208.956375122,
            209.560089111,
            210.163803101,
            210.75592041,
            211.348022461,
            211.95173645,
            212.555465698,
            213.159179688,
            213.762893677,
            214.366622925,
            214.970336914,
            215.562438965,
            216.154556274,
            216.758270264,
            217.361984253,
            217.965713501,
            218.56942749,
            219.173141479,
            219.765258789,
            220.35736084,
            220.961090088,
            221.564804077,
            222.168518066,
            222.772247314,
            223.375961304,
            223.979675293,
            224.571792603,
            225.163894653,
            225.767608643,
            226.371337891,
            226.97505188,
            227.578765869,
            228.182495117,
            228.786209106,
            229.378311157,
            229.970428467,
            230.574142456,
            231.177871704,
            231.781585693,
            232.385299683,
            232.989013672,
            233.581130981,
            234.173233032,
            234.77696228,
            235.38067627,
            235.984390259,
            236.588119507,
            237.191833496,
            237.795547485,
            238.387664795,
            238.979766846,
            239.583480835,
            240.187210083,
            240.790924072,
            241.394638062,
            241.99836731,
            242.602081299,
            243.19418335,
            243.786300659,
            244.390014648,
            244.993743896,
            245.597457886,
            246.201171875,
            246.804885864,
            247.397003174,
            247.989105225,
            248.592834473,
            249.196548462,
            249.800262451,
            250.403991699,
            251.007705688,
            251.599807739,
            252.191925049,
            252.795639038,
            253.399353027,
            254.003082275,
            254.606796265,
            255.210510254,
            255.814239502,
            256.406341553,
            256.998443604,
            257.602172852,
            258.2059021,
            258.80960083,
            259.413330078,
            260.017059326,
            260.609161377,
            261.201263428,
            261.804992676,
            262.408691406,
            263.012420654,
            263.616149902,
            264.219848633,
            264.811981201,
            265.404083252,
            266.007781982,
            266.61151123,
            267.192016602,
            267.760894775,
            268.329803467,
            268.898681641,
            269.479187012,
            270.082885742,
            270.675018311,
            271.278717041,
            271.882446289,
            272.486175537,
            273.078277588,
            273.681976318,
            274.274108887,
            274.877807617,
            275.481536865,
            276.096862793,
            276.700592041,
            277.292694092,
            277.884796143,
            278.488525391,
            279.080627441,
            279.684356689,
            280.28805542,
            280.891784668,
            281.483886719,
            282.07598877,
            282.679718018,
            283.271820068,
            283.875549316,
            284.479278564,
            285.082977295,
            285.675109863,
            286.278808594,
            286.882537842,
            287.474639893,
            288.078369141,
            288.682067871,
            289.285797119,
            289.889526367,
            290.481628418,
            291.085357666,
            291.689056396,
            292.292785645,
            292.896514893,
            293.500213623,
            294.103942871,
            294.696044922,
            295.29977417,
            295.891876221,
            296.495605469,
            297.099304199,
            297.703033447,
            298.306762695,
            298.910461426,
            299.502593994,
            300.106292725,
            300.698394775,
            301.302124023,
            301.905853271,
            302.509552002,
            303.10168457,
            303.705383301,
            304.297485352,
            304.9012146,
            305.504943848,
            306.097045898,
            306.700775146,
            307.292877197,
            307.896606445,
            308.500305176,
            309.104034424,
            309.707763672,
            310.311462402,
            310.91519165,
            311.507293701,
            312.111022949,
            312.703125,
            313.306854248,
            313.910552979,
            314.514282227,
            315.118011475,
            315.721710205,
            316.313812256,
            316.905944824,
            317.509643555,
            318.113372803,
            318.717102051,
            319.320800781,
            319.924530029,
            320.528259277,
            321.120361328,
            321.712463379,
            322.316192627,
            322.919891357,
            323.523620605,
            324.127349854,
            324.731048584,
            325.323181152,
            325.915283203,
            326.518981934,
            327.122711182,
            327.72644043,
            328.33013916,
            328.933868408,
            329.537597656,
            330.129699707,
            330.721801758,
            331.325531006,
            331.929260254,
            332.532958984,
            333.136688232,
            333.728790283,
            334.320892334,
            334.924621582,
            335.52835083,
            336.132049561,
            336.735778809,
            337.339508057,
            337.943206787,
            338.535308838,
            339.127441406,
            339.731140137,
            340.334869385,
            340.938598633,
            341.542297363,
            342.146026611,
            342.749755859,
            343.35345459,
            343.980407715,
            344.607330322,
            345.234283447,
            345.861206055,
            346.48815918,
            347.091888428,
            347.695587158,
            348.299316406,
            348.903045654,
            349.506744385,
            350.110473633,
            350.714202881,
            351.294677734,
            351.898406982,
            352.50213623,
            353.105834961,
            353.709564209,
            354.313293457,
            354.905395508,
            355.497497559,
            356.101226807,
            356.704925537,
            357.308654785,
            357.912384033,
            358.516082764,
            359.119812012,
            359.72354126,
            360.315643311,
            360.907745361,
            361.511474609,
            362.11517334,
            362.718902588,
            363.322631836,
            363.926330566,
            364.518463135,
            365.110565186,
            365.714263916,
            366.317993164,
            366.921722412,
            367.525421143,
            368.105926514,
            368.709655762,
            369.31338501,
            369.91708374,
            370.520812988,
            371.124542236,
            371.728240967,
            372.320343018,
            372.912475586,
            373.516174316,
            374.119903564,
            374.723632812,
            375.327331543,
            375.919464111,
            376.511566162,
            377.115264893,
            377.718994141,
            378.322723389,
            378.926422119,
            379.530151367,
            380.122253418,
            380.714355469,
            381.318084717,
            381.921813965,
            382.525512695,
            383.129241943,
            383.732971191,
            384.336669922,
            384.92880249,
            385.520904541,
            386.124633789,
            386.72833252,
            387.332061768,
            387.935760498,
            388.539489746,
            389.143218994,
            389.735321045,
            390.327423096,
            390.931152344,
            391.534881592,
            392.138580322,
            392.74230957,
            393.346038818,
            393.949737549,
            394.5418396,
            395.133972168,
            395.737670898,
            396.341400146,
            396.945129395,
            397.548828125,
            398.152557373,
            398.744659424,
            399.336761475,
            399.940490723,
            400.544219971,
            401.147918701,
            401.751647949,
            402.355377197,
            402.959075928,
            403.562805176,
            404.166534424,
            404.770233154,
            405.362365723,
            405.954467773,
            406.558166504,
            407.161895752,
            407.765625,
            408.36932373,
            408.973052979,
            409.565155029,
            410.15725708,
            410.760986328,
            411.364715576,
            411.968414307,
            412.572143555,
            413.175872803,
            413.767974854,
            414.360076904,
            414.963806152,
            415.5675354,
            416.171234131,
            416.774963379,
            417.378662109,
            417.982391357,
            418.574493408,
            419.166625977,
            419.770324707,
            420.374053955,
            420.977783203,
            421.581481934,
            422.185211182,
            422.78894043,
            423.427490234,
            424.066009521,
            424.704559326,
            425.319915771,
            425.946838379,
            426.573791504,
            427.200714111,
            427.839263916,
            428.466217041,
            429.081542969,
            429.685241699,
            430.288970947,
            430.892700195,
            431.496398926,
            432.100128174,
            432.692230225,
            433.284332275,
            433.888061523,
            434.491790771,
            435.095489502,
            435.69921875,
            436.302947998,
            436.906646729,
            437.498779297,
            438.090881348,
            438.694580078,
            439.298309326,
            439.902038574,
            440.505737305,
            441.109466553,
            441.713195801,
            442.305297852,
            442.897399902,
            443.50112915,
            444.104858398,
            444.708557129,
            445.30065918,
            445.892791748,
            446.496490479,
            447.100219727,
            447.692321777,
            448.296051025,
            448.899749756,
            449.503479004,
            450.107208252,
            450.699310303,
            451.303039551,
            451.906738281,
            452.510467529,
            453.114196777,
            453.717895508,
            454.310028076,
            454.902130127,
            455.505828857,
            456.109558105,
            456.713287354,
            457.316986084,
            457.920715332,
            458.512817383,
            459.104949951,
            459.708648682,
            460.31237793,
            460.91607666,
            461.519805908,
            462.18157959,
            462.831726074,
            463.470275879,
            464.03918457,
            464.619659424,
            465.235015869,
            465.850341797,
            466.465667725,
            467.069366455,
            467.673095703,
            468.276824951,
            468.892150879,
        ],
        "beats_loudness": {
            "dmean2": 0.0825308188796,
            "median": 0.0359352231026,
            "min": 1.83989157243e-10,
            "dvar2": 0.00314171635546,
            "dvar": 0.000814661907498,
            "dmean": 0.0418311133981,
            "max": 0.121387630701,
            "var": 0.00122407265007,
            "mean": 0.049061499536,
        },
        "danceability": 1.34079182148,
        "bpm_histogram_second_peak_weight": {
            "dmean2": 0,
            "median": 0.00641848519444,
            "min": 0.00641848519444,
            "dvar2": 0,
            "dvar": 0,
            "dmean": 0,
            "max": 0.00641848519444,
            "var": 0,
            "mean": 0.00641848519444,
        },
        "beats_count": 780,
        "bpm_histogram_second_peak_bpm": {
            "dmean2": 0,
            "median": 94,
            "min": 94,
            "dvar2": 0,
            "dvar": 0,
            "dmean": 0,
            "max": 94,
            "var": 0,
            "mean": 94,
        },
        "bpm_histogram_first_peak_spread": {
            "dmean2": 0,
            "median": 0.2734375,
            "min": 0.2734375,
            "dvar2": 0,
            "dvar": 0,
            "dmean": 0,
            "max": 0.2734375,
            "var": 0,
            "mean": 0.2734375,
        },
        "beats_loudness_band_ratio": {
            "dmean2": [
                0.430711448193,
                0.188225373626,
                0.192341089249,
                0.0548661835492,
                0.0471438392997,
                0.123224511743,
            ],
            "median": [
                0.710864305496,
                0.115177638829,
                0.0495216995478,
                0.020622305572,
                0.00899726618081,
                0.0268207900226,
            ],
            "min": [
                0.00397313572466,
                0.00232740468346,
                0.000627535802778,
                0.000159295930644,
                3.33271077579e-06,
                2.05074557016e-05,
            ],
            "dvar2": [
                0.0628691762686,
                0.025976035744,
                0.0532752200961,
                0.0034635476768,
                0.00604925304651,
                0.0200368855149,
            ],
            "dvar": [
                0.0190303269774,
                0.00874643959105,
                0.0149922650307,
                0.00136204250157,
                0.00214308989234,
                0.00627325056121,
            ],
            "dmean": [
                0.220870420337,
                0.104567043483,
                0.099620424211,
                0.0303768031299,
                0.0256413463503,
                0.0636089965701,
            ],
            "max": [
                0.976348400116,
                0.550616443157,
                0.743173718452,
                0.422599494457,
                0.658635556698,
                0.877433359623,
            ],
            "var": [
                0.0393318757415,
                0.0109097696841,
                0.0161587037146,
                0.00192682177294,
                0.00177863473073,
                0.00867664348334,
            ],
            "mean": [
                0.672453582287,
                0.14296464622,
                0.100197598338,
                0.035744804889,
                0.0233048740774,
                0.0617416091263,
            ],
        },
    },
    "lowlevel": {
        "barkbands_spread": {
            "dmean2": 11.3085775375,
            "median": 12.6962938309,
            "min": 0.302790343761,
            "dvar2": 220.751251221,
            "dvar": 103.028068542,
            "dmean": 7.43946075439,
            "max": 113.481361389,
            "var": 325.057189941,
            "mean": 19.2823753357,
        },
        "melbands_spread": {
            "dmean2": 10.1318311691,
            "median": 10.1432170868,
            "min": 0.279888927937,
            "dvar2": 228.005065918,
            "dvar": 114.305671692,
            "dmean": 6.58419561386,
            "max": 196.989822388,
            "var": 340.562133789,
            "mean": 15.8502044678,
        },
        "hfc": {
            "dmean2": 9.66177272797,
            "median": 9.96537971497,
            "min": 1.07952132284e-16,
            "dvar2": 322.719940186,
            "dvar": 157.343002319,
            "dmean": 7.03391361237,
            "max": 154.550186157,
            "var": 353.103057861,
            "mean": 15.7598428726,
        },
        "barkbands_kurtosis": {
            "dmean2": 15.5694160461,
            "median": 5.25297260284,
            "min": -1.94565689564,
            "dvar2": 1111.80249023,
            "dvar": 458.765808105,
            "dmean": 9.81207084656,
            "max": 485.822357178,
            "var": 826.78503418,
            "mean": 15.2445526123,
        },
        "gfcc": {
            "mean": [
                -52.8757591248,
                107.735832214,
                -81.0579147339,
                31.3055839539,
                -45.5405044556,
                17.9729709625,
                -21.1462440491,
                5.70709228516,
                -12.8363389969,
                1.68357098103,
                -9.73576259613,
                -2.00269675255,
                -7.53992891312,
            ],
            "icov": [
                [
                    0.00010386921349,
                    -0.000106655752461,
                    0.000111980138172,
                    -0.000111873210699,
                    0.000189964135643,
                    -0.000229979938013,
                    0.000213177758269,
                    -0.000271449534921,
                    0.000279503467027,
                    -6.60322839394e-05,
                    0.000166915968293,
                    5.55644219276e-05,
                    0.00016976367624,
                ],
                [
                    -0.000106655752461,
                    0.00149726681411,
                    -0.00029822596116,
                    0.000676356139593,
                    -0.000958638149314,
                    0.000967702420894,
                    -0.000732818734832,
                    0.00196211785078,
                    -0.00184333114885,
                    0.00178882759064,
                    -0.00155100796837,
                    0.00153711787425,
                    -0.00130420573987,
                ],
                [
                    0.000111980138172,
                    -0.00029822596116,
                    0.00103786541149,
                    -1.55373709276e-05,
                    0.000404983060434,
                    -0.000631500442978,
                    0.000882507534698,
                    -0.000837513129227,
                    0.000998365110718,
                    -0.000635055708699,
                    0.000573393073864,
                    0.000478364148876,
                    -0.000210827318369,
                ],
                [
                    -0.000111873210699,
                    0.000676356139593,
                    -1.55373709276e-05,
                    0.00213184463792,
                    -0.000775675289333,
                    0.000699523487128,
                    -0.000228347169468,
                    0.000810330093373,
                    0.000204399126233,
                    -2.07752473216e-05,
                    -0.000575851649046,
                    0.000312113843393,
                    2.17651540879e-05,
                ],
                [
                    0.000189964135643,
                    -0.000958638149314,
                    0.000404983060434,
                    -0.000775675289333,
                    0.00441662222147,
                    -0.0018977324944,
                    0.00093678291887,
                    -0.000779778463766,
                    -0.000238973618252,
                    0.00124784593936,
                    -0.00172325293534,
                    0.000882573018316,
                    -0.000355818134267,
                ],
                [
                    -0.000229979938013,
                    0.000967702420894,
                    -0.000631500442978,
                    0.000699523487128,
                    -0.0018977324944,
                    0.00470419740304,
                    -0.00224701920524,
                    0.000846366921905,
                    0.000901055056602,
                    -0.00165795383509,
                    0.0019007694209,
                    -0.00330009195022,
                    0.000795492320322,
                ],
                [
                    0.000213177758269,
                    -0.000732818734832,
                    0.000882507534698,
                    -0.000228347169468,
                    0.00093678291887,
                    -0.00224701920524,
                    0.00696297176182,
                    -0.0026064470876,
                    -8.30165154184e-05,
                    0.00128976954147,
                    -0.00322702690028,
                    0.00428045261651,
                    -0.0017247867072,
                ],
                [
                    -0.000271449534921,
                    0.00196211785078,
                    -0.000837513129227,
                    0.000810330093373,
                    -0.000779778463766,
                    0.000846366921905,
                    -0.0026064470876,
                    0.010320478119,
                    -0.00472758943215,
                    0.000610328454059,
                    0.000745555968024,
                    -0.000348137982655,
                    0.00116989726666,
                ],
                [
                    0.000279503467027,
                    -0.00184333114885,
                    0.000998365110718,
                    0.000204399126233,
                    -0.000238973618252,
                    0.000901055056602,
                    -8.30165154184e-05,
                    -0.00472758943215,
                    0.0120761645958,
                    -0.00565708614886,
                    0.00197687884793,
                    0.000191203667782,
                    0.000477980007418,
                ],
                [
                    -6.60322839394e-05,
                    0.00178882759064,
                    -0.000635055708699,
                    -2.07752473216e-05,
                    0.00124784593936,
                    -0.00165795383509,
                    0.00128976954147,
                    0.000610328454059,
                    -0.00565708614886,
                    0.0153611283749,
                    -0.0074116284959,
                    0.00313461828046,
                    -0.000616872508544,
                ],
                [
                    0.000166915968293,
                    -0.00155100796837,
                    0.000573393073864,
                    -0.000575851649046,
                    -0.00172325293534,
                    0.0019007694209,
                    -0.00322702690028,
                    0.000745555968024,
                    0.00197687884793,
                    -0.0074116284959,
                    0.0204007960856,
                    -0.00898791570216,
                    0.00370677234605,
                ],
                [
                    5.55644219276e-05,
                    0.00153711787425,
                    0.000478364148876,
                    0.000312113843393,
                    0.000882573018316,
                    -0.00330009195022,
                    0.00428045261651,
                    -0.000348137982655,
                    0.000191203667782,
                    0.00313461828046,
                    -0.00898791570216,
                    0.0203682091087,
                    -0.00913562625647,
                ],
                [
                    0.00016976367624,
                    -0.00130420573987,
                    -0.000210827318369,
                    2.17651540879e-05,
                    -0.000355818134267,
                    0.000795492320322,
                    -0.0017247867072,
                    0.00116989726666,
                    0.000477980007418,
                    -0.000616872508544,
                    0.00370677234605,
                    -0.00913562625647,
                    0.0166303087026,
                ],
            ],
            "cov": [
                [
                    17690.7089844,
                    -4189.91357422,
                    126.300872803,
                    1082.3527832,
                    -1009.1784668,
                    1426.14001465,
                    -691.036804199,
                    529.490356445,
                    -788.191101074,
                    285.409301758,
                    -406.658721924,
                    264.882141113,
                    -438.328033447,
                ],
                [
                    -4189.91357422,
                    5079.68652344,
                    -852.493041992,
                    -792.726806641,
                    834.966552734,
                    -1248.83447266,
                    554.589172363,
                    -540.169067383,
                    608.643310547,
                    -423.881378174,
                    278.276367188,
                    -389.223266602,
                    295.430206299,
                ],
                [
                    126.300872803,
                    -852.493041992,
                    1537.67004395,
                    13.2496614456,
                    -212.534973145,
                    335.952209473,
                    -228.548202515,
                    111.474601746,
                    -184.409698486,
                    95.5868225098,
                    -111.175376892,
                    65.111869812,
                    -31.4369678497,
                ],
                [
                    1082.3527832,
                    -792.726806641,
                    13.2496614456,
                    688.815307617,
                    -58.7191429138,
                    135.787139893,
                    -96.6784744263,
                    34.5018119812,
                    -126.266029358,
                    56.3887863159,
                    -19.1238422394,
                    50.1215591431,
                    -56.6393089294,
                ],
                [
                    -1009.1784668,
                    834.966552734,
                    -212.534973145,
                    -58.7191429138,
                    443.4609375,
                    -114.242698669,
                    106.349227905,
                    -90.794921875,
                    100.969955444,
                    -79.0163803101,
                    75.178276062,
                    -45.6535148621,
                    57.8662796021,
                ],
                [
                    1426.14001465,
                    -1248.83447266,
                    335.952209473,
                    135.787139893,
                    -114.242698669,
                    665.476196289,
                    -93.4698257446,
                    126.453094482,
                    -192.028656006,
                    113.603408813,
                    -70.1643753052,
                    142.460281372,
                    -57.6503334045,
                ],
                [
                    -691.036804199,
                    554.589172363,
                    -228.548202515,
                    -96.6784744263,
                    106.349227905,
                    -93.4698257446,
                    286.254364014,
                    -7.24755954742,
                    101.515708923,
                    -40.7418060303,
                    53.4621047974,
                    -71.0922241211,
                    29.3220424652,
                ],
                [
                    529.490356445,
                    -540.169067383,
                    111.474601746,
                    34.5018119812,
                    -90.794921875,
                    126.453094482,
                    -7.24755954742,
                    206.569717407,
                    0.97103369236,
                    60.0731735229,
                    -37.1339569092,
                    17.8584671021,
                    -49.3860282898,
                ],
                [
                    -788.191101074,
                    608.643310547,
                    -184.409698486,
                    -126.266029358,
                    100.969955444,
                    -192.028656006,
                    101.515708923,
                    0.97103369236,
                    220.32913208,
                    -4.45077848434,
                    39.7392883301,
                    -67.9825820923,
                    22.7107410431,
                ],
                [
                    285.409301758,
                    -423.881378174,
                    95.5868225098,
                    56.3887863159,
                    -79.0163803101,
                    113.603408813,
                    -40.7418060303,
                    60.0731735229,
                    -4.45077848434,
                    133.429000854,
                    5.01334619522,
                    26.9678840637,
                    -31.8196659088,
                ],
                [
                    -406.658721924,
                    278.276367188,
                    -111.175376892,
                    -19.1238422394,
                    75.178276062,
                    -70.1643753052,
                    53.4621047974,
                    -37.1339569092,
                    39.7392883301,
                    5.01334619522,
                    96.0722427368,
                    6.16748142242,
                    18.7299880981,
                ],
                [
                    264.882141113,
                    -389.223266602,
                    65.111869812,
                    50.1215591431,
                    -45.6535148621,
                    142.460281372,
                    -71.0922241211,
                    17.8584671021,
                    -67.9825820923,
                    26.9678840637,
                    6.16748142242,
                    124.394592285,
                    21.0249347687,
                ],
                [
                    -438.328033447,
                    295.430206299,
                    -31.4369678497,
                    -56.6393089294,
                    57.8662796021,
                    -57.6503334045,
                    29.3220424652,
                    -49.3860282898,
                    22.7107410431,
                    -31.8196659088,
                    18.7299880981,
                    21.0249347687,
                    103.502845764,
                ],
            ],
        },
        "spectral_energyband_middle_low": {
            "dmean2": 0.0102198179811,
            "median": 0.00536039331928,
            "min": 2.65123048843e-22,
            "dvar2": 0.000328054215061,
            "dvar": 0.000197467088583,
            "dmean": 0.00720286648721,
            "max": 0.119120843709,
            "var": 0.000343762105331,
            "mean": 0.0114525295794,
        },
        "melbands_crest": {
            "dmean2": 6.53237104416,
            "median": 17.9162158966,
            "min": 1.67271459103,
            "dvar2": 31.6969070435,
            "dvar": 13.6536588669,
            "dmean": 4.13504600525,
            "max": 36.3514518738,
            "var": 45.5407485962,
            "mean": 18.4556522369,
        },
        "spectral_kurtosis": {
            "dmean2": 11.2094621658,
            "median": 3.03414392471,
            "min": -1.27646434307,
            "dvar2": 1162.87890625,
            "dvar": 516.966552734,
            "dmean": 7.35913610458,
            "max": 625.964172363,
            "var": 1592.42443848,
            "mean": 13.7717342377,
        },
        "spectral_rms": {
            "dmean2": 0.00157959479839,
            "median": 0.00466633308679,
            "min": 3.12093211517e-12,
            "dvar2": 2.90444609163e-06,
            "dvar": 1.51608685428e-06,
            "dmean": 0.00108874298166,
            "max": 0.0143991792575,
            "var": 8.33909598441e-06,
            "mean": 0.00527398148552,
        },
        "zerocrossingrate": {
            "dmean2": 0.0154096977785,
            "median": 0.03369140625,
            "min": 0.001953125,
            "dvar2": 0.000311016134219,
            "dvar": 0.000255326158367,
            "dmean": 0.0135301901028,
            "max": 0.53173828125,
            "var": 0.00270974566229,
            "mean": 0.0507398732007,
        },
        "silence_rate_60dB": {
            "dmean2": 0.0973053127527,
            "median": 0,
            "min": 0,
            "dvar2": 0.115952201188,
            "dvar": 0.0462943948805,
            "dmean": 0.0486502535641,
            "max": 1,
            "var": 0.0738631635904,
            "mean": 0.0802887231112,
        },
        "erbbands_kurtosis": {
            "dmean2": 2.95349383354,
            "median": -0.0600297451019,
            "min": -1.9039914608,
            "dvar2": 32.2196807861,
            "dvar": 14.8086624146,
            "dmean": 1.914244771,
            "max": 109.734512329,
            "var": 22.2196083069,
            "mean": 1.34698784351,
        },
        "erbbands": {
            "dmean2": [
                0.428299844265,
                2.28906655312,
                5.51969909668,
                12.5672893524,
                21.8281002045,
                29.313495636,
                16.3954944611,
                10.3064498901,
                8.35011768341,
                14.7721195221,
                31.08735466,
                10.5252399445,
                13.8429956436,
                10.2353038788,
                12.4269170761,
                15.4577627182,
                7.70342493057,
                9.83148288727,
                9.0091753006,
                5.50162410736,
                8.03603172302,
                5.58365821838,
                7.62825536728,
                5.08629179001,
                4.8826174736,
                6.02267599106,
                5.06897115707,
                3.89897060394,
                4.18314743042,
                3.49176216125,
                5.57253408432,
                3.77836751938,
                2.91096735001,
                1.80456626415,
                1.10676884651,
                0.416215091944,
                0.128760591149,
                0.0229548234493,
                0.0030636980664,
                0.000156342808623,
            ],
            "median": [
                0.283721119165,
                0.927432239056,
                1.47792208195,
                3.52597641945,
                2.42209553719,
                3.7360265255,
                4.3532910347,
                3.73197412491,
                2.65285634995,
                5.40560531616,
                8.99517536163,
                4.68354463577,
                4.5760307312,
                3.41258049011,
                5.04569816589,
                5.86556816101,
                3.6113049984,
                3.37496948242,
                2.82456660271,
                2.26255869865,
                2.13879799843,
                1.67384028435,
                1.41590988636,
                1.1372115612,
                1.05508136749,
                1.21863031387,
                1.16889476776,
                0.912000060081,
                0.915995657444,
                0.67924708128,
                0.686301469803,
                0.477991074324,
                0.319812089205,
                0.204466596246,
                0.114510826766,
                0.0470657609403,
                0.0110522108153,
                0.00186074071098,
                0.000260208500549,
                8.15202583908e-05,
            ],
            "min": [
                1.73851526564e-22,
                3.92272553265e-21,
                4.44086710627e-20,
                2.42112437185e-20,
                1.94110467692e-19,
                1.17262933093e-19,
                1.98267761128e-19,
                3.97561329613e-19,
                2.18353477084e-19,
                8.14600849797e-19,
                9.92887946907e-19,
                2.17544613353e-18,
                1.40501031779e-18,
                2.23168986569e-18,
                3.23321078645e-18,
                4.46148548777e-18,
                6.14916017127e-18,
                5.52335130217e-18,
                3.36392207364e-18,
                8.55628346435e-18,
                9.33644961243e-18,
                7.00916537849e-18,
                8.54520089851e-18,
                1.17213478032e-17,
                1.73005751558e-17,
                1.7268897447e-17,
                1.54648430008e-17,
                1.47321280686e-17,
                2.49232910003e-17,
                1.74896487915e-17,
                1.97446026983e-17,
                1.73287886321e-17,
                1.25015776442e-17,
                9.77732860712e-18,
                9.27149525483e-18,
                5.97360887064e-18,
                2.9045715155e-18,
                1.33250659293e-18,
                3.08723036554e-19,
                3.08572360436e-20,
            ],
            "dvar2": [
                0.589720308781,
                15.8781013489,
                115.261634827,
                894.74395752,
                3623.12988281,
                7216.13037109,
                1346.86328125,
                395.219970703,
                271.484313965,
                883.684448242,
                3534.11352539,
                445.965484619,
                962.390808105,
                561.608764648,
                816.409118652,
                1031.81359863,
                194.976074219,
                469.042724609,
                484.088256836,
                167.000671387,
                411.591308594,
                267.27520752,
                709.139465332,
                153.441650391,
                139.300918579,
                290.952606201,
                140.435256958,
                69.9862442017,
                73.9900283813,
                51.5217933655,
                215.226226807,
                78.0563201904,
                44.971824646,
                18.3356552124,
                10.0238132477,
                1.39749252796,
                0.152415767312,
                0.00447462266311,
                8.89804796316e-05,
                2.12016431078e-07,
            ],
            "dvar": [
                0.215524703264,
                7.72521686554,
                80.2513198853,
                455.896606445,
                1555.41345215,
                2612.75439453,
                480.685638428,
                158.112503052,
                99.4391098022,
                426.479644775,
                1852.37145996,
                222.037109375,
                523.996582031,
                305.131774902,
                385.197235107,
                484.387969971,
                81.6952133179,
                205.287582397,
                213.223922729,
                66.5508499146,
                173.345169067,
                118.786193848,
                367.967529297,
                77.0198516846,
                67.1517333984,
                160.670516968,
                65.487197876,
                35.6394195557,
                35.1753883362,
                23.6361217499,
                93.6900482178,
                38.116645813,
                22.9980621338,
                9.08386993408,
                4.69902420044,
                0.651064157486,
                0.0717034339905,
                0.00214108382352,
                4.24883910455e-05,
                9.9780372409e-08,
            ],
            "dmean": [
                0.285809576511,
                1.71466720104,
                4.78485679626,
                9.31142616272,
                13.9938602448,
                16.6350879669,
                9.57824516296,
                6.50806808472,
                5.05367469788,
                9.50606632233,
                20.6629753113,
                6.76035642624,
                9.00918102264,
                6.65080547333,
                8.12446498871,
                9.74140357971,
                4.73407125473,
                6.18055009842,
                5.77913284302,
                3.41051983833,
                5.03814935684,
                3.6214363575,
                4.92935085297,
                3.33136463165,
                3.14136767387,
                3.96647572517,
                3.27122926712,
                2.54569911957,
                2.78428292274,
                2.35334944725,
                3.64336013794,
                2.66048502922,
                2.00047636032,
                1.23166310787,
                0.78520399332,
                0.289730489254,
                0.0912321954966,
                0.015846285969,
                0.00216961093247,
                0.000110848726763,
            ],
            "max": [
                9.64658355713,
                31.4366493225,
                96.8313522339,
                196.806808472,
                335.627746582,
                405.507202148,
                318.389892578,
                425.480987549,
                155.056884766,
                734.217224121,
                942.432678223,
                450.749542236,
                967.919799805,
                676.453918457,
                748.619262695,
                896.790100098,
                260.385131836,
                327.264251709,
                355.615600586,
                249.552474976,
                314.737304688,
                559.337402344,
                506.524536133,
                278.813781738,
                154.522460938,
                336.360961914,
                208.021591187,
                148.887634277,
                151.330780029,
                101.265739441,
                153.754882812,
                95.8621826172,
                106.083877563,
                54.0735244751,
                36.8974952698,
                10.6748008728,
                3.31087422371,
                0.541660666466,
                0.0808046162128,
                0.00396120175719,
            ],
            "var": [
                0.381921380758,
                26.4123821259,
                222.696472168,
                638.73651123,
                1439.33251953,
                1684.1505127,
                392.004180908,
                183.072143555,
                90.0787277222,
                885.989501953,
                4406.39013672,
                503.672363281,
                1249.47509766,
                741.50958252,
                774.553833008,
                842.99206543,
                140.2137146,
                283.121612549,
                276.191925049,
                91.2408294678,
                259.221588135,
                241.70489502,
                608.700134277,
                205.818191528,
                143.899810791,
                318.06237793,
                165.342025757,
                84.3325576782,
                81.7364196777,
                55.5553512573,
                134.187164307,
                77.3074874878,
                48.5985832214,
                18.1627349854,
                8.14870166779,
                0.956533193588,
                0.0977488458157,
                0.00271429261193,
                5.59098298254e-05,
                1.41564086675e-07,
            ],
            "mean": [
                0.477559149265,
                3.27270174026,
                7.60931921005,
                11.1678476334,
                12.6747789383,
                13.5848760605,
                10.8409786224,
                7.9783949852,
                5.86515951157,
                14.7864227295,
                34.0759735107,
                11.0554237366,
                14.3407850266,
                10.1547403336,
                12.8605480194,
                14.7355899811,
                7.50419521332,
                8.41241931915,
                7.65559196472,
                5.26523971558,
                7.26457834244,
                6.14167642593,
                7.44393730164,
                6.10683917999,
                5.52889490128,
                6.9977273941,
                6.09786462784,
                4.76159763336,
                5.06575870514,
                4.18718147278,
                5.28951311111,
                4.17526912689,
                3.10808396339,
                1.84764695168,
                1.06080031395,
                0.381296306849,
                0.113579489291,
                0.0184997897595,
                0.00259469938464,
                0.000180276590982,
            ],
        },
        "spectral_strongpeak": {
            "dmean2": 0.464798301458,
            "median": 0.361429721117,
            "min": 0,
            "dvar2": 0.545165300369,
            "dvar": 0.234720677137,
            "dmean": 0.286188274622,
            "max": 11.5523529053,
            "var": 0.389024376869,
            "mean": 0.591991603374,
        },
        "spectral_energy": {
            "dmean2": 0.0197877436876,
            "median": 0.0223190300167,
            "min": 9.98372264228e-21,
            "dvar2": 0.000644606188871,
            "dvar": 0.000289402203634,
            "dmean": 0.0129658170044,
            "max": 0.2125197649,
            "var": 0.00117197574582,
            "mean": 0.0370581746101,
        },
        "average_loudness": 0.875494062901,
        "spectral_rolloff": {
            "dmean2": 960.533081055,
            "median": 495.263671875,
            "min": 43.06640625,
            "dvar2": 3546625.75,
            "dvar": 1597838.5,
            "dmean": 574.075500488,
            "max": 19250.6835938,
            "var": 3974413.25,
            "mean": 1086.88134766,
        },
        "spectral_centroid": {
            "dmean2": 420.67791748,
            "median": 810.38659668,
            "min": 108.107566833,
            "dvar2": 241400.015625,
            "dvar": 134993.796875,
            "dmean": 284.24029541,
            "max": 10686.7919922,
            "var": 578495.375,
            "mean": 1054.03820801,
        },
        "pitch_salience": {
            "dmean2": 0.142301797867,
            "median": 0.513008773327,
            "min": 0.0365366339684,
            "dvar2": 0.0186522137374,
            "dvar": 0.00812033563852,
            "dmean": 0.0912069603801,
            "max": 0.904991447926,
            "var": 0.0302368402481,
            "mean": 0.486002385616,
        },
        "spectral_energyband_middle_high": {
            "dmean2": 0.00129647296853,
            "median": 0.00109805946704,
            "min": 1.59428602143e-21,
            "dvar2": 3.59975365427e-06,
            "dvar": 1.95909206013e-06,
            "dmean": 0.000891717558261,
            "max": 0.0392765961587,
            "var": 5.41546842214e-06,
            "mean": 0.00189269217663,
        },
        "spectral_contrast_coeffs": {
            "dmean2": [
                0.078412592411,
                0.0850035324693,
                0.0669980943203,
                0.0519671812654,
                0.0410364679992,
                0.0324285030365,
            ],
            "median": [
                -0.712934792042,
                -0.743565022945,
                -0.804856359959,
                -0.822204768658,
                -0.819852888584,
                -0.799151182175,
            ],
            "min": [
                -0.967198193073,
                -0.974618017673,
                -0.972245693207,
                -0.961651921272,
                -0.964111566544,
                -0.961030483246,
            ],
            "dvar2": [
                0.00545508880168,
                0.00506833242252,
                0.00283919996582,
                0.00171206286177,
                0.00133325520437,
                0.00108349311631,
            ],
            "dvar": [
                0.00259663071483,
                0.00218783551827,
                0.00130078371149,
                0.000817677413579,
                0.000717440620065,
                0.000583040935453,
            ],
            "dmean": [
                0.050219014287,
                0.0553187951446,
                0.0431638620794,
                0.033497761935,
                0.0269822217524,
                0.0216719079763,
            ],
            "max": [
                -0.362060725689,
                -0.415127366781,
                -0.449471920729,
                -0.544864594936,
                -0.520337045193,
                -0.484779447317,
            ],
            "var": [
                0.00896039698273,
                0.00595588609576,
                0.00468455068767,
                0.00240010186099,
                0.00273609859869,
                0.00380731164478,
            ],
            "mean": [
                -0.691217899323,
                -0.738405883312,
                -0.792845070362,
                -0.815919578075,
                -0.808208405972,
                -0.783209085464,
            ],
        },
        "melbands_skewness": {
            "dmean2": 2.03852105141,
            "median": 3.49205732346,
            "min": -5.09098958969,
            "dvar2": 5.59213733673,
            "dvar": 2.60251045227,
            "dmean": 1.34666419029,
            "max": 27.9477214813,
            "var": 10.3399658203,
            "mean": 4.20400619507,
        },
        "spectral_spread": {
            "dmean2": 1288822,
            "median": 5559910,
            "min": 202146.75,
            "dvar2": 2488438882300.0,
            "dvar": 1429272657920.0,
            "dmean": 912083.75,
            "max": 47627192,
            "var": 9248252624900.0,
            "mean": 5486277,
        },
        "dissonance": {
            "dmean2": 0.0456466600299,
            "median": 0.461005300283,
            "min": 0.0437632985413,
            "dvar2": 0.00274136965163,
            "dvar": 0.00109660299495,
            "dmean": 0.0279056765139,
            "max": 0.500000059605,
            "var": 0.00284485798329,
            "mean": 0.445629894733,
        },
        "spectral_skewness": {
            "dmean2": 0.794989943504,
            "median": 1.50365900993,
            "min": -0.509248197079,
            "dvar2": 1.80770647526,
            "dvar": 0.896960437298,
            "dmean": 0.536588907242,
            "max": 21.0161762238,
            "var": 4.03656053543,
            "mean": 2.08940839767,
        },
        "spectral_flux": {
            "dmean2": 0.0560869723558,
            "median": 0.0768227279186,
            "min": 1.56993041855e-06,
            "dvar2": 0.00292630377226,
            "dvar": 0.00139451096766,
            "dmean": 0.0342274866998,
            "max": 0.346184521914,
            "var": 0.00482846889645,
            "mean": 0.0971085876226,
        },
        "spectral_contrast_valleys": {
            "dmean2": [
                0.721830368042,
                0.903648912907,
                0.712160050869,
                0.610355079174,
                0.583003401756,
                0.655298888683,
            ],
            "median": [
                -6.59833145142,
                -6.94885444641,
                -7.49168157578,
                -8.04998779297,
                -8.20836257935,
                -10.1902523041,
            ],
            "min": [
                -27.0580844879,
                -26.7953968048,
                -27.0427837372,
                -27.444858551,
                -27.150718689,
                -27.1886482239,
            ],
            "dvar2": [
                0.450322687626,
                0.736437559128,
                0.416774332523,
                0.365079373121,
                0.383671492338,
                0.714411377907,
            ],
            "dvar": [
                0.274560123682,
                0.357804059982,
                0.205317720771,
                0.192964762449,
                0.214226126671,
                0.383591622114,
            ],
            "dmean": [
                0.495899945498,
                0.603728413582,
                0.457975208759,
                0.399977326393,
                0.413742393255,
                0.499295979738,
            ],
            "max": [
                -5.08935785294,
                -4.38697957993,
                -5.28223896027,
                -5.58056020737,
                -6.22064733505,
                -7.39333581924,
            ],
            "var": [
                1.47113585472,
                1.41246891022,
                1.57198941708,
                1.60442912579,
                2.06623411179,
                4.46363019943,
            ],
            "mean": [
                -6.87335252762,
                -7.15114831924,
                -7.72637891769,
                -8.31610107422,
                -8.46786499023,
                -10.7827568054,
            ],
        },
        "erbbands_flatness_db": {
            "dmean2": 0.0375987179577,
            "median": 0.137290820479,
            "min": 0.0375320166349,
            "dvar2": 0.00167702429462,
            "dvar": 0.000914512726013,
            "dmean": 0.027081925422,
            "max": 0.576275110245,
            "var": 0.00752041861415,
            "mean": 0.161212623119,
        },
        "spectral_energyband_high": {
            "dmean2": 0.000732991029508,
            "median": 0.000172440675669,
            "min": 7.32384502664e-21,
            "dvar2": 2.84155316876e-06,
            "dvar": 1.41091561545e-06,
            "dmean": 0.000526301038917,
            "max": 0.017098184675,
            "var": 3.01054546981e-06,
            "mean": 0.000937026168685,
        },
        "spectral_entropy": {
            "dmean2": 0.404551714659,
            "median": 7.50951766968,
            "min": 3.83585262299,
            "dvar2": 0.203517630696,
            "dvar": 0.113898038864,
            "dmean": 0.29108941555,
            "max": 9.81510448456,
            "var": 0.692627191544,
            "mean": 7.35735177994,
        },
        "silence_rate_20dB": {
            "dmean2": 0,
            "median": 1,
            "min": 1,
            "dvar2": 0,
            "dvar": 0,
            "dmean": 0,
            "max": 1,
            "var": 0,
            "mean": 1,
        },
        "melbands_flatness_db": {
            "dmean2": 0.0618936605752,
            "median": 0.276825994253,
            "min": 0.002832879778,
            "dvar2": 0.00364537141286,
            "dvar": 0.00214522774331,
            "dmean": 0.045175999403,
            "max": 0.766786575317,
            "var": 0.0141341816634,
            "mean": 0.288144052029,
        },
        "barkbands": {
            "dmean2": [
                0.000780046277214,
                0.0112598799169,
                0.00605708220974,
                0.00587175553665,
                0.00767562072724,
                0.00122650107369,
                0.000665556231979,
                0.00140987348277,
                0.000565772177652,
                0.000346536544384,
                0.000235375904595,
                0.000462280353531,
                0.000173911525053,
                0.00019429254462,
                0.000148159728269,
                0.000128144325572,
                0.000106870829768,
                0.000137910654303,
                0.000100147895864,
                0.000120911368867,
                0.000115335969895,
                0.000109506108856,
                0.000169414444827,
                0.000173168242327,
                0.000152851440362,
                7.42633710615e-05,
                1.64063621924e-05,
            ],
            "median": [
                0.00051947060274,
                0.00655899569392,
                0.00139943067916,
                0.000909918686375,
                0.00112726807129,
                0.000451086089015,
                0.000139944080729,
                0.000426110171247,
                0.000153119675815,
                9.37489749049e-05,
                6.31682341918e-05,
                0.000157135087647,
                6.7123779445e-05,
                5.83748333156e-05,
                4.5447894081e-05,
                3.86669562431e-05,
                2.67468385573e-05,
                2.32060701819e-05,
                1.99756977963e-05,
                2.44901912083e-05,
                2.81980246655e-05,
                2.38180637098e-05,
                2.30469813687e-05,
                2.0819465135e-05,
                1.68694987224e-05,
                7.75625903771e-06,
                1.76939909124e-06,
            ],
            "min": [
                1.05992921059e-23,
                1.01882080107e-23,
                3.81650085531e-23,
                9.58720013055e-24,
                6.05843723706e-23,
                3.97942082707e-23,
                1.94826602816e-23,
                4.55673290334e-23,
                8.03923036692e-23,
                6.13023998774e-23,
                3.90747860578e-23,
                1.36834159167e-22,
                1.19486830182e-22,
                9.59428157712e-23,
                7.36139500768e-23,
                2.05417839863e-22,
                1.69648021278e-22,
                1.62890563033e-22,
                3.11427467418e-22,
                3.05703189767e-22,
                3.80435328089e-22,
                6.14622508647e-22,
                6.26719974618e-22,
                8.48117814015e-22,
                1.07863463304e-21,
                1.69885611036e-21,
                2.13010882298e-21,
            ],
            "dvar2": [
                1.47044806909e-06,
                0.000373879738618,
                0.000165687393746,
                0.000221965063247,
                0.000413867877796,
                6.03771377428e-06,
                2.01042939807e-06,
                6.88658747094e-06,
                1.74298861566e-06,
                5.89952264818e-07,
                4.87073350541e-07,
                1.03171248611e-06,
                1.07554612327e-07,
                1.63596951097e-07,
                1.26727968564e-07,
                9.0355520399e-08,
                1.02161081372e-07,
                1.85926893437e-07,
                7.48928954408e-08,
                1.11035916461e-07,
                6.02020833185e-08,
                4.97267329536e-08,
                1.73844824758e-07,
                1.58650124149e-07,
                1.6659740254e-07,
                4.58809203963e-08,
                2.44244335867e-09,
            ],
            "dvar": [
                5.69594305944e-07,
                0.000150891966769,
                0.000102631311165,
                9.98085743049e-05,
                0.000165758596268,
                2.23823212764e-06,
                8.41339669932e-07,
                3.64793049812e-06,
                9.1262540991e-07,
                3.29465478899e-07,
                2.32748931239e-07,
                5.15360568443e-07,
                4.57060664871e-08,
                7.06937512973e-08,
                5.41987397185e-08,
                3.7454729096e-08,
                4.461882952e-08,
                9.44123073054e-08,
                3.58138905199e-08,
                6.19630355914e-08,
                2.9774760435e-08,
                2.44501130453e-08,
                7.74448452034e-08,
                8.03016817486e-08,
                7.94691246142e-08,
                2.18788205331e-08,
                1.17777132491e-09,
            ],
            "dmean": [
                0.000512506987434,
                0.00804802589118,
                0.00487215537578,
                0.00382397091016,
                0.00462136603892,
                0.000744397111703,
                0.000406474980991,
                0.000937627162784,
                0.000366782565834,
                0.000220809830353,
                0.000151705185999,
                0.000295725563774,
                0.000106656254502,
                0.00012199585035,
                9.30279566091e-05,
                7.9521589214e-05,
                6.80271550664e-05,
                8.83774118847e-05,
                6.44517785986e-05,
                7.93256476754e-05,
                7.51946136006e-05,
                7.46458317735e-05,
                0.000111750843644,
                0.000121858320199,
                0.00010798213043,
                5.2450595831e-05,
                1.15770853881e-05,
            ],
            "max": [
                0.0238662660122,
                0.195193648338,
                0.0952763408422,
                0.0813259333372,
                0.0981372743845,
                0.0360920317471,
                0.029116846621,
                0.0528674200177,
                0.032112069428,
                0.0240354128182,
                0.0265638101846,
                0.0361663326621,
                0.00565500324592,
                0.00523700891063,
                0.00579286180437,
                0.00454595778137,
                0.00979127082974,
                0.00769131816924,
                0.00685597769916,
                0.0066013825126,
                0.00420426204801,
                0.003395519685,
                0.00438667321578,
                0.00533647229895,
                0.00484169786796,
                0.00184860487934,
                0.000426471000537,
            ],
            "var": [
                1.00052716334e-06,
                0.000443879165687,
                0.000208414290682,
                9.92849600152e-05,
                0.000132293673232,
                2.05686956178e-06,
                1.2623826251e-06,
                8.76024478202e-06,
                2.01141006073e-06,
                8.06357945748e-07,
                4.44669126409e-07,
                9.52840196078e-07,
                7.04275890939e-08,
                9.74497922357e-08,
                6.93088750836e-08,
                5.45285452347e-08,
                8.23804526817e-08,
                1.70030460822e-07,
                7.66214398595e-08,
                1.23539535934e-07,
                7.75202835257e-08,
                5.95994933406e-08,
                1.22098853694e-07,
                1.74581757051e-07,
                1.51459090603e-07,
                3.05719431992e-08,
                1.57505808396e-09,
            ],
            "mean": [
                0.000792860577349,
                0.0153801292181,
                0.00665125902742,
                0.00363414245658,
                0.00425389688462,
                0.000888265087269,
                0.00047852890566,
                0.0015550450189,
                0.000540278153494,
                0.000325157860061,
                0.000214053303353,
                0.000429682782851,
                0.00015268351126,
                0.00016140457592,
                0.000122115612612,
                0.000112896683277,
                0.000105671220808,
                0.000135471418616,
                0.000110447574116,
                0.000138279981911,
                0.000139797382872,
                0.000137177747092,
                0.000168910919456,
                0.000193774962099,
                0.000155140223796,
                6.66902997182e-05,
                1.43114566526e-05,
            ],
        },
        "erbbands_skewness": {
            "dmean2": 0.806777954102,
            "median": 0.440305769444,
            "min": -4.64436674118,
            "dvar2": 0.675009250641,
            "dvar": 0.328243792057,
            "dmean": 0.546455204487,
            "max": 8.12401199341,
            "var": 1.04869437218,
            "mean": 0.503021776676,
        },
        "erbbands_spread": {
            "dmean2": 20.0970649719,
            "median": 42.1646537781,
            "min": 0.278548508883,
            "dvar2": 522.817504883,
            "dvar": 262.181243896,
            "dmean": 13.6425132751,
            "max": 183.475982666,
            "var": 954.214477539,
            "mean": 47.5880241394,
        },
        "melbands_kurtosis": {
            "dmean2": 37.6371421814,
            "median": 17.5584392548,
            "min": -1.83117628098,
            "dvar2": 5883.89160156,
            "dvar": 2582.94384766,
            "dmean": 24.0055179596,
            "max": 1107.68518066,
            "var": 5390.95068359,
            "mean": 42.5645561218,
        },
        "melbands": {
            "dmean2": [
                0.00329641532153,
                0.0026374112349,
                0.00187228771392,
                0.00180036283564,
                0.0007303962484,
                0.000222255082917,
                0.000132661632961,
                0.00020058186783,
                0.000218769302592,
                7.24005949451e-05,
                7.06409045961e-05,
                4.27709564974e-05,
                3.79567354685e-05,
                5.88957736909e-05,
                2.37249169004e-05,
                1.84442196769e-05,
                1.82501553354e-05,
                1.56433070515e-05,
                9.16564476938e-06,
                7.00420378053e-06,
                1.01661744338e-05,
                6.29408714303e-06,
                6.71077714287e-06,
                6.99026259099e-06,
                4.12339386457e-06,
                4.15688373323e-06,
                4.44842953584e-06,
                3.88687976738e-06,
                3.67111715605e-06,
                2.61515447164e-06,
                2.38516872741e-06,
                2.41933616962e-06,
                2.26173710871e-06,
                2.03485728889e-06,
                3.53071436621e-06,
                2.68972803497e-06,
                2.38401025854e-06,
                2.23127585741e-06,
                1.68457290783e-06,
                1.60106765179e-06,
            ],
            "median": [
                0.00216405838728,
                0.00101412762888,
                0.000571827986278,
                0.000202020863071,
                0.00016626593424,
                7.9059296695e-05,
                4.09869862779e-05,
                7.25362333469e-05,
                6.23455853201e-05,
                2.26373686019e-05,
                2.20830006583e-05,
                1.21264438349e-05,
                1.26217200886e-05,
                1.92631814571e-05,
                8.60928776092e-06,
                7.06569699105e-06,
                5.57656539968e-06,
                4.13972065871e-06,
                3.38378731612e-06,
                2.46519289249e-06,
                2.28213752962e-06,
                1.57693932579e-06,
                1.32363095418e-06,
                1.03902482351e-06,
                8.31881152408e-07,
                6.98770861618e-07,
                8.20869786367e-07,
                6.86328689881e-07,
                7.16525391908e-07,
                5.77442165195e-07,
                4.89849981022e-07,
                4.82333689433e-07,
                4.68203353421e-07,
                3.40057397352e-07,
                4.30950535701e-07,
                2.96965936286e-07,
                2.67273350119e-07,
                2.37223986232e-07,
                1.66473483887e-07,
                1.67262996342e-07,
            ],
            "min": [
                2.54779412357e-24,
                1.59138993003e-23,
                9.79070021169e-24,
                1.23866726345e-23,
                6.20370194956e-24,
                8.31699838907e-24,
                3.46986464504e-24,
                9.82299065512e-24,
                8.27121290213e-24,
                1.3206985424e-23,
                6.71308718543e-24,
                9.21575629586e-24,
                8.26029664492e-24,
                1.66434662644e-23,
                1.14134225925e-23,
                1.12782867751e-23,
                1.02581649846e-23,
                4.38057261125e-24,
                7.50343667118e-24,
                1.66349386781e-23,
                1.01460639057e-23,
                7.12410659402e-24,
                8.232875051e-24,
                8.27436992347e-24,
                9.45578063691e-24,
                1.17401498467e-23,
                1.32374251998e-23,
                1.02838187413e-23,
                8.10053416782e-24,
                9.78477586629e-24,
                7.83643694738e-24,
                1.7413921467e-23,
                8.2819603431e-24,
                9.63422728455e-24,
                1.12002281997e-23,
                1.24978428576e-23,
                9.36531012399e-24,
                8.62730944791e-24,
                8.31138879917e-24,
                8.07182751947e-24,
            ],
            "dvar2": [
                3.41360646416e-05,
                1.77424899448e-05,
                1.60067065735e-05,
                2.49053100561e-05,
                3.13707209898e-06,
                1.89071258205e-07,
                6.62138788243e-08,
                1.31598056896e-07,
                1.75494065502e-07,
                2.65618265161e-08,
                2.46251730118e-08,
                1.05109352333e-08,
                1.12366285165e-08,
                1.85752213611e-08,
                2.04274308579e-09,
                1.38876810052e-09,
                1.47249523685e-09,
                1.39897116114e-09,
                3.35575400801e-10,
                2.33118274684e-10,
                6.82675915797e-10,
                4.3367004432e-10,
                6.24023721585e-10,
                3.55194929025e-10,
                1.20648546709e-10,
                1.40170333673e-10,
                1.71406291938e-10,
                1.14362519454e-10,
                8.40982630756e-11,
                3.15224409075e-11,
                2.62250204192e-11,
                2.74679064216e-11,
                2.2249901574e-11,
                1.64916916928e-11,
                8.66626423401e-11,
                4.29321578288e-11,
                3.092853057e-11,
                2.80370379691e-11,
                1.48893328222e-11,
                1.55275462627e-11,
            ],
            "dvar": [
                1.25958504213e-05,
                1.19361502584e-05,
                9.92833702185e-06,
                1.01751447801e-05,
                1.10249095542e-06,
                7.38648679999e-08,
                2.46767069001e-08,
                6.59268906134e-08,
                9.28329342287e-08,
                1.36733522282e-08,
                1.33950210923e-08,
                5.78824854713e-09,
                5.27800469996e-09,
                9.34593735735e-09,
                8.1471057678e-10,
                6.13167738805e-10,
                6.4058813809e-10,
                6.15332840237e-10,
                1.42515957369e-10,
                9.12699776867e-11,
                2.86098922331e-10,
                1.81715559266e-10,
                3.07557562751e-10,
                1.68111913279e-10,
                6.04145206085e-11,
                6.53835111053e-11,
                9.30251431441e-11,
                6.14560000112e-11,
                3.70867225818e-11,
                1.61549211508e-11,
                1.28530094554e-11,
                1.32515188059e-11,
                1.05031313508e-11,
                7.0189813163e-12,
                3.6923960417e-11,
                2.02655461856e-11,
                1.42444450332e-11,
                1.33324263088e-11,
                7.44717742335e-12,
                7.52735981185e-12,
            ],
            "dmean": [
                0.00229502934963,
                0.00207201717421,
                0.00148841179907,
                0.00110294274054,
                0.000417607021518,
                0.000138642484671,
                8.079289546e-05,
                0.000131005115691,
                0.000145740428707,
                4.67124446004e-05,
                4.60060946352e-05,
                2.76701957773e-05,
                2.46403087658e-05,
                3.7861718738e-05,
                1.42134094858e-05,
                1.14496015158e-05,
                1.1467674085e-05,
                9.96491326077e-06,
                5.73495390199e-06,
                4.29265037383e-06,
                6.35433207208e-06,
                3.98508973376e-06,
                4.25367534262e-06,
                4.42740110884e-06,
                2.68380449597e-06,
                2.6487989544e-06,
                2.8972558539e-06,
                2.54162932833e-06,
                2.325889227e-06,
                1.69365830516e-06,
                1.53014161697e-06,
                1.59524222454e-06,
                1.51298979745e-06,
                1.31523813707e-06,
                2.27577038459e-06,
                1.79932794708e-06,
                1.64016375948e-06,
                1.49033098751e-06,
                1.15183547678e-06,
                1.07529763227e-06,
            ],
            "max": [
                0.0677252113819,
                0.0378952510655,
                0.0290289148688,
                0.0254700183868,
                0.0128895640373,
                0.00856050662696,
                0.00286759086885,
                0.00716800382361,
                0.00654651876539,
                0.00379466195591,
                0.00477699656039,
                0.00299851293676,
                0.00376314716414,
                0.00523377442732,
                0.000584416906349,
                0.000505314266775,
                0.000558701285627,
                0.0005260904436,
                0.000225639698328,
                0.000301965657854,
                0.000398421369027,
                0.000744273711462,
                0.000533478625584,
                0.000286504684482,
                0.000261136126937,
                0.000268742762273,
                0.000268247065833,
                0.000205783173442,
                0.000170988860191,
                8.68624047143e-05,
                7.21269097994e-05,
                0.000123736841488,
                7.81475755502e-05,
                4.78184156236e-05,
                0.000100603712781,
                8.45123504405e-05,
                5.73282668483e-05,
                9.03406980797e-05,
                5.86272690271e-05,
                5.47057570657e-05,
            ],
            "var": [
                3.04177119688e-05,
                4.68787911814e-05,
                1.81959476322e-05,
                8.42998906592e-06,
                7.75159548994e-07,
                8.02486823659e-08,
                2.52603289397e-08,
                1.48078854068e-07,
                2.24927916292e-07,
                2.99575368956e-08,
                3.19193311782e-08,
                1.40635734169e-08,
                1.02791402057e-08,
                1.71800866866e-08,
                1.16958798202e-09,
                9.48678025026e-10,
                8.88495277351e-10,
                7.94154908501e-10,
                2.05291103561e-10,
                1.29107141644e-10,
                4.27535090664e-10,
                2.93017693442e-10,
                4.73644123922e-10,
                3.10595688058e-10,
                1.52411028242e-10,
                1.2613242395e-10,
                1.5855858293e-10,
                1.29483784805e-10,
                7.99841026744e-11,
                4.30840640941e-11,
                2.8866565388e-11,
                2.75744739542e-11,
                2.49805558183e-11,
                1.52014841687e-11,
                4.94557797492e-11,
                3.47199664852e-11,
                2.74030364372e-11,
                2.57780706503e-11,
                1.54394483309e-11,
                1.40736120674e-11,
            ],
            "mean": [
                0.00421799859032,
                0.00423632748425,
                0.00198257528245,
                0.000957176904194,
                0.000434520887211,
                0.000167578022229,
                9.5177514595e-05,
                0.000210603539017,
                0.00024211226264,
                7.01813114574e-05,
                7.22835611668e-05,
                4.05879654863e-05,
                3.73676557501e-05,
                5.41154404345e-05,
                2.00823214982e-05,
                1.64985340234e-05,
                1.50245914483e-05,
                1.25500946524e-05,
                8.03009515948e-06,
                6.3551688072e-06,
                8.80254174263e-06,
                6.16126408204e-06,
                6.2302265178e-06,
                6.67886297379e-06,
                4.73709496873e-06,
                4.31715216109e-06,
                4.50706738775e-06,
                4.40146368419e-06,
                4.02836485591e-06,
                2.9554471439e-06,
                2.69067754743e-06,
                2.76429022961e-06,
                2.71651742878e-06,
                2.17673414227e-06,
                3.17735816679e-06,
                2.68415283244e-06,
                2.49323466051e-06,
                2.23538791033e-06,
                1.75996285634e-06,
                1.54595397817e-06,
            ],
        },
        "spectral_complexity": {
            "dmean2": 4.85211372375,
            "median": 11,
            "min": 0,
            "dvar2": 21.6541137695,
            "dvar": 12.252790451,
            "dmean": 3.37941265106,
            "max": 43,
            "var": 52.0905303955,
            "mean": 11.7429180145,
        },
        "spectral_energyband_low": {
            "dmean2": 0.0149820484221,
            "median": 0.0116189969704,
            "min": 5.17603856995e-23,
            "dvar2": 0.00048074303777,
            "dvar": 0.000204645111808,
            "dmean": 0.0103260846809,
            "max": 0.209965258837,
            "var": 0.00083902978804,
            "mean": 0.0249669943005,
        },
        "erbbands_crest": {
            "dmean2": 5.19815206528,
            "median": 8.93636989594,
            "min": 2.15950489044,
            "dvar2": 22.1311225891,
            "dvar": 10.3251180649,
            "dmean": 3.33805418015,
            "max": 33.2750091553,
            "var": 31.5032939911,
            "mean": 10.3855142593,
        },
        "silence_rate_30dB": {
            "dmean2": 0.0619035847485,
            "median": 1,
            "min": 0,
            "dvar2": 0.0776470303535,
            "dvar": 0.0299905408174,
            "dmean": 0.0309502612799,
            "max": 1,
            "var": 0.0223165713251,
            "mean": 0.977159261703,
        },
        "barkbands_skewness": {
            "dmean2": 1.44653308392,
            "median": 2.15599632263,
            "min": -4.99289512634,
            "dvar2": 2.84916734695,
            "dvar": 1.25199246407,
            "dmean": 0.951523184776,
            "max": 18.4443359375,
            "var": 5.02271461487,
            "mean": 2.63306665421,
        },
        "barkbands_flatness_db": {
            "dmean2": 0.0515989176929,
            "median": 0.178841590881,
            "min": 0.0207624137402,
            "dvar2": 0.00268978346139,
            "dvar": 0.00151978223585,
            "dmean": 0.0371669046581,
            "max": 0.588909327984,
            "var": 0.00901084393263,
            "mean": 0.189883828163,
        },
        "dynamic_complexity": 3.91312026978,
        "mfcc": {
            "mean": [
                -700.66998291,
                135.489395142,
                28.421787262,
                23.1075897217,
                8.46113491058,
                14.6377601624,
                6.19332933426,
                8.12203979492,
                3.99862265587,
                6.61921405792,
                2.94504475594,
                5.26969957352,
                1.69772768021,
            ],
            "icov": [
                [
                    0.000107419175038,
                    8.07790347608e-05,
                    4.01940233132e-05,
                    1.41072532642e-05,
                    3.17661724694e-05,
                    -7.08577063051e-05,
                    -9.35970456339e-06,
                    -0.000100402481621,
                    0.000114859867608,
                    -9.32187540457e-05,
                    7.22726763343e-05,
                    -7.69687903812e-05,
                    0.00014752781135,
                ],
                [
                    8.07790347608e-05,
                    0.000406299368478,
                    0.000213704173802,
                    3.07764312311e-05,
                    -0.00013552600285,
                    0.000182592688361,
                    0.00012160334154,
                    0.000219605542952,
                    -0.000212934159208,
                    -4.89489757456e-05,
                    1.42763483382e-05,
                    0.000134202622576,
                    0.000169955193996,
                ],
                [
                    4.01940233132e-05,
                    0.000213704173802,
                    0.000946900865529,
                    0.000225221272558,
                    -0.000283261353616,
                    4.93922234455e-05,
                    -0.000170131010236,
                    0.00063844473334,
                    -0.000424777012086,
                    5.35618892172e-06,
                    -3.6012417695e-05,
                    0.000732303771656,
                    -0.000106887986476,
                ],
                [
                    1.41072532642e-05,
                    3.07764312311e-05,
                    0.000225221272558,
                    0.00193217326887,
                    -0.000282399530988,
                    0.000215311141801,
                    2.67071982307e-07,
                    0.00030652814894,
                    0.000730566098355,
                    -0.000450449471828,
                    0.000128779895022,
                    -0.000382249534596,
                    0.00046093785204,
                ],
                [
                    3.17661724694e-05,
                    -0.00013552600285,
                    -0.000283261353616,
                    -0.000282399530988,
                    0.00403568474576,
                    -0.00131330266595,
                    0.000184118820471,
                    6.65877596475e-05,
                    -0.000588293594774,
                    0.0022953953594,
                    -0.00182200025301,
                    0.00155140575953,
                    -0.00139354949351,
                ],
                [
                    -7.08577063051e-05,
                    0.000182592688361,
                    4.93922234455e-05,
                    0.000215311141801,
                    -0.00131330266595,
                    0.00576621666551,
                    -0.00270392908715,
                    -0.000217271212023,
                    -1.60099043569e-05,
                    -7.43252239772e-05,
                    0.00248048570938,
                    -0.00210075359792,
                    -0.000226830641623,
                ],
                [
                    -9.35970456339e-06,
                    0.00012160334154,
                    -0.000170131010236,
                    2.67071982307e-07,
                    0.000184118820471,
                    -0.00270392908715,
                    0.0077719129622,
                    -0.00278255506419,
                    0.000178223548573,
                    0.000260046246694,
                    -0.00136796431616,
                    0.00183528801426,
                    -0.00172712933272,
                ],
                [
                    -0.000100402481621,
                    0.000219605542952,
                    0.00063844473334,
                    0.00030652814894,
                    6.65877596475e-05,
                    -0.000217271212023,
                    -0.00278255506419,
                    0.0108746550977,
                    -0.00441821618006,
                    0.00173715199344,
                    -0.000915269250982,
                    0.000479113426991,
                    -0.000192808452994,
                ],
                [
                    0.000114859867608,
                    -0.000212934159208,
                    -0.000424777012086,
                    0.000730566098355,
                    -0.000588293594774,
                    -1.60099043569e-05,
                    0.000178223548573,
                    -0.00441821618006,
                    0.0135310757905,
                    -0.00714344065636,
                    0.00168303831015,
                    -0.0014511711197,
                    0.00106327817775,
                ],
                [
                    -9.32187540457e-05,
                    -4.89489757456e-05,
                    5.35618892172e-06,
                    -0.000450449471828,
                    0.0022953953594,
                    -7.43252239772e-05,
                    0.000260046246694,
                    0.00173715199344,
                    -0.00714344065636,
                    0.0186834186316,
                    -0.00981860049069,
                    0.00231390399858,
                    -0.00192968046758,
                ],
                [
                    7.22726763343e-05,
                    1.42763483382e-05,
                    -3.6012417695e-05,
                    0.000128779895022,
                    -0.00182200025301,
                    0.00248048570938,
                    -0.00136796431616,
                    -0.000915269250982,
                    0.00168303831015,
                    -0.00981860049069,
                    0.0199193935841,
                    -0.0106352949515,
                    0.00361294415779,
                ],
                [
                    -7.69687903812e-05,
                    0.000134202622576,
                    0.000732303771656,
                    -0.000382249534596,
                    0.00155140575953,
                    -0.00210075359792,
                    0.00183528801426,
                    0.000479113426991,
                    -0.0014511711197,
                    0.00231390399858,
                    -0.0106352949515,
                    0.0232046339661,
                    -0.0109363310039,
                ],
                [
                    0.00014752781135,
                    0.000169955193996,
                    -0.000106887986476,
                    0.00046093785204,
                    -0.00139354949351,
                    -0.000226830641623,
                    -0.00172712933272,
                    -0.000192808452994,
                    0.00106327817775,
                    -0.00192968046758,
                    0.00361294415779,
                    -0.0109363310039,
                    0.0196638610214,
                ],
            ],
            "cov": [
                [
                    12126.3984375,
                    -2809.12280273,
                    -96.7183837891,
                    -85.6093063354,
                    -142.921066284,
                    369.096466064,
                    244.115966797,
                    210.12538147,
                    -83.0255813599,
                    -16.950138092,
                    -48.6743812561,
                    47.0859146118,
                    -9.6302986145,
                ],
                [
                    -2809.12280273,
                    3659.98071289,
                    -633.989257812,
                    86.3335189819,
                    19.0077934265,
                    -274.307128906,
                    -212.871582031,
                    -111.485351562,
                    41.133228302,
                    37.7287139893,
                    31.5271949768,
                    -30.5362644196,
                    -58.9328155518,
                ],
                [
                    -96.7183837891,
                    -633.989257812,
                    1356.80432129,
                    -153.434661865,
                    98.7311553955,
                    48.7583999634,
                    48.0191307068,
                    -39.584274292,
                    21.167137146,
                    -17.4020252228,
                    -33.5541763306,
                    -60.2577857971,
                    -1.63439774513,
                ],
                [
                    -85.6093063354,
                    86.3335189819,
                    -153.434661865,
                    567.058959961,
                    11.0189790726,
                    -40.2089500427,
                    -34.1776237488,
                    -36.5306549072,
                    -42.4389686584,
                    2.53039526939,
                    9.81129646301,
                    8.50957870483,
                    -11.8001689911,
                ],
                [
                    -142.921066284,
                    19.0077934265,
                    98.7311553955,
                    11.0189790726,
                    312.80770874,
                    82.4246749878,
                    31.4400959015,
                    4.76846504211,
                    -5.12284135818,
                    -42.4019470215,
                    -8.1498708725,
                    -10.4739961624,
                    18.9017391205,
                ],
                [
                    369.096466064,
                    -274.307128906,
                    48.7583999634,
                    -40.2089500427,
                    82.4246749878,
                    282.404968262,
                    117.230125427,
                    44.6262054443,
                    0.698531210423,
                    -29.1598892212,
                    -30.9757270813,
                    13.9199514389,
                    31.1762313843,
                ],
                [
                    244.115966797,
                    -212.871582031,
                    48.0191307068,
                    -34.1776237488,
                    31.4400959015,
                    117.230125427,
                    200.592071533,
                    64.6621398926,
                    13.1711330414,
                    -6.63176584244,
                    -2.64764499664,
                    2.61026930809,
                    23.4787330627,
                ],
                [
                    210.12538147,
                    -111.485351562,
                    -39.584274292,
                    -36.5306549072,
                    4.76846504211,
                    44.6262054443,
                    64.6621398926,
                    133.662643433,
                    42.5779304504,
                    5.81215953827,
                    6.34349679947,
                    7.00799417496,
                    8.87113571167,
                ],
                [
                    -83.0255813599,
                    41.133228302,
                    21.167137146,
                    -42.4389686584,
                    -5.12284135818,
                    0.698531210423,
                    13.1711330414,
                    42.5779304504,
                    116.71862793,
                    51.1233444214,
                    23.580953598,
                    11.0932693481,
                    3.13923668861,
                ],
                [
                    -16.950138092,
                    37.7287139893,
                    -17.4020252228,
                    2.53039526939,
                    -42.4019470215,
                    -29.1598892212,
                    -6.63176584244,
                    5.81215953827,
                    51.1233444214,
                    106.574928284,
                    59.1461105347,
                    23.1877384186,
                    5.50347471237,
                ],
                [
                    -48.6743812561,
                    31.5271949768,
                    -33.5541763306,
                    9.81129646301,
                    -8.1498708725,
                    -30.9757270813,
                    -2.64764499664,
                    6.34349679947,
                    23.580953598,
                    59.1461105347,
                    103.604682922,
                    46.4150886536,
                    9.88276195526,
                ],
                [
                    47.0859146118,
                    -30.5362644196,
                    -60.2577857971,
                    8.50957870483,
                    -10.4739961624,
                    13.9199514389,
                    2.61026930809,
                    7.00799417496,
                    11.0932693481,
                    23.1877384186,
                    46.4150886536,
                    85.4878540039,
                    39.7927474976,
                ],
                [
                    -9.6302986145,
                    -58.9328155518,
                    -1.63439774513,
                    -11.8001689911,
                    18.9017391205,
                    31.1762313843,
                    23.4787330627,
                    8.87113571167,
                    3.13923668861,
                    5.50347471237,
                    9.88276195526,
                    39.7927474976,
                    76.2382049561,
                ],
            ],
        },
        "spectral_decrease": {
            "dmean2": 5.08372233199e-09,
            "median": -5.58059509714e-09,
            "min": -5.60004380645e-08,
            "dvar2": 4.32927042758e-17,
            "dvar": 1.93053019642e-17,
            "dmean": 3.32266036907e-09,
            "max": 2.9793302381e-18,
            "var": 7.8599522368e-17,
            "mean": -9.43999456382e-09,
        },
        "barkbands_crest": {
            "dmean2": 5.16687965393,
            "median": 12.3140182495,
            "min": 2.29649472237,
            "dvar2": 21.2763710022,
            "dvar": 8.37867069244,
            "dmean": 3.1435611248,
            "max": 25.9979152679,
            "var": 29.487985611,
            "mean": 13.2201347351,
        },
    },
}


def test_load_extractor():
    path = "tests/resources/mir_datasets/acousticbrainz_genre/acousticbrainz-mediaeval-validation/be/be9e01e5-8f93-494d-bbaa-ddcc5a52f629.json"
    extractor_data = acousticbrainz_genre.load_extractor(path)

    assert type(extractor_data) == dict
    assert extractor_data == features


def test_to_jams(httpserver):
    data_home = "tests/resources/mir_datasets/acousticbrainz_genre"
    trackid = "tagtraum#validation#be9e01e5-8f93-494d-bbaa-ddcc5a52f629#2b6bfcfd-46a5-3f98-a58f-2c51d7c9e960#trance########"

    httpserver.serve_content(
        open(
            "tests/resources/download/acousticbrainz_genre_dataset_little_test.json.zip",
            "rb",
        ).read()
    )
    remote_index = {
        "index": download_utils.RemoteFileMetadata(
            filename="acousticbrainz_genre_dataset_little_test.json.zip",
            url=httpserver.url,
            checksum="c5fbdd4f8b7de383796a34143cb44c4f",
            destination_dir="",
        )
    }
    track = acousticbrainz_genre.Track(
        trackid,
        data_home=data_home,
        remote_index=remote_index,
        remote_index_name="acousticbrainz_genre_dataset_little_test.json",
    )

    jam_ground_truth = jams_utils.jams_converter(
        metadata={
            "features": features,
            "duration": features["metadata"]["audio_properties"]["length"],
        }
    )
    jam = track.to_jams()
    os.remove(
        os.path.join(
            "mirdata",
            "datasets/indexes",
            "acousticbrainz_genre_dataset_little_test.json",
        )
    )

    assert jam_ground_truth == jam


def test_filter_index(httpserver):
    httpserver.serve_content(
        open(
            "tests/resources/download/acousticbrainz_genre_dataset_little_test.json.zip",
            "rb",
        ).read()
    )

    remote_index = {
        "index": download_utils.RemoteFileMetadata(
            filename="acousticbrainz_genre_dataset_little_test.json.zip",
            url=httpserver.url,
            checksum="c5fbdd4f8b7de383796a34143cb44c4f",
            destination_dir="",
        )
    }
    DATA_test = core.LargeData(
        "acousticbrainz_genre_dataset_little_test.json", remote_index=remote_index
    )
    dataset = acousticbrainz_genre.Dataset(index=DATA_test.index)
    index = dataset.load_all_train()
    assert len(index) == 8
    index = dataset.load_all_validation()
    assert len(index) == 8
    index = dataset.load_tagtraum_validation()
    assert len(index) == 2
    index = dataset.load_tagtraum_train()
    assert len(index) == 2
    index = dataset.load_allmusic_validation()
    assert len(index) == 2
    index = dataset.load_lastfm_train()
    assert len(index) == 2
    index = dataset.load_lastfm_validation()
    assert len(index) == 2
    index = dataset.load_discogs_train()
    assert len(index) == 2
    index = dataset.load_discogs_validation()
    assert len(index) == 2
    os.remove(
        os.path.join(
            "mirdata",
            "datasets/indexes",
            "acousticbrainz_genre_dataset_little_test.json",
        )
    )


# TODO Fix this test
@pytest.mark.skip("This isn't working now with the new dataset API")
def test_download(httpserver):

    data_home = "tests/resources/mir_datasets/acousticbrainz_genre2"

    if os.path.exists(data_home):
        shutil.rmtree(data_home)

    # download the full dataset

    httpserver.serve_content(
        open(
            "tests/resources/download/acousticbrainz-mediaeval-features-validation-01.tar.bz2",
            "rb",
        ).read()
    )

    remotes = {
        "validation-01": download_utils.RemoteFileMetadata(
            filename="acousticbrainz-mediaeval-features-validation-01.tar.bz2",
            url=httpserver.url,
            checksum="2cc101d8a6e388ff27048c0d693ae141",
            destination_dir="temp",
        )
    }
    dataset = acousticbrainz_genre.Dataset(data_home)
    dataset.remotes = remotes
    dataset.download(data_home, False, False)

    assert os.path.exists(data_home)
    assert os.path.exists(
        os.path.join(data_home, "acousticbrainz-mediaeval-validation")
    )
    assert os.path.exists(
        os.path.join(data_home, "acousticbrainz-mediaeval-validation", "01")
    )
    assert os.path.exists(
        os.path.join(
            data_home,
            "acousticbrainz-mediaeval-validation",
            "01",
            "01a1a77a-f81e-49a5-9fea-7060394409ec.json",
        )
    )

    httpserver.serve_content(
        open(
            "tests/resources/download/acousticbrainz-mediaeval-features-train-01.tar.bz2",
            "rb",
        ).read()
    )

    remotes = {
        "train-01": download_utils.RemoteFileMetadata(
            filename="acousticbrainz-mediaeval-features-train-01.tar.bz2",
            url=httpserver.url,
            checksum="eb155784e1d4de0f35aa23ded4d34849",
            destination_dir="temp",
        )
    }
    dataset = acousticbrainz_genre.Dataset(data_home)
    dataset.remotes = remotes
    dataset.download(data_home, False, False)

    assert os.path.exists(data_home)
    assert os.path.exists(os.path.join(data_home, "acousticbrainz-mediaeval-train"))
    assert os.path.exists(
        os.path.join(data_home, "acousticbrainz-mediaeval-train", "01")
    )
    assert os.path.exists(
        os.path.join(
            data_home,
            "acousticbrainz-mediaeval-train",
            "01",
            "01a0a332-d340-4806-a88b-cb60a05355c0.json",
        )
    )

    shutil.rmtree(data_home)
