# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/API_Client.ipynb (unless otherwise specified).

__all__ = ["Client"]

# Cell

from typing import *

# Internal Cell

import os

import pandas as pd

# from fastcore.foundation import patch

import captn
from ..logger import get_logger, set_level
from ..helper import get_base_url  # , get_data, post_data

from airt.client import Client as AirtClient

# Internal Cell

logger = get_logger(__name__)

# Internal Cell


def _get_credentials(
    username: Optional[str] = None, password: Optional[str] = None
) -> Tuple[(str, str)]:
    """Returns the username and password.

    If username is **None**, retrive the value from CAPTN_SERVICE_USERNAME environment variable.
    If password is **None**, retrive the value from CAPTN_SERVICE_PASSWORD environment variable.

    Args:
        username: Username for your developer account.
        password: Password for your developer account.

    Returns:
        The username and password as a tuple.

    Raises:
        Key Error, if the environment variables are not set.
    """

    username = (
        username if username is not None else os.environ.get("CAPTN_SERVICE_USERNAME")
    )

    password = (
        password if password is not None else os.environ.get("CAPTN_SERVICE_PASSWORD")
    )

    if not username and not password:
        raise KeyError(
            f"The username and password are neither passed as parameters nor set in the environment variables "
            f"`CAPTN_SERVICE_USERNAME` and `CAPTN_SERVICE_PASSWORD`."
        )

    elif not username:
        raise KeyError(
            f"The username is neither passed as parameter nor set in the environment variable CAPTN_SERVICE_USERNAME."
        )

    elif not password:
        raise KeyError(
            f"The password is neither passed as parameter nor set in the environment variable CAPTN_SERVICE_PASSWORD."
        )

    return (username, password)


# Cell


class Client:
    """A class for authenticating and accessing the captn service.

    Before you can use the service, you must acquire a username and password for your developer account. Please contact us by email **info@captn.ai** to get one.

    The username, password, and server address can either be passed explicitly while calling the `get_token` method or stored
    in environment variables **CAPTN_SERVICE_USERNAME**, **CAPTN_SERVICE_PASSWORD**, and **CAPTN_SERVER_URL**.

    Upon successful authentication, the captn services will be available.
    """

    @classmethod
    def get_token(
        cls, *, username: str = None, password: str = None, server: str = None
    ):
        """Get application token for captn service from a username/password pair.

        This method calls the captn services for validating the supplied username and password. Upon successful
        authentication, an auth token will be returned and will be implicitly used in all the subsequent interactions with the server.

        Args:
            username: Username for your developer account. If not set (default value **None**), it will try to
                use the value from environment variable **CAPTN_SERVICE_USERNAME**.
            password: Password for your developer account. If not set (default value **None**), it will try to
                use the value from environment variable **CAPTN_SERVICE_PASSWORD**.
            server: Server address used to connect to. If not set (default value **None**), it will try to
                use the value from environment variable **CAPTN_SERVER_URL**. If the variable is not set as well,
                then the default public server will be used. You should leave this to default value unless you
                are running your own server (please contact us for that possibility by email info@captn.ai).

        Raises:
            ValueException: If the username/password pair does not match the one for sevice hosted at server.
            ConnectionError: If the server address is invalid or not reachable.
            KeyError: if username or password are None and there are not stored in environment variables

        """

        server = get_base_url(server)
        username, password = _get_credentials(username, password)

        AirtClient.get_token(username=username, password=password, server=server)

    @staticmethod
    def version() -> pd.Series:
        """Return client and server version.

        Returns:
            A pandas series containing the client and server version.

        Raises:
            ConnectionError: If the server address is invalid or not reachable.

        An example to get the client and server version:

        ```python
        Client.version()
        ```
        """
        server_version = AirtClient.version()["airt-service"]
        version = {
            "client version": captn.__version__,
            "server version": server_version,
        }

        return pd.Series(version, name="Version")
