"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
function deploy(stack, props = {}) {
    return new lib_1.KinesisStreamsToKinesisFirehoseToS3(stack, 'test-stream-firehose-s3', props);
}
test('test kinesisFirehose override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
test('test kinesisFirehose.deliveryStreamType override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisFirehoseProps: {
            deliveryStreamType: 'DirectPut'
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        DeliveryStreamType: 'KinesisStreamAsSource'
    });
});
test('test kinesisFirehose.kinesisStreamSourceConfiguration override ', () => {
    const stack = new cdk.Stack();
    const kinesisStream = defaults.buildKinesisStream(stack, {
        existingStreamObj: undefined,
        kinesisStreamProps: undefined
    });
    deploy(stack, {
        kinesisFirehoseProps: {
            kinesisStreamSourceConfiguration: {
                kinesisStreamArn: kinesisStream.streamArn,
                roleArn: new aws_iam_1.Role(stack, 'myRole', {
                    assumedBy: new aws_iam_1.ServicePrincipal('firehose.amazonaws.com')
                }).roleArn
            }
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        KinesisStreamSourceConfiguration: {
            KinesisStreamARN: {
                "Fn::GetAtt": [
                    "teststreamfirehoses3KinesisStream3165E68E",
                    "Arn"
                ]
            },
            RoleARN: {
                "Fn::GetAtt": [
                    "KinesisStreamsRole2BFD39A5",
                    "Arn"
                ]
            }
        }
    });
});
test('test kinesisStreamProps override ', () => {
    const stack = new cdk.Stack();
    deploy(stack, {
        kinesisStreamProps: {
            shardCount: 3
        }
    });
    expect(stack).toHaveResourceLike("AWS::Kinesis::Stream", {
        ShardCount: 3
    });
});
test('Test All properties', () => {
    const stack = new cdk.Stack();
    const construct = deploy(stack);
    expect(construct.cloudwatchAlarms).not.toEqual(undefined);
    expect(construct.kinesisFirehose).not.toEqual(undefined);
    expect(construct.kinesisFirehoseRole).not.toEqual(undefined);
    expect(construct.kinesisFirehoseLogGroup).not.toEqual(undefined);
    expect(construct.kinesisStream).not.toEqual(undefined);
    expect(construct.kinesisStreamRole).not.toEqual(undefined);
    expect(construct.s3Bucket).not.toEqual(undefined);
    expect(construct.s3LoggingBucket).not.toEqual(undefined);
});
test('Test properties with no CW Alarms', () => {
    const stack = new cdk.Stack();
    const construct = deploy(stack, {
        createCloudWatchAlarms: false
    });
    expect(construct.cloudwatchAlarms).toEqual(undefined);
});
test('Test properties with existing S3 bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = defaults.CreateScrapBucket(stack, {});
    const mybucket = s3.Bucket.fromBucketName(stack, 'mybucket', existingBucket.bucketName);
    const construct = deploy(stack, {
        existingBucketObj: mybucket
    });
    expect(construct.s3Bucket).toEqual(undefined);
    expect(construct.s3LoggingBucket).toEqual(undefined);
});
test('Test properties with existing logging S3 bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = defaults.CreateScrapBucket(stack, {});
    const myLoggingBucket = s3.Bucket.fromBucketName(stack, 'myLoggingBucket', existingBucket.bucketName);
    const construct = deploy(stack, {
        existingLoggingBucketObj: myLoggingBucket
    });
    expect(construct.s3Bucket).not.toEqual(undefined);
    expect(construct.s3LoggingBucket).toEqual(undefined);
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new cdk.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.KinesisStreamsToKinesisFirehoseToS3(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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