"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
/**
 * The base class for ScheduledEc2Task and ScheduledFargateTask tasks.
 */
class ScheduledTaskBase extends core_1.Construct {
    /**
     * Constructs a new instance of the ScheduledTaskBase class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredTaskCount !== undefined && props.desiredTaskCount < 1) {
            throw new Error('You must specify a desiredTaskCount greater than 0');
        }
        this.desiredTaskCount = props.desiredTaskCount || 1;
        this.subnetSelection = props.subnetSelection || { subnetType: aws_ec2_1.SubnetType.PRIVATE };
        // An EventRule that describes the event trigger (in this case a scheduled run)
        this.eventRule = new aws_events_1.Rule(this, 'ScheduledEventRule', {
            schedule: props.schedule,
        });
    }
    /**
     * Create an ECS task using the task definition provided and add it to the scheduled event rule.
     *
     * @param taskDefinition the TaskDefinition to add to the event rule
     */
    addTaskDefinitionToEventTarget(taskDefinition) {
        // Use the EcsTask as the target of the EventRule
        const eventRuleTarget = new aws_events_targets_1.EcsTask({
            cluster: this.cluster,
            taskDefinition,
            taskCount: this.desiredTaskCount,
            subnetSelection: this.subnetSelection,
        });
        this.eventRule.addTarget(eventRuleTarget);
        return eventRuleTarget;
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Create an AWS Log Driver with the provided streamPrefix
     *
     * @param prefix the Cloudwatch logging prefix
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.ScheduledTaskBase = ScheduledTaskBase;
//# sourceMappingURL=data:application/json;base64,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