"""Classes to manage the state of nodes within kedro pipelines"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/20_state_mgmt.ipynb.

# %% auto 0
__all__ = ['Flag', 'States']

# %% ../nbs/20_state_mgmt.ipynb 3
import logging


class Flag:
    """
    A class to represent a flag with a state that can be checked, set, or retrieved.

    Attributes:
        name (str): The name of the flag.
        _state (bool or None): The state of the flag. Defaults to None.

    Methods:
        check(): Verifies the state of the flag. Raises an error if not set or if False.
        set(state): Sets the state of the flag.
        get(): Returns the current state of the flag.
    """

    def __init__(self, name, state=None):
        """
        Initializes the flag with a given name and optional initial state.

        Args:
            name (str): The name of the flag.
            state (bool or None): The initial state of the flag. Defaults to None.
        """
        self.name = name
        self._state = state

    def check(self):
        """
        Checks the state of the flag.

        Raises:
            ValueError: If the flag's state is None.
            AssertionError: If the flag's state is False.

        Logs:
            An info message if the flag's state is True.
        """
        if self._state is None:
            raise ValueError(f"Flag '{self.name}' is not set.")
        elif not self._state:  # This checks if the state is False (False, 0, '', etc.)
            raise AssertionError(f"Flag '{self.name}' is False.")
        else:
            logging.info(f"Flag '{self.name}' is True. Everything is good.")

    def set(self, state):
        """
        Sets the state of the flag.

        Args:
            state (bool): The new state to set for the flag.
        """
        self._state = state

    def get(self):
        """
        Retrieves the current state of the flag.

        Returns:
            bool or None: The current state of the flag.
        """
        return self._state

# %% ../nbs/20_state_mgmt.ipynb 4
class States:
    """
    A class to manage the states of nodes within the ETL pipeline.
    The are verified function checks if all the states have been verified and the corresponding nodes has run successfully.
    """

    @staticmethod
    def are_verified(*states: Flag):
        """
        Checks if all the states have been verified.
        """
        return all(state.get() for state in states)
