"""Pre-specified nodes that write data into the database"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/30_ETL_db_writers.ipynb.

# %% auto 0
__all__ = ['logger', 'write_company_name', 'write_categories', 'write_category_level_descriptions', 'write_category_level',
           'write_products', 'write_stores', 'get_region_ids', 'add_region_ids', 'write_skus', 'get_product_ids',
           'get_store_ids', 'write_datapoints', 'write_sales', 'write_prices', 'write_sold_flag',
           'write_SKU_date_specific_data', 'get_sku_ids', 'get_datapoint_ids', 'get_date_ids',
           'write_time_region_features']

# %% ../nbs/30_ETL_db_writers.ipynb 3
import pandas as pd
import psycopg2
import psycopg2.extras
import typing as t
from tqdm import tqdm

import os
import csv
import shutil

from .db_mgmt import get_db_credentials, insert_multi_rows

# %% ../nbs/30_ETL_db_writers.ipynb 5
import logging

logger = logging.getLogger(__name__)


def write_company_name(
    name: str,
    additional_info: t.Dict = None,
    min_date=None,
    max_date=None,
    ignore_company_if_exist: bool = True,
) -> int:
    """
    This function writes the company name to the database and any additional info.
    Each key in `additional_info` becomes a column in the database table if it doesn't exist,
    and the associated value is written to that column.

    If `ignore_company_if_exist` is False and the company name already exists, an error is raised.
    If `ignore_company_if_exist` is True, a warning is logged and the existing record is updated if additional info differs.

    Returns the ID that the database has assigned to the company name.
    """

    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:
                # Try to insert the company name
                cur.execute(
                    """
                    INSERT INTO companies (name, min_date, max_date)
                    VALUES (%s, %s, %s)
                    ON CONFLICT (name) DO NOTHING
                    RETURNING "ID";
                    """,
                    (name, min_date, max_date),
                )
                result = cur.fetchone()

                if result is None:
                    # Company exists, handle based on ignore_company_if_exist flag
                    cur.execute(
                        """
                        SELECT "ID" FROM companies WHERE name = %s;
                        """,
                        (name,),
                    )
                    company_id = cur.fetchone()[0]

                    if not ignore_company_if_exist:
                        raise ValueError(f"Company '{name}' already exists.")

                    logger.warning("Company already exists, ignoring new entry")

                    # Check if additional info needs to be updated
                    if additional_info is not None:
                        for key, value in additional_info.items():
                            # Add column if it doesn't exist
                            cur.execute(
                                f"""
                                ALTER TABLE companies
                                ADD COLUMN IF NOT EXISTS {key} TEXT;
                                """
                            )

                            # Check current value before updating
                            cur.execute(
                                f"""
                                SELECT {key} FROM companies WHERE "ID" = %s;
                                """,
                                (company_id,),
                            )
                            current_value = cur.fetchone()[0]

                            # Only update if the value is different
                            if current_value != value:
                                logger.warning(
                                    f"Overwriting '{key}' for company '{name}' from '{current_value}' to '{value}'."
                                )
                                cur.execute(
                                    f"""
                                    UPDATE companies
                                    SET {key} = %s
                                    WHERE "ID" = %s;
                                    """,
                                    (value, company_id),
                                )
                else:
                    company_id = result[0]

                    # Insert additional information for new entry
                    if additional_info is not None:
                        for key, value in additional_info.items():
                            # Add column if it doesn't exist
                            cur.execute(
                                f"""
                                ALTER TABLE companies
                                ADD COLUMN IF NOT EXISTS {key} TEXT;
                                """
                            )
                            # Update the row with the additional information
                            cur.execute(
                                f"""
                                UPDATE companies
                                SET {key} = %s
                                WHERE "ID" = %s;
                                """,
                                (value, company_id),
                            )

    except Exception as e:
        logger.error(f"Got error while writing company name to database: {e}")
        raise e

    return company_id

# %% ../nbs/30_ETL_db_writers.ipynb 7
def write_categories(
    categories: dict, company_id: int, category_level_names: list = None
) -> t.List[int]:
    """
    This function writes the categories to the database.

    The categories are expected to be in a dictionary with the following structure:
    {1:
        {
            "category_name": None
        }
    2:
        {
            "category_name": ["parent_category_1", "parent_category_2", ...]
        }
    ...
    }

    For the first level, the parent list should be None.

    If a category on a lower level has another parent from 2 or more levels above, the cateogry should be
    listed under the lowest level parent. (such that the write db function can first write all parents and then the children)
    """

    if category_level_names is not None:
        write_category_level_descriptions(category_level_names, company_id)

    for i, level in categories.items():
        write_category_level(level, company_id)

    return True

# %% ../nbs/30_ETL_db_writers.ipynb 9
def write_category_level_descriptions(category_level_names, company_id):
    """
    This function writes the names of the category levels to the database.
    """

    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:
                for i, name in enumerate(category_level_names):
                    cur.execute(
                        """
                        INSERT INTO category_level_descriptions ("companyID", level, name)
                        VALUES (%s, %s, %s)
                        ON CONFLICT ("companyID", level) 
                        DO UPDATE SET name = EXCLUDED.name;
                        """,
                        (company_id, i + 1, name),
                    )
    except Exception as e:
        logger.error(f"Got error while writing category level names to database: {e}")
        raise e


def write_category_level(categories: list, company_id: int) -> t.Dict[str, int]:
    """
    This function writes one level of categories into the database and returns a list of the IDs that the database has assigned.
    The purpose is to call this function repeatedly for each level of categories.

    It will add data to two tables:
     - `categories` with the category names and a flag if it is a leaf category
     - `category_relations` with the parent-child relationships between categories

    """

    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:
                for key, value in categories.items():
                    parents = value
                    cur.execute(
                        """
                        WITH ins AS (
                            INSERT INTO categories ("companyID", name)
                            VALUES (%s, %s)
                            ON CONFLICT ("companyID", name) DO NOTHING
                            RETURNING "ID"
                        )
                        SELECT "ID" FROM ins
                        UNION ALL
                        SELECT "ID" FROM categories
                        WHERE "companyID" = %s AND name = %s
                        LIMIT 1;
                        """,
                        (company_id, key, company_id, key),
                    )
                    category_id = cur.fetchone()[0]

                    if parents is not None:

                        for parent in parents:
                            cur.execute(
                                """
                                SELECT "ID" FROM categories
                                WHERE "companyID" = %s AND name = %s;
                                """,
                                (company_id, parent),
                            )
                            parent_id = cur.fetchone()[0]
                            cur.execute(
                                """
                                INSERT INTO category_relations ("subID", "parentID")
                                VALUES (%s, %s)
                                ON CONFLICT ("subID", "parentID") 
                                DO NOTHING;
                                """,
                                (category_id, parent_id),
                            )
        return

    except Exception as e:
        logger.error(f"Got error while writing category level names to database: {e}")
        raise e

# %% ../nbs/30_ETL_db_writers.ipynb 11
def write_products(products: pd.DataFrame, company_id: int) -> None:
    """
    This function writes the products to the database.

    The input must be a dataframe with the following structure:
    First column: product name (column name is irrelevant)
    Second column: category name (column name is irrelevant)

    Note that each product may have more than one category.

    """

    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:
                cur.execute(
                    """ 
                    SELECT products.name
                    FROM companies 
                    JOIN categories ON companies."ID" = categories."companyID"
                    JOIN product_categories ON product_categories."categoryID" = categories."ID"
                    JOIN products ON product_categories."productID" = products."ID"
                    ;
                    """,
                    (company_id,),
                )

                names = cur.fetchall()
                names = [name[0] for name in names]

                products_filtered = products[~products.iloc[:, 0].isin(names)]

                products_filtered_list = products_filtered.iloc[:, 0].tolist()
                products_filtered_list = list(set(products_filtered_list))

                inserted_ids = []
                for product in products_filtered_list:
                    cur.execute(
                        """
                        INSERT INTO products (name)
                        VALUES (%s)
                        RETURNING "ID";
                        """,
                        (product,),
                    )
                    inserted_id = cur.fetchone()[0]  # Fetch the generated ID
                    inserted_ids.append(inserted_id)

                cur.execute(
                    """ 
                    SELECT categories."ID", categories.name
                    FROM companies 
                    JOIN categories ON companies."ID" = categories."companyID"
                    WHERE companies."ID" = %s;
                    """,
                    (company_id,),
                )

                category_names = cur.fetchall()

                category_names_df = pd.DataFrame(category_names, columns=["ID", "name"])

                products_filtered = products_filtered.merge(
                    category_names_df,
                    left_on=products_filtered.columns[1],
                    right_on="name",
                    how="left",
                )
                products_filtered["product_id"] = inserted_ids

                products_filtered = products_filtered[["product_id", "ID"]]
                products_filtered["product_id"] = products_filtered[
                    "product_id"
                ].astype(int)
                products_filtered["ID"] = products_filtered["ID"].astype(int)

                values_to_insert = [
                    tuple(row) for row in products_filtered.itertuples(index=False)
                ]

                cur.executemany(
                    """
                    INSERT INTO product_categories ("productID", "categoryID")
                    VALUES (%s, %s);
                    """,
                    values_to_insert,  # Use the converted list of tuples
                )

    except Exception as e:
        logger.error(f"Got error while writing products to database: {e}")
        raise e

# %% ../nbs/30_ETL_db_writers.ipynb 13
def write_stores(store_regions: pd.DataFrame, company_id) -> None:
    """
    This function writes the store data to the database.
    """

    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:

                store_regions = add_region_ids(store_regions, cur)

                cur.executemany(
                    """
                    INSERT INTO stores ("name", "regionID", "companyID")
                    VALUES (%s, %s, %s)
                    ON CONFLICT ("name", "companyID") DO NOTHING;
                    """,
                    (
                        store_regions[["name", "region_id"]]
                        .assign(companyID=company_id)
                        .values.tolist()
                    ),
                )

    except Exception as e:
        logger.error(f"Got error while writing stores to database: {e}")
        raise e

# %% ../nbs/30_ETL_db_writers.ipynb 15
def get_region_ids(cur, country, abbreviation, type_):

    cur.execute(
        """
        WITH RECURSIVE RegionHierarchy AS (
            -- Base case: Find the ID of the specified country region
            SELECT "ID", "abbreviation", "type"
            FROM regions
            WHERE "abbreviation" = %s AND "type" = %s

            UNION ALL

            -- Recursive case: Find all regions whose parent_regionID matches the ID of the regions in the previous level
            SELECT r."ID", r."abbreviation", r."type"
            FROM regions r
            INNER JOIN RegionHierarchy rh ON r."parent_regionID" = rh."ID"
        )
        -- Select the desired region with the specified abbreviation and type
        SELECT "ID"
        FROM RegionHierarchy
        WHERE "abbreviation" = %s AND "type" = %s;
        """,
        (country, "country", abbreviation, type_),
    )
    region_id = cur.fetchone()

    return region_id


def add_region_ids(data, cur):
    """
    Adds region IDs to the given DataFrame by mapping region, type, and country.

    Args:
        data (pd.DataFrame): Input DataFrame containing `region`, `type`, and `country` columns.
        cur (psycopg2.cursor): Database cursor for querying region IDs.

    Returns:
        pd.DataFrame: DataFrame with an additional `region_id` column.
    """
    # Extract unique region, type, and country combinations
    region_identifier = data[["region", "type", "country"]].drop_duplicates()

    # Create a mapping for region IDs
    region_id_mapping = []
    for abbreviation, type_, country in region_identifier.itertuples(index=False):
        region_id = get_region_ids(cur, country, abbreviation, type_)

        if region_id:
            region_id = int(region_id[0])
            region_id_mapping.append((abbreviation, type_, country, region_id))
        else:
            # Raise an error if the region ID is not found
            raise ValueError(
                f"No matching region found for abbreviation={abbreviation}, type={type_}, country={country}."
            )

    # Convert mapping to a DataFrame
    region_id_mapping_df = pd.DataFrame(
        region_id_mapping, columns=["region", "type", "country", "region_id"]
    )

    # Merge the region ID mapping back into the original data
    data = data.merge(
        region_id_mapping_df, on=["region", "type", "country"], how="left"
    )

    # Check for any unmatched rows (this should not happen due to the error raised earlier)
    if data["region_id"].isnull().any():
        raise ValueError("Some rows could not be assigned a region_id.")

    return data

# %% ../nbs/30_ETL_db_writers.ipynb 17
def write_skus(store_item_combinations: pd.DataFrame, company_id: int) -> None:
    """
    Writes SKU data to the database.

    For each store-item combination:
    - Fetches the product ID by matching `item_name` in the `products` table, filtering by company.
    - Fetches the store ID by matching `store_name` in the `stores` table, filtering by company.
    - Inserts the resulting combinations into the `sku_table`.

    Args:
        store_item_combinations (pd.DataFrame): DataFrame with columns `store_name` and `item_name`.
        company_id (int): ID of the company for filtering relevant data.

    Raises:
        Exception: If any errors occur during the database operation.
    """
    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:
                # Fetch product IDs
                product_mapping = get_product_ids(
                    cur=cur,
                    company_id=company_id,
                    item_name_list=store_item_combinations["item_name"]
                    .unique()
                    .tolist(),
                )

                # Fetch store IDs
                store_mapping = get_store_ids(
                    cur=cur,
                    company_id=company_id,
                    store_name_list=store_item_combinations["store_name"]
                    .unique()
                    .tolist(),
                )

                # Merge product and store IDs with the input DataFrame
                merged_data = store_item_combinations.merge(
                    product_mapping, on="item_name", how="left"
                ).merge(store_mapping, on="store_name", how="left")

                # Check for unmatched rows
                if merged_data["productID"].isnull().any():
                    unmatched_items = merged_data.loc[
                        merged_data["productID"].isnull(), "item_name"
                    ].unique()
                    raise ValueError(f"Unmatched item_names: {unmatched_items}")
                if merged_data["storeID"].isnull().any():
                    unmatched_stores = merged_data.loc[
                        merged_data["storeID"].isnull(), "store_name"
                    ].unique()
                    raise ValueError(f"Unmatched store_names: {unmatched_stores}")

                # Prepare data for insertion
                sku_data = (
                    merged_data[["productID", "storeID"]]
                    .drop_duplicates()
                    .values.tolist()
                )

                # Insert data into the SKU table
                cur.executemany(
                    """
                    INSERT INTO sku_table ("productID", "storeID")
                    VALUES (%s, %s)
                    ON CONFLICT ("productID", "storeID") DO NOTHING;
                    """,
                    sku_data,
                )
                conn.commit()

    except Exception as e:
        logger.error(f"Error while writing SKUs to the database: {e}")
        raise e

# %% ../nbs/30_ETL_db_writers.ipynb 19
def get_product_ids(cur, company_id, item_name_list):
    """
    Fetch product IDs for a given company and a list of item names.

    Args:
        cur (psycopg2.cursor): Database cursor for executing SQL commands.
        company_id (int): ID of the company.
        item_name_list (list): List of item names.

    Returns:
        pd.DataFrame: A DataFrame mapping product IDs to item names.
    """
    cur.execute(
        """
        SELECT products."ID" AS productID, products.name AS item_name
        FROM products
        JOIN product_categories ON products."ID" = product_categories."productID"
        JOIN categories ON product_categories."categoryID" = categories."ID"
        WHERE categories."companyID" = %s AND products.name = ANY(%s);
        """,
        (company_id, item_name_list),
    )
    product_mapping = cur.fetchall()
    return pd.DataFrame(product_mapping, columns=["productID", "item_name"])


def get_store_ids(cur, company_id, store_name_list):
    """
    Fetch store IDs for a given company and a list of store names.

    Args:
        cur (psycopg2.cursor): Database cursor for executing SQL commands.
        company_id (int): ID of the company.
        store_name_list (list): List of store names.

    Returns:
        pd.DataFrame: A DataFrame mapping store IDs to store names.
    """
    cur.execute(
        """
        SELECT "ID", name 
        FROM stores 
        WHERE "companyID" = %s AND name = ANY(%s);
        """,
        (company_id, list(store_name_list)),
    )
    store_id_mapping = cur.fetchall()
    return pd.DataFrame(store_id_mapping, columns=["storeID", "store_name"])

# %% ../nbs/30_ETL_db_writers.ipynb 21
def write_datapoints(sales: pd.DataFrame, company_id: int) -> None:
    """
    Writes datapoints to the `datapoints` table in the database.

    The datapoints consist of `skuID` and `dateID`, resolved based on `store_name`, `item_name`, and `date`.

    Args:
        sales (pd.DataFrame): DataFrame containing `store_name`, `item_name`, `date`, and additional data.
        company_id (int): ID of the company for filtering relevant data.

    Raises:
        ValueError: If any `store_name`, `item_name`, or `date` cannot be matched or if duplicate rows are found.
    """
    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:
                # Step 1: Resolve SKU IDs
                store_product_names = sales[
                    ["store_name", "item_name"]
                ].drop_duplicates()
                sku_ids = get_sku_ids(cur, store_product_names, company_id)

                # Merge SKU IDs into the sales DataFrame based on `store_name` and `item_name`
                sales = sales.merge(sku_ids, on=["store_name", "item_name"], how="left")

                # Step 2: Resolve Date IDs
                unique_dates = sales["date"].drop_duplicates().tolist()
                date_ids = get_date_ids(cur, unique_dates)
                date_ids["date"] = pd.to_datetime(date_ids["date"], errors="coerce")

                # Merge Date IDs into the sales DataFrame
                sales["date"] = pd.to_datetime(sales["date"], errors="coerce")
                sales = sales.merge(date_ids, on="date", how="left")

                # Check for unmatched rows
                if sales["skuID"].isnull().any():
                    unmatched_skus = sales.loc[
                        sales["skuID"].isnull(), ["store_name", "item_name"]
                    ].drop_duplicates()
                    raise ValueError(
                        f"Unmatched SKUs: {unmatched_skus.to_dict(orient='records')}"
                    )
                if sales["dateID"].isnull().any():
                    unmatched_dates = sales.loc[
                        sales["dateID"].isnull(), "date"
                    ].unique()
                    raise ValueError(f"Unmatched dates: {unmatched_dates}")

                # Check for duplicate rows in `skuID` and `dateID`
                if sales[["skuID", "dateID"]].duplicated().any():
                    duplicate_rows = sales[
                        sales[["skuID", "dateID"]].duplicated(keep=False)
                    ]
                    raise ValueError(
                        f"Duplicate rows found in the data: {duplicate_rows}"
                    )

                datapoints_data = sales[["skuID", "dateID"]]
                datapoints_data = datapoints_data.iloc[:1000]

                # Insert data into the `datapoints` table in batches
                datapoint_ids = insert_multi_rows(
                    data_to_insert=datapoints_data,
                    table_name="datapoints",
                    column_names=["skuID", "dateID"],
                    types=[int, int],
                    cur=cur,
                    conn=conn,
                    return_with_ids=True,
                    unique_columns=["skuID", "dateID"],
                )
        return datapoint_ids

    except Exception as e:
        logger.error(f"Error while writing datapoints to the database: {e}")
        raise e

# %% ../nbs/30_ETL_db_writers.ipynb 23
def write_sales(sales: pd.DataFrame, company_id, datapoint_ids) -> None:
    """
    This function writes the sales data to the database.

    """

    write_SKU_date_specific_data(
        data=sales,
        datapoint_ids=datapoint_ids,
        variable_name="sales",
        variable_type=float,
        table_name="sales",
        company_id=company_id,
    )


def write_prices(prices: pd.DataFrame, company_id, datapoint_ids) -> None:
    """
    This function writes the prices data to the database.
    """

    write_SKU_date_specific_data(
        data=prices,
        datapoint_ids=datapoint_ids,
        variable_name="price",
        variable_type=float,
        table_name="prices",
        company_id=company_id,
    )


def write_sold_flag(sold_flags: pd.DataFrame, company_id, datapoint_ids) -> None:
    """
    This function writes the sold flag data to the database.
    """

    write_SKU_date_specific_data(
        data=sold_flags,
        datapoint_ids=datapoint_ids,
        variable_name="name",
        variable_type=str,
        table_name="flags",
        company_id=company_id,
        name_in_df="sold_flag",
    )

# %% ../nbs/30_ETL_db_writers.ipynb 24
def write_SKU_date_specific_data(
    data: pd.DataFrame,
    datapoint_ids,
    variable_name: str,
    variable_type: callable,
    table_name: str,
    company_id: int,
    name_in_df=None,
) -> None:
    """
    Writes SKU and date-specific data to the database using the new `datapointID` schema.

    Args:
        data (pd.DataFrame): Input data containing `item_name`, `store_name`, `date`, and the variable to insert.
        variable_name (str): The name of the variable/column to be inserted into the database.
        variable_type (callable): The type to cast the variable's values (e.g., int, float, str).
        table_name (str): The name of the database table to insert into.
        company_id (int): The company ID for filtering relevant records.
        name_in_df (str, optional): Column name in the DataFrame for the variable. Defaults to `variable_name`.
    """
    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:
                # Fetch `skuID` mappings
                logger.info("-- in write SKU date specific data -- getting sku IDs")
                store_product_names = data[
                    ["store_name", "item_name"]
                ].drop_duplicates()
                sku_mapping = get_sku_ids(cur, store_product_names, company_id)

                # Fetch `dateID` mappings
                logger.info("-- in write SKU date specific data -- getting date IDs")
                unique_dates = data["date"].drop_duplicates()
                date_mapping = get_date_ids(cur, unique_dates)
                date_mapping["date"] = pd.to_datetime(
                    date_mapping["date"], errors="coerce"
                )

                # Merge `skuID` and `dateID` into the input data
                logger.info(
                    "-- in write SKU date specific data -- merging sku IDs and date IDs"
                )
                data = data.merge(
                    sku_mapping, on=["store_name", "item_name"], how="left"
                )
                data["date"] = pd.to_datetime(data["date"], errors="coerce")
                data = data.merge(date_mapping, on="date", how="left")
                data.drop(columns=["store_name", "item_name", "date"], inplace=True)

                # show ram usage of data
                logger.info(
                    f"-- in write SKU date specific data -- Memory usage of data: {data.memory_usage().sum() / 1024 / 1024 ** 2:.2f} GB"
                )

                # Check for unmatched mappings
                if data["skuID"].isnull().any():
                    unmatched_skus = data.loc[
                        data["skuID"].isnull(), ["store_name", "item_name"]
                    ].drop_duplicates()
                    raise ValueError(
                        f"Unmatched SKUs: {unmatched_skus.to_dict(orient='records')}"
                    )
                if data["dateID"].isnull().any():
                    unmatched_dates = data.loc[data["dateID"].isnull(), "date"].unique()
                    raise ValueError(f"Unmatched dates: {unmatched_dates}")

                logger.info(
                    "-- in write SKU date specific data -- getting checking for duplicates"
                )
                # Fetch `datapointID` for `skuID` and `dateID` combinations
                datapoint_combinations = data[["skuID", "dateID"]]

                # Check for duplicate combinations of `skuID` and `dateID`
                if datapoint_combinations.duplicated().any():
                    duplicate_rows = datapoint_combinations[
                        datapoint_combinations.duplicated(keep=False)
                    ]
                    raise ValueError(
                        f"Duplicate rows found in the data: {duplicate_rows}"
                    )
                data.drop(columns=["storeID", "productID"], inplace=True)

                # Merge `datapointID` into the input data
                # rename column ID to datapointID in the datapoint_IDs
                logger.info(
                    "-- in write SKU date specific data -- merging datapoint IDs"
                )
                datapoint_ids = datapoint_ids.rename(columns={"ID": "datapointID"})
                data = data.merge(datapoint_ids, on=["skuID", "dateID"], how="left")

                # Check for unmatched `datapointID`
                logger.info(
                    "-- in write SKU date specific data -- checking for unmatched datapoints"
                )
                # if data["datapointID"].isnull().any():
                #     # unmatched_datapoints = data.loc[data["datapointID"].isnull(), ["skuID", "dateID"]].drop_duplicates()
                #     # raise ValueError(f"Unmatched datapoints: {unmatched_datapoints.to_dict(orient='records')}")
                #     raise ValueError(f"Unmatched datapoints")
                data.drop(columns=["skuID", "dateID"], inplace=True)

                # Prepare data for insertion
                logger.info(
                    "-- in write SKU date specific data -- preparing data for insertion"
                )
                if name_in_df is None:
                    name_in_df = variable_name
                data_to_write = data[["datapointID", name_in_df]].copy()
                data_to_write[name_in_df] = data_to_write[name_in_df].astype(
                    variable_type
                )

                # Insert data into the specified table
                logger.info("-- in write SKU date specific data -- inserting data")
                insert_multi_rows(
                    data_to_insert=data_to_write,
                    table_name=table_name,
                    column_names=["datapointID", variable_name],
                    types=[int, variable_type],
                    cur=cur,
                    conn=conn,
                )

    except Exception as e:
        logger.error(f"Error while writing {variable_name} data to the database: {e}")
        raise e

# %% ../nbs/30_ETL_db_writers.ipynb 26
def get_sku_ids(
    cur, store_product_names: pd.DataFrame, company_id: int
) -> pd.DataFrame:
    """
    Fetch skuIDs for given combinations of `store_name` and `item_name`.

    Args:
        cur (psycopg2.cursor): Database cursor for executing SQL commands.
        store_product_names (pd.DataFrame): DataFrame with two columns: `store_name` and `item_name`.
        company_id (int): ID of the company for filtering relevant data.

    Returns:
        pd.DataFrame: DataFrame mapping `skuID`, `storeID`, `productID`, `store_name`, and `item_name`.
    """
    # Step 1: Resolve store IDs
    store_ids = get_store_ids(
        cur=cur,
        company_id=company_id,
        store_name_list=store_product_names["store_name"].unique().tolist(),
    )

    # Step 2: Resolve product IDs
    product_ids = get_product_ids(
        cur=cur,
        company_id=company_id,
        item_name_list=store_product_names["item_name"].unique().tolist(),
    )

    # Step 3: Merge store and product IDs with input DataFrame
    store_product_ids = store_product_names.merge(
        store_ids, on="store_name", how="left"
    )
    store_product_ids = store_product_ids.merge(product_ids, on="item_name", how="left")

    # Check for unmatched rows
    if store_product_ids["storeID"].isnull().any():
        unmatched_stores = store_product_ids.loc[
            store_product_ids["storeID"].isnull(), "store_name"
        ].unique()
        raise ValueError(f"Unmatched store names: {unmatched_stores}")
    if store_product_ids["productID"].isnull().any():
        unmatched_products = store_product_ids.loc[
            store_product_ids["productID"].isnull(), "item_name"
        ].unique()
        raise ValueError(f"Unmatched product names: {unmatched_products}")

    # Step 4: Use a temporary table for efficient querying
    temp_table_name = "temp_store_product"
    temp_data = store_product_ids[["storeID", "productID"]]

    # Create temporary table
    cur.execute(
        f"""
        CREATE TEMP TABLE {temp_table_name} (
            storeID INT,
            productID INT
        ) ON COMMIT DROP;
    """
    )

    # Insert data into the temporary table
    psycopg2.extras.execute_batch(
        cur,
        f"""
        INSERT INTO {temp_table_name} (storeID, productID)
        VALUES (%s, %s);
        """,
        temp_data.values.tolist(),
    )

    # Query sku_table using a JOIN
    cur.execute(
        f"""
        SELECT sku_table."ID" AS skuID, sku_table."storeID", sku_table."productID"
        FROM sku_table
        INNER JOIN {temp_table_name}
        ON sku_table."storeID" = {temp_table_name}.storeID
        AND sku_table."productID" = {temp_table_name}.productID;
    """
    )

    # Fetch and return results
    result = cur.fetchall()
    sku_df = pd.DataFrame(result, columns=["skuID", "storeID", "productID"])

    # Merge the original store_name and item_name back into the results
    final_result = sku_df.merge(
        store_product_ids, on=["storeID", "productID"], how="left"
    )

    return final_result[["skuID", "storeID", "productID", "store_name", "item_name"]]


def get_datapoint_ids(cur, datapoint_combinations: pd.DataFrame) -> pd.DataFrame:
    """
    Fetch `datapointID` for given combinations of `skuID` and `dateID`.

    Args:
        cur (psycopg2.cursor): Database cursor for executing SQL commands.
        datapoint_combinations (pd.DataFrame): DataFrame with two columns: `skuID` and `dateID`.

    Returns:
        pd.DataFrame: DataFrame mapping `datapointID`, `skuID`, and `dateID`.
    """
    # Check for valid columns
    if not {"skuID", "dateID"}.issubset(datapoint_combinations.columns):
        raise ValueError("Input DataFrame must contain 'skuID' and 'dateID' columns.")

    # Convert combinations to a list of tuples for use in the query
    combinations_list = (
        datapoint_combinations[["skuID", "dateID"]].drop_duplicates().values.tolist()
    )

    try:
        # Create a temporary table to store the combinations
        cur.execute(
            """
            CREATE TEMP TABLE temp_datapoints (
                "skuID" INTEGER,
                "dateID" INTEGER
            ) ON COMMIT DROP;
        """
        )

        logger.info("Adding into table for temp_datapoints")
        # Insert the combinations into the temporary table
        psycopg2.extras.execute_values(
            cur,
            """
            INSERT INTO temp_datapoints ("skuID", "dateID")
            VALUES %s;
            """,
            combinations_list,
        )

        # Query for datapointIDs
        cur.execute(
            """
            SELECT d."ID", d."skuID", d."dateID"
            FROM datapoints d
            INNER JOIN temp_datapoints t
            ON d."skuID" = t."skuID" AND d."dateID" = t."dateID";
        """
        )

        # Fetch results and return as a DataFrame
        result = cur.fetchall()
        return pd.DataFrame(result, columns=["datapointID", "skuID", "dateID"])

    except Exception as e:
        logger.error(f"Error while fetching datapoint IDs: {e}")
        raise e


def get_date_ids(cur, dates_list):
    """
    Fetch date IDs for a given list of dates.

    Args:
        cur (psycopg2.cursor): Database cursor for executing SQL commands.
        dates_list (list): List of dates.

    Returns:
        pd.DataFrame: A DataFrame mapping date IDs to dates.
    """
    cur.execute(
        """
        SELECT "ID", date 
        FROM dates 
        WHERE date = ANY(%s::date[]);
        """,
        (list(dates_list),),
    )
    date_id_mapping = cur.fetchall()
    return pd.DataFrame(date_id_mapping, columns=["dateID", "date"])

# %% ../nbs/30_ETL_db_writers.ipynb 28
def write_time_region_features(
    time_region_features: pd.DataFrame,
    name_description: [str, str],  # containing name and description of the feature
    company_id: int,
):
    """

    This function writes data into the database whose values are specific to a
    time-stamps and regions

    """

    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:

                # add name and description to the time_region_features_description table
                cur.execute(
                    """
                    WITH inserted AS (
                        INSERT INTO time_region_features_description ("name", "description")
                        VALUES (%s, %s)
                        ON CONFLICT ("name") DO NOTHING
                        RETURNING "ID"
                    )
                    SELECT "ID" FROM inserted
                    UNION ALL
                    SELECT "ID" FROM time_region_features_description WHERE "name" = %s;
                    """,
                    (name_description[0], name_description[1], name_description[0]),
                )

                feature_id = cur.fetchone()[0]

                time_region_features["trfID"] = feature_id

                # add link to company to the time_region_features_description table
                cur.execute(
                    """
                    INSERT INTO time_region_features_company ("companyID", "trfID")
                    VALUES (%s, %s)
                    ON CONFLICT DO NOTHING;
                    """,
                    (company_id, feature_id),
                )

                # add features to the time_region_features table
                time_region_features = add_region_ids(time_region_features, cur)

                # add date features
                time_region_features["date"] = pd.to_datetime(
                    time_region_features["date"], errors="coerce"
                )
                date_ids = get_date_ids(cur, time_region_features["date"].unique())
                date_ids["date"] = pd.to_datetime(date_ids["date"], errors="coerce")
                time_region_features = time_region_features.merge(
                    date_ids, on="date", how="left"
                )

                cur.executemany(
                    """
                    INSERT INTO time_region_features ("dateID", "regionID", "trfID", "value")
                    VALUES (%s, %s, %s, %s)
                    ON CONFLICT ("dateID", "regionID", "trfID") DO NOTHING;
                    """,
                    (
                        time_region_features[
                            ["dateID", "region_id", "trfID", "feature_value"]
                        ].values.tolist()
                    ),
                )

                conn.commit()

    except Exception as e:
        logger.error(f"Got error while writing time region features to database: {e}")
        raise e
