"""Helper functions to retrieve specific data from a database"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/11_db_retrievers.ipynb.

# %% auto 0
__all__ = ['get_company_id', 'get_date_id', 'get_norm_param', 'get_norm_param_by_company']

# %% ../nbs/11_db_retrievers.ipynb 3
from kedro.config import OmegaConfigLoader
from kedro.framework.project import settings
from pathlib import Path

import pandas as pd
import numpy as np
from tqdm import tqdm
import typing as t


import psycopg2

# %% ../nbs/11_db_retrievers.ipynb 5
def get_company_id(company_name: str, credentials_con: str):
    """
    Function to get the company_id from the company_name

    Args:
        company_name (str): The company name
        credentials_con (str): The credentials connection string

    """

    try:
        with psycopg2.connect(credentials_con) as conn:
            with conn.cursor() as cur:
                cur.execute(
                    """
                    SELECT "ID"
                    FROM companies
                    WHERE name = %s
                    """,
                    (company_name,),
                )
                # assert that only one company_id is returned
                if cur.rowcount < 1:
                    raise ValueError("Company name not found")
                elif cur.rowcount > 1:
                    raise ValueError("Multiple company names found")
                company_id = cur.fetchone()[0]
        return company_id
    except Exception as e:
        print(e)
        raise e

# %% ../nbs/11_db_retrievers.ipynb 6
def get_date_id(date: str, credentials_con: str):
    """
    Function to get the date_id from a date

    Args:
        date (str): A date
        credentials_con (str): The credentials connection string

    """

    try:
        with psycopg2.connect(credentials_con) as conn:
            with conn.cursor() as cur:
                cur.execute(
                    """
                    SELECT "ID"
                    FROM dates
                    WHERE date = %s
                    """,
                    (date,),
                )
                # assert that only one company_id is returned
                if cur.rowcount < 1:
                    raise ValueError("Date not found")
                elif cur.rowcount > 1:
                    raise ValueError("Multiple dates found")
                company_id = cur.fetchone()[0]
        return company_id

    except Exception as e:
        print(e)
        raise e

# %% ../nbs/11_db_retrievers.ipynb 7
def get_norm_param(
    table_name: str,
    key_columns: list,
    key_values: list,
    normalization_type: str,
    credentials: str,
    additional_columns: list = None,
):
    """
    Retrieves normalization parameters from a database table.

    Args:
        table_name (str): The name of the table containing normalization parameters.
        key_columns (list): List of column names used as keys for filtering the data.
        key_values (list): List of values corresponding to the key columns.
        normalization_type (str): Type of normalization ('standardize' or 'scale').
        credentials (str): Database connection string.
        additional_columns (list, optional): Additional columns to include in the result.

    Returns:
        dict: A dictionary containing the normalization parameters (and additional columns if specified).

    Raises:
        ValueError: If the normalization type is not recognized.
        psycopg2.DatabaseError: If there is a database error during execution.
    """

    try:
        # Determine the columns to fetch based on normalization type
        if normalization_type == "standardize":
            value_columns = ["mean", "std"]
        elif normalization_type == "scale":
            value_columns = ["min", "max"]
        elif normalization_type == ["standardize", "scale"]:
            value_columns = ["mean", "std", "min", "max"]
        elif normalization_type == ["scale", "standardize"]:
            value_columns = ["min", "max", "mean", "std"]
        else:
            raise ValueError("Normalization type not recognized")

        # Include additional columns if specified
        if additional_columns:
            value_columns = additional_columns + value_columns

        # Build the query string for fetching data
        quoted_key_columns = [f'"{col}"' for col in key_columns]
        key_conditions = " AND ".join([f"{col} = %s" for col in quoted_key_columns])
        quoted_table_name = f'"{table_name}"'
        query = f"""
            SELECT {', '.join([f'"{col}"' for col in value_columns])}
            FROM {quoted_table_name}
            WHERE {key_conditions};
        """

        # Connect to the database and execute the query
        with psycopg2.connect(credentials) as conn:
            with conn.cursor() as cur:
                cur.execute(query, key_values)
                if additional_columns:
                    result = cur.fetchall()
                else:
                    result = cur.fetchone()

        # Check if the result is empty
        if not result:
            raise ValueError(
                "No matching normalization parameters found in the database."
            )

        # Map the result to a dictionary
        if additional_columns:
            return {
                row[0]: dict(
                    zip(
                        value_columns[len(additional_columns) :],
                        row[len(additional_columns) :],
                    )
                )
                for row in result
            }
        else:
            return dict(zip(value_columns, result))

    except psycopg2.DatabaseError as e:
        raise RuntimeError(f"Database error occurred: {e}") from e
    except Exception as e:
        raise RuntimeError(f"An error occurred: {e}") from e

# %% ../nbs/11_db_retrievers.ipynb 8
def get_norm_param_by_company(
    table_name: str,
    key_columns: list,
    key_values: list,
    normalization_type: str,
    credentials: str,
    company_filter: t.List[int],
    additional_columns: list = None,
):
    """
    Retrieves normalization parameters filtered by company IDs from a database table.

    Args:
        table_name (str): The name of the table containing normalization parameters.
        key_columns (list): List of column names used as keys for filtering the data.
        key_values (list): List of values corresponding to the key columns.
        normalization_type (str): Type of normalization ('standardize' or 'scale').
        credentials (str): Database connection string.
        company_filter (list[int]): List of company IDs to filter the SKUs.
        additional_columns (list, optional): Additional columns to include in the result.

    Returns:
        dict: A dictionary containing the normalization parameters (and additional columns if specified).

    Raises:
        ValueError: If the normalization type is not recognized or required columns are missing.
        psycopg2.DatabaseError: If there is a database error during execution.
    """
    try:
        # Ensure "skuID" is included in additional columns
        if additional_columns is None or "skuID" not in additional_columns:
            raise ValueError(
                '"skuID" must be included in additional_columns for company filtering.'
            )

        # Determine the columns to fetch based on normalization type
        if normalization_type == "standardize":
            value_columns = ["mean", "std"]
        elif normalization_type == "scale":
            value_columns = ["min", "max"]
        else:
            raise ValueError("Normalization type not recognized")

        # Include additional columns
        value_columns = additional_columns + value_columns

        # Build the query string for fetching data
        quoted_key_columns = [f'"{col}"' for col in key_columns]
        key_conditions = " AND ".join([f"{col} = %s" for col in quoted_key_columns])
        quoted_table_name = f'"{table_name}"'

        # Create the company filter condition via a join with sku_table and stores
        company_filter_str = ", ".join(map(str, company_filter))
        company_filter_condition = f"""
            "skuID" IN (
                SELECT s."ID"
                FROM "sku_table" s
                JOIN "stores" st ON s."storeID" = st."ID"
                WHERE st."companyID" IN ({company_filter_str})
            )
        """

        query = f"""
            SELECT {', '.join([f'"{col}"' for col in value_columns])}
            FROM {quoted_table_name}
            WHERE {key_conditions} AND {company_filter_condition};
        """

        # Connect to the database and execute the query
        with psycopg2.connect(credentials) as conn:
            with conn.cursor() as cur:
                cur.execute(query, key_values)
                result = cur.fetchall()

        # Check if the result is empty
        if not result:
            raise ValueError(
                "No matching normalization parameters found in the database."
            )

        # Map the result to a dictionary
        return {
            row[0]: dict(
                zip(
                    value_columns[len(additional_columns) :],
                    row[len(additional_columns) :],
                )
            )
            for row in result
        }

    except psycopg2.DatabaseError as e:
        raise RuntimeError(f"Database error occurred: {e}") from e
    except Exception as e:
        raise RuntimeError(f"An error occurred: {e}") from e
