"""Additional datasets for kedro pipelines"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/50_custom_datasets.ipynb.

# %% auto 0
__all__ = ['logger', 'AddRowDataset', 'DynamicPathJSONDataset']

# %% ../nbs/50_custom_datasets.ipynb 3
from kedro.io import AbstractDataset

from kedro.config import OmegaConfigLoader
from pathlib import Path

from kedro.framework.project import settings

import psycopg2
from psycopg2 import sql

import typing as t

import pandas as pd
import numpy as np

import json

# %% ../nbs/50_custom_datasets.ipynb 5
import logging

logger = logging.getLogger(__name__)


class AddRowDataset(AbstractDataset):
    """
    Adds or update one row to a SQL table, if it does not exist.

    """

    def __init__(
        self,
        table: str,
        column_names: t.List,
        unique_columns: t.List,
        db_credentials: t.Dict,
        credentials: str = None,
        load_args=None,
        save_args=None,
    ):

        self.unique_columns = unique_columns
        self.table = table
        self.column_names = column_names
        self.db_credentials = db_credentials
        self.save_args = save_args or {}
        self.load_args = load_args or {}

    def _describe(self) -> t.Dict[str, t.Any]:
        """Returns a dict that describes the attributes of the dataset."""
        return dict(
            table=self.table,
            column_names=self.column_names,
        )

    def _load(self) -> pd.DataFrame:
        """
        Get all entries from all specified columns in the table.
        """

        return_all_columns = self.load_args.get("return_all_columns", False)

        try:
            with psycopg2.connect(self.db_credentials["con"]) as conn:
                with conn.cursor() as cursor:

                    if return_all_columns:

                        # Fetch all rows
                        cursor.execute(f"SELECT * FROM {self.table}")
                        data = cursor.fetchall()

                        # Fetch column names in the correct order from the database
                        cursor.execute(
                            """
                            SELECT a.attname
                            FROM pg_attribute a
                            JOIN pg_class c ON a.attrelid = c.oid
                            JOIN pg_namespace n ON c.relnamespace = n.oid
                            WHERE c.relname = %s AND a.attnum > 0 AND NOT a.attisdropped
                            ORDER BY a.attnum
                            """,
                            (self.table,),
                        )
                        columns = [row[0] for row in cursor.fetchall()]

                    else:
                        # Use the specified column names
                        columns = self.column_names
                        cursor.execute(
                            f"SELECT {','.join(self.column_names)} FROM {self.table}"
                        )
                        data = cursor.fetchall()

            # Create a DataFrame with the data and columns
            data = pd.DataFrame(data, columns=columns)

        except Exception as e:
            logger.error(f"Error loading data from table {self.table}: {e}")
            raise e

        return data

    def _save(self, data: pd.DataFrame) -> None:
        """
        Save the data to the table.

        If unique constraints are violated, updates the conflicting rows with new values for non-conflicting columns.
        """

        verbose = self.save_args.get("verbose", 1)

        try:
            # Connect to the database
            with psycopg2.connect(self.db_credentials["con"]) as conn:
                with conn.cursor() as cursor:
                    # Prepare data insertion
                    for _, row in data.iterrows():
                        # Ensure all data is properly converted to standard Python types
                        row_data = tuple(
                            (
                                row[col].item()
                                if isinstance(row[col], (np.generic, np.ndarray))
                                else row[col]
                            )
                            for col in self.column_names
                        )

                        # Determine the update clause (exclude unique columns)
                        updatable_columns = [
                            col
                            for col in self.column_names
                            if col not in self.unique_columns
                        ]

                        # Only create an update clause if there are columns to update
                        if updatable_columns:
                            update_clause = sql.SQL(", ").join(
                                sql.SQL("{} = EXCLUDED.{}").format(
                                    sql.Identifier(col), sql.Identifier(col)
                                )
                                for col in updatable_columns
                            )

                            # Specify the conflict resolution based on unique columns
                            conflict_clause = sql.SQL(", ").join(
                                sql.Identifier(col) for col in self.unique_columns
                            )

                            # Build the SQL query dynamically
                            query = sql.SQL(
                                """
                                INSERT INTO {table} ({columns})
                                VALUES ({values})
                                ON CONFLICT ({conflict_clause}) DO UPDATE SET
                                    {update_clause}
                                RETURNING xmax = 0 AS is_inserted
                            """
                            ).format(
                                table=sql.Identifier(self.table),
                                columns=sql.SQL(", ").join(
                                    sql.Identifier(col) for col in self.column_names
                                ),
                                values=sql.SQL(", ").join(
                                    sql.Placeholder() for _ in self.column_names
                                ),
                                conflict_clause=conflict_clause,
                                update_clause=update_clause,
                            )
                        else:
                            # Build the SQL query for insertion without an update clause
                            query = sql.SQL(
                                """
                                INSERT INTO {table} ({columns})
                                VALUES ({values})
                                ON CONFLICT DO NOTHING
                                RETURNING xmax = 0 AS is_inserted
                            """
                            ).format(
                                table=sql.Identifier(self.table),
                                columns=sql.SQL(", ").join(
                                    sql.Identifier(col) for col in self.column_names
                                ),
                                values=sql.SQL(", ").join(
                                    sql.Placeholder() for _ in self.column_names
                                ),
                            )

                        # Execute the query with properly cast values
                        cursor.execute(query, row_data)

                        # Check if the operation was an insert or an update
                        is_inserted = cursor.fetchone()[0]

                        if verbose > 0:
                            if is_inserted:
                                logger.info(
                                    f"Inserted new row: {dict(zip(self.column_names, row_data))}"
                                )
                            else:
                                logger.info(
                                    f"Updated row (or skipped due to conflict): {dict(zip(self.column_names, row_data))}"
                                )

                    # Commit the transaction
                    conn.commit()

        except Exception as e:
            logger.error(f"Error saving data to table {self.table}: {e}")
            raise e

# %% ../nbs/50_custom_datasets.ipynb 6
class DynamicPathJSONDataset(AbstractDataset):
    """
    Custom dataset to dynamically resolve a JSON file path from parameters.
    """

    def __init__(self, path_param: str):
        """
        Initializes the ConditionedJSONDataset.

        Args:
            path_param (str): The parameter key that contains the file path.
        """

        config_path = str(Path(settings.CONF_SOURCE))
        self.config_loader = OmegaConfigLoader(config_path)
        self.path_param = path_param

    def _load(self) -> dict:
        """
        Loads the JSON data from the file specified in the parameter.
        """
        # Load parameters
        params_path = self.config_loader["parameters"][self.path_param]

        # Resolve the file path from parameters
        if not params_path:
            raise ValueError(
                f"Path parameter '{self.path_param}' not found in parameters."
            )

        # Load and return JSON data
        full_path = Path(params_path)
        if not full_path.exists():
            raise FileNotFoundError(f"File not found at path: {params_path}")

        with open(params_path, "r") as json_file:
            return (json.load(json_file), params_path)

    def _save(self, data: dict) -> None:
        """
        (Optional) Save method if you want to support writing JSON data.
        """
        raise NotImplementedError("Saving is not supported for DynamicPathJSONDataset.")

    def _describe(self) -> dict:
        """
        Returns a description of the dataset.
        """
        return {"path_param": self.path_param}
