"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * Property to specify if the compute environment
 * uses On-Demand, SpotFleet, Fargate, or Fargate Spot compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
    /**
     * Resources will be Fargate resources.
     */
    ComputeResourceType["FARGATE"] = "FARGATE";
    /**
     * Resources will be Fargate Spot resources.
     *
     * Fargate Spot uses spare capacity in the AWS cloud to run your fault-tolerant,
     * time-flexible jobs at up to a 70% discount. If AWS needs the resources back,
     * jobs running on Fargate Spot will be interrupted with two minutes of notification.
     */
    ComputeResourceType["FARGATE_SPOT"] = "FARGATE_SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends core_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_batch_ComputeEnvironmentProps(props);
        const isFargate = ComputeResourceType.FARGATE === ((_b = props.computeResources) === null || _b === void 0 ? void 0 : _b.type)
            || ComputeResourceType.FARGATE_SPOT === ((_c = props.computeResources) === null || _c === void 0 ? void 0 : _c.type);
        ;
        this.validateProps(props, isFargate);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                placementGroup: props.computeResources.placementGroup,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole === null || spotFleetRole === void 0 ? void 0 : spotFleetRole.roleArn,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
                ...(!isFargate ? {
                    allocationStrategy: props.computeResources.allocationStrategy
                        || (props.computeResources.type === ComputeResourceType.SPOT
                            ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                            : AllocationStrategy.BEST_FIT),
                    instanceRole: props.computeResources.instanceRole
                        ? props.computeResources.instanceRole
                        : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                            roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                    assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                    managedPolicies: [
                                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                    ],
                                }).roleName],
                        }).attrArn,
                    instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                    minvCpus: props.computeResources.minvCpus || 0,
                } : {}),
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: (_e = (_d = props.serviceRole) === null || _d === void 0 ? void 0 : _d.roleArn) !== null && _e !== void 0 ? _e : new iam.Role(this, 'Resource-Service-Instance-Role', {
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                ],
                assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
            }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.splitArn(computeEnvironmentArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        var _b;
        return (_b = props.managed) !== null && _b !== void 0 ? _b : true;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props, isFargate) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        if (props.computeResources) {
            if (isFargate) {
                // VALIDATE FOR FARGATE
                // Bid percentage cannot be set for Fargate evnvironments
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Bid percentage must not be set for Fargate compute environments');
                }
                // Allocation strategy cannot be set for Fargate evnvironments
                if (props.computeResources.allocationStrategy !== undefined) {
                    throw new Error('Allocation strategy must not be set for Fargate compute environments');
                }
                // Desired vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.desiredvCpus !== undefined) {
                    throw new Error('Desired vCPUs must not be set for Fargate compute environments');
                }
                // Image ID cannot be set for Fargate evnvironments
                if (props.computeResources.image !== undefined) {
                    throw new Error('Image must not be set for Fargate compute environments');
                }
                // Instance types cannot be set for Fargate evnvironments
                if (props.computeResources.instanceTypes !== undefined) {
                    throw new Error('Instance types must not be set for Fargate compute environments');
                }
                // EC2 key pair cannot be set for Fargate evnvironments
                if (props.computeResources.ec2KeyPair !== undefined) {
                    throw new Error('EC2 key pair must not be set for Fargate compute environments');
                }
                // Instance role cannot be set for Fargate evnvironments
                if (props.computeResources.instanceRole !== undefined) {
                    throw new Error('Instance role must not be set for Fargate compute environments');
                }
                // Launch template cannot be set for Fargate evnvironments
                if (props.computeResources.launchTemplate !== undefined) {
                    throw new Error('Launch template must not be set for Fargate compute environments');
                }
                // Min vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.minvCpus !== undefined) {
                    throw new Error('Min vCPUs must not be set for Fargate compute environments');
                }
                // Placement group cannot be set for Fargate evnvironments
                if (props.computeResources.placementGroup !== undefined) {
                    throw new Error('Placement group must not be set for Fargate compute environments');
                }
                // Spot fleet role cannot be set for Fargate evnvironments
                if (props.computeResources.spotFleetRole !== undefined) {
                    throw new Error('Spot fleet role must not be set for Fargate compute environments');
                }
            }
            else {
                // VALIDATE FOR ON_DEMAND AND SPOT
                if (props.computeResources.minvCpus) {
                    // minvCpus cannot be less than 0
                    if (props.computeResources.minvCpus < 0) {
                        throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                    }
                    // minvCpus cannot exceed max vCpus
                    if (props.computeResources.maxvCpus &&
                        props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                        throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                    }
                }
                // Setting a bid percentage is only allowed on SPOT resources +
                // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
                if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                    // VALIDATE FOR ON_DEMAND
                    // Bid percentage is not allowed
                    if (props.computeResources.bidPercentage !== undefined) {
                        throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                    }
                    // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                    if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                        throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                    }
                }
                else if (props.computeResources.type === ComputeResourceType.SPOT) {
                    // VALIDATE FOR SPOT
                    // Bid percentage must be from 0 - 100
                    if (props.computeResources.bidPercentage !== undefined &&
                        (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                        throw new Error('Bid percentage can only be a value between 0 and 100');
                    }
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _b;
        if (((_b = props.computeResources) === null || _b === void 0 ? void 0 : _b.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
_a = JSII_RTTI_SYMBOL_1;
ComputeEnvironment[_a] = { fqn: "@aws-cdk/aws-batch.ComputeEnvironment", version: "1.151.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcHV0ZS1lbnZpcm9ubWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvbXB1dGUtZW52aXJvbm1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyx3Q0FBc0U7QUFFdEUsdURBQTBEO0FBRTFEOzs7R0FHRztBQUNILElBQVksbUJBd0JYO0FBeEJELFdBQVksbUJBQW1CO0lBQzdCOztPQUVHO0lBQ0gsd0NBQWlCLENBQUE7SUFFakI7O09BRUc7SUFDSCxvQ0FBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCwwQ0FBbUIsQ0FBQTtJQUVuQjs7Ozs7O09BTUc7SUFDSCxvREFBNkIsQ0FBQTtBQUMvQixDQUFDLEVBeEJXLG1CQUFtQixHQUFuQiwyQkFBbUIsS0FBbkIsMkJBQW1CLFFBd0I5QjtBQUVEOzs7R0FHRztBQUNILElBQVksa0JBcUJYO0FBckJELFdBQVksa0JBQWtCO0lBQzVCOzs7T0FHRztJQUNILDJDQUFxQixDQUFBO0lBRXJCOzs7O09BSUc7SUFDSCxtRUFBNkMsQ0FBQTtJQUU3Qzs7Ozs7T0FLRztJQUNILHlFQUFtRCxDQUFBO0FBQ3JELENBQUMsRUFyQlcsa0JBQWtCLEdBQWxCLDBCQUFrQixLQUFsQiwwQkFBa0IsUUFxQjdCO0FBMlBEOzs7O0dBSUc7QUFDSCxNQUFhLGtCQUFtQixTQUFRLGVBQVE7SUFrQzlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBaUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUU7O1FBQ3pHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxzQkFBc0I7U0FDM0MsQ0FBQyxDQUFDOztRQUVILE1BQU0sU0FBUyxHQUFHLG1CQUFtQixDQUFDLE9BQU8sWUFBSyxLQUFLLENBQUMsZ0JBQWdCLDBDQUFFLElBQUksQ0FBQTtlQUN6RSxtQkFBbUIsQ0FBQyxZQUFZLFlBQUssS0FBSyxDQUFDLGdCQUFnQiwwQ0FBRSxJQUFJLENBQUEsQ0FBQztRQUFBLENBQUM7UUFFeEUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFckMsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ25ELElBQUksZ0JBQTRFLENBQUM7UUFFakYsaUVBQWlFO1FBQ2pFLElBQUksS0FBSyxDQUFDLGdCQUFnQixJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDbkQsZ0JBQWdCLEdBQUc7Z0JBQ2pCLGFBQWEsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYTtnQkFDbkQsWUFBWSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZO2dCQUNqRCxVQUFVLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFVBQVU7Z0JBQzdDLE9BQU8sRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU87Z0JBQzVGLGNBQWMsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYztnQkFDckQsUUFBUSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLElBQUksR0FBRztnQkFDaEQsY0FBYyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjO2dCQUNyRCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDO2dCQUMvRyxnQkFBZ0IsRUFBRSxhQUFhLGFBQWIsYUFBYSx1QkFBYixhQUFhLENBQUUsT0FBTztnQkFDeEMsT0FBTyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsQ0FBQyxTQUFTO2dCQUM5RixJQUFJLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLG9CQUFvQjtnQkFDakQsSUFBSSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLElBQUksbUJBQW1CLENBQUMsU0FBUztnQkFDbEUsR0FBRyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztvQkFDZixrQkFBa0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCOzJCQUN4RCxDQUNELEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEtBQUssbUJBQW1CLENBQUMsSUFBSTs0QkFDdEQsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLHVCQUF1Qjs0QkFDNUMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FDaEM7b0JBQ0gsWUFBWSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZO3dCQUMvQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFlBQVk7d0JBQ3JDLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7NEJBQ3JELEtBQUssRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUU7b0NBQzlDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztvQ0FDeEQsZUFBZSxFQUFFO3dDQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsa0RBQWtELENBQUM7cUNBQy9GO2lDQUNGLENBQUMsQ0FBQyxRQUFRLENBQUM7eUJBQ2IsQ0FBQyxDQUFDLE9BQU87b0JBQ1osYUFBYSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDO29CQUM1RSxRQUFRLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsSUFBSSxDQUFDO2lCQUMvQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDUixDQUFDO1NBQ0g7UUFFRCxNQUFNLGtCQUFrQixHQUFHLElBQUksdUNBQXFCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNyRSxzQkFBc0IsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUN6QyxnQkFBZ0I7WUFDaEIsV0FBVyxjQUFFLEtBQUssQ0FBQyxXQUFXLDBDQUFFLE9BQU8sbUNBQ2xDLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsZ0NBQWdDLEVBQUU7Z0JBQ3RELGVBQWUsRUFBRTtvQkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLGtDQUFrQyxDQUFDO2lCQUMvRTtnQkFDRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMscUJBQXFCLENBQUM7YUFDM0QsQ0FBQyxDQUFDLE9BQU87WUFDWixLQUFLLEVBQUUsS0FBSyxDQUFDLE9BQU8sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztZQUN6RixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxXQUFXO1NBQ3RELENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLGdCQUFnQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUU7WUFDeEQsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsSUFBSSxDQUFDLHFCQUFxQixHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLEVBQUU7WUFDaEYsT0FBTyxFQUFFLE9BQU87WUFDaEIsUUFBUSxFQUFFLHFCQUFxQjtZQUMvQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDaEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUNyRjtJQTVHRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMseUJBQXlCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUscUJBQTZCO1FBQ2pHLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDOUIsTUFBTSxzQkFBc0IsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLHFCQUFxQixFQUFFLGdCQUFTLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxZQUFhLENBQUM7UUFFbEgsTUFBTSxNQUFPLFNBQVEsZUFBUTtZQUE3Qjs7Z0JBQ2tCLDBCQUFxQixHQUFHLHFCQUFxQixDQUFDO2dCQUM5QywyQkFBc0IsR0FBRyxzQkFBc0IsQ0FBQztZQUNsRSxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztLQUM5QjtJQTZGTyxTQUFTLENBQUMsS0FBOEI7O1FBQzlDLGFBQU8sS0FBSyxDQUFDLE9BQU8sbUNBQUksSUFBSSxDQUFDO0tBQzlCO0lBRUQ7O09BRUc7SUFDSyxhQUFhLENBQUMsS0FBOEIsRUFBRSxTQUFrQjtRQUN0RSxJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDdkIsT0FBTztTQUNSO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxDQUFDLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtZQUNsRSxNQUFNLElBQUksS0FBSyxDQUFDLG1GQUFtRixDQUFDLENBQUM7U0FDdEc7UUFFRCxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxDQUFDLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtZQUNqRSxNQUFNLElBQUksS0FBSyxDQUFDLDJFQUEyRSxDQUFDLENBQUM7U0FDOUY7UUFFRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixJQUFJLFNBQVMsRUFBRTtnQkFDYix1QkFBdUI7Z0JBRXZCLHlEQUF5RDtnQkFDekQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtvQkFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO2lCQUNwRjtnQkFFRCw4REFBOEQ7Z0JBQzlELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGtCQUFrQixLQUFLLFNBQVMsRUFBRTtvQkFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRUFBc0UsQ0FBQyxDQUFDO2lCQUN6RjtnQkFFRCx3REFBd0Q7Z0JBQ3hELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7b0JBQ3JELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztpQkFDbkY7Z0JBRUQsbURBQW1EO2dCQUNuRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFO29CQUM5QyxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7aUJBQzNFO2dCQUVELHlEQUF5RDtnQkFDekQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtvQkFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO2lCQUNwRjtnQkFFRCx1REFBdUQ7Z0JBQ3ZELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFVBQVUsS0FBSyxTQUFTLEVBQUU7b0JBQ25ELE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQztpQkFDbEY7Z0JBRUQsd0RBQXdEO2dCQUN4RCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUFFO29CQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxDQUFDLENBQUM7aUJBQ25GO2dCQUVELDBEQUEwRDtnQkFDMUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxLQUFLLFNBQVMsRUFBRTtvQkFDdkQsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO2lCQUNyRjtnQkFFRCxvREFBb0Q7Z0JBQ3BELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsS0FBSyxTQUFTLEVBQUU7b0JBQ2pELE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztpQkFDL0U7Z0JBRUQsMERBQTBEO2dCQUMxRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEtBQUssU0FBUyxFQUFFO29CQUN2RCxNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUM7aUJBQ3JGO2dCQUVELDBEQUEwRDtnQkFDMUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtvQkFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO2lCQUNyRjthQUNGO2lCQUFNO2dCQUNMLGtDQUFrQztnQkFDbEMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxFQUFFO29CQUNuQyxpQ0FBaUM7b0JBQ2pDLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsR0FBRyxDQUFDLEVBQUU7d0JBQ3ZDLE1BQU0sSUFBSSxLQUFLLENBQUMscUVBQXFFLENBQUMsQ0FBQztxQkFDeEY7b0JBRUQsbUNBQW1DO29CQUNuQyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRO3dCQUNqQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLEVBQUU7d0JBQ25FLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQztxQkFDM0U7aUJBQ0Y7Z0JBQ0QsK0RBQStEO2dCQUMvRCwwREFBMEQ7Z0JBQzFELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLElBQUksS0FBSyxtQkFBbUIsQ0FBQyxTQUFTLEVBQUU7b0JBQ2pFLHlCQUF5QjtvQkFFekIsZ0NBQWdDO29CQUNoQyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO3dCQUN0RCxNQUFNLElBQUksS0FBSyxDQUFDLG1HQUFtRyxDQUFDLENBQUM7cUJBQ3RIO29CQUVELG9EQUFvRDtvQkFDcEQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGtCQUFrQixLQUFLLGtCQUFrQixDQUFDLHVCQUF1QixFQUFFO3dCQUN6SSxNQUFNLElBQUksS0FBSyxDQUFDLHVIQUF1SCxDQUFDLENBQUM7cUJBQzFJO2lCQUNGO3FCQUFNLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLElBQUksS0FBSyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUU7b0JBQ25FLG9CQUFvQjtvQkFFcEIsc0NBQXNDO29CQUN0QyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEtBQUssU0FBUzt3QkFDcEQsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxHQUFHLENBQUMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxHQUFHLEdBQUcsQ0FBQyxFQUFFO3dCQUMxRixNQUFNLElBQUksS0FBSyxDQUFDLHNEQUFzRCxDQUFDLENBQUM7cUJBQ3pFO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGO0lBRU8sa0JBQWtCLENBQUMsYUFBa0M7UUFDM0QsSUFBSSxhQUFhLEtBQUssU0FBUyxFQUFFO1lBQy9CLE9BQU87Z0JBQ0wsU0FBUzthQUNWLENBQUM7U0FDSDtRQUVELE9BQU8sYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQXNCLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0tBQ3ZFO0lBRU8scUJBQXFCLENBQUMsR0FBYSxFQUFFLGNBQXFDO1FBQ2hGLElBQUksY0FBYyxLQUFLLFNBQVMsRUFBRTtZQUNoQyxPQUFPO2dCQUNMLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLGVBQWU7YUFDaEYsQ0FBQztTQUNIO1FBRUQsT0FBTyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBeUIsRUFBRSxFQUFFLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDO0tBQ2pGO0lBRUQ7Ozs7O09BS0c7SUFDSyxnQkFBZ0IsQ0FBQyxLQUE4Qjs7UUFDckQsSUFBSSxPQUFBLEtBQUssQ0FBQyxnQkFBZ0IsMENBQUUsa0JBQWtCLEtBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGtCQUFrQixLQUFLLGtCQUFrQixDQUFDLFFBQVEsRUFBRTtZQUMzSCxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQzFCLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsRUFBRTtnQkFDeEMsT0FBTyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDO2FBQzdDO2lCQUFNLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLElBQUksS0FBSyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUU7Z0JBQ25FLE9BQU8sR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUN6RCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxTQUFTLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyw4RUFBOEUsQ0FBQyxDQUFDO2FBQ3pJO1NBQ0Y7UUFFRCxPQUFPLFNBQVMsQ0FBQztLQUNsQjs7QUEvUUgsZ0RBZ1JDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQXJuRm9ybWF0LCBJUmVzb3VyY2UsIFJlc291cmNlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5Db21wdXRlRW52aXJvbm1lbnQgfSBmcm9tICcuL2JhdGNoLmdlbmVyYXRlZCc7XG5cbi8qKlxuICogUHJvcGVydHkgdG8gc3BlY2lmeSBpZiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudFxuICogdXNlcyBPbi1EZW1hbmQsIFNwb3RGbGVldCwgRmFyZ2F0ZSwgb3IgRmFyZ2F0ZSBTcG90IGNvbXB1dGUgcmVzb3VyY2VzLlxuICovXG5leHBvcnQgZW51bSBDb21wdXRlUmVzb3VyY2VUeXBlIHtcbiAgLyoqXG4gICAqIFJlc291cmNlcyB3aWxsIGJlIEVDMiBPbi1EZW1hbmQgcmVzb3VyY2VzLlxuICAgKi9cbiAgT05fREVNQU5EID0gJ0VDMicsXG5cbiAgLyoqXG4gICAqIFJlc291cmNlcyB3aWxsIGJlIEVDMiBTcG90RmxlZXQgcmVzb3VyY2VzLlxuICAgKi9cbiAgU1BPVCA9ICdTUE9UJyxcblxuICAvKipcbiAgICogUmVzb3VyY2VzIHdpbGwgYmUgRmFyZ2F0ZSByZXNvdXJjZXMuXG4gICAqL1xuICBGQVJHQVRFID0gJ0ZBUkdBVEUnLFxuXG4gIC8qKlxuICAgKiBSZXNvdXJjZXMgd2lsbCBiZSBGYXJnYXRlIFNwb3QgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBGYXJnYXRlIFNwb3QgdXNlcyBzcGFyZSBjYXBhY2l0eSBpbiB0aGUgQVdTIGNsb3VkIHRvIHJ1biB5b3VyIGZhdWx0LXRvbGVyYW50LFxuICAgKiB0aW1lLWZsZXhpYmxlIGpvYnMgYXQgdXAgdG8gYSA3MCUgZGlzY291bnQuIElmIEFXUyBuZWVkcyB0aGUgcmVzb3VyY2VzIGJhY2ssXG4gICAqIGpvYnMgcnVubmluZyBvbiBGYXJnYXRlIFNwb3Qgd2lsbCBiZSBpbnRlcnJ1cHRlZCB3aXRoIHR3byBtaW51dGVzIG9mIG5vdGlmaWNhdGlvbi5cbiAgICovXG4gIEZBUkdBVEVfU1BPVCA9ICdGQVJHQVRFX1NQT1QnLFxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGhvdyB0byBwcmVwYXJlIGNvbXB1dGUgcmVzb3VyY2VzXG4gKiB0aGF0IGFyZSBwcm92aXNpb25lZCBmb3IgYSBjb21wdXRlIGVudmlyb25tZW50LlxuICovXG5leHBvcnQgZW51bSBBbGxvY2F0aW9uU3RyYXRlZ3kge1xuICAvKipcbiAgICogQmF0Y2ggd2lsbCB1c2UgdGhlIGJlc3QgZml0dGluZyBpbnN0YW5jZSB0eXBlIHdpbGwgYmUgdXNlZFxuICAgKiB3aGVuIGFzc2lnbmluZyBhIGJhdGNoIGpvYiBpbiB0aGlzIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqL1xuICBCRVNUX0ZJVCA9ICdCRVNUX0ZJVCcsXG5cbiAgLyoqXG4gICAqIEJhdGNoIHdpbGwgc2VsZWN0IGFkZGl0aW9uYWwgaW5zdGFuY2UgdHlwZXMgdGhhdCBhcmUgbGFyZ2UgZW5vdWdoIHRvXG4gICAqIG1lZXQgdGhlIHJlcXVpcmVtZW50cyBvZiB0aGUgam9icyBpbiB0aGUgcXVldWUsIHdpdGggYSBwcmVmZXJlbmNlIGZvclxuICAgKiBpbnN0YW5jZSB0eXBlcyB3aXRoIGEgbG93ZXIgY29zdCBwZXIgdW5pdCB2Q1BVLlxuICAgKi9cbiAgQkVTVF9GSVRfUFJPR1JFU1NJVkUgPSAnQkVTVF9GSVRfUFJPR1JFU1NJVkUnLFxuXG4gIC8qKlxuICAgKiBUaGlzIGlzIG9ubHkgYXZhaWxhYmxlIGZvciBTcG90IEluc3RhbmNlIGNvbXB1dGUgcmVzb3VyY2VzIGFuZCB3aWxsIHNlbGVjdFxuICAgKiBhZGRpdGlvbmFsIGluc3RhbmNlIHR5cGVzIHRoYXQgYXJlIGxhcmdlIGVub3VnaCB0byBtZWV0IHRoZSByZXF1aXJlbWVudHMgb2ZcbiAgICogdGhlIGpvYnMgaW4gdGhlIHF1ZXVlLCB3aXRoIGEgcHJlZmVyZW5jZSBmb3IgaW5zdGFuY2UgdHlwZXMgdGhhdCBhcmUgbGVzc1xuICAgKiBsaWtlbHkgdG8gYmUgaW50ZXJydXB0ZWQuXG4gICAqL1xuICBTUE9UX0NBUEFDSVRZX09QVElNSVpFRCA9ICdTUE9UX0NBUEFDSVRZX09QVElNSVpFRCcsXG59XG5cbi8qKlxuICogTGF1bmNoIHRlbXBsYXRlIHByb3BlcnR5IHNwZWNpZmljYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb24ge1xuICAvKipcbiAgICogVGhlIExhdW5jaCB0ZW1wbGF0ZSBuYW1lXG4gICAqL1xuICByZWFkb25seSBsYXVuY2hUZW1wbGF0ZU5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBsYXVuY2ggdGVtcGxhdGUgdmVyc2lvbiB0byBiZSB1c2VkIChvcHRpb25hbCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRlZmF1bHQgdmVyc2lvbiBvZiB0aGUgbGF1bmNoIHRlbXBsYXRlXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9uPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlZmluaW5nIHRoZSBzdHJ1Y3R1cmUgb2YgdGhlIGJhdGNoIGNvbXB1dGUgY2x1c3Rlci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb21wdXRlUmVzb3VyY2VzIHtcbiAgLyoqXG4gICAqIFRoZSBhbGxvY2F0aW9uIHN0cmF0ZWd5IHRvIHVzZSBmb3IgdGhlIGNvbXB1dGUgcmVzb3VyY2UgaW4gY2FzZSBub3QgZW5vdWdoIGluc3RhbmNlcyBvZiB0aGUgYmVzdFxuICAgKiBmaXR0aW5nIGluc3RhbmNlIHR5cGUgY2FuIGJlIGFsbG9jYXRlZC4gVGhpcyBjb3VsZCBiZSBkdWUgdG8gYXZhaWxhYmlsaXR5IG9mIHRoZSBpbnN0YW5jZSB0eXBlIGluXG4gICAqIHRoZSByZWdpb24gb3IgQW1hem9uIEVDMiBzZXJ2aWNlIGxpbWl0cy4gSWYgdGhpcyBpcyBub3Qgc3BlY2lmaWVkLCB0aGUgZGVmYXVsdCBmb3IgdGhlIEVDMlxuICAgKiBDb21wdXRlUmVzb3VyY2VUeXBlIGlzIEJFU1RfRklULCB3aGljaCB3aWxsIHVzZSBvbmx5IHRoZSBiZXN0IGZpdHRpbmcgaW5zdGFuY2UgdHlwZSwgd2FpdGluZyBmb3JcbiAgICogYWRkaXRpb25hbCBjYXBhY2l0eSBpZiBpdCdzIG5vdCBhdmFpbGFibGUuIFRoaXMgYWxsb2NhdGlvbiBzdHJhdGVneSBrZWVwcyBjb3N0cyBsb3dlciBidXQgY2FuIGxpbWl0XG4gICAqIHNjYWxpbmcuIElmIHlvdSBhcmUgdXNpbmcgU3BvdCBGbGVldHMgd2l0aCBCRVNUX0ZJVCB0aGVuIHRoZSBTcG90IEZsZWV0IElBTSBSb2xlIG11c3QgYmUgc3BlY2lmaWVkLlxuICAgKiBCRVNUX0ZJVF9QUk9HUkVTU0lWRSB3aWxsIHNlbGVjdCBhbiBhZGRpdGlvbmFsIGluc3RhbmNlIHR5cGUgdGhhdCBpcyBsYXJnZSBlbm91Z2ggdG8gbWVldCB0aGVcbiAgICogcmVxdWlyZW1lbnRzIG9mIHRoZSBqb2JzIGluIHRoZSBxdWV1ZSwgd2l0aCBhIHByZWZlcmVuY2UgZm9yIGFuIGluc3RhbmNlIHR5cGUgd2l0aCBhIGxvd2VyIGNvc3QuXG4gICAqIFRoZSBkZWZhdWx0IHZhbHVlIGZvciB0aGUgU1BPVCBpbnN0YW5jZSB0eXBlIGlzIFNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVELCB3aGljaCBpcyBvbmx5IGF2YWlsYWJsZSBmb3JcbiAgICogZm9yIHRoaXMgdHlwZSBvZiBjb21wdXRlIHJlc291cmNlcyBhbmQgd2lsbCBzZWxlY3QgYW4gYWRkaXRpb25hbCBpbnN0YW5jZSB0eXBlIHRoYXQgaXMgbGFyZ2UgZW5vdWdoXG4gICAqIHRvIG1lZXQgdGhlIHJlcXVpcmVtZW50cyBvZiB0aGUgam9icyBpbiB0aGUgcXVldWUsIHdpdGggYSBwcmVmZXJlbmNlIGZvciBhbiBpbnN0YW5jZSB0eXBlIHRoYXQgaXNcbiAgICogbGVzcyBsaWtlbHkgdG8gYmUgaW50ZXJydXB0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IEFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVFxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb2NhdGlvblN0cmF0ZWd5PzogQWxsb2NhdGlvblN0cmF0ZWd5O1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIEVDUyBpbnN0YW5jZSBwcm9maWxlIGFwcGxpZWQgdG8gQW1hem9uIEVDMiBpbnN0YW5jZXMgaW4gYSBjb21wdXRlIGVudmlyb25tZW50LiBZb3UgY2FuIHNwZWNpZnlcbiAgICogdGhlIHNob3J0IG5hbWUgb3IgZnVsbCBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiBhbiBpbnN0YW5jZSBwcm9maWxlLiBGb3IgZXhhbXBsZSwgZWNzSW5zdGFuY2VSb2xlIG9yXG4gICAqIGFybjphd3M6aWFtOjo8YXdzX2FjY291bnRfaWQ+Omluc3RhbmNlLXByb2ZpbGUvZWNzSW5zdGFuY2VSb2xlIC4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBBbWF6b24gRUNTXG4gICAqIEluc3RhbmNlIFJvbGUgaW4gdGhlIEFXUyBCYXRjaCBVc2VyIEd1aWRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IHJvbGUgd2lsbCBiZSBjcmVhdGVkLlxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9pbnN0YW5jZV9JQU1fcm9sZS5odG1sXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVJvbGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIGxhdW5jaCB0ZW1wbGF0ZSB0byBhc3NvY2lhdGUgd2l0aCB5b3VyIGNvbXB1dGUgcmVzb3VyY2VzLlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIFJFQURNRSBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGN1c3RvbSBsYXVuY2ggdGVtcGxhdGUgd2lsbCBiZSB1c2VkXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL2xhdW5jaC10ZW1wbGF0ZXMuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgbGF1bmNoVGVtcGxhdGU/OiBMYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlcyBvZiBFQzIgaW5zdGFuY2VzIHRoYXQgbWF5IGJlIGxhdW5jaGVkIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LiBZb3UgY2FuIHNwZWNpZnkgaW5zdGFuY2VcbiAgICogZmFtaWxpZXMgdG8gbGF1bmNoIGFueSBpbnN0YW5jZSB0eXBlIHdpdGhpbiB0aG9zZSBmYW1pbGllcyAoZm9yIGV4YW1wbGUsIGM0IG9yIHAzKSwgb3IgeW91IGNhbiBzcGVjaWZ5XG4gICAqIHNwZWNpZmljIHNpemVzIHdpdGhpbiBhIGZhbWlseSAoc3VjaCBhcyBjNC44eGxhcmdlKS4gWW91IGNhbiBhbHNvIGNob29zZSBvcHRpbWFsIHRvIHBpY2sgaW5zdGFuY2UgdHlwZXNcbiAgICogKGZyb20gdGhlIEMsIE0sIGFuZCBSIGluc3RhbmNlIGZhbWlsaWVzKSBvbiB0aGUgZmx5IHRoYXQgbWF0Y2ggdGhlIGRlbWFuZCBvZiB5b3VyIGpvYiBxdWV1ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG9wdGltYWxcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZXM/OiBlYzIuSW5zdGFuY2VUeXBlW107XG5cbiAgLyoqXG4gICAqIFRoZSBFQzIgc2VjdXJpdHkgZ3JvdXAocykgYXNzb2NpYXRlZCB3aXRoIGluc3RhbmNlcyBsYXVuY2hlZCBpbiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBV1MgZGVmYXVsdCBzZWN1cml0eSBncm91cC5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzPzogZWMyLklTZWN1cml0eUdyb3VwW107XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgbmV0d29yayB0aGF0IGFsbCBjb21wdXRlIHJlc291cmNlcyB3aWxsIGJlIGNvbm5lY3RlZCB0by5cbiAgICovXG4gIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgc3VibmV0cyBpbnRvIHdoaWNoIHRoZSBjb21wdXRlIHJlc291cmNlcyBhcmUgbGF1bmNoZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gcHJpdmF0ZSBzdWJuZXRzIG9mIHRoZSBzdXBwbGllZCBWUEMuXG4gICAqL1xuICByZWFkb25seSB2cGNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgY29tcHV0ZSBlbnZpcm9ubWVudDogT05fREVNQU5ELCBTUE9ULCBGQVJHQVRFLCBvciBGQVJHQVRFX1NQT1QuXG4gICAqXG4gICAqIEBkZWZhdWx0IE9OX0RFTUFORFxuICAgKi9cbiAgcmVhZG9ubHkgdHlwZT86IENvbXB1dGVSZXNvdXJjZVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoaXMgcHJvcGVydHkgd2lsbCBiZSBpZ25vcmVkIGlmIHlvdSBzZXQgdGhlIGVudmlyb25tZW50IHR5cGUgdG8gT05fREVNQU5ELlxuICAgKlxuICAgKiBUaGUgbWF4aW11bSBwZXJjZW50YWdlIHRoYXQgYSBTcG90IEluc3RhbmNlIHByaWNlIGNhbiBiZSB3aGVuIGNvbXBhcmVkIHdpdGggdGhlIE9uLURlbWFuZCBwcmljZSBmb3JcbiAgICogdGhhdCBpbnN0YW5jZSB0eXBlIGJlZm9yZSBpbnN0YW5jZXMgYXJlIGxhdW5jaGVkLiBGb3IgZXhhbXBsZSwgaWYgeW91ciBtYXhpbXVtIHBlcmNlbnRhZ2UgaXMgMjAlLFxuICAgKiB0aGVuIHRoZSBTcG90IHByaWNlIG11c3QgYmUgYmVsb3cgMjAlIG9mIHRoZSBjdXJyZW50IE9uLURlbWFuZCBwcmljZSBmb3IgdGhhdCBFQzIgaW5zdGFuY2UuIFlvdSBhbHdheXNcbiAgICogcGF5IHRoZSBsb3dlc3QgKG1hcmtldCkgcHJpY2UgYW5kIG5ldmVyIG1vcmUgdGhhbiB5b3VyIG1heGltdW0gcGVyY2VudGFnZS4gSWYgeW91IGxlYXZlIHRoaXMgZmllbGQgZW1wdHksXG4gICAqIHRoZSBkZWZhdWx0IHZhbHVlIGlzIDEwMCUgb2YgdGhlIE9uLURlbWFuZCBwcmljZS5cbiAgICpcbiAgICogQGRlZmF1bHQgMTAwXG4gICAqL1xuICByZWFkb25seSBiaWRQZXJjZW50YWdlPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgZGVzaXJlZCBudW1iZXIgb2YgRUMyIHZDUFVTIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGRlc2lyZWQgdmNwdSB2YWx1ZSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBkZXNpcmVkdkNwdXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiBFQzIgdkNQVXMgdGhhdCBhbiBlbnZpcm9ubWVudCBjYW4gcmVhY2guIEVhY2ggdkNQVSBpcyBlcXVpdmFsZW50IHRvXG4gICAqIDEsMDI0IENQVSBzaGFyZXMuIFlvdSBtdXN0IHNwZWNpZnkgYXQgbGVhc3Qgb25lIHZDUFUuXG4gICAqXG4gICAqIEBkZWZhdWx0IDI1NlxuICAgKi9cbiAgcmVhZG9ubHkgbWF4dkNwdXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIG51bWJlciBvZiBFQzIgdkNQVXMgdGhhdCBhbiBlbnZpcm9ubWVudCBzaG91bGQgbWFpbnRhaW4gKGV2ZW4gaWYgdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQgc3RhdGUgaXMgRElTQUJMRUQpLlxuICAgKiBFYWNoIHZDUFUgaXMgZXF1aXZhbGVudCB0byAxLDAyNCBDUFUgc2hhcmVzLiBCeSBrZWVwaW5nIHRoaXMgc2V0IHRvIDAgeW91IHdpbGwgbm90IGhhdmUgaW5zdGFuY2UgdGltZSB3YXN0ZWQgd2hlblxuICAgKiB0aGVyZSBpcyBubyB3b3JrIHRvIGJlIHJ1bi4gSWYgeW91IHNldCB0aGlzIGFib3ZlIHplcm8geW91IHdpbGwgbWFpbnRhaW4gdGhhdCBudW1iZXIgb2YgdkNQVXMgYXQgYWxsIHRpbWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAwXG4gICAqL1xuICByZWFkb25seSBtaW52Q3B1cz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBFQzIgcGxhY2VtZW50IGdyb3VwIHRvIGFzc29jaWF0ZSB3aXRoIHlvdXIgY29tcHV0ZSByZXNvdXJjZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gcGxhY2VtZW50IGdyb3VwIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IHBsYWNlbWVudEdyb3VwPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgRUMyIGtleSBwYWlyIHRoYXQgaXMgdXNlZCBmb3IgaW5zdGFuY2VzIGxhdW5jaGVkIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKiBJZiBubyBrZXkgaXMgZGVmaW5lZCwgdGhlbiBTU0ggYWNjZXNzIGlzIG5vdCBhbGxvd2VkIHRvIHByb3Zpc2lvbmVkIGNvbXB1dGUgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIFNTSCBhY2Nlc3Mgd2lsbCBiZSBwb3NzaWJsZS5cbiAgICovXG4gIHJlYWRvbmx5IGVjMktleVBhaXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gTWFjaGluZSBJbWFnZSAoQU1JKSBJRCB1c2VkIGZvciBpbnN0YW5jZXMgbGF1bmNoZWQgaW4gdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gaW1hZ2Ugd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2U/OiBlYzIuSU1hY2hpbmVJbWFnZTtcblxuICAvKipcbiAgICogVGhpcyBwcm9wZXJ0eSB3aWxsIGJlIGlnbm9yZWQgaWYgeW91IHNldCB0aGUgZW52aXJvbm1lbnQgdHlwZSB0byBPTl9ERU1BTkQuXG4gICAqXG4gICAqIFRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiB0aGUgQW1hem9uIEVDMiBTcG90IEZsZWV0IElBTSByb2xlIGFwcGxpZWQgdG8gYSBTUE9UIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgQW1hem9uIEVDMiBTcG90IEZsZWV0IFJvbGUgaW4gdGhlIEFXUyBCYXRjaCBVc2VyIEd1aWRlLlxuICAgKlxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9zcG90X2ZsZWV0X0lBTV9yb2xlLmh0bWxcbiAgICogQGRlZmF1bHQgLSBubyBmbGVldCByb2xlIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IHNwb3RGbGVldFJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIEtleS12YWx1ZSBwYWlyIHRhZ3MgdG8gYmUgYXBwbGllZCB0byByZXNvdXJjZXMgdGhhdCBhcmUgbGF1bmNoZWQgaW4gdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqIEZvciBBV1MgQmF0Y2gsIHRoZXNlIHRha2UgdGhlIGZvcm0gb2YgXCJTdHJpbmcxXCI6IFwiU3RyaW5nMlwiLCB3aGVyZSBTdHJpbmcxIGlzIHRoZSB0YWcga2V5IGFuZFxuICAgKiBTdHJpbmcyIGlzIHRoZSB0YWcgdmFsdWXigJRmb3IgZXhhbXBsZSwgeyBcIk5hbWVcIjogXCJBV1MgQmF0Y2ggSW5zdGFuY2UgLSBDNE9uRGVtYW5kXCIgfS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyB0YWdzIHdpbGwgYmUgYXNzaWduZWQgb24gY29tcHV0ZSByZXNvdXJjZXMuXG4gICAqL1xuICByZWFkb25seSBjb21wdXRlUmVzb3VyY2VzVGFncz86IHtcbiAgICBba2V5OiBzdHJpbmddOiBzdHJpbmdcbiAgfTtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBjcmVhdGluZyBhIG5ldyBDb21wdXRlIEVudmlyb25tZW50XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29tcHV0ZUVudmlyb25tZW50UHJvcHMge1xuICAvKipcbiAgICogQSBuYW1lIGZvciB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogVXAgdG8gMTI4IGxldHRlcnMgKHVwcGVyY2FzZSBhbmQgbG93ZXJjYXNlKSwgbnVtYmVycywgaHlwaGVucywgYW5kIHVuZGVyc2NvcmVzIGFyZSBhbGxvd2VkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENsb3VkRm9ybWF0aW9uLWdlbmVyYXRlZCBuYW1lXG4gICAqL1xuICByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnROYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZGV0YWlscyBvZiB0aGUgcmVxdWlyZWQgY29tcHV0ZSByZXNvdXJjZXMgZm9yIHRoZSBtYW5hZ2VkIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIElmIHNwZWNpZmllZCwgYW5kIHRoaXMgaXMgYW4gdW5tYW5hZ2VkIGNvbXB1dGUgZW52aXJvbm1lbnQsIHdpbGwgdGhyb3cgYW4gZXJyb3IuXG4gICAqXG4gICAqIEJ5IGRlZmF1bHQsIEFXUyBCYXRjaCBtYW5hZ2VkIGNvbXB1dGUgZW52aXJvbm1lbnRzIHVzZSBhIHJlY2VudCwgYXBwcm92ZWQgdmVyc2lvbiBvZiB0aGVcbiAgICogQW1hem9uIEVDUy1vcHRpbWl6ZWQgQU1JIGZvciBjb21wdXRlIHJlc291cmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDbG91ZEZvcm1hdGlvbiBkZWZhdWx0c1xuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcHJvcGVydGllcy1iYXRjaC1jb21wdXRlZW52aXJvbm1lbnQtY29tcHV0ZXJlc291cmNlcy5odG1sXG4gICAqL1xuICByZWFkb25seSBjb21wdXRlUmVzb3VyY2VzPzogQ29tcHV0ZVJlc291cmNlcztcblxuICAvKipcbiAgICogVGhlIHN0YXRlIG9mIHRoZSBjb21wdXRlIGVudmlyb25tZW50LiBJZiB0aGUgc3RhdGUgaXMgc2V0IHRvIHRydWUsIHRoZW4gdGhlIGNvbXB1dGVcbiAgICogZW52aXJvbm1lbnQgYWNjZXB0cyBqb2JzIGZyb20gYSBxdWV1ZSBhbmQgY2FuIHNjYWxlIG91dCBhdXRvbWF0aWNhbGx5IGJhc2VkIG9uIHF1ZXVlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gcm9sZSB1c2VkIGJ5IEJhdGNoIHRvIG1ha2UgY2FsbHMgdG8gb3RoZXIgQVdTIHNlcnZpY2VzIG9uIHlvdXIgYmVoYWxmIGZvciBtYW5hZ2luZ1xuICAgKiB0aGUgcmVzb3VyY2VzIHRoYXQgeW91IHVzZSB3aXRoIHRoZSBzZXJ2aWNlLiBCeSBkZWZhdWx0LCB0aGlzIHJvbGUgaXMgY3JlYXRlZCBmb3IgeW91IHVzaW5nXG4gICAqIHRoZSBBV1MgbWFuYWdlZCBzZXJ2aWNlIHBvbGljeSBmb3IgQmF0Y2guXG4gICAqXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL3NlcnZpY2VfSUFNX3JvbGUuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJvbGUgdXNpbmcgdGhlICdzZXJ2aWNlLXJvbGUvQVdTQmF0Y2hTZXJ2aWNlUm9sZScgcG9saWN5LlxuICAgKi9cbiAgcmVhZG9ubHkgc2VydmljZVJvbGU/OiBpYW0uSVJvbGUsXG5cbiAgLyoqXG4gICAqIERldGVybWluZXMgaWYgQVdTIHNob3VsZCBtYW5hZ2UgdGhlIGFsbG9jYXRpb24gb2YgY29tcHV0ZSByZXNvdXJjZXMgZm9yIHByb2Nlc3Npbmcgam9icy5cbiAgICogSWYgc2V0IHRvIGZhbHNlLCB0aGVuIHlvdSBhcmUgaW4gY2hhcmdlIG9mIHByb3ZpZGluZyB0aGUgY29tcHV0ZSByZXNvdXJjZSBkZXRhaWxzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBtYW5hZ2VkPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIG9mIGEgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQ29tcHV0ZUVudmlyb25tZW50IGV4dGVuZHMgSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhpcyBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnRBcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhpcyBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnROYW1lOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQmF0Y2ggQ29tcHV0ZSBFbnZpcm9ubWVudC5cbiAqXG4gKiBEZWZpbmVzIGEgYmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudCB0byBydW4gYmF0Y2ggam9icyBvbi5cbiAqL1xuZXhwb3J0IGNsYXNzIENvbXB1dGVFbnZpcm9ubWVudCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUNvbXB1dGVFbnZpcm9ubWVudCB7XG4gIC8qKlxuICAgKiBGZXRjaGVzIGFuIGV4aXN0aW5nIGJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnQgYnkgaXRzIGFtYXpvbiByZXNvdXJjZSBuYW1lLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGVcbiAgICogQHBhcmFtIGlkXG4gICAqIEBwYXJhbSBjb21wdXRlRW52aXJvbm1lbnRBcm5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNvbXB1dGVFbnZpcm9ubWVudEFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBjb21wdXRlRW52aXJvbm1lbnRBcm46IHN0cmluZyk6IElDb21wdXRlRW52aXJvbm1lbnQge1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IGNvbXB1dGVFbnZpcm9ubWVudE5hbWUgPSBzdGFjay5zcGxpdEFybihjb21wdXRlRW52aXJvbm1lbnRBcm4sIEFybkZvcm1hdC5TTEFTSF9SRVNPVVJDRV9OQU1FKS5yZXNvdXJjZU5hbWUhO1xuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQ29tcHV0ZUVudmlyb25tZW50IHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnRBcm4gPSBjb21wdXRlRW52aXJvbm1lbnRBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50TmFtZSA9IGNvbXB1dGVFbnZpcm9ubWVudE5hbWU7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNvbXB1dGVFbnZpcm9ubWVudEFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGlzIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnROYW1lOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENvbXB1dGVFbnZpcm9ubWVudFByb3BzID0geyBlbmFibGVkOiB0cnVlLCBtYW5hZ2VkOiB0cnVlIH0pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuY29tcHV0ZUVudmlyb25tZW50TmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGlzRmFyZ2F0ZSA9IENvbXB1dGVSZXNvdXJjZVR5cGUuRkFSR0FURSA9PT0gcHJvcHMuY29tcHV0ZVJlc291cmNlcz8udHlwZVxuICAgICAgfHwgQ29tcHV0ZVJlc291cmNlVHlwZS5GQVJHQVRFX1NQT1QgPT09IHByb3BzLmNvbXB1dGVSZXNvdXJjZXM/LnR5cGU7O1xuXG4gICAgdGhpcy52YWxpZGF0ZVByb3BzKHByb3BzLCBpc0ZhcmdhdGUpO1xuXG4gICAgY29uc3Qgc3BvdEZsZWV0Um9sZSA9IHRoaXMuZ2V0U3BvdEZsZWV0Um9sZShwcm9wcyk7XG4gICAgbGV0IGNvbXB1dGVSZXNvdXJjZXM6IENmbkNvbXB1dGVFbnZpcm9ubWVudC5Db21wdXRlUmVzb3VyY2VzUHJvcGVydHkgfCB1bmRlZmluZWQ7XG5cbiAgICAvLyBPbmx5IGFsbG93IGNvbXB1dGUgcmVzb3VyY2VzIHRvIGJlIHNldCB3aGVuIHVzaW5nIE1BTkFHRUQgdHlwZVxuICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzICYmIHRoaXMuaXNNYW5hZ2VkKHByb3BzKSkge1xuICAgICAgY29tcHV0ZVJlc291cmNlcyA9IHtcbiAgICAgICAgYmlkUGVyY2VudGFnZTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlLFxuICAgICAgICBkZXNpcmVkdkNwdXM6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuZGVzaXJlZHZDcHVzLFxuICAgICAgICBlYzJLZXlQYWlyOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmVjMktleVBhaXIsXG4gICAgICAgIGltYWdlSWQ6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW1hZ2UgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbWFnZS5nZXRJbWFnZSh0aGlzKS5pbWFnZUlkLFxuICAgICAgICBsYXVuY2hUZW1wbGF0ZTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5sYXVuY2hUZW1wbGF0ZSxcbiAgICAgICAgbWF4dkNwdXM6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubWF4dkNwdXMgfHwgMjU2LFxuICAgICAgICBwbGFjZW1lbnRHcm91cDogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5wbGFjZW1lbnRHcm91cCxcbiAgICAgICAgc2VjdXJpdHlHcm91cElkczogdGhpcy5idWlsZFNlY3VyaXR5R3JvdXBJZHMocHJvcHMuY29tcHV0ZVJlc291cmNlcy52cGMsIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuc2VjdXJpdHlHcm91cHMpLFxuICAgICAgICBzcG90SWFtRmxlZXRSb2xlOiBzcG90RmxlZXRSb2xlPy5yb2xlQXJuLFxuICAgICAgICBzdWJuZXRzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnZwYy5zZWxlY3RTdWJuZXRzKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudnBjU3VibmV0cykuc3VibmV0SWRzLFxuICAgICAgICB0YWdzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmNvbXB1dGVSZXNvdXJjZXNUYWdzLFxuICAgICAgICB0eXBlOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnR5cGUgfHwgQ29tcHV0ZVJlc291cmNlVHlwZS5PTl9ERU1BTkQsXG4gICAgICAgIC4uLighaXNGYXJnYXRlID8ge1xuICAgICAgICAgIGFsbG9jYXRpb25TdHJhdGVneTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5hbGxvY2F0aW9uU3RyYXRlZ3lcbiAgICAgICAgICAgIHx8IChcbiAgICAgICAgICAgICAgcHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLlNQT1RcbiAgICAgICAgICAgICAgICA/IEFsbG9jYXRpb25TdHJhdGVneS5TUE9UX0NBUEFDSVRZX09QVElNSVpFRFxuICAgICAgICAgICAgICAgIDogQWxsb2NhdGlvblN0cmF0ZWd5LkJFU1RfRklUXG4gICAgICAgICAgICApLFxuICAgICAgICAgIGluc3RhbmNlUm9sZTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVJvbGVcbiAgICAgICAgICAgID8gcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVJvbGVcbiAgICAgICAgICAgIDogbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUodGhpcywgJ0luc3RhbmNlLVByb2ZpbGUnLCB7XG4gICAgICAgICAgICAgIHJvbGVzOiBbbmV3IGlhbS5Sb2xlKHRoaXMsICdFY3MtSW5zdGFuY2UtUm9sZScsIHtcbiAgICAgICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgICAgICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkVDMkNvbnRhaW5lclNlcnZpY2Vmb3JFQzJSb2xlJyksXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSkucm9sZU5hbWVdLFxuICAgICAgICAgICAgfSkuYXR0ckFybixcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzOiB0aGlzLmJ1aWxkSW5zdGFuY2VUeXBlcyhwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmluc3RhbmNlVHlwZXMpLFxuICAgICAgICAgIG1pbnZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzIHx8IDAsXG4gICAgICAgIH0gOiB7fSksXG4gICAgICB9O1xuICAgIH1cblxuICAgIGNvbnN0IGNvbXB1dGVFbnZpcm9ubWVudCA9IG5ldyBDZm5Db21wdXRlRW52aXJvbm1lbnQodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgY29tcHV0ZUVudmlyb25tZW50TmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICBjb21wdXRlUmVzb3VyY2VzLFxuICAgICAgc2VydmljZVJvbGU6IHByb3BzLnNlcnZpY2VSb2xlPy5yb2xlQXJuXG4gICAgICAgID8/IG5ldyBpYW0uUm9sZSh0aGlzLCAnUmVzb3VyY2UtU2VydmljZS1JbnN0YW5jZS1Sb2xlJywge1xuICAgICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTQmF0Y2hTZXJ2aWNlUm9sZScpLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2JhdGNoLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgfSkucm9sZUFybixcbiAgICAgIHN0YXRlOiBwcm9wcy5lbmFibGVkID09PSB1bmRlZmluZWQgPyAnRU5BQkxFRCcgOiAocHJvcHMuZW5hYmxlZCA/ICdFTkFCTEVEJyA6ICdESVNBQkxFRCcpLFxuICAgICAgdHlwZTogdGhpcy5pc01hbmFnZWQocHJvcHMpID8gJ01BTkFHRUQnIDogJ1VOTUFOQUdFRCcsXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcyAmJiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnZwYykge1xuICAgICAgdGhpcy5ub2RlLmFkZERlcGVuZGVuY3kocHJvcHMuY29tcHV0ZVJlc291cmNlcy52cGMpO1xuICAgIH1cblxuICAgIHRoaXMuY29tcHV0ZUVudmlyb25tZW50QXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShjb21wdXRlRW52aXJvbm1lbnQucmVmLCB7XG4gICAgICBzZXJ2aWNlOiAnYmF0Y2gnLFxuICAgICAgcmVzb3VyY2U6ICdjb21wdXRlLWVudmlyb25tZW50JyxcbiAgICAgIHJlc291cmNlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgfSk7XG4gICAgdGhpcy5jb21wdXRlRW52aXJvbm1lbnROYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUoY29tcHV0ZUVudmlyb25tZW50LnJlZik7XG4gIH1cblxuICBwcml2YXRlIGlzTWFuYWdlZChwcm9wczogQ29tcHV0ZUVudmlyb25tZW50UHJvcHMpOiBib29sZWFuIHtcbiAgICByZXR1cm4gcHJvcHMubWFuYWdlZCA/PyB0cnVlO1xuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlcyB0aGUgcHJvcGVydGllcyBwcm92aWRlZCBmb3IgYSBuZXcgYmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVQcm9wcyhwcm9wczogQ29tcHV0ZUVudmlyb25tZW50UHJvcHMsIGlzRmFyZ2F0ZTogYm9vbGVhbikge1xuICAgIGlmIChwcm9wcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKCF0aGlzLmlzTWFuYWdlZChwcm9wcykgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0l0IGlzIG5vdCBhbGxvd2VkIHRvIHNldCBjb21wdXRlUmVzb3VyY2VzIG9uIGFuIEFXUyB1bm1hbmFnZWQgY29tcHV0ZSBlbnZpcm9ubWVudCcpO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmlzTWFuYWdlZChwcm9wcykgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2NvbXB1dGVSZXNvdXJjZXMgaXMgbWlzc2luZyBidXQgcmVxdWlyZWQgb24gYSBtYW5hZ2VkIGNvbXB1dGUgZW52aXJvbm1lbnQnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcykge1xuICAgICAgaWYgKGlzRmFyZ2F0ZSkge1xuICAgICAgICAvLyBWQUxJREFURSBGT1IgRkFSR0FURVxuXG4gICAgICAgIC8vIEJpZCBwZXJjZW50YWdlIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0JpZCBwZXJjZW50YWdlIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gQWxsb2NhdGlvbiBzdHJhdGVneSBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0FsbG9jYXRpb24gc3RyYXRlZ3kgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBEZXNpcmVkIHZDUFVzIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5kZXNpcmVkdkNwdXMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignRGVzaXJlZCB2Q1BVcyBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEltYWdlIElEIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbWFnZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdJbWFnZSBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEluc3RhbmNlIHR5cGVzIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVR5cGVzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0luc3RhbmNlIHR5cGVzIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gRUMyIGtleSBwYWlyIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5lYzJLZXlQYWlyICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VDMiBrZXkgcGFpciBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEluc3RhbmNlIHJvbGUgY2Fubm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBldm52aXJvbm1lbnRzXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmluc3RhbmNlUm9sZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdJbnN0YW5jZSByb2xlIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gTGF1bmNoIHRlbXBsYXRlIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5sYXVuY2hUZW1wbGF0ZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdMYXVuY2ggdGVtcGxhdGUgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBNaW4gdkNQVXMgY2Fubm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBldm52aXJvbm1lbnRzXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pbiB2Q1BVcyBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFBsYWNlbWVudCBncm91cCBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMucGxhY2VtZW50R3JvdXAgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUGxhY2VtZW50IGdyb3VwIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gU3BvdCBmbGVldCByb2xlIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5zcG90RmxlZXRSb2xlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Nwb3QgZmxlZXQgcm9sZSBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gVkFMSURBVEUgRk9SIE9OX0RFTUFORCBBTkQgU1BPVFxuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5taW52Q3B1cykge1xuICAgICAgICAgIC8vIG1pbnZDcHVzIGNhbm5vdCBiZSBsZXNzIHRoYW4gMFxuICAgICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzIDwgMCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaW5pbXVtIHZDcHVzIGZvciBhIGJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnQgY2Fubm90IGJlIGxlc3MgdGhhbiAwJyk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgLy8gbWludkNwdXMgY2Fubm90IGV4Y2VlZCBtYXggdkNwdXNcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5tYXh2Q3B1cyAmJlxuICAgICAgICAgICAgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5taW52Q3B1cyA+IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubWF4dkNwdXMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTWluaW11bSB2Q3B1cyBjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIHRoZSBtYXhpbXVtIHZDcHVzJyk7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIC8vIFNldHRpbmcgYSBiaWQgcGVyY2VudGFnZSBpcyBvbmx5IGFsbG93ZWQgb24gU1BPVCByZXNvdXJjZXMgK1xuICAgICAgICAvLyBDYW5ub3QgdXNlIFNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVEIHdoZW4gdXNpbmcgT05fREVNQU5EXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnR5cGUgPT09IENvbXB1dGVSZXNvdXJjZVR5cGUuT05fREVNQU5EKSB7XG4gICAgICAgICAgLy8gVkFMSURBVEUgRk9SIE9OX0RFTUFORFxuXG4gICAgICAgICAgLy8gQmlkIHBlcmNlbnRhZ2UgaXMgbm90IGFsbG93ZWRcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignU2V0dGluZyB0aGUgYmlkIHBlcmNlbnRhZ2UgaXMgb25seSBhbGxvd2VkIGZvciBTUE9UIHR5cGUgcmVzb3VyY2VzIG9uIGEgYmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudCcpO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIC8vIFNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVEIGFsbG9jYXRpb24gaXMgbm90IGFsbG93ZWRcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5hbGxvY2F0aW9uU3RyYXRlZ3kgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5hbGxvY2F0aW9uU3RyYXRlZ3kgPT09IEFsbG9jYXRpb25TdHJhdGVneS5TUE9UX0NBUEFDSVRZX09QVElNSVpFRCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgU1BPVF9DQVBBQ0lUWV9PUFRJTUlaRUQgYWxsb2NhdGlvbiBzdHJhdGVneSBpcyBvbmx5IGFsbG93ZWQgaWYgdGhlIGVudmlyb25tZW50IGlzIGEgU1BPVCB0eXBlIGNvbXB1dGUgZW52aXJvbm1lbnQnKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLlNQT1QpIHtcbiAgICAgICAgICAvLyBWQUxJREFURSBGT1IgU1BPVFxuXG4gICAgICAgICAgLy8gQmlkIHBlcmNlbnRhZ2UgbXVzdCBiZSBmcm9tIDAgLSAxMDBcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlICE9PSB1bmRlZmluZWQgJiZcbiAgICAgICAgICAgIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmJpZFBlcmNlbnRhZ2UgPCAwIHx8IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYmlkUGVyY2VudGFnZSA+IDEwMCkpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQmlkIHBlcmNlbnRhZ2UgY2FuIG9ubHkgYmUgYSB2YWx1ZSBiZXR3ZWVuIDAgYW5kIDEwMCcpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgYnVpbGRJbnN0YW5jZVR5cGVzKGluc3RhbmNlVHlwZXM/OiBlYzIuSW5zdGFuY2VUeXBlW10pOiBzdHJpbmdbXSB7XG4gICAgaWYgKGluc3RhbmNlVHlwZXMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAgJ29wdGltYWwnLFxuICAgICAgXTtcbiAgICB9XG5cbiAgICByZXR1cm4gaW5zdGFuY2VUeXBlcy5tYXAoKHR5cGU6IGVjMi5JbnN0YW5jZVR5cGUpID0+IHR5cGUudG9TdHJpbmcoKSk7XG4gIH1cblxuICBwcml2YXRlIGJ1aWxkU2VjdXJpdHlHcm91cElkcyh2cGM6IGVjMi5JVnBjLCBzZWN1cml0eUdyb3Vwcz86IGVjMi5JU2VjdXJpdHlHcm91cFtdKTogc3RyaW5nW10gfCB1bmRlZmluZWQge1xuICAgIGlmIChzZWN1cml0eUdyb3VwcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gW1xuICAgICAgICBuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgJ1Jlc291cmNlLVNlY3VyaXR5LUdyb3VwJywgeyB2cGMgfSkuc2VjdXJpdHlHcm91cElkLFxuICAgICAgXTtcbiAgICB9XG5cbiAgICByZXR1cm4gc2VjdXJpdHlHcm91cHMubWFwKChncm91cDogZWMyLklTZWN1cml0eUdyb3VwKSA9PiBncm91cC5zZWN1cml0eUdyb3VwSWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlcyBhbiBBV1MgSUFNIHJvbGUgZm9yIHByb3Zpc2lvbmluZyBzcG90ZmxlZXQgcmVzb3VyY2VzXG4gICAqIGlmIHRoZSBhbGxvY2F0aW9uIHN0cmF0ZWd5IGlzIHNldCB0byBCRVNUX0ZJVCBvciBub3QgZGVmaW5lZC5cbiAgICpcbiAgICogQHBhcmFtIHByb3BzIC0gdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQgY29uc3RydWN0IHByb3BlcnRpZXNcbiAgICovXG4gIHByaXZhdGUgZ2V0U3BvdEZsZWV0Um9sZShwcm9wczogQ29tcHV0ZUVudmlyb25tZW50UHJvcHMpOiBpYW0uSVJvbGUgfCB1bmRlZmluZWQge1xuICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzPy5hbGxvY2F0aW9uU3RyYXRlZ3kgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5hbGxvY2F0aW9uU3RyYXRlZ3kgIT09IEFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcykge1xuICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuc3BvdEZsZWV0Um9sZSkge1xuICAgICAgICByZXR1cm4gcHJvcHMuY29tcHV0ZVJlc291cmNlcy5zcG90RmxlZXRSb2xlO1xuICAgICAgfSBlbHNlIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnR5cGUgPT09IENvbXB1dGVSZXNvdXJjZVR5cGUuU1BPVCkge1xuICAgICAgICByZXR1cm4gaWFtLlJvbGUuZnJvbVJvbGVBcm4odGhpcywgJ1Jlc291cmNlLVNwb3RGbGVldC1Sb2xlJyxcbiAgICAgICAgICBgYXJuOiR7dGhpcy5zdGFjay5wYXJ0aXRpb259OmlhbTo6JHt0aGlzLnN0YWNrLmFjY291bnR9OnJvbGUvYXdzLXNlcnZpY2Utcm9sZS9zcG90ZmxlZXQuYW1hem9uYXdzLmNvbS9BV1NTZXJ2aWNlUm9sZUZvckVDMlNwb3RGbGVldGApO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cbn1cbiJdfQ==