"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
var KeyLength;
(function (KeyLength) {
    KeyLength[KeyLength["L2048"] = 2048] = "L2048";
    KeyLength[KeyLength["L4096"] = 4096] = "L4096";
})(KeyLength = exports.KeyLength || (exports.KeyLength = {}));
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
* An EC2 Key Pair
*/
class KeyPair extends cdk.Construct {
    /**
    * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
    */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
        * ARN of the private key in AWS Secrets Manager
        */
        this.arn = '';
        /**
        * Name of the Key Pair
        */
        this.name = '';
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        const tags = props.tags || {};
        tags.CreatedBy = ID;
        const key = new cfn.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                name: props.name,
                description: props.description || '',
                keyLength: props.keyLength || KeyLength.L2048,
                kms: ((_a = props.kms) === null || _a === void 0 ? void 0 : _a.keyArn) || 'alias/aws/secretsmanager',
                StackName: stack,
                tags: tags,
            }
        });
        this.arn = key.getAttString('PrivateKeyARN');
        this.name = key.getAttString('KeyPairName');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'ec2:CreateKeyPair',
                        'ec2:DeleteKeyPair',
                        'ec2:DescribeKeyPairs',
                        'secretsmanager:ListSecrets',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:TagResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:RequestTag/CreatedBy': ID,
                        }
                    },
                }),
                new iam.PolicyStatement({
                    actions: [
                        'secretsmanager:DeleteResourcePolicy',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:DescribeSecret',
                        'secretsmanager:GetResourcePolicy',
                        'secretsmanager:ListSecretVersionIds',
                        'secretsmanager:PutResourcePolicy',
                        'secretsmanager:PutSecretValue',
                        'secretsmanager:RestoreSecret',
                        'secretsmanager:UntagResource',
                        'secretsmanager:UpdateSecret',
                        'secretsmanager:UpdateSecretVersionStage',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:ResourceTag/CreatedBy': ID,
                        }
                    },
                }),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ]
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout)
        });
        return fn;
    }
    /**
    * Grants read access to the private key in the AWS Secrets Manager
    */
    grantRead(grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [this.arn],
            scope: this
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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