# LLM-as-a-Judge 활용 가이드

HRET(Haerae Evaluation Toolkit)는 정답이 정해져 있지 않은 생성형 작업을 평가하기 위해 강력한 언어 모델을 '심판(Judge)'으로 활용하는 **LLM-as-a-Judge** 기능을 제공합니다. 이 가이드에서는 k2\_eval 데이터셋을 평가하는 실제 예제를 통해 단계별 사용 방법을 설명합니다.

## 목차

1. [LLM-as-a-Judge란?](#llm-as-a-judget란)
2. [사용 방법](#사용-방법)
3. [주요 Judge 유형 및 프롬프트](#주요-judge-유형-및-프롬프트)
4. [결과 확인 및 분석](#결과-확인-및-분석)
5. [주의사항 및 팁](#주의사항-및-팁)

---

## 1. LLM-as-a-Judge란?

* 정답이 정해져 있지 않거나 단순 문자열 비교가 어려운 **생성형 작업**을 평가할 때, GPT-4o와 같은 강력한 LLM을 채점자(Judge)로 활용하는 방식입니다.
* HRET에서는 `LLMJudgeEvaluator`를 통해 아래 기능을 제공하여, 평가 기준이 담긴 프롬프트를 기반으로 심판 모델을 지정해 자동화된 평가를 수행할 수 있습니다.

## 2. 사용 방법

### 설정해야 할 3가지 요소

1. **평가 대상 모델**: 평가할 모델 백엔드 (예: 로컬 vLLM 서버로 실행 중인 EXAONE 모델)
2. **심판 모델**: 채점에 사용할 LLM (예: gpt-4o)
3. **데이터셋**: 평가에 사용할 데이터셋 (예: `k2_eval`)

### 전체 실행 예제

```python
import os
from llm_eval.evaluator import Evaluator
import pandas as pd

# 1) Evaluator 인스턴스 생성
evaluator = Evaluator()

# API 키는 환경 변수에서 가져옵니다.
openai_api_key = os.getenv("OPENAI_API_KEY")

# 2) LLM-as-a-Judge 평가 실행
results = evaluator.run(
    # 평가 대상 모델 설정
    model="openai",  # vLLM을 OpenAI 호환 API로 사용
    model_params={
        "api_base": "http://0.0.0.0:8000/v1/chat/completions",
        "model_name": "LGAI-EXAONE/EXAONE-3.5-2.4B-Instruct",
        "batch_size": 3
    },

    # 데이터셋 설정
    dataset="k2_eval",
    split="test",

    # 심판 모델 및 평가 방식 설정
    evaluation_method="llm_judge",
    judge_model="openai_judge",  # 심판으로 OpenAI 모델 사용
    judge_params={
        "api_base": "https://api.openai.com/v1",
        "model_name": "gpt-4o",
        "api_key": openai_api_key,
        "batch_size": 5
    },
    
    # k2_eval용 프롬프트와 파서 사용
    evaluator_params={"default_judge_type": "k2_eval"},
)

# 3) 결과 확인 및 저장
print(results)
if results:
    res_df = results.to_dataframe()
    print(res_df.head())
    res_df.to_parquet('./exaone3b_judged_results.parquet')
```

## 3. 주요 Judge 유형 및 프롬프트

`evaluator_params["default_judge_type"]`를 변경하여 다양한 채점 방식을 선택할 수 있습니다.

| Judge 유형                   | 설명                                       | 사용 데이터셋 예시      |
| -------------------------- | ---------------------------------------- | --------------- |
| `k2_eval`                  | K2-Eval 전용. 유창성, 논리성 등 4가지 기준으로 1\~5점 평가 | k2\_eval        |
| `rubric_and_response`      | 주어진 루브릭에 따라 모델 답변을 1\~5점 척도로 평가          | 모든 생성형 태스크      |
| `response_comparison`      | 두 모델(A, B) 답변 중 더 나은 쪽 선택                | KUDGE Pointwise |
| `rubric_response_and_gold` | 정답(gold)과 비교해 맞으면 `true`, 아니면 `false` 반환 | KUDGE Pairwise  |

### k2\_eval 프롬프트 구조

```text
You are tasked with evaluating the responses generated by a Korean language model.
Please evaluate the response according to the following criteria:
1. Completeness: Does the response fully address the question?
2. Logical Consistency: Is the reasoning clear and logically sound?
3. Fluency: Is the language natural and coherent?
4. Cultural Appropriateness: Does the response adhere to Korean linguistic and cultural norms?

Provide your evaluation in 4-5 sentences, then assign a score from 1 to 5 in the following format:
Evaluation: [Your evaluation here]
Score: [Numeric score here]

Below is the question and the model's response. Please evaluate the response based on the criteria provided.

**Question:**
{input}

**Model's Response:**
{response}

[[End of conversation. Begin evaluation.]]
```

## 4. 결과 확인 및 분석

* `EvaluationResult` 객체 반환
* `to_dataframe()`으로 DataFrame 생성

**주요 컬럼 예시**

* `judge_evaluation`: 원본 평가 텍스트
* `judge_score`: 파싱된 점수
* `language_penalizer`: 한국어 외 포함 비율

```python
# 예: 언어 패널티 적용 후 점수 조정
res['adjusted_judge_score'] = res.apply(
    lambda row: max(row['judge_score'] - 1, 1.0)
                if row['language_penalizer'] != 1.0 else row['judge_score'],
    axis=1
)
final_score = res['adjusted_judge_score'].mean()
print(f"언어 패널티 적용 후 최종 평균 점수: {final_score}")
```

## 5. 주의사항 및 팁

* **파서 안정성**: 심판 출력 형식이 바뀌면 파싱 오류가 발생할 수 있습니다. `ResponseParser`를 확장해 다양한 포맷에 대응하세요.

