# LLM-as-a-Judge Usage Guide

HRET (HaeRae Evaluation Toolkit) offers a powerful **LLM-as-a-Judge** feature that leverages large language models as automated assessors for generative tasks without fixed answers. In this guide, we demonstrate step-by-step usage with the `k2_eval` dataset.

## Table of Contents

1. [What is LLM-as-a-Judge?](#what-is-llm-as-a-judge)
2. [How to Use](#how-to-use)
3. [Judge Types & Prompts](#judge-types--prompts)
4. [Reviewing & Analyzing Results](#reviewing--analyzing-results)
5. [Notes & Tips](#notes--tips)

---

## 1. What is LLM-as-a-Judge?

* For generative tasks without clear ground truths or where simple string matching falls short, you can employ a powerful LLM (e.g., GPT-4o) as an automated judge.
* HRET’s `LLMJudgeEvaluator` supports this by allowing you to specify a judge model and prompt, then run automated scoring based on your evaluation criteria.

## 2. How to Use

### Three Required Components

1. **Target Model**: The model under evaluation (e.g., EXAONE running on a local vLLM server).
2. **Judge Model**: The LLM used for scoring (e.g., GPT-4o).
3. **Dataset**: The evaluation dataset (e.g., `k2_eval`).

### Full Example

```python
import os
from llm_eval.evaluator import Evaluator

# 1) Initialize evaluator
evaluator = Evaluator()

# Load API key from environment
openai_api_key = os.getenv("OPENAI_API_KEY")

# 2) Run LLM-as-a-Judge evaluation
results = evaluator.run(
    # Target model settings
    model="openai",  # Use the local vLLM server via OpenAI-compatible API
    model_params={
        "api_base": "http://0.0.0.0:8000/v1/chat/completions",
        "model_name": "LGAI-EXAONE/EXAONE-3.5-2.4B-Instruct",
        "batch_size": 3
    },

    # Dataset settings
    dataset="k2_eval",
    split="test",

    # Judge settings
    evaluation_method="llm_judge",
    judge_model="openai_judge",  # Use OpenAI model as judge
    judge_params={
        "api_base": "https://api.openai.com/v1",
        "model_name": "gpt-4o",
        "api_key": openai_api_key,
        "batch_size": 5
    },

    # Use k2_eval default prompt and parser
    evaluator_params={"default_judge_type": "k2_eval"}
)

# 3) Review and save results
print(results)
if results:
    df = results.to_dataframe()
    print(df.head())
    df.to_parquet('./exaone3b_judged_results.parquet')
```

---

## 3. Judge Types & Prompts

Select different scoring schemes by setting `evaluator_params["default_judge_type"]`.

| Judge Type                 | Description                                                      | Example Dataset     |
| -------------------------- | ---------------------------------------------------------------- | ------------------- |
| `k2_eval`                  | K2-Eval: score 1–5 on Completeness, Logic, Fluency, Cultural Fit | `k2_eval`           |
| `rubric_and_response`      | 1–5 scale based on a provided rubric                             | Any generative task |
| `response_comparison`      | Choose the better answer between two model outputs               | KUDGE Pointwise     |
| `rubric_response_and_gold` | Compare to gold response, return `true` or `false`               | KUDGE Pairwise      |

### k2\_eval Prompt Template

```text
You are tasked with evaluating the responses generated by a Korean language model.
Please evaluate the response according to the following criteria:
1. Completeness: Does the response fully address the question?
2. Logical Consistency: Is the reasoning clear and logically sound?
3. Fluency: Is the language natural and coherent?
4. Cultural Appropriateness: Does the response adhere to Korean linguistic and cultural norms?

Provide your evaluation in 4–5 sentences, then assign a score from 1 to 5 in the following format:
Evaluation: [Your evaluation here]
Score: [Numeric score here]

Below is the question and the model's response. Please evaluate the response based on the criteria provided.

**Question:**
{input}

**Model's Response:**
{response}

[[End of conversation. Begin evaluation.]]
```

---

## 4. Reviewing & Analyzing Results

* The evaluator returns an `EvaluationResult` object.
* Call `to_dataframe()` to convert to a pandas DataFrame.

**Key Columns**

* `judge_evaluation`: Raw evaluation text.
* `judge_score`: Parsed numeric score.
* `language_penalizer`: Ratio of non-Korean content.

```python
# Example: adjust score if a language penalty applies
df['adjusted_score'] = df.apply(
    lambda row: max(row['judge_score'] - 1, 1.0)
                if row['language_penalizer'] != 1.0 else row['judge_score'],
    axis=1
)
mean_score = df['adjusted_score'].mean()
print(f"Final average score after penalty: {mean_score}")
```

---

## 5. Notes & Tips

* **Parser Robustness**: If the judge’s output format changes, parsing may fail. Extend `ResponseParser` to handle multiple formats.
