// Copyright (c) Mito
import React, { useEffect, useState } from 'react';
import "../../../css/tour.css";
import XIcon from '../icons/XIcon';
import TextButton from '../elements/TextButton';
import { TourName, TourPopupLocation, tours } from './Tours';
import { classNames } from '../../utils/classNames';
import Row from '../spacing/Row';
import Col from '../spacing/Col';
const locationToClassNamesMapping = {
    [TourPopupLocation.BOTTOM_LEFT]: 'tour-container-bottom-left',
    [TourPopupLocation.BOTTOM_RIGHT]: 'tour-container-bottom-right',
    [TourPopupLocation.TOP_LEFT]: 'tour-container-top-left',
    [TourPopupLocation.TOP_RIGHT]: 'tour-container-top-right',
};
const Tour = (props) => {
    var _a;
    const [stepNumber, setStepNumber] = useState(0);
    const [skippedTour, setSkippedTour] = useState(false);
    // Construct full list of steps to show by appending the steps of each tour
    // we want to display together. 
    const steps = [];
    props.tourNames.forEach((tourName) => {
        steps.push(...tours[tourName]);
    });
    // Log the beginning of the tour. Because the useEffect has [] as
    // its second param, it only runs once. 
    useEffect(() => {
        void props.mitoAPI.log('begin_tour', {
            'tour_names': props.tourNames,
            'total_number_of_tour_steps': steps.length
        });
    }, []);
    /*
        Each tour is able to register actions that occur at the beginning of specific steps.
        The useEffect is fired each time the stepNumber changes.
    */
    useEffect(() => {
        if (steps[stepNumber].tourName === TourName.PIVOT && steps[stepNumber].stepNumber === 1) {
            props.setHighlightPivotTableButton(true);
        }
        else {
            props.setHighlightPivotTableButton(false);
            props.setHighlightAddColButton(false);
        }
    }, [stepNumber]);
    // Go to stepNumber if it exists, otherwise close the tour
    const goToStep = (newStepNumber) => {
        if (newStepNumber <= steps.length - 1) {
            // Log switching steps
            void props.mitoAPI.log('switched_tour_step', {
                'old_tour_name': steps[stepNumber].tourName,
                'old_tour_step': stepNumber,
                'new_tour_name': steps[newStepNumber].tourName,
                'new_tour_step': newStepNumber,
                'tour_names': props.tourNames,
                'total_number_of_tour_steps': steps.length
            });
            // Update the step number
            setStepNumber(newStepNumber);
        }
        else {
            void closeTour();
        }
    };
    // Called if you close the tour before you are on the final step
    const closeTourEarly = async () => {
        void props.mitoAPI.log('closed_tour_early', {
            'tour_names': props.tourNames,
            'tour_name': steps[stepNumber].tourName,
            'relative_tour_step_number': steps[stepNumber].stepNumber,
            'absolute_tour_step_number': stepNumber,
            'total_number_of_tour_steps': steps.length // Log the length of the entire tour if they had taken it all
        });
        // Mark that the user skipped the tour so we don't log a finished_tour event
        setSkippedTour(true);
        if (steps[stepNumber].tourName === TourName.INTRO) {
            // Display the final tutorial step if this is the intro tour always
            setStepNumber(steps.length - 1);
        }
        else {
            // Otherwise end the tour
            void closeTour(true);
        }
    };
    // Called if you close the tour on the final step
    const closeTour = async (_skippedTour) => {
        // Log finishing the tour, if you didn't skip to the final step
        if (!skippedTour && !_skippedTour) {
            void props.mitoAPI.log('finished_tour', {
                'tour_names': props.tourNames,
                'total_number_of_tour_steps': steps.length
            });
        }
        // Make sure that the pivot & Add Col button is not stuck in pulsing mode. 
        props.setHighlightPivotTableButton(false);
        props.setHighlightAddColButton(false);
        // Send the closeTour message to the backend. We still record each tour individually
        // so we can later add new tours to display to users
        await props.mitoAPI.updateCloseTour(props.tourNames);
    };
    /*
        Retrieve the finalStepText, which is the text displayed in the body of the tour popup.
        Although not enforced by the type system, a tour step should either have a defined stepText of stepTextFunction.
    */
    const stepText = steps[stepNumber].stepText;
    const stepTextFunction = steps[stepNumber].stepTextFunction;
    const finalStepText = stepText || (stepTextFunction && stepTextFunction(((_a = props.sheetData) === null || _a === void 0 ? void 0 : _a.data[0].columnID) || ''));
    const hideXIcon = steps[stepNumber].hideXIcon === true;
    // If we're at a valid step number
    return (React.createElement("div", { className: classNames('tour-container', locationToClassNamesMapping[steps[stepNumber].location]), key: stepNumber },
        React.createElement(Row, { justify: 'space-between', align: 'center' },
            React.createElement(Col, null,
                React.createElement("p", { className: 'text-header-2' },
                    stepNumber + 1,
                    "/",
                    steps.length)),
            !hideXIcon &&
                React.createElement(Col, null,
                    React.createElement(XIcon, { variant: 'light', onClick: async () => {
                            if (stepNumber >= steps.length - 1) {
                                await closeTour();
                            }
                            else {
                                await closeTourEarly();
                            }
                        } }))),
        React.createElement(Row, null,
            React.createElement(Col, null,
                React.createElement("p", { className: 'text-header-2 text-color-white-important' }, steps[stepNumber].stepHeader))),
        React.createElement("div", { className: 'text-overflow-wrap mb-20px' }, finalStepText),
        React.createElement(Row, { justify: 'space-between' },
            stepNumber - 1 >= 0 &&
                React.createElement(Col, null,
                    React.createElement(TextButton, { variant: 'dark', width: 'small', onClick: () => goToStep(stepNumber - 1) }, "Back")),
            stepNumber - 1 <= 0 &&
                // Add a space so that the advance button is always in the same place
                React.createElement(Col, null),
            React.createElement(Col, null,
                React.createElement(TextButton, { variant: 'light', width: 'small', onClick: () => goToStep(stepNumber + 1) }, steps[stepNumber].advanceButtonText)))));
};
export default Tour;
//# sourceMappingURL=Tour.js.map