// Copyright (c) Mito
import React, { useEffect, useRef, useState } from 'react';
import { classNames } from '../../utils/classNames';
import Input from '../elements/Input';
import { focusGrid } from '../endo/focusUtils';
// import icons
import SelectedSheetTabDropdownIcon from '../icons/SelectedSheetTabDropdownIcon';
import UnselectedSheetTabDropdownIcon from '../icons/UnselectedSheetTabDropdownIcon';
import { TaskpaneType } from '../taskpanes/taskpanes';
import { ModalEnum } from '../modals/modals';
import GraphIcon from '../icons/GraphIcon';
import DataSheetTabActions from './DataSheetTabActions';
import GraphSheetTabActions from './GraphSheetTabActions';
export const selectPreviousGraphSheetTab = (graphDataDict, prevGraphIndex, setUIState) => {
    /*
        Try to select the graph that is at the same index as the previous graph.
        If no graph exists at that index, then select the graph at the previous index.
        If there are no graphs, then select the last sheet index
    */
    const graphIDs = Object.keys(graphDataDict || {});
    let newGraphID = undefined;
    if (graphIDs.length > prevGraphIndex) {
        // If the the number of graphIDs is larger than the prevGraphIndex, 
        // then a new graphID is at the prevGraphIndex
        newGraphID = graphIDs[prevGraphIndex];
    }
    else if (graphIDs.length > 0) {
        // Otherwise if the prevGraphIndex was the highest index graphID 
        // and there is another graph, get the previous index
        newGraphID = graphIDs[prevGraphIndex - 1];
    }
    if (newGraphID !== undefined) {
        // If there is a graph, then keep displaying graphs, otherwise display a data tab
        // Safely mark as GraphID because of the check above that the compiler is unable to understand
        const _newGraphID = newGraphID;
        setUIState((prevUIState) => {
            return Object.assign(Object.assign({}, prevUIState), { selectedGraphID: _newGraphID, selectedTabType: 'graph', currOpenTaskpane: { type: TaskpaneType.GRAPH, graphID: _newGraphID } });
        });
        return _newGraphID;
    }
    else {
        // If there are no more graphs, close the graph taskpane and display a data sheet instead
        setUIState((prevUIState) => {
            return Object.assign(Object.assign({}, prevUIState), { selectedGraphID: undefined, selectedTabType: 'data', currOpenTaskpane: { type: TaskpaneType.NONE } });
        });
        return undefined;
    }
};
/*
    Component that displays a dataframe name at the bottom of the sheet, and
    furthermore renders the sheet actions if the sheet action dropdown is
    clicked.
*/
export default function SheetTab(props) {
    // We only set this as open if it the currOpenSheetTabActions
    const openDropdownDivRef = useRef(null);
    const [displayActions, setDisplayActions] = useState(false);
    const [isRename, setIsRename] = useState(false);
    const [newTabName, setNewTabName] = useState(props.tabName);
    // Make sure that if we change the sheet tab name that is displayed, we default to 
    // the correct new name as well
    useEffect(() => {
        setNewTabName(props.tabName);
    }, [props.tabName]);
    const onRename = async () => {
        var _a;
        void props.closeOpenEditingPopups(); // First, make sure to close open editing popups
        if (props.tabIDObj.tabType === 'data') {
            await props.mitoAPI.editDataframeRename(props.tabIDObj.sheetIndex, newTabName);
        }
        else {
            await props.mitoAPI.editGraphRename(props.tabIDObj.graphID, newTabName);
        }
        setDisplayActions(false);
        setIsRename(false);
        // Focus back on the grid
        const endoGridContainer = (_a = props.mitoContainerRef.current) === null || _a === void 0 ? void 0 : _a.querySelector('.endo-grid-container');
        focusGrid(endoGridContainer);
    };
    return (React.createElement("div", { className: classNames('tab', { 'tab-graph': props.tabIDObj.tabType === 'graph' }, { 'tab-selected': props.isSelectedTab }), onClick: () => {
            if (props.tabIDObj.tabType === 'graph') {
                // If opening a graph tab, close the cell editor 
                props.setEditorState(undefined);
            }
            props.setUIState(prevUIState => {
                if (props.tabIDObj.tabType === 'data') {
                    // If the user clicks on a data sheet tab, switch to it and make sure the graph taskpane is not open
                    const taskpaneInfo = prevUIState.currOpenTaskpane.type === TaskpaneType.GRAPH ?
                        { type: TaskpaneType.NONE } : prevUIState.currOpenTaskpane;
                    return Object.assign(Object.assign({}, prevUIState), { selectedTabType: 'data', selectedSheetIndex: props.tabIDObj.sheetIndex, currOpenTaskpane: taskpaneInfo });
                }
                else {
                    return Object.assign(Object.assign({}, prevUIState), { selectedTabType: 'graph', selectedGraphID: props.tabIDObj.graphID, currOpenModal: { type: ModalEnum.None }, currOpenTaskpane: {
                            type: TaskpaneType.GRAPH,
                            graphID: props.tabIDObj.graphID
                        } });
                }
            });
        }, onDoubleClick: () => { setIsRename(true); }, onContextMenu: (e) => {
            var _a;
            // If the user right clicks, show the dropdown for the sheet tabs
            e.preventDefault();
            (_a = openDropdownDivRef.current) === null || _a === void 0 ? void 0 : _a.click();
        } },
        React.createElement("div", { className: 'tab-content' },
            props.tabIDObj.tabType === 'graph' &&
                /* Put it inside a div so everything is spaced correctly */
                React.createElement("div", null,
                    React.createElement(GraphIcon, { variant: props.isSelectedTab ? 'light' : undefined })),
            isRename &&
                React.createElement("form", { onSubmit: async (e) => { e.preventDefault(); await onRename(); }, onBlur: onRename },
                    React.createElement(Input, { value: newTabName, onChange: (e) => { setNewTabName(e.target.value); }, autoFocus: true })),
            !isRename &&
                React.createElement("p", null, props.tabName),
            React.createElement("div", { ref: openDropdownDivRef, className: 'sheet-tab-dropdown-button-div', onClick: () => { setDisplayActions(true); } }, props.isSelectedTab ? React.createElement(SelectedSheetTabDropdownIcon, null) : React.createElement(UnselectedSheetTabDropdownIcon, null))),
        props.tabIDObj.tabType === 'data' &&
            React.createElement(DataSheetTabActions, { setDisplayActions: setDisplayActions, setUIState: props.setUIState, closeOpenEditingPopups: props.closeOpenEditingPopups, setIsRename: setIsRename, sheetIndex: props.tabIDObj.sheetIndex, mitoAPI: props.mitoAPI, graphDataDict: props.graphDataDict, sheetDataArray: props.sheetDataArray, display: displayActions && props.tabIDObj.tabType === 'data' }),
        props.tabIDObj.tabType === 'graph' &&
            React.createElement(GraphSheetTabActions, { setDisplayActions: setDisplayActions, setUIState: props.setUIState, closeOpenEditingPopups: props.closeOpenEditingPopups, setIsRename: setIsRename, graphID: props.tabIDObj.graphID, mitoAPI: props.mitoAPI, graphDataDict: props.graphDataDict, display: displayActions && props.tabIDObj.tabType === 'graph' })));
}
//# sourceMappingURL=SheetTab.js.map