from logging import StreamHandler

import jwt
from functools import wraps
from firebase_admin import firestore
from flask import request, abort
from os import getenv
import logging

logger = logging.getLogger("ace_authorize")
handler = StreamHandler()
formatter = logging.Formatter('[%(levelname)s] [%(name)s] %(message)s')
handler.setFormatter(formatter)
logger.addHandler(handler)


def authorize(scope):
    def decorator(endpoint_func):
        @wraps(endpoint_func)
        def decorated_function(*args, **kwargs):
            tenant_id = kwargs.get('tenant_id')

            if tenant_id is not None:
                scopes = get_scopes(tenant_id)
                if scope in scopes:
                    return endpoint_func(*args, **kwargs)
                else:
                    return abort(403, "NOT ALLOWED")
            # TODO: (cjk) Implement flow for (admin) users without a tenant
            else:
                return abort(501, "NOT IMPLEMENTED")
        return decorated_function
    return decorator


def get_scopes(tenant_id):
    # TODO: (cjk) Check for jwt payload in the headers when dealing with requests passed through an api gateway
    # jwt_payload = request.headers.get("X-Apigateway-Api-Userinfo")
    auth_header = request.headers.get("Authorization")

    if auth_header:
        auth_header = auth_header.rpartition(' ')[2]
        decoded_token = jwt.decode(auth_header, options={"verify_signature": False})
        user_roles = decoded_token.get('roles')
        host = getenv("ROLES_HOST")

        if tenant_id == decoded_token.get('firebase').get('tenant'):
            if host:
                roles = get_roles()
                scopes = []
                for role in roles:
                    if role['name'] in user_roles:
                        scopes.extend(role["scopes"])
                return scopes
            else:
                logger.error("Failed to retrieve roles. ROLES_HOST is not set.")
                return abort(500, "INTERNAL SERVER ERROR")
    return abort(403, "NOT ALLOWED")


def get_roles():
    db = firestore.client()
    roles = []

    roles_ref = db.collection(u'roles')
    docs = roles_ref.stream()

    for doc in docs:
        roles.append(doc.to_dict())

    return roles
