"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_s3_1 = require("@aws-cdk/aws-s3");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../../core/lib");
const ec2 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'can create Windows user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.forWindows();
        userData.addCommands('command1', 'command2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>command1\ncommand2</powershell>');
        test.done();
    },
    'can create Windows user data with commands on exit'(test) {
        // GIVEN
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addCommands('command1', 'command2');
        userData.addOnExitCommands('onexit1', 'onexit2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>trap {\n' +
            '$success=($PSItem.Exception.Message -eq "Success")\n' +
            'onexit1\n' +
            'onexit2\n' +
            'break\n' +
            '}\n' +
            'command1\n' +
            'command2\n' +
            'throw "Success"</powershell>');
        test.done();
    },
    'can create Windows with Signal Command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addSignalOnExitCommand(resource);
        userData.addCommands('command1');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>trap {\n' +
            '$success=($PSItem.Exception.Message -eq "Success")\n' +
            'cfn-signal --stack Default --resource RESOURCE1989552F --region ${Token[AWS::Region.4]} --success ($success.ToString().ToLower())\n' +
            'break\n' +
            '}\n' +
            'command1\n' +
            'throw "Success"</powershell>');
        test.done();
    },
    'can windows userdata download S3 files'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.forWindows();
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        const bucket2 = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket2', 'test2');
        // WHEN
        userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.bat',
        });
        userData.addS3DownloadCommand({
            bucket: bucket2,
            bucketKey: 'filename2.bat',
            localFile: 'c:\\test\\location\\otherScript.bat',
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>mkdir (Split-Path -Path \'C:/temp/filename.bat\' ) -ea 0\n' +
            'Read-S3Object -BucketName \'test\' -key \'filename.bat\' -file \'C:/temp/filename.bat\' -ErrorAction Stop\n' +
            'mkdir (Split-Path -Path \'c:\\test\\location\\otherScript.bat\' ) -ea 0\n' +
            'Read-S3Object -BucketName \'test2\' -key \'filename2.bat\' -file \'c:\\test\\location\\otherScript.bat\' -ErrorAction Stop</powershell>');
        test.done();
    },
    'can windows userdata execute files'(test) {
        // GIVEN
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addExecuteFileCommand({
            filePath: 'C:\\test\\filename.bat',
        });
        userData.addExecuteFileCommand({
            filePath: 'C:\\test\\filename2.bat',
            arguments: 'arg1 arg2 -arg $variable',
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>&\'C:\\test\\filename.bat\' undefined\n' +
            'if (!$?) { Write-Error \'Failed to execute the file "C:\\test\\filename.bat"\' -ErrorAction Stop }\n' +
            '&\'C:\\test\\filename2.bat\' arg1 arg2 -arg $variable\n' +
            'if (!$?) { Write-Error \'Failed to execute the file "C:\\test\\filename2.bat"\' -ErrorAction Stop }</powershell>');
        test.done();
    },
    'can create Linux user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.forLinux();
        userData.addCommands('command1', 'command2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\ncommand1\ncommand2');
        test.done();
    },
    'can create Linux user data with commands on exit'(test) {
        // GIVEN
        const userData = ec2.UserData.forLinux();
        // WHEN
        userData.addCommands('command1', 'command2');
        userData.addOnExitCommands('onexit1', 'onexit2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            'onexit1\n' +
            'onexit2\n' +
            '}\n' +
            'trap exitTrap EXIT\n' +
            'command1\n' +
            'command2');
        test.done();
    },
    'can create Linux with Signal Command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        // WHEN
        const userData = ec2.UserData.forLinux();
        userData.addCommands('command1');
        userData.addSignalOnExitCommand(resource);
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Default --resource RESOURCE1989552F --region ${Token[AWS::Region.4]} -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            '}\n' +
            'trap exitTrap EXIT\n' +
            'command1');
        test.done();
    },
    'can linux userdata download S3 files'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.forLinux();
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        const bucket2 = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket2', 'test2');
        // WHEN
        userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.sh',
        });
        userData.addS3DownloadCommand({
            bucket: bucket2,
            bucketKey: 'filename2.sh',
            localFile: 'c:\\test\\location\\otherScript.sh',
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'mkdir -p $(dirname \'/tmp/filename.sh\')\n' +
            'aws s3 cp \'s3://test/filename.sh\' \'/tmp/filename.sh\'\n' +
            'mkdir -p $(dirname \'c:\\test\\location\\otherScript.sh\')\n' +
            'aws s3 cp \'s3://test2/filename2.sh\' \'c:\\test\\location\\otherScript.sh\'');
        test.done();
    },
    'can linux userdata execute files'(test) {
        // GIVEN
        const userData = ec2.UserData.forLinux();
        // WHEN
        userData.addExecuteFileCommand({
            filePath: '/tmp/filename.sh',
        });
        userData.addExecuteFileCommand({
            filePath: '/test/filename2.sh',
            arguments: 'arg1 arg2 -arg $variable',
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'set -e\n' +
            'chmod +x \'/tmp/filename.sh\'\n' +
            '\'/tmp/filename.sh\' undefined\n' +
            'set -e\n' +
            'chmod +x \'/test/filename2.sh\'\n' +
            '\'/test/filename2.sh\' arg1 arg2 -arg $variable');
        test.done();
    },
    'can create Custom user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.custom('Some\nmultiline\ncontent');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, 'Some\nmultiline\ncontent');
        test.done();
    },
    'Custom user data throws when adding on exit commands'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.custom('');
        // THEN
        test.throws(() => userData.addOnExitCommands('a command goes here'));
        test.done();
    },
    'Custom user data throws when adding signal command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        // WHEN
        const userData = ec2.UserData.custom('');
        // THEN
        test.throws(() => userData.addSignalOnExitCommand(resource));
        test.done();
    },
    'Custom user data throws when downloading file'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.custom('');
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        // WHEN
        // THEN
        test.throws(() => userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.sh',
        }));
        test.done();
    },
    'Custom user data throws when executing file'(test) {
        // GIVEN
        const userData = ec2.UserData.custom('');
        // WHEN
        // THEN
        test.throws(() => userData.addExecuteFileCommand({
            filePath: '/tmp/filename.sh',
        }));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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