import { Construct } from '@aws-cdk/core';
import { UserData } from './user-data';
import { WindowsVersion } from './windows-versions';
/**
 * Interface for classes that can select an appropriate machine image to use
 */
export interface IMachineImage {
    /**
     * Return the image to use in the given context
     */
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
export declare abstract class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version: WindowsVersion, props?: WindowsImageProps): IMachineImage;
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props?: AmazonLinuxImageProps): IMachineImage;
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap: Record<string, string>, props?: GenericLinuxImageProps): IMachineImage;
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap: Record<string, string>, props?: GenericWindowsImageProps): IMachineImage;
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static lookup(props: LookupMachineImageProps): IMachineImage;
}
/**
 * Configuration for a machine image
 */
export interface MachineImageConfig {
    /**
     * The AMI ID of the image to use
     */
    readonly imageId: string;
    /**
     * Operating system type for this image
     */
    readonly osType: OperatingSystemType;
    /**
     * Initial UserData for this image
     */
    readonly userData: UserData;
}
/**
 * Configuration options for WindowsImage
 */
export interface WindowsImageProps {
    /**
     * Initial user data
     *
     * @default - Empty UserData for Windows machines
     */
    readonly userData?: UserData;
}
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
export declare class WindowsImage implements IMachineImage {
    private readonly version;
    private readonly props;
    constructor(version: WindowsVersion, props?: WindowsImageProps);
    /**
     * Return the image to use in the given context
     */
    getImage(scope: Construct): MachineImageConfig;
    /**
     * Construct the SSM parameter name for the given Windows image
     */
    private imageParameterName;
}
/**
 * CPU type
 */
export declare enum AmazonLinuxCpuType {
    /**
     * arm64 CPU type
     */
    ARM_64 = "arm64",
    /**
     * x86_64 CPU type
     */
    X86_64 = "x86_64"
}
/**
 * Amazon Linux image properties
 */
export interface AmazonLinuxImageProps {
    /**
     * What generation of Amazon Linux to use
     *
     * @default AmazonLinux
     */
    readonly generation?: AmazonLinuxGeneration;
    /**
     * What edition of Amazon Linux to use
     *
     * @default Standard
     */
    readonly edition?: AmazonLinuxEdition;
    /**
     * Virtualization type
     *
     * @default HVM
     */
    readonly virtualization?: AmazonLinuxVirt;
    /**
     * What storage backed image to use
     *
     * @default GeneralPurpose
     */
    readonly storage?: AmazonLinuxStorage;
    /**
     * Initial user data
     *
     * @default - Empty UserData for Linux machines
     */
    readonly userData?: UserData;
    /**
     * CPU Type
     *
     * @default X86_64
     */
    readonly cpuType?: AmazonLinuxCpuType;
}
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
export declare class AmazonLinuxImage implements IMachineImage {
    private readonly props;
    private readonly generation;
    private readonly edition;
    private readonly virtualization;
    private readonly storage;
    private readonly cpu;
    constructor(props?: AmazonLinuxImageProps);
    /**
     * Return the image to use in the given context
     */
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * What generation of Amazon Linux to use
 */
export declare enum AmazonLinuxGeneration {
    /**
     * Amazon Linux
     */
    AMAZON_LINUX = "amzn",
    /**
     * Amazon Linux 2
     */
    AMAZON_LINUX_2 = "amzn2"
}
/**
 * Amazon Linux edition
 */
export declare enum AmazonLinuxEdition {
    /**
     * Standard edition
     */
    STANDARD = "standard",
    /**
     * Minimal edition
     */
    MINIMAL = "minimal"
}
/**
 * Virtualization type for Amazon Linux
 */
export declare enum AmazonLinuxVirt {
    /**
     * HVM virtualization (recommended)
     */
    HVM = "hvm",
    /**
     * PV virtualization
     */
    PV = "pv"
}
export declare enum AmazonLinuxStorage {
    /**
     * EBS-backed storage
     */
    EBS = "ebs",
    /**
     * S3-backed storage
     */
    S3 = "ebs",
    /**
     * General Purpose-based storage (recommended)
     */
    GENERAL_PURPOSE = "gp2"
}
/**
 * Configuration options for GenericLinuxImage
 */
export interface GenericLinuxImageProps {
    /**
     * Initial user data
     *
     * @default - Empty UserData for Linux machines
     */
    readonly userData?: UserData;
}
/**
 * Configuration options for GenericWindowsImage
 */
export interface GenericWindowsImageProps {
    /**
     * Initial user data
     *
     * @default - Empty UserData for Windows machines
     */
    readonly userData?: UserData;
}
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
export declare class GenericLinuxImage implements IMachineImage {
    private readonly amiMap;
    private readonly props;
    constructor(amiMap: {
        [region: string]: string;
    }, props?: GenericLinuxImageProps);
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
export declare class GenericWindowsImage implements IMachineImage {
    private readonly amiMap;
    private readonly props;
    constructor(amiMap: {
        [region: string]: string;
    }, props?: GenericWindowsImageProps);
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * The OS type of a particular image
 */
export declare enum OperatingSystemType {
    LINUX = 0,
    WINDOWS = 1
}
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
export declare class LookupMachineImage implements IMachineImage {
    private readonly props;
    constructor(props: LookupMachineImageProps);
    getImage(scope: Construct): MachineImageConfig;
}
/**
 * Properties for looking up an image
 */
export interface LookupMachineImageProps {
    /**
     * Name of the image (may contain wildcards)
     */
    readonly name: string;
    /**
     * Owner account IDs or aliases
     *
     * @default - All owners
     */
    readonly owners?: string[];
    /**
     * Additional filters on the AMI
     *
     * @see https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html
     * @default - No additional filters
     */
    readonly filters?: {
        [key: string]: string[];
    };
    /**
     * Look for Windows images
     *
     * @default false
     */
    readonly windows?: boolean;
    /**
     * Custom userdata for this image
     *
     * @default - Empty user data appropriate for the platform type
     */
    readonly userData?: UserData;
}
