"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * The type of the CodeBuild action that determines its CodePipeline Category -
 * Build, or Test.
 * The default is Build.
 */
var CodeBuildActionType;
(function (CodeBuildActionType) {
    /**
     * The action will have the Build Category.
     * This is the default.
     */
    CodeBuildActionType[CodeBuildActionType["BUILD"] = 0] = "BUILD";
    /**
     * The action will have the Test Category.
     */
    CodeBuildActionType[CodeBuildActionType["TEST"] = 1] = "TEST";
})(CodeBuildActionType = exports.CodeBuildActionType || (exports.CodeBuildActionType = {}));
/**
 * CodePipeline build action that uses AWS CodeBuild.
 */
class CodeBuildAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: props.type === CodeBuildActionType.TEST
                ? codepipeline.ActionCategory.TEST
                : codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.input, ...props.extraInputs || []],
            resource: props.project,
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuild project this action points to.
     * Variables in CodeBuild actions are defined using the 'exported-variables' subsection of the 'env'
     * section of the buildspec.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'exported-variables' section of the buildspec
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-syntax
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // check for a cross-account action if there are any outputs
        if ((this.actionProperties.outputs || []).length > 0) {
            const pipelineStack = cdk.Stack.of(scope);
            const projectStack = cdk.Stack.of(this.props.project);
            if (pipelineStack.account !== projectStack.account) {
                throw new Error('A cross-account CodeBuild action cannot have outputs. ' +
                    'This is a known CodeBuild limitation. ' +
                    'See https://github.com/aws/aws-cdk/issues/4169 for details');
            }
        }
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.project.projectArn],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (this.props.project.role) {
            if ((this.actionProperties.outputs || []).length > 0) {
                options.bucket.grantReadWrite(this.props.project);
            }
            else {
                options.bucket.grantRead(this.props.project);
            }
        }
        if (this.props.project instanceof codebuild.Project) {
            this.props.project.bindToCodePipeline(scope, {
                artifactBucket: options.bucket,
            });
        }
        const configuration = {
            ProjectName: this.props.project.projectName,
            EnvironmentVariables: this.props.environmentVariables &&
                cdk.Stack.of(scope).toJsonString(codebuild.Project.serializeEnvVariables(this.props.environmentVariables)),
        };
        if ((this.actionProperties.inputs || []).length > 1) {
            // lazy, because the Artifact name might be generated lazily
            configuration.PrimarySource = cdk.Lazy.stringValue({ produce: () => this.props.input.artifactName });
        }
        return {
            configuration,
        };
    }
}
exports.CodeBuildAction = CodeBuildAction;
//# sourceMappingURL=data:application/json;base64,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