#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (c) 2020-2021 The WfCommons Team.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

import json
import logging
import pickle
import uuid

import pandas as pd
import networkx as nx
import numpy as np

from abc import ABC, abstractmethod
from os import path
from logging import Logger
from typing import Any, Dict, List, Optional, Set

from wfcommons.common.file import File, FileLink
from wfcommons.common.task import Task, TaskType
from wfcommons.common.workflow import Workflow
from wfcommons.utils import generate_rvs
from wfcommons.wfchef.duplicate import duplicate


class WorkflowRecipe(ABC):
    """An abstract class of workflow recipes for creating synthetic workflow instances.

    :param name: The workflow recipe name.
    :type name: str
    :param data_footprint: The upper bound for the workflow total data footprint (in bytes).
    :type data_footprint: int
    :param num_tasks: The upper bound for the total number of tasks in the workflow.
    :type num_tasks: int
    :param runtime_factor: The factor of which tasks runtime will be increased/decreased.
    :type runtime_factor: float
    :param input_file_size_factor: The factor of which tasks input files size will be increased/decreased.
    :type input_file_size_factor: float
    :param output_file_size_factor: The factor of which tasks output files size will be increased/decreased.
    :type output_file_size_factor: float
    :param logger: The logger where to log information/warning or errors (optional).
    :type logger: Logger
    """

    def __init__(self, name: str,
                 data_footprint: Optional[int],
                 num_tasks: Optional[int],
                 exclude_graphs: Set[str] = set(),
                 runtime_factor: Optional[float] = 1.0,
                 input_file_size_factor: Optional[float] = 1.0,
                 output_file_size_factor: Optional[float] = 1.0,
                 logger: Optional[Logger] = None,
                 this_dir=None) -> None:
        """Create an object of the workflow recipe."""
        # sanity checks
        if runtime_factor <= 0.0:
            raise ValueError("The runtime factor should be a number higher than 0.0.")
        if input_file_size_factor <= 0.0:
            raise ValueError("The input file size factor should be a number higher than 0.0.")
        if output_file_size_factor <= 0.0:
            raise ValueError("The output file size factor should be a number higher than 0.0.")

        self.logger = logging.getLogger(__name__) if logger is None else logger
        self.name = name
        self.data_footprint = data_footprint
        self.num_tasks = num_tasks
        self.exclude_graphs = exclude_graphs
        self.runtime_factor = runtime_factor
        self.input_file_size_factor = input_file_size_factor
        self.output_file_size_factor = output_file_size_factor
        self.workflows: List[Workflow] = []
        self.tasks_files: Dict[str, List[File]] = {}
        self.task_id_counter = 1
        self.this_dir = this_dir

    @abstractmethod
    def _workflow_recipe(self) -> Dict[str, Any]:
        """Recipe for generating synthetic instances for a workflow. Recipes can be
        generated by using the :class:`~wfcommons.wfinstances.instance_analyzer.InstanceAnalyzer`.

        :return: A recipe in the form of a dictionary in which keys are task prefixes.
        :rtype: Dict[str, Any]
        """

    @classmethod
    @abstractmethod
    def from_num_tasks(cls,
                       num_tasks: int,
                       runtime_factor: Optional[float] = 1.0,
                       input_file_size_factor: Optional[float] = 1.0,
                       output_file_size_factor: Optional[float] = 1.0
                       ) -> 'WorkflowRecipe':
        """
        Instantiate a workflow recipe that will generate synthetic workflows up to the
        total number of tasks provided.

        :param num_tasks: The upper bound for the total number of tasks in the workflow.
        :type num_tasks: int
        :param runtime_factor: The factor of which tasks runtime will be increased/decreased.
        :type runtime_factor: float
        :param input_file_size_factor: The factor of which tasks input files size will be increased/decreased.
        :type input_file_size_factor: float
        :param output_file_size_factor: The factor of which tasks output files size will be increased/decreased.
        :type output_file_size_factor: float

        :return: A workflow recipe object that will generate synthetic workflows up to
                 the total number of tasks provided.
        :rtype: WorkflowRecipe
        """

    def generate_nx_graph(self) -> nx.DiGraph:
        summary_path = self.this_dir.joinpath("microstructures", "summary.json")
        summary = json.loads(summary_path.read_text())

        metric_path = self.this_dir.joinpath("microstructures", "metric", "err.csv")
        df = pd.read_csv(str(metric_path), index_col=0)
        df = df.drop(self.exclude_graphs, axis=0, errors="ignore")
        df = df.drop(self.exclude_graphs, axis=1, errors="ignore")
        for col in df.columns:
            df.loc[col, col] = np.nan

        reference_orders = [summary["base_graphs"][col]["order"] for col in df.columns]
        idx = np.argmin([abs(self.num_tasks - ref_num_tasks) for ref_num_tasks in reference_orders])
        reference = df.columns[idx]

        base = df.index[df[reference].argmin()]
        graph = duplicate(self.this_dir.joinpath("microstructures"), base, self.num_tasks)
        return graph

    def build_workflow(self, workflow_name: Optional[str] = None) -> Workflow:
        """Generate a synthetic workflow instance.

        :param workflow_name: The workflow name
        :type workflow_name: int

        :return: A synthetic workflow instance object.
        :rtype: Workflow
        """
        workflow = Workflow(name=self.name + "-synthetic-instance" if not workflow_name else workflow_name, makespan=None)
        graph = self.generate_nx_graph()

        task_names = {}
        for node in graph.nodes:
            if node in ["SRC", "DST"]:
                continue
            node_type = graph.nodes[node]["type"]
            task_name = self._generate_task_name(node_type)
            task = self._generate_task(node_type, task_name)
            workflow.add_node(task_name, task=task)

            task_names[node] = task_name

        for (src, dst) in graph.edges:
            if src in ["SRC", "DST"] or dst in ["SRC", "DST"]:
                continue
            workflow.add_edge(task_names[src], task_names[dst])

        workflow.nxgraph = graph
        self.workflows.append(workflow)
        return workflow

    def _load_base_graph(self) -> nx.DiGraph:
        return pickle.loads(self.this_dir.joinpath("base_graph.pickle").read_bytes())

    def _load_microstructures(self) -> Dict:
        return json.loads(self.this_dir.joinpath("microstructures.json").read_text())

    def _generate_task(self, task_name: str,
                       task_id: str,
                       input_files: Optional[List[File]] = None,
                       files_recipe: Optional[Dict[FileLink, Dict[str, int]]] = None
                       ) -> Task:
        """Generate a synthetic task.

        :param task_name: task name.
        :type task_name: str
        :param task_id: task ID.
        :type task_id: str
        :param input_files: List of input files to be included.
        :type input_files: List[File]
        :param files_recipe: Recipe for generating task files.
        :type files_recipe: Dict[FileLink, Dict[str, int]]

        :return: A task object.
        :rtype: task
        """
        task_recipe = self._workflow_recipe()[task_name]
        # runtime
        runtime: float = float(format(
            self.runtime_factor * generate_rvs(task_recipe['runtime']['distribution'],
                                               task_recipe['runtime']['min'],
                                               task_recipe['runtime']['max']), '.3f'))

        # linking previous generated output files as input files
        self.tasks_files[task_id] = []
        if input_files:
            for f in input_files:
                if f.link == FileLink.OUTPUT:
                    self.tasks_files[task_id].append(File(name=f.name, size=f.size, link=FileLink.INPUT))

        # generate additional in/output files
        self._generate_files(task_id, task_recipe['input'], FileLink.INPUT, files_recipe)
        self._generate_files(task_id, task_recipe['output'], FileLink.OUTPUT, files_recipe)

        return Task(
            name=task_id,
            task_type=TaskType.COMPUTE,
            runtime=runtime,
            machine=None,
            args=[],
            cores=1,
            avg_cpu=None,
            bytes_read=None,
            bytes_written=None,
            memory=None,
            energy=None,
            avg_power=None,
            priority=None,
            files=self.tasks_files[task_id]
        )

    def _generate_task_name(self, prefix: str) -> str:
        """Generate a task name from a prefix appended with an ID.

        :param prefix: task prefix.
        :type prefix: str

        :return: task name from prefix appended with an ID.
        :rtype: str
        """
        task_name = "{}_{:08d}".format(prefix, self.task_id_counter)
        self.task_id_counter += 1
        return task_name

    def _generate_files(self, task_id: str, recipe: Dict[str, Any], link: FileLink,
                        files_recipe: Optional[Dict[FileLink, Dict[str, int]]] = None) -> None:
        """Generate files for a specific task ID.

        :param task_id: task ID.
        :type task_id: str
        :param recipe: Recipe for generating the task.
        :type recipe: Dict[str, Any]
        :param link: Type of file link.
        :type link: FileLink
        :param files_recipe: Recipe for generating task files.
        :type files_recipe: Dict[FileLink, Dict[str, int]]
        """
        extension_list: List[str] = []
        for f in self.tasks_files[task_id]:
            if f.link == link:
                extension_list.append(path.splitext(f.name)[1] if '.' in f.name else f.name)

        for extension in recipe:
            if extension not in extension_list:
                num_files = 1
                if files_recipe and link in files_recipe and extension in files_recipe[link]:
                    num_files = files_recipe[link][extension]
                for _ in range(0, num_files):
                    self.tasks_files[task_id].append(self._generate_file(extension, recipe, link))

    def _generate_file(self, extension: str, recipe: Dict[str, Any], link: FileLink) -> File:
        """Generate a file according to a file recipe.

        :param extension:
        :type extension: str
        :param recipe: Recipe for generating the file.
        :type recipe: Dict[str, Any]
        :param link: Type of file link.
        :type link: FileLink

        :return: The generated file.
        :rtype: File
        """
        size = int((self.input_file_size_factor if link == FileLink.INPUT
                    else self.output_file_size_factor) * generate_rvs(recipe[extension]['distribution'],
                                                                      recipe[extension]['min'],
                                                                      recipe[extension]['max']))
        return File(name=str(uuid.uuid4()) + extension,
                    link=link,
                    size=size)

    def _get_files_by_task_and_link(self, task_id: str, link: FileLink) -> List[File]:
        """Get the list of files for a task ID and link type.

        :param task_id: task ID.
        :type task_id: str
        :param link: Type of file link.
        :type link: FileLink

        :return: List of files for a task ID and link type.
        :rtype: List[File]
        """
        files_list: List[File] = []
        for f in self.tasks_files[task_id]:
            if f.link == link:
                files_list.append(f)
        return files_list
