# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2015-2018 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports :class:`HassaniAtkinson2020SInter`
               :class:`HassaniAtkinson2020SSlab`
               :class:`HassaniAtkinson2020Asc`
"""
import math

import numpy as np

from openquake.hazardlib import const
from openquake.hazardlib.gsim.base import GMPE, CoeffsTable
from openquake.hazardlib.imt import PGA, SA, PGV

CONSTANTS = {"mlf0": 5.5, "mlf1": 7, "f1": 0, "f3": 98.1,
             "b1": -1.3, "b2": -0.5, "v0": 100, "v1": 250, "v2": 1000,
             "zx0": 150, "zx1": 800, "zx2": 4200,
             "cfp0": -0.011, "cfp1": 0.421, "cfp2": -0.604, "cfp3": 0.086,
             "x0": 0.5, "x1": 1, "x2": 2.5, "vref": 760, "vmin": 360}


def _clf(suffix, C, mag):
    """
    Low frequency calibration factor.
    """
    clf0 = C['clf0' + suffix]
    clf1 = C['clf1' + suffix]
    mlf0 = CONSTANTS["mlf0"]

    if mag > mlf0:
        return clf0 + clf1 * (min(mag, CONSTANTS["mlf1"]) - mlf0)
    return clf0


def _dsigma(creg, hypo_depth):
    """
    Hypocentre depth factor.
    """
    out = creg['cd0']
    dp0 = creg['dp0']
    if hypo_depth > dp0:
        out += creg['cd1'] * (min(hypo_depth, creg['dp1']) - dp0)
    return 10 ** out


def _fds_ha18(C, mag, dsigma):
    """
    Dsigma factor.
    """
    eds1 = np.polyval([C['g14'], C['g13'], C['g12'], C['g11'], C['g10']],
                      mag)
    eds2 = np.polyval([C['g24'], C['g23'], C['g22'], C['g21'], C['g20']],
                      mag)
    eds0 = -2 * eds1 - 4 * eds2

    return eds0 + eds1 * math.log10(dsigma) \
        + eds2 * math.log10(dsigma) ** 2


def _ffpeak(C, imt, fpeak):
    """
    Fpeak factor.
    """
    if imt.string[:2] != "SA" or max(fpeak) <= 0:
        # pgv, pga or unknown fpeak
        return 0

    s = CONSTANTS
    x = fpeak / 10 ** C['f']

    ffpeak = np.where(fpeak <= 0, 0, s['cfp0'])

    idx = np.where((s['x0'] < x) & (x <= s['x1']))
    ffpeak[idx] = s['cfp0'] + s['cfp1'] * np.log10(x[idx] / s['x0'])

    idx = np.where((s['x1'] < x) & (x <= s['x2']))
    ffpeak[idx] = s['cfp0'] + s['cfp1'] * math.log10(s['x1'] / s['x0']) \
        + s['cfp2'] * np.log10(x[idx] / s['x1'])

    idx = np.where(s['x2'] < x)
    ffpeak[idx] = s['cfp0'] + s['cfp1'] * math.log10(s['x1'] / s['x0']) \
        + s['cfp2'] * math.log10(s['x2'] / s['x1']) \
        + s['cfp3'] * np.log10(x[idx] / s['x2'])

    return ffpeak


def _fgamma(suffix, backarc, forearc_ne, forearc_sw, C, rrup):
    """
    Gamma factor.
    """
    # proportion sum for normalised values with rrup factor
    p_sum = rrup / (backarc + forearc_ne + forearc_sw)

    return C['barc' + suffix] * backarc * p_sum \
        + C['farc_ne' + suffix] * forearc_ne * p_sum \
        + C['farc_sw' + suffix] * forearc_sw * p_sum


def _fkp_ha18(kappa, C, mag, dsigma):
    """
    Kappa factor for B/C site condition of Japan.
    """
    l10kp = math.log10(kappa)

    p = np.zeros(4)
    ek0 = np.zeros(4)
    for i in range(4):
        for j in range(4):
            p[j] = np.polyval([C[f'd{i}{j}2'], C[f'd{i}{j}1'],
                               C[f'd{i}{j}0']], math.log10(dsigma))
        ek0[i] = np.polyval(p[::-1], math.log10(mag))
    return 3 * ek0[0] - 9 * ek0[1] + 27 * ek0[2] - 81 * ek0[3] \
        + ek0[0] * l10kp + ek0[1] * l10kp ** 2 \
        + ek0[2] * l10kp ** 3 + ek0[3] * l10kp ** 4


def _fm_ha18(C, mag):
    """
    Magnitude factor.
    """
    if mag <= C['mh']:
        return C['e0'] + C['e1'] * (mag - C['mh']) \
               + C['e2'] * (mag - C['mh']) ** 2
    return C['e0'] + C['e3'] * (mag - C['mh'])


def _fsnonlin_ss14(C, vs30, pga_rock):
    """
    Non-linear factor.
    """
    s = CONSTANTS

    f2 = C['f4'] * (np.exp(C['f5']
                           * (np.minimum(vs30, s['vref']) - s['vmin']))
                    - math.exp(C['f5'] * (s['vref'] - s['vmin'])))

    return s['f1'] + f2 * np.log((pga_rock + s['f3']) / s['f3'])


def _fvs30(C, vs30):
    """
    Vs30 factor.
    """
    s = CONSTANTS
    fvs30 = np.where(vs30 <= s['v0'],
                     C['cv1'] * math.log10(s['v0'] / s['vref'])
                     + (C['cv2'] - C['cv1'])
                     * math.log10(s['v1'] / s['vref']),
                     C['cv2'] * math.log10(s['v2'] / s['vref']))
    fvs30 = np.where((s['v0'] < vs30) & (vs30 <= s['v1']),
                     C['cv1'] * np.log10(vs30 / s['vref'])
                     + (C['cv2'] - C['cv1'])
                     * math.log10(s['v1'] / s['vref']), fvs30)
    return np.where((s['v1'] < vs30) & (vs30 <= s['v2']),
                    C['cv2'] * np.log10(vs30 / s['vref']), fvs30)


def _fz2pt5(C, z2pt5):
    """
    Z2pt5 factor.
    """
    s = CONSTANTS
    fz2pt5 = np.where(z2pt5 >= 0, C['cz0'], 0)

    idx = np.where((s['zx0'] < z2pt5) & (z2pt5 <= s['zx1']))
    fz2pt5[idx] = C['cz0'] + C['cz1'] * np.log10(z2pt5[idx] / s['zx0'])

    idx = np.where((s['zx1'] < z2pt5) & (z2pt5 <= s['zx2']))
    fz2pt5[idx] = C['cz0'] + C['cz1'] * math.log10(s['zx1'] / s['zx0']) \
        + C['cz2'] * np.log10(z2pt5[idx] / s['zx1'])

    idx = np.where(s['zx2'] < z2pt5)
    fz2pt5[idx] = C['cz0'] + C['cz1'] * math.log10(s['zx1'] / s['zx0']) \
        + C['cz2'] * math.log10(s['zx2'] / s['zx1'])

    return fz2pt5


def _fz_ha18(rt, C, mag, rrup):
    """
    Z factor.
    """
    s = CONSTANTS
    h = 10 ** (-0.405 + 0.235 * mag)
    ref = np.sqrt(rrup ** 2 + h ** 2)
    rref = math.sqrt(1 ** 2 + h ** 2)

    return np.where(ref <= rt, s['b1'] * np.log10(ref)
                    + (C['b3'] + C['b4'] * mag) * np.log10(ref / rref),
                    s['b1'] * math.log10(rt)
                    + s['b2'] * np.log10(ref / rt)
                    + (C['b3'] + C['b4'] * mag) * np.log10(ref / rref))


def get_stddevs(suffix, C):
    """
    Between event standard deviations as tau.
    Intra event from site to site stddev and within site stddev.
    Total given in COEFFS to 3dp.
    """
    return [C['s' + suffix], C['tau'],
            math.sqrt(C['ps2s'] ** 2 + C['pss' + suffix] ** 2)]


class HassaniAtkinson2020SInter(GMPE):
    """
    Hassani Atkinson (2020) for Subduction Interface.
    """

    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.SUBDUCTION_INTERFACE

    #: Supported intensity measure types are spectral acceleration,
    #: peak ground acceleration and peak ground velocity
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGV, PGA, SA}

    #: Supported intensity measure component is the geometric mean component
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.AVERAGE_HORIZONTAL

    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL, const.StdDev.INTER_EVENT, const.StdDev.INTRA_EVENT}

    REQUIRES_DISTANCES = {'rrup'}

    REQUIRES_RUPTURE_PARAMETERS = {'hypo_depth', 'mag'}

    REQUIRES_SITES_PARAMETERS = {'fpeak', 'vs30', 'z2pt5'}

    REQUIRES_ATTRIBUTES = {'kappa', 'backarc', 'forearc_ne', 'forearc_sw'}

    def __init__(self, kappa=0.04, backarc=0, forearc_ne=1, forearc_sw=0,
                 **kwargs):
        """
        Aditional parameters.
        """
        super().__init__(kappa=kappa, backarc=backarc, forearc_ne=forearc_ne,
                         forearc_sw=forearc_sw, **kwargs)
        # kappa parameter
        self.kappa = kappa
        # set proportion of rrups in backarc, forearc_ne and forearc_sw
        self.backarc = backarc
        self.forearc_ne = forearc_ne
        self.forearc_sw = forearc_sw

    def compute(self, ctx, imts, mean, sig, tau, phi):
        """
        See :meth:`superclass method
        <.base.GroundShakingIntensityModel.compute>`
        for spec of input and result values.
        """

        C_PGA = self.COEFFS[PGA()]
        dsigma = _dsigma(self.CONST_REGION, ctx.hypo_depth)
        fm_pga = _fm_ha18(C_PGA, ctx.mag)
        fz_pga = _fz_ha18(self.CONST_REGION['rt'], C_PGA, ctx.mag, ctx.rrup)
        fdsigma_pga = _fds_ha18(C_PGA, ctx.mag, dsigma)
        fgamma_pga = _fgamma(self.SUFFIX, self.backarc, self.forearc_ne,
                             self.forearc_sw, C_PGA, ctx.rrup)
        fkappa_pga = _fkp_ha18(self.kappa, C_PGA, ctx.mag, dsigma)
        clf_pga = _clf(self.SUFFIX, C_PGA, ctx.mag)
        pga_rock = 10 ** (fm_pga + fz_pga + fdsigma_pga + fkappa_pga +
                          fgamma_pga + self.CONST_REGION['cc'] +
                          clf_pga + C_PGA['chf'] + C_PGA['amp_cr'])
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            fm = _fm_ha18(C, ctx.mag)
            fz = _fz_ha18(self.CONST_REGION['rt'], C, ctx.mag, ctx.rrup)
            fdsigma = _fds_ha18(C, ctx.mag, dsigma)
            fkappa = _fkp_ha18(self.kappa, C, ctx.mag, dsigma)
            fgamma = _fgamma(self.SUFFIX, self.backarc, self.forearc_ne,
                             self.forearc_sw, C, ctx.rrup)
            clf = _clf(self.SUFFIX, C, ctx.mag)
            fsnonlin = _fsnonlin_ss14(C, ctx.vs30, pga_rock)
            fvs30 = _fvs30(C, ctx.vs30)
            fz2pt5 = _fz2pt5(C, ctx.z2pt5)
            ffpeak = _ffpeak(C, imt, ctx.fpeak)

            mean[m] = 10 ** (fm + fdsigma + fz + fkappa + fgamma
                             + self.CONST_REGION['cc'] + clf + C['chf']
                             + C['amp_cr'] + fvs30 + fz2pt5 + ffpeak +
                             fsnonlin)
            if imt.string != "PGV":
                # pgv in cm/s
                # sa and psa in cm/s^2
                mean[m] /= 981.
            mean[m] = np.log(mean[m])

            sig[m], tau[m], phi[m] = get_stddevs(self.SUFFIX, C)

    # periods given by 1 / 10 ** COEFFS['f']
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT       f   amp_cr b3     b4      chf    clf0_if clf1_if clf0_is clf1_is clf0_cr clf1_cr cv1    cv2      cz0      cz1      cz2      barc_if  farc_ne_if farc_sw_if barc_is  farc_ne_is farc_sw_is barc_cr   farc_ne_cr  farc_sw_cr  d000        d001      d002         d010        d011        d012         d020        d021        d022        d030        d031       d032        d100       d101        d102        d110        d111        d112        d120        d121       d122        d130        d131       d132       d200       d201       d202       d210        d211       d212       d220        d221       d222       d230       d231       d232      d300      d301       d302      d310       d311      d312       d320       d321      d322      d330      d331      d332     mh   e0      e1       e2      e3       f4      f5      g10       g11       g12       g13       g14       g20       g21       g22       g23       g24      tau   ps2s  pss_if pss_is pss_cr s_if  s_is  s_cr
    pgv      -9   0     -0.4414 0.0394  0.204  0       0       0       0       0        0     -0.667 -0.866   -0.046    0.166   -0.068   -0.00281 -0.0015    -0.00048   -0.00327 -0.00148   -0.0024    -0.00406  -0.00242    -0.00278   -40.826164   40.113039 -5.102129   155.252313 -182.36868    27.033349  -196.671754  258.210795  -42.36533    82.357479 -116.454408  20.537609  -31.558927  36.874774   -5.739549  125.277771 -161.306518   27.874045 -163.436869  223.266447 -41.391833   69.903157  -99.25266   19.350221 -10.37628   13.051401  -2.256509  41.867738  -56.055028  10.530332 -55.192594   76.696014 -15.238462  23.77607  -33.832713   6.996974 -1.229433  1.604051  -0.296017  4.993482  -6.816711  1.350202  -6.609082   9.262415 -1.92463   2.854701 -4.066423  0.874373 8.8  3.30208 0.43897 -0.01672 0.41464 -0.0434 -0.0084  0.444251  0.056972  0.003644 -0.004046  0.000353 -0.094203  0.053343 -0.014163  0.001919 -0.0001   0.482 0.293 0.386  0.376  0.401  0.683 0.678 0.692
    pga      -9   0     -0.4565 0.0382  0.574  0       0       0       0       0        0     -0.16  -0.612    0        0        0       -0.00684 -0.00273   -0.00456   -0.00541 -0.00167   -0.00396   -0.00688  -0.00476    -0.00484   -51.098718   32.578796 -0.602711   193.376939 -154.536656    8.970141  -246.377014  224.30934   -19.056039  103.867407 -102.862801  10.767404  -44.562063  38.708279   -3.922374  176.725188 -170.793471   21.015529 -230.932652  237.86616  -33.083719   99.136459 -106.274886  16.078647 -16.422438  15.847931  -2.168642  65.997737  -67.973039  10.385514 -87.070182   93.033435 -15.329737  37.638662 -41.09486    7.148396 -2.134042  2.150893  -0.337012  8.618758  -9.089596  1.542771 -11.416684  12.32449  -2.213236  4.950046 -5.410116  1.012283 7.4  4.37376 0.33001 -0.0094  0.30153 -0.0651 -0.007   0.709663 -0.038392  0.013081 -0.001138  0.000024 -0.032385  0.019043 -0.004517  0.000375 -0.000009 0.513 0.422 0.411  0.452  0.496  0.781 0.803 0.829
     0.01     2.0 0.678 -0.4321 0.0353 -0.09   0       0       0       0       0        0     -0.163 -0.598    0        0        0       -0.00686 -0.00274   -0.00457   -0.00544 -0.00168   -0.00398   -0.00693  -0.0048     -0.00489   -41.010798   20.984834  4.332932   143.725915 -112.573964   -8.798757  -185.502717  174.262586    2.145698   78.793199  -83.128703   2.380509  -35.56513   27.797776    0.50508   130.2847   -131.140942    4.976749 -173.646134  190.435518 -13.863902   75.476551  -87.53478    8.453463 -13.145747  11.601671  -0.552587  48.068218  -52.451102   4.478831 -64.765944   74.387032  -8.20745   28.388967 -33.704532   4.310521 -1.715884  1.576223  -0.13206   6.206072  -6.977314  0.786463  -8.387874   9.776758 -1.295275  3.688342 -4.397405  0.644874 7.15 4.64514 0.34404 -0.00816 0.30565 -0.0644 -0.007   1.472416 -0.541245  0.133779 -0.013606  0.000492 -0.222277  0.146276 -0.035353  0.00358  -0.00013  0.513 0.423 0.411  0.450  0.496  0.782 0.803 0.830
     0.01259  1.9 0.677 -0.4335 0.036  -0.089  0       0       0       0       0        0     -0.165 -0.593    0        0        0       -0.00688 -0.00275   -0.00458   -0.00546 -0.00169   -0.00399   -0.00696  -0.00482    -0.00492   -36.828522   19.569252  6.125909   133.799687 -108.928441  -14.579428  -175.108029  171.43759     8.453326   74.844597  -82.543522   0.054735  -31.17228   25.351334    2.531652  118.752325 -123.338464   -1.805442 -161.173356  182.153805  -6.194116   70.665151  -84.601815   5.529234 -11.281178  10.204519   0.356264  42.754302  -47.641396   1.35097  -58.876236   68.854243  -4.5789    26.088198 -31.574179   2.895149 -1.449797  1.338067   0.003661  5.403172  -6.129093  0.310717  -7.485634   8.768873 -0.734349  3.333472 -3.997333  0.422956 7.4  4.69775 0.33848 -0.00825 0.3049  -0.0642 -0.0071  1.554224 -0.561608  0.1314   -0.012766  0.000443 -0.26475   0.165203 -0.037932  0.003683 -0.000129 0.513 0.424 0.411  0.451  0.496  0.782 0.804 0.830
     0.01585  1.8 0.676 -0.4282 0.0351 -0.092  0       0       0       0       0        0     -0.168 -0.587    0        0        0       -0.0069  -0.00276   -0.00459   -0.00547 -0.0017    -0.004     -0.00699  -0.00485    -0.00494   -33.843087   21.852772  6.415627   132.533545 -119.657522  -14.289425  -178.466424  186.886925    6.835534   77.884983  -89.602175   1.106684  -28.577173  27.547838    3.040521  118.556209 -134.307003   -2.241058 -166.146429  198.547813  -7.035861   74.454737  -92.292678   6.313616 -10.409129  11.098325   0.623752  43.180509  -52.297863   0.97445  -61.611402   75.987898  -4.711912  27.946807 -34.97869    3.144502 -1.350461  1.465128   0.046401  5.522435  -6.807942  0.240613  -7.948998   9.823776 -0.734948  3.628394 -4.505746  0.452039 7.5  4.69913 0.33602 -0.00894 0.30604 -0.0641 -0.0072  1.714652 -0.6427    0.145153 -0.013649  0.000459 -0.34998   0.21312  -0.047606  0.004516 -0.000155 0.513 0.424 0.412  0.451  0.496  0.783 0.805 0.831
     0.01995  1.7 0.672 -0.4209 0.0344 -0.102  0       0       0       0       0        0     -0.171 -0.578    0        0        0       -0.00693 -0.00277   -0.00461   -0.00549 -0.00171   -0.00402   -0.00704  -0.00488    -0.00498   -42.827733   41.107588  1.767006   175.999485 -195.196015    4.995207  -233.844386  282.760331  -18.604775  100.440836 -129.207506  11.927987  -36.362289  45.183985   -0.914573  157.333408 -204.021974   14.524812 -216.204378  287.545597 -29.45818    95.015702 -129.226338  15.940451 -13.041058  17.526987  -0.713597  56.776641  -77.875531   6.778782 -79.377281  108.793041 -12.585143  35.289638 -48.639106   6.556052 -1.659777  2.280063  -0.111625  7.178509 -10.068016  0.942713 -10.137835  14.021054 -1.700233  4.53801  -6.258265  0.873887 7.8  4.75561 0.32694 -0.01031 0.30339 -0.0639 -0.0073  1.912143 -0.751222  0.167414 -0.015674  0.000528 -0.447888  0.269464 -0.059636  0.005644 -0.000194 0.515 0.426 0.412  0.452  0.496  0.785 0.806 0.832
     0.02512  1.6 0.667 -0.4098 0.0331 -0.109  0       0       0       0       0        0     -0.174 -0.557    0        0        0       -0.007   -0.0028    -0.00466   -0.00554 -0.00173   -0.00405   -0.00715  -0.00497    -0.00508   -48.479722   55.7785   -4.013399   203.174441 -245.465754   25.938538  -266.708211  340.075945  -43.592027  113.93429  -151.062403  21.79371   -42.838283  60.841219   -6.458254  188.720316 -259.25022    35.007115 -255.708778  352.33067  -54.311583  111.649687 -154.593101  25.898767 -15.836581  23.981785  -2.828785  70.439152 -101.059928  14.712066 -96.989307  136.457275 -22.332544  42.812057 -59.638179  10.503095 -2.064808  3.184205  -0.391809  9.16487  -13.355305  2.005714 -12.736528  17.987595 -3.018564  5.657377 -7.850825  1.411863 8.95 5.06565 0.29638 -0.01193 0.29694 -0.0653 -0.0074  2.153914 -0.892759  0.198231 -0.018627  0.000633 -0.571979  0.342742 -0.075675  0.007183 -0.000249 0.519 0.428 0.414  0.455  0.496  0.790 0.812 0.836
     0.03162  1.5 0.658 -0.4075 0.0327 -0.105  0       0       0       0       0        0     -0.17  -0.522    0        0        0       -0.00712 -0.00285   -0.00474   -0.00565 -0.00177   -0.00411   -0.00737  -0.00513    -0.00524   -36.454347   50.747067 -7.854934   150.78109  -212.767887   36.872429  -186.167731  284.265373  -53.190934   75.382423 -122.580834  24.242413  -31.514516  58.500243  -10.430031  140.047747 -237.762319   46.97275  -180.764568  311.480621 -65.709208   75.653022 -132.580092  29.251673 -11.52124   23.815783  -4.438532  52.126256  -95.730629  19.745268 -68.777242  124.527905 -27.359875  29.228937 -52.757417  12.088909 -1.492253  3.233297  -0.61419   6.757361 -12.929789  2.718016  -9.027904  16.764317 -3.750698  3.868932 -7.087305  1.651818 8.25 4.80422 0.30666 -0.01375 0.30784 -0.068  -0.0073  2.277944 -0.964799  0.213693 -0.020064  0.000681 -0.682212  0.407086 -0.089573  0.008493 -0.000294 0.528 0.433 0.417  0.459  0.496  0.800 0.822 0.844
     0.03981  1.4 0.643 -0.3943 0.0311 -0.087  0       0       0       0       0        0     -0.154 -0.466    0        0        0       -0.00733 -0.00291   -0.00488   -0.0058  -0.00182   -0.0042    -0.00769  -0.00536    -0.00547   -10.329681    9.721835 -2.428972    43.883537  -42.554849   10.081315   -39.435885   56.809793  -13.528548   10.662441  -23.902673   5.852527  -11.523864  27.277669   -7.084519   59.612501 -107.034131   29.134065  -69.920435  135.78203  -38.279502   26.646272  -56.063821  16.245126  -5.065915  13.83251   -3.600758  26.701714  -53.571284  14.804109 -33.683484   67.573516 -19.42278   13.70416  -27.873688   8.233201 -0.742827  2.08603   -0.542555  3.869187  -8.046313  2.231618  -5.039166  10.136535 -2.928032  2.104843 -4.183466  1.24144  8.1  4.71025 0.30748 -0.01584 0.30733 -0.0727 -0.007   2.425762 -1.048628  0.232494 -0.021999  0.000756 -0.811617  0.481302 -0.105652  0.010042 -0.00035  0.544 0.441 0.422  0.466  0.498  0.818 0.841 0.860
     0.05012  1.3 0.62  -0.3872 0.0303 -0.037  0       0       0       0       0        0     -0.134 -0.368    0        0        0       -0.00761 -0.00299   -0.00508   -0.00598 -0.00187   -0.00431   -0.00806  -0.0056     -0.00571    27.085064  -63.719099 13.834259  -111.853248  249.144665  -59.288528   168.949914 -319.851084   80.35663   -78.758592  134.893708 -35.057929   23.637977 -38.463885    6.9208    -85.747788  155.869439  -31.159347  125.072682 -205.760151  43.923771  -57.313362   88.712074 -19.794897   8.194186 -10.055622   1.357385 -27.794144   42.435636  -6.686058  39.56707   -57.711116  10.038086 -17.922068  25.449342  -4.743307  0.997066 -0.962233   0.078463 -3.249763   4.250549 -0.473888   4.544393  -5.964419  0.796484 -2.04153   2.690324 -0.404916 7.4  4.42934 0.31661 -0.02109 0.31654 -0.0835 -0.0065  2.215697 -0.886227  0.191129 -0.017643  0.000592 -0.856308  0.494428 -0.106361  0.009953 -0.000343 0.563 0.455 0.426  0.474  0.501  0.840 0.865 0.881
     0.06310  1.2 0.59  -0.3816 0.0302  0.076  0       0       0       0       0        0     -0.097 -0.249    0        0        0       -0.00794 -0.00307   -0.00535   -0.00618 -0.00188   -0.00445   -0.00842  -0.0058     -0.00595    51.166641 -130.666484 34.967023  -210.719519  498.951885 -142.564784   294.373089 -626.397294  186.575608 -129.806949  258.885062 -79.262114   46.691884 -99.668192   25.418002 -179.966129  386.435965 -104.697277  245.410515 -490.923198 138.370241 -106.562235  204.801216 -59.317817  16.994651 -32.642702   8.000598 -63.51458   127.996417 -33.238646  85.315364 -163.991495  44.279226 -36.683227  68.862481 -19.117712  2.163192 -3.880007   0.920976 -7.954353  15.342591 -3.853207  10.577232 -19.779754  5.165791 -4.517738  8.345031 -2.242844 6.85 4.18536 0.32679 -0.02767 0.32638 -0.0951 -0.006   1.501368 -0.385625  0.067907 -0.004809  0.00011  -0.780202  0.424442 -0.086695  0.00774  -0.000255 0.577 0.476 0.430  0.482  0.506  0.863 0.890 0.903
     0.07943  1.1 0.55  -0.3694 0.0284  0.179  0       0       0       0       0        0     -0.01  -0.205    0        0        0       -0.00819 -0.00314   -0.00558   -0.0063  -0.00186   -0.00457   -0.0086   -0.00589    -0.00603    -7.763642  -66.149606 26.952316     8.97581   219.649317 -103.379076    15.405799 -240.457636  127.104568  -12.572634   86.735333 -50.71788     0.064604 -52.779583   20.630675   -6.928645  179.893747  -79.818698   26.556561 -201.956797  99.041055  -14.74867    74.692232 -39.890352   1.371378 -17.938684   6.785619  -5.622      62.274894 -26.43558   12.308399  -71.110282  33.03583   -6.09636   26.730934 -13.400089  0.278993 -2.198066   0.810585 -0.973691   7.731894 -3.174988   1.792855  -8.934275  3.98935  -0.841024  3.395967 -1.626842 6.1  3.86847 0.33558 -0.04859 0.33554 -0.1038 -0.0057  0.234847  0.470479 -0.13762   0.016237 -0.000672 -0.55048   0.252179 -0.042792  0.003069 -0.000077 0.577 0.498 0.434  0.490  0.512  0.877 0.906 0.918
     0.1      1.0 0.501 -0.3524 0.0265  0.204  0       0       0       0       0        0      0.119 -0.304    0        0        0       -0.00829 -0.00318   -0.00571   -0.00633 -0.00183   -0.00457   -0.00852  -0.00582    -0.00584   -68.458481   28.829689  4.605562   237.82803  -164.266697   -8.836444  -274.764841  260.022051   -0.933937  107.138761 -125.374848   5.317074  -49.29738   21.631191    3.518638  177.983317 -122.615957   -7.061507 -206.833056  194.017768   0.06255    81.241286  -93.659822   3.599962 -15.600124   6.982888   1.149204  57.790512  -39.46235   -2.385281 -67.51015    62.458687   0.211002  26.679145 -30.191159   1.065238 -1.812877  0.816535   0.13729   6.835188  -4.612347 -0.294438  -8.019372   7.308278  0.048217  3.184593 -3.538278  0.113795 5.9  3.73063 0.34123 -0.06587 0.34086 -0.1082 -0.0056 -1.574873  1.682296 -0.42639   0.045587 -0.001754 -0.151856 -0.03276   0.027751 -0.004278  0.000198 0.560 0.512 0.436  0.494  0.520  0.875 0.905 0.920
     0.12589  0.9 0.448 -0.3418 0.0252  0.167  0       0       0       0       0        0      0.239 -0.531    0        0        0       -0.00823 -0.00317   -0.00571   -0.00622 -0.00178   -0.00447   -0.00818  -0.00565    -0.00541   -57.661409   52.210153 -6.534009   177.243465 -234.368311   33.479926  -183.23805   322.707297  -52.189627   62.323522 -140.821243  25.32931   -42.110184  41.044747   -5.388708  133.944972 -182.937031   27.078094 -138.757168  251.119517 -41.730651   47.425905 -109.454024  20.112694 -13.483786  13.758383  -1.88179   43.870741  -61.008017   9.304197 -45.579787   83.569682 -14.2053    15.662299 -36.39955    6.807954 -1.58149   1.661143  -0.233866  5.224073  -7.340905  1.143415  -5.441877  10.042552 -1.734476  1.878435 -4.372726  0.828058 5.7  3.59587 0.34574 -0.0912  0.34554 -0.111  -0.0057 -3.703106  3.066794 -0.749185  0.077859 -0.002928  0.381196 -0.399304  0.116176 -0.013321  0.000533 0.534 0.509 0.435  0.493  0.528  0.856 0.887 0.907
     0.15849  0.8 0.4   -0.3386 0.0252  0.116  0       0       0       0       0        0      0.298 -0.774    0        0        0       -0.00805 -0.00312   -0.00563   -0.00601 -0.00174   -0.00426   -0.00772  -0.00535    -0.00491   -24.177127   31.505749 -6.301202    42.856647 -131.442129   28.590153   -12.536823  166.623715  -40.386786   -9.725772  -66.403364  18.060282  -17.00014   26.159992   -5.545691   31.632532 -107.972584   24.720766   -8.170103  136.316367 -34.518011   -7.964177  -54.294589  15.318878  -5.341652   9.099537  -2.01801   10.347768  -37.283796   8.875047  -2.606959   46.936269 -12.284179  -2.643744 -18.685157   5.420392 -0.62009   1.12699   -0.257915  1.236343  -4.595344  1.123957  -0.31152    5.773783 -1.546427 -0.314799 -2.297616  0.679679 5.5  3.44976 0.35261 -0.12539 0.35251 -0.1113 -0.0059 -5.461329  4.154378 -0.991766  0.101168 -0.003747  0.890829 -0.737591  0.195269 -0.021186  0.000816 0.509 0.492 0.434  0.485  0.532  0.830 0.858 0.886
     0.19953  0.7 0.356 -0.3262 0.0244  0.072  0       0       0       0       0        0      0.245 -0.963    0        0        0       -0.0077  -0.00304   -0.00544   -0.0057  -0.00171   -0.00401   -0.00712  -0.00496    -0.00445   -23.660344   26.464031 -5.426108    55.791231 -112.060621   24.636964   -47.626855  146.51806   -35.041156   11.280835  -60.888577  15.837994  -16.828254  21.447259   -4.63974    41.810074  -89.728015   20.704709  -35.876491  116.590896 -29.106723    8.681616  -48.298949  13.051859  -5.288994   7.319696  -1.66621   13.58268   -30.351731   7.324407 -11.670177   39.245374 -10.194418   2.853302 -16.212637   4.540574 -0.611269  0.893067  -0.211358  1.602063  -3.68013   0.919167  -1.373911   4.741356 -1.2702    0.336447 -1.954443  0.562998 5.55 3.39236 0.3603  -0.14134 0.35792 -0.1071 -0.0061 -6.701839  4.842848 -1.128498  0.112821 -0.004109  1.341265 -1.022197  0.25856  -0.027186  0.001023 0.498 0.470 0.433  0.475  0.531  0.810 0.834 0.866
     0.25119  0.6 0.318 -0.3078 0.0219  0.043  0       0       0       0       0        0      0.104 -1.076    0        0        0       -0.00721 -0.00292   -0.00517   -0.00535 -0.00167   -0.00375   -0.0064   -0.00452    -0.00404   -14.445389   18.496667 -3.731559    24.548144  -76.141144   16.60388    -11.975991   97.214791  -23.169101   -2.324617  -39.552576  10.302406   -9.917548  15.168281   -3.258594   17.609536  -61.529618   14.221533   -8.005212   77.893013 -19.573251   -2.102768  -31.522354   8.616295  -2.996519   5.146787  -1.183173   5.366689  -20.638785   5.071155  -2.11815    25.924107  -6.892427  -0.886377 -10.42874    3.006621 -0.33248   0.62014   -0.150501  0.586776  -2.465349  0.636599  -0.184214   3.076553 -0.85724  -0.133498 -1.230828  0.371403 5.75 3.40241 0.35949 -0.14483 0.35761 -0.1022 -0.0064 -7.161937  4.9757   -1.125765  0.109686 -0.003908  1.647322 -1.197509  0.293011 -0.030007  0.001106 0.501 0.443 0.429  0.463  0.527  0.795 0.813 0.851
     0.31623  0.5 0.287 -0.3085 0.0223  0.006  0       0       0       0       0        0     -0.117 -1.124    0        0        0       -0.00661 -0.00276   -0.00482   -0.00496 -0.00161   -0.00348   -0.00568  -0.00406    -0.00374   -12.457198   13.452604 -2.296076    25.041919  -56.590695   10.665132   -20.429261   74.083436  -15.332033    4.098787  -30.944402   6.964706   -8.496994  11.193186   -2.133387   17.892765  -46.107003    9.553985  -14.466532   59.589844 -13.399191    2.838654  -24.677585   5.981079  -2.575012   3.845386  -0.820948   5.531501  -15.578438   3.558698  -4.345622   19.906482  -4.882163   0.78274   -8.173392   2.145381 -0.287494  0.467366  -0.108764  0.618599  -1.870106  0.461095  -0.465766   2.367969 -0.622809  0.072099 -0.965107  0.270594 5.9  3.39048 0.36101 -0.14953 0.36028 -0.0931 -0.0068 -6.541656  4.34703  -0.933851  0.086626 -0.002949  1.717464 -1.199601  0.282951 -0.028015  0.001001 0.510 0.416 0.424  0.453  0.522  0.783 0.799 0.840
     0.39810  0.4 0.26  -0.2966 0.0212 -0.036  0       0       0       0       0        0     -0.39  -1.125    0        0        0       -0.006   -0.00254   -0.00444   -0.00454 -0.00152   -0.00327   -0.00504  -0.00363    -0.00348   -11.171429   13.42476  -2.312676    23.828545  -55.212966   10.512492   -20.516676   71.266493  -14.856756    4.629921  -29.469083   6.663018   -8.78464   12.532116   -2.431218   21.431952  -50.473874   10.609339  -20.202742   64.49718  -14.605681    5.595161  -26.559974   6.436415  -2.889597   4.488225  -0.964385   7.435233  -17.804979   4.085591  -7.148481   22.53445   -5.513494   2.060159  -9.219452   2.395702 -0.339177  0.554542  -0.128487  0.890425  -2.174981  0.533644  -0.853481   2.73019  -0.710181  0.245231 -1.109827  0.305441 6.2  3.43447 0.35471 -0.14217 0.35926 -0.0852 -0.0071 -4.815439  2.969329 -0.56384   0.045648 -0.001339  1.511291 -1.006868  0.22483  -0.02103   0.00071  0.511 0.393 0.420  0.440  0.512  0.769 0.780 0.823
     0.50119  0.3 0.236 -0.2813 0.0202 -0.068  0       0       0       0       0        0     -0.682 -1.105   -0.044    0.19    -0.202   -0.00534 -0.00228   -0.00397   -0.00411 -0.00142   -0.00295   -0.00443  -0.00316    -0.00318   -11.812408   13.810381 -2.093803    31.444778  -57.894833    9.877971   -33.699963   76.336005  -14.273426   11.435244  -32.281562   6.49964    -9.513158  13.27323    -2.411077   27.756507  -54.194612   10.699909  -30.847394   70.408832 -14.890386   11.001245  -29.52527    6.614863  -3.121738   4.754601  -0.982266   9.388295  -19.088831   4.203363 -10.460817   24.522232  -5.710879   3.74871  -10.200835   2.494789 -0.363552  0.584758  -0.132423  1.10234   -2.319098  0.55367   -1.219101   2.952493 -0.740247  0.433316 -1.219535  0.319613 6.45 3.4499  0.36074 -0.13591 0.36323 -0.0759 -0.0074 -2.478112  1.203996 -0.107894 -0.003352  0.000541  1.121871 -0.689355  0.136977 -0.011065  0.000311 0.503 0.380 0.417  0.426  0.497  0.756 0.761 0.803
     0.63096  0.2 0.217 -0.265  0.0185 -0.123  0       0       0       0       0        0     -0.992 -1.079   -0.064    0.244   -0.193   -0.00468 -0.00201   -0.00345   -0.0037  -0.00135   -0.00258   -0.0037   -0.00276    -0.00294   -13.540851   17.055776 -2.804665    41.258829  -70.294928   12.710257   -48.22679    92.592344  -18.045488   18.39273   -39.458509   8.170882  -10.562035  16.115678   -3.036013   33.662132  -64.896749   13.201129  -39.691195   84.206632 -18.222828   15.298954  -35.52086    8.087504  -3.348491   5.651546  -1.17536   10.766342  -22.430135   4.978746 -12.609158   28.801371  -6.748344   4.830425 -12.053701   2.95528  -0.376572  0.680627  -0.152351  1.204459  -2.672388  0.633989  -1.39319    3.40324  -0.848446  0.527028 -1.414898  0.368028 6.8  3.52557 0.36154 -0.1252  0.36002 -0.0669 -0.0078  0.52648  -0.970882  0.435757 -0.060272  0.002678  0.511349 -0.224278  0.014578  0.002319 -0.000209 0.491 0.374 0.414  0.418  0.476  0.743 0.745 0.779
     0.79433  0.1 0.201 -0.265  0.0188 -0.158  0       0       0       0       0        0     -1.223 -1.056   -0.083    0.296   -0.175   -0.00402 -0.00175   -0.00294   -0.0033  -0.00127   -0.00216   -0.00298  -0.00231    -0.00274    -9.791213   13.769991 -2.058023    28.968177  -57.064269    9.664295   -33.509414   75.202024  -13.947098   12.611617  -31.974638   6.360024   -8.121296  14.810735   -2.766638   25.532426  -59.413122   12.080842  -29.81416    76.696416 -16.65731    11.338682  -32.161415   7.365446  -2.735425   5.545806  -1.166749   8.765384  -21.891175   4.931566 -10.186372   27.94221   -6.652381   3.856514 -11.620005   2.896002 -0.324788  0.698059  -0.158691  1.044729  -2.726708  0.657992  -1.203449   3.453349 -0.875747  0.451513 -1.427377  0.377501 7.1  3.57136 0.35653 -0.11932 0.35702 -0.0576 -0.0082  3.425185 -2.991325  0.921437 -0.10922   0.00445  -0.149513  0.260358 -0.108515  0.015353 -0.000702 0.483 0.375 0.409  0.416  0.458  0.736 0.740 0.764
     1.0      0.0 0.189 -0.2662 0.0201 -0.177  0       0       0       0       0        0     -1.308 -1.028   -0.103    0.341   -0.168   -0.00341 -0.00154   -0.00251   -0.0029  -0.00121   -0.00183   -0.00234  -0.00187    -0.00261    -7.325605   13.614002 -2.386626    20.485389  -55.648435   10.84517    -22.683851   72.724816  -15.358254    8.102878  -30.762067   6.924623   -6.60002   15.889768   -3.329883   20.220607  -62.917428   14.24876   -22.94835    80.523509 -19.391391    8.442674  -33.570142   8.501097  -2.399851   6.205665  -1.42624    7.626841  -24.238636   5.9471    -8.706471   30.726631  -7.950366   3.229578 -12.720858   3.440594 -0.304166  0.805239  -0.196305  0.983223  -3.119945  0.806391  -1.12337    3.932134 -1.066722  0.417681 -1.620806  0.458069 7.4  3.61643 0.35977 -0.11193 0.35949 -0.0457 -0.0084  5.949047 -4.671327  1.304611 -0.14567   0.00569  -0.807548  0.726202 -0.222759  0.027034 -0.001128 0.472 0.383 0.404  0.413  0.439  0.730 0.735 0.750
     1.25892 -0.1 0.179 -0.2782 0.0225 -0.177 -0.007  -0.052   0.019  -0.034  -0.017    0     -1.209 -1.015   -0.12     0.388   -0.146   -0.00282 -0.00137   -0.00209   -0.00255 -0.00118   -0.00147   -0.002    -0.00136    -0.00225    -8.939958   14.674685 -2.50588     28.48766   -60.533728   11.447177   -33.74954    79.839562  -16.272449   13.110558  -34.093951   7.363532   -6.823299  16.349183   -3.378252   21.945586  -64.788429   14.473168  -25.43702    83.014323 -19.700187    9.622979  -34.666492   8.636822  -2.230057   6.227992  -1.418814   7.139707  -24.239425   5.910868  -8.089198   30.632422  -7.8882     2.98057  -12.650068   3.407453 -0.262857  0.794673  -0.192817  0.833591  -3.058636  0.790049  -0.924593   3.831317 -1.041746  0.332274 -1.570612  0.445901 7.7  3.65455 0.3576  -0.10572 0.35653 -0.0344 -0.0083  7.368131 -5.499974  1.459253 -0.156522  0.00591  -1.284003  1.045681 -0.29604   0.033957 -0.001359 0.457 0.390 0.403  0.408  0.429  0.724 0.727 0.738
     1.58489 -0.2 0.168 -0.2895 0.0246 -0.177  0.017  -0.118  -0.006  -0.033  -0.015    0     -1.038 -1.003   -0.134    0.411   -0.126   -0.00227 -0.00124   -0.00162   -0.00228 -0.0012    -0.00102   -0.00185  -0.00098    -0.00195   -15.626776   19.508059 -3.560531    56.985234  -81.272413   15.949639   -71.946571  108.418298  -22.436338   29.788741  -46.830402  10.092356  -11.768413  20.485075   -4.373144   43.018477  -82.174059   18.668326  -53.780421  106.598489 -25.382767   22.029363  -45.049168  11.130096  -3.820418   7.664231  -1.778515  13.91816   -30.2218     7.420629 -17.228546   38.688876  -9.924873   6.988037 -16.176321   4.297893 -0.449459  0.970762  -0.237817  1.629299  -3.788807  0.978539  -1.999441   4.811048 -1.295541  0.804229 -1.998174  0.556679 7    3.28577 0.60973 -0.08858 0.3841  -0.0247 -0.0073  7.648531 -5.478711  1.393184 -0.143506  0.00521  -1.556335  1.20724  -0.326813  0.03613  -0.001401 0.435 0.394 0.407  0.400  0.422  0.714 0.710 0.723
     1.99526 -0.3 0.155 -0.3002 0.0273 -0.177  0.029  -0.167  -0.038  -0.022   0.012    0     -0.877 -1.005   -0.141    0.422   -0.074   -0.00174 -0.00116   -0.0011    -0.00209 -0.00121   -0.00055   -0.00183  -0.00081    -0.00177   -20.196064   21.605585 -3.967159    76.955882  -91.937133   18.009102   -99.17849   124.888604  -25.625691   41.854937  -54.793264  11.641878  -14.237646  21.4583     -4.630733   54.204954  -87.447024   19.980398  -69.387392  115.040466 -27.41286    29.065915  -49.224908  12.113529  -4.450964   7.879438  -1.854116  16.869748  -31.468194   7.806779 -21.431777   40.755464 -10.521754   8.912003 -17.220691   4.586453 -0.512949  0.990385  -0.246374  1.934245  -3.908465  1.022173  -2.440762   5.014605 -1.362831  1.0086   -2.102644  0.589139 7.05 3.205   0.68673 -0.07303 0.40216 -0.0158 -0.0048  6.500901 -4.413972  1.059609 -0.101708  0.003404 -1.529644  1.14411  -0.298233  0.031732 -0.001184 0.421 0.392 0.409  0.394  0.421  0.706 0.697 0.713
     2.51189 -0.4 0.142 -0.3243 0.0323 -0.177  0.046  -0.200  -0.067  -0.014   0.070    0     -0.787 -0.989   -0.145    0.405   -0.012   -0.00127 -0.0011    -0.00071   -0.00188 -0.00117   -0.00018   -0.00176  -0.00081    -0.00169   -21.025654   20.914681 -3.65759     80.254159  -89.929962   16.838693  -102.867295  122.982399  -24.163864   43.215191  -54.20406   11.039272  -14.058824  20.33818    -4.315139   53.578857  -83.280809   18.729752  -68.270807  109.860542 -25.776517   28.477607  -47.081453  11.408887  -4.220327   7.384426  -1.728543  16.004347  -29.551045   7.303437 -20.246004   38.293665  -9.856116   8.382355 -16.175944   4.297066 -0.471012  0.920307  -0.229126  1.77601   -3.633509  0.952913  -2.231373   4.65817  -1.271094  0.917647 -1.950334  0.549199 7.2  3.15761 0.71829 -0.06331 0.42226 -0.0092 -0.0027  4.471238 -2.731619  0.575555 -0.044609  0.001042 -1.296131  0.930304 -0.231209  0.023231 -0.000811 0.422 0.390 0.401  0.386  0.413  0.701 0.693 0.708
     3.16228 -0.5 0.129 -0.3363 0.0348 -0.177  0.068  -0.233  -0.029  -0.039   0.146    0     -0.739 -0.957   -0.147    0.38     0.088   -0.00087 -0.00108   -0.00052   -0.00167 -0.00112    0         -0.00177  -0.00082    -0.00153   -22.741065   22.411323 -4.096293    86.63212   -97.020411   18.794135  -110.234252  133.365048  -26.940861   46.034113  -59.021914  12.307843  -15.263972  21.425521   -4.717232   58.109433  -88.201374   20.443717  -73.576682  116.846948 -28.124593   30.529589  -50.252     12.45084   -4.61871    7.768048  -1.881403  17.508884  -31.215017   7.940036 -22.024665   40.586946 -10.711395   9.075048 -17.193783   4.670423 -0.520072  0.970347  -0.249365  1.961472  -3.844432  1.036151  -2.451769   4.94279  -1.381728  1.003782 -2.074601  0.597048 7.25 3.07444 0.79168 -0.0485  0.43945 -0.005  -0.0017  1.70041  -0.551013 -0.02409   0.023559 -0.001692 -0.856979  0.570437 -0.12792   0.010975 -0.0003   0.427 0.387 0.386  0.381  0.401  0.694 0.691 0.702
     3.98107 -0.6 0.114 -0.353  0.0379 -0.177  0.097  -0.252   0.047  -0.071   0.222    0     -0.715 -0.892   -0.148    0.331    0.199   -0.0006  -0.00104   -0.00046   -0.00149 -0.0011     0         -0.00188  -0.00064    -0.00117   -26.264268   23.771252 -4.060054   100.472301 -103.646308   18.814282  -127.685929  143.161093  -27.132822   53.278586  -63.57275   12.444219  -17.751774  21.96526    -4.627096   68.116276  -91.178818   20.183777  -86.467332  121.521772 -27.87899    35.975417  -52.500411  12.374512  -5.401081   7.859048  -1.842007  20.706062  -31.830584   7.810706 -26.201897   41.632888 -10.56744   10.860052 -17.718161   4.616053 -0.611228  0.975707  -0.244152  2.338044  -3.894667  1.018348  -2.948565   5.036198 -1.361074  1.217733 -2.123459  0.588934 7.35 3.00688 0.83749 -0.03816 0.45682 -0.0015 -0.0015 -1.21286   1.659951 -0.609563  0.087637 -0.004167 -0.330958  0.159703 -0.015464 -0.001793  0.000211 0.429 0.381 0.368  0.373  0.395  0.681 0.684 0.697
     5.01187 -0.7 0.101 -0.3689 0.041  -0.177  0.104  -0.261   0.119  -0.088   0.298    0     -0.727 -0.81    -0.14     0.266    0.313   -0.00045 -0.00095   -0.00026   -0.00133 -0.00109    0         -0.00191  -0.00048    -0.00078   -25.705775   24.076487 -3.870359    96.629913 -105.020322   18.090218  -120.70367   145.026697  -26.228237   49.610109  -64.360305  12.071078  -16.84009   21.159053   -4.272283   63.439586  -87.991892   18.758983  -79.104292  117.348296 -26.011167   32.390672  -50.693008  11.572278  -4.974984   7.359398  -1.677937  18.699921  -29.825945   7.145804 -23.214685   38.994195  -9.688467   9.45493  -16.577788   4.235971 -0.5486    0.896211  -0.220551  2.05407   -3.574324  0.922361  -2.537028   4.613769 -1.233766  1.027578 -1.940914  0.533737 7.4  2.8864  0.87617 -0.03008 0.49447 -0.0001 -0.0014 -3.889662  3.647243 -1.123764  0.142588 -0.006239  0.20708  -0.251394  0.094773 -0.014072  0.000694 0.416 0.368 0.350  0.356  0.384  0.657 0.660 0.676
     6.30957 -0.8 0.089 -0.3897 0.0446 -0.177  0.127  -0.265   0.151  -0.101   0.370    0     -0.744 -0.721   -0.126    0.2      0.395   -0.00035 -0.00088   -0.00008   -0.00115 -0.00105    0         -0.00168  -0.00035    -0.00054   -24.77411    26.224526 -4.200479    90.85638  -112.942284   19.297359  -110.881403  154.63744   -27.703051   44.643788  -68.197225  12.669506  -16.399063  22.499296   -4.454105   60.312115  -92.857      19.391601  -73.538359  123.20744  -26.757826   29.522271  -53.029484  11.868641  -4.882647   7.738961  -1.723739  17.92392   -31.186636   7.297028 -21.769156   40.625276  -9.860528   8.69684  -17.22899    4.30281  -0.541658  0.936975  -0.225009  1.98114   -3.719234  0.936385  -2.394693   4.787192 -1.249196  0.951699 -2.010391  0.539632 7.4  2.71458 0.91369 -0.02248 0.54244  0.0001 -0.0014 -6.070844  5.214049 -1.513383  0.182277 -0.007654  0.699129 -0.617005  0.189848 -0.024315  0.001083 0.402 0.349 0.331  0.340  0.367  0.627 0.631 0.647
     7.94328 -0.9 0.08  -0.4051 0.0472 -0.177  0.161  -0.270   0.184  -0.115   0.422    0     -0.758 -0.623   -0.111    0.151    0.42    -0.00032 -0.00091   -0.00003   -0.00093 -0.00098    0         -0.00124  -0.00031    -0.00049   -23.2554     27.269174 -4.355136    83.882593 -116.925661   19.896133  -100.723717  159.512369  -28.450802   39.959542  -70.128341  12.972515  -15.584863  23.227516   -4.54862    56.449509  -95.598076   19.739125  -67.769774  126.51015  -27.164746   26.812408  -54.31465   12.021431  -4.678472   7.958405  -1.748796  16.923356  -32.002733   7.384073 -20.239549   41.594188  -9.954634   7.966198 -17.599496   4.334423 -0.52156   0.961043  -0.22741   1.879887  -3.807686  0.944166  -2.236969   4.890666 -1.256707  0.87537  -2.049251  0.541691 7.45 2.59172 0.94723 -0.0155  0.57479  0      -0.0014 -7.967855  6.53621  -1.828753  0.212558 -0.008649  1.149224 -0.944491  0.272771 -0.032948  0.001398 0.392 0.321 0.312  0.327  0.334  0.596 0.603 0.607
    10.000   -1.0 0.073 -0.4203 0.0498 -0.177  0.223  -0.271   0.246  -0.139   0.451    0     -0.729 -0.514   -0.098    0.121    0.395   -0.0003  -0.00106   -0.00022   -0.00072 -0.00092    0         -0.00076  -0.00018    -0.00062   -21.116565   26.950596 -4.295478    74.840821 -115.458924   19.652997   -88.397431  157.343534  -28.131669   34.546793  -69.08876   12.833902  -14.309344  23.080364   -4.515178   51.065975  -94.912402   19.60508   -60.439985  125.48986  -26.99314    23.597517  -53.822052  11.948725  -4.333707   7.93206   -1.740815  15.469595  -31.875118   7.352615 -18.261556   41.39963   -9.915528   7.099148 -17.503328   4.318339 -0.486169  0.95965   -0.226702  1.730713  -3.800122  0.941414  -2.034029   4.878348 -1.253388  0.786412 -2.042806  0.540368 7.45 2.41501 0.97695 -0.00945 0.61307  0      -0.0014 -8.935422  7.144835 -1.952403  0.221535 -0.008813  1.461981 -1.163289  0.32556  -0.038128  0.001573 0.392 0.289 0.293  0.324  0.303  0.568 0.585 0.573
    """)

    # constant table suffix
    SUFFIX = "_if"

    CONST_REGION = {"cc": 0.85, "rt": 150,
                    "cd0": 1.606, "cd1": 0.0097, "dp0": 25, "dp1": 55}


class HassaniAtkinson2020SSlab(HassaniAtkinson2020SInter):
    """
    Hassani Atkinson (2020) for Subduction IntraSlab.
    """
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.SUBDUCTION_INTRASLAB

    # constant table suffix
    SUFFIX = "_is"

    CONST_REGION = {"cc": 0.9, "rt": 250,
                    "cd0": 1.9241, "cd1": 0.0133, "dp0": 40, "dp1": 90}


class HassaniAtkinson2020Asc(HassaniAtkinson2020SInter):
    """
    Hassani Atkinson (2020) for Crustal.
    """
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    # constant table suffix
    SUFFIX = "_cr"

    CONST_REGION = {"cc": 0.45, "rt": 50,
                    "cd0": 2.5011, "cd1": 0, "dp0": 0, "dp1": 30}
