# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2014-2021 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.
"""
Module exports :class:`CauzziEtAl2014`,
               :class:`CauzziEtAl2014NoSOF`,
               :class:`CauzziEtAl2014FixedVs30`,
               :class:`CauzziEtAl2014FixedVs30NoSOF`,
               :class:`CauzziEtAl2014Eurocode8`,
               :class:`CauzziEtAl2014Eurocode8NoSOF`,
               :class:`CauzziEtAl2014Eurocode8scaled`
"""
import numpy as np
from scipy.constants import g

from openquake.baselib.general import CallableDict
from openquake.hazardlib.gsim.base import CoeffsTable, GMPE
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA


def _compute_mean(clsname, sof, adjustment_factor, C, ctx, imt):
    """
    Returns the mean ground motion acceleration and velocity
    """
    if sof:
        sof_term = _get_style_of_faulting_term(C, ctx.rake)
    else:
        sof_term = 0.
    if clsname.endswith("Germany"):
        if ctx.width > 1.0E-3:
            # Finite rupture source used
            rrup = np.copy(ctx.rrup)
        else:
            # Point source MSR used - convert rhypo to rrup
            rrup = rhypo_to_rrup(ctx.rhypo, ctx.mag)
    else:
        rrup = ctx.rrup
    mean = (_get_magnitude_scaling_term(C, ctx.mag) +
            _get_distance_scaling_term(C, ctx.mag, rrup) +
            sof_term + _get_site_amplification_term(
                clsname, sof, C, ctx.vs30))
    # convert from cm/s**2 to g for SA and from cm/s**2 to g for PGA (PGV
    # is already in cm/s) and also convert from base 10 to base e.
    if imt.string == "PGA":
        mean = np.log((10 ** mean) * ((2 * np.pi / 0.01) ** 2) *
                      1e-2 / g)
    elif imt.string[:2] == "SA":
        mean = np.log((10 ** mean) * ((2 * np.pi / imt.period) ** 2) *
                      1e-2 / g)
    else:
        mean = np.log(10 ** mean)

    return mean + adjustment_factor


def _get_distance_scaling_term(C, mag, rrup):
    """
    Returns the distance scaling parameter
    """
    return (C["r1"] + C["r2"] * mag) * np.log10(rrup + C["r3"])


def _get_magnitude_scaling_term(C, mag):
    """
    Returns the magnitude term
    """
    return C["c1"] + (C["m1"] * mag) + (C["m2"] * (mag ** 2.))


_get_site_amplification_term = CallableDict()


@_get_site_amplification_term.add(
    "CauzziEtAl2014", "CauzziEtAl2014NoSOF", "CauzziEtAl2014Armenia",
    "CauzziEtAl2014RhypoGermany")
def _get_site_amplification_term_1(clsname, sof, C, vs30):
    """
    Returns the site amplification scaling term assuming
    period dependent reference vs30
    """
    return C["bV"] * np.log10(vs30 / C["VA"])


@_get_site_amplification_term.add(
    "CauzziEtAl2014FixedVs30", "CauzziEtAl2014FixedVs30NoSOF")
def _get_site_amplification_term_2(clsname, sof, C, vs30):
    """
    Returns the site amplification scaling term assuming
    period dependent reference vs30
    """
    return C["bV800"] * np.log10(vs30 / 800.)


@_get_site_amplification_term.add("CauzziEtAl2014Eurocode8")
def _get_site_amplification_term_4(clsname, sof, C, vs30):
    """
    Returns the site amplification term on the basis of Eurocode 8
    site class
    """
    s_b, s_c, s_d = _get_site_dummy_variables(sof, vs30)
    return (C["sB"] * s_b) + (C["sC"] * s_c) + (C["sD"] * s_d)


@_get_site_amplification_term.add("CauzziEtAl2014Eurocode8NoSOF")
def _get_site_amplification_term_5(clsname, sof, C, vs30):
    """
    Returns the site amplification term on the basis of Eurocode 8
    site class
    """
    s_b, s_c, s_d = _get_site_dummy_variables(sof, vs30)
    return (C["sB"] * s_b) + (C["sC"] * s_c) + (C["sD"] * s_d)


@_get_site_amplification_term.add("CauzziEtAl2014Eurocode8scaled")
def _get_site_amplification_term_6(clsname, sof, C, vs30):
    """
    Returns the site amplification term on the basis of Eurocode 8
    site class
    """
    s_b = np.zeros_like(vs30)
    s_c = np.zeros_like(vs30)
    s_d = np.zeros_like(vs30)
    s_b[np.logical_and(vs30 >= 360., vs30 < 800.)] = 1.0
    s_c[np.logical_and(vs30 >= 180., vs30 < 360.)] = 1.0
    s_d[vs30 < 180] = 1.0
    return C["sB"] * s_b + C["sC"] * s_c + C["sD"] * s_d


_get_site_dummy_variables = CallableDict()


@_get_site_dummy_variables.add(True)
def _get_site_dummy_variables_1(sof, vs30):
    """
    Returns the Eurocode 8 site class dummy variable
    """
    s_b = np.zeros_like(vs30)
    s_c = np.zeros_like(vs30)
    s_d = np.zeros_like(vs30)
    s_b[np.logical_and(vs30 >= 360., vs30 < 800.)] = 1.0
    s_c[np.logical_and(vs30 >= 180., vs30 < 360.)] = 1.0
    s_d[vs30 < 180] = 1.0
    return s_b, s_c, s_d


@_get_site_dummy_variables.add(False)
def _get_site_dummy_variables_2(sof, vs30):
    """
    Returns the Eurocode 8 site class dummy variable
    """
    s_b = np.zeros_like(vs30)
    s_c = np.zeros_like(vs30)
    s_d = np.zeros_like(vs30)
    s_b[np.logical_and(vs30 >= 360., vs30 < 800.)] = 1.0
    s_c[np.logical_and(vs30 >= 180., vs30 < 360.)] = 1.0
    s_d[vs30 < 180.] = 1.0
    return s_b, s_c, s_d


_get_stddevs = CallableDict()


@_get_stddevs.add(True)
def _get_stddevs_1(sof, C):
    """
    Return standard deviations assuming style-of-faulting is known
    """
    return [np.log(10.0 ** C['sM']),
            np.log(10.0 ** C["tM"]),
            np.log(10.0 ** C['f'])]


@_get_stddevs.add(False)
def _get_stddevs_2(sof, C):
    """
    Returns the standard deviation terms assuming no style-of-faulting
    is known
    """
    return [np.log(10.0 ** C['s']),
            np.log(10.0 ** C["t"]),
            np.log(10.0 ** C['f'])]


def _get_style_of_faulting_term(C, rake):
    """
    Returns the style of faulting term. Cauzzi et al. determind SOF from
    the plunge of the B-, T- and P-axes. For consistency with existing
    GMPEs the Wells & Coppersmith model is preferred
    """
    if rake > -150.0 and rake <= -30.0:
        return C['fN']
    elif rake > 30.0 and rake <= 150.0:
        return C['fR']
    else:
        return C['fSS']


class CauzziEtAl2014(GMPE):
    """
    Implements GMPE developed by Carlo Cauzzi et al (2014) and published
    as C.Cauzzi, E. Faccioli, M. Vanini and A. Bianchini (2014) "Updated
    predictive equations for broadband (0.0 - 10.0 s) horizontal response
    spectra and peak ground motions, based on a global dataset of digital
    acceleration records", Bulletin of Earthquake Engineering, In Press

    Spectral acceleration (SA) values are obtained from displacement response
    spectrum  (DSR) values (as provided by the original equations) using the
    following formula ::

        SA = DSR * (2 * π / T) ** 2
    """
    #: Supported tectonic region type is active shallow crust,
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    #: Supported intensity measure types are spectral acceleration, peak
    #: ground acceleration and peak ground velocity.
    #: The original paper provides coefficients for PGA and PGV, while SA
    #: is obtained from displacement response spectrum values.
    #: Coefficients for PGA are taken from the SA (0.01 s) spectral
    #: acceleration, as indicated in Page 11 (at the time of writing)
    #: of Cauzzi et al. (2014)
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, PGV, SA}

    #: Supported intensity measure component is the geometric mean of two
    #: horizontal components
    #: :attr:`~openquake.hazardlib.const.IMC.AVERAGE_HORIZONTAL`,
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.AVERAGE_HORIZONTAL

    #: Supported standard deviation types are inter-event, intra-event and
    #: total
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.INTER_EVENT, const.StdDev.INTRA_EVENT, const.StdDev.TOTAL}

    #: Required site parameter is only Vs30
    REQUIRES_SITES_PARAMETERS = {'vs30'}

    #: Required rupture parameters are magnitude and rake
    REQUIRES_RUPTURE_PARAMETERS = {'rake', 'mag'}

    #: Required distance measure is Rrup,
    REQUIRES_DISTANCES = {'rrup'}

    #: The reference rock conditions. The definition of this parameter is
    #: unclear in the paper so we assume a value of 800 m/s
    DEFINED_FOR_REFERENCE_VELOCITY = 800.0

    #: style of faulting term
    sof = True

    def __init__(self, adjustment_factor=1.0, **kwargs):
        super().__init__(adjustment_factor=adjustment_factor, **kwargs)
        self.adjustment_factor = np.log(adjustment_factor)

    def compute(self, ctx, imts, mean, sig, tau, phi):
        """
        See :meth:`superclass method
        <.base.GroundShakingIntensityModel.compute>`
        for spec of input and result values.
        """
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            mean[m] = _compute_mean(
                self.__class__.__name__, self.sof, self.adjustment_factor,
                C, ctx, imt)
            sig[m], tau[m], phi[m] = _get_stddevs(self.sof, C)

    #: Coefficient table constructed from the electronic suplements of the
    #: original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    imt                       c1                   m1                    m2                    r1                   r2                    r3                   sB                   sC                   sD                    bV                 bV800                       VA                    fN                    fR                   fSS                    f                    t                    s                   tM                   sM
    pgv       0.4422159946365680   0.5482239378818140   -0.0319470258028777   -2.8457788432226700   0.2406737047414070   6.51696666287798000   0.1919277313361110   0.3706196443240230   0.4978018189249720   -0.6909580227999990   -0.7596804404000000     883.9565406477700000   -0.1433313027208760    0.0184633160924233    0.0049897699311183   0.2398935826956730   0.2213004346208430   0.3263783286033860   0.2149404298227920   0.3220998664588330
    pga      -2.1961743955816300   0.5237450060972680   -0.0609447663010394   -3.8019035608295600   0.3550808121411740   11.6415555876916000   0.2106985279596590   0.2825106921224770   0.2828846140789600   -0.3100704816000000   -0.7024376883999990    2319.1859784562300000   -0.0241122431339601    0.0724633666482452   -0.0563165754085399   0.2589229720745850   0.2214506097241730   0.3407073201666570   0.2162221044760760   0.3373323345485870
    0.0100   -2.1961743955816300   0.5237450060972680   -0.0609447663010394   -3.8019035608295600   0.3550808121411740   11.6415555876916000   0.2106985279596590   0.2825106921224770   0.2828846140789600   -0.3100704816000000   -0.7024376883999990    2319.1859784562300000   -0.0241122431339601    0.0724633666482452   -0.0563165754085399   0.2589229720745850   0.2214506097241730   0.3407073201666570   0.2162221044760760   0.3373323345485870
    0.0200   -1.8195694621090300   0.6453679917780000   -0.0718828302259198   -3.8644883929436800   0.3593203519694770   12.1350058517031000   0.1992376827391620   0.2657559899035650   0.2597832779404480   -0.2838417797000000   -0.6697416278000000    2528.8050326671200000   -0.0272261357768130    0.0770853044555010   -0.0589959439154124   0.2592700212170700   0.2266408762153660   0.3443646768667910   0.2208310066257620   0.3405690493706790
    0.0300   -1.2943572408433100   0.6982707545157320   -0.0795637931948720   -4.0443482492067200   0.3765163250150630   13.3154555674385000   0.1781152657707780   0.2315977870956060   0.2152815315755380   -0.2288226422000000   -0.6191416576999990    3228.4186287294800000   -0.0333340581611655    0.0835931466396944   -0.0620510202333671   0.2610359421517940   0.2329378875769420   0.3498568601069340   0.2262184621898820   0.3454193910750090
    0.0400   -0.6931089735055480   0.7088746113257160   -0.0846090922165347   -4.2569567103123300   0.3973949524463510   14.9701297686778000   0.1812772493016900   0.2182921854255120   0.1831111109272540   -0.1573206416000000   -0.6076234879999990    8228.7311143995100000   -0.0383834128731151    0.0898513499539263   -0.0655039981465477   0.2655226521597100   0.2370812514738440   0.3559631983932190   0.2293557959689310   0.3508651592199890
    0.0500   -0.1738965355678180   0.7038280147755070   -0.0875507720994941   -4.4155335972196800   0.4140306595178590   15.9634085011888000   0.1678531023791990   0.1863342553869610   0.1392855607298930   -0.1000000000000000   -0.5639501998999990   30552.3271673005000000   -0.0389955855635790    0.0960304396771990   -0.0706779148995298   0.2702569776994650   0.2423624069839720   0.3630129065395790   0.2337736891265200   0.3573359367920780
    0.0600    0.2069820440679000   0.6906315133559440   -0.0888413835088005   -4.4921203206778700   0.4232967413275050   16.1995006962009000   0.1627597795917690   0.1676876852934890   0.1025963949386600   -0.1000000000000000   -0.5367391490999990   23999.0695317075000000   -0.0490906833424132    0.1014786096166360   -0.0714347704520967   0.2750551534173430   0.2452399150958750   0.3685077385587310   0.2354962432697930   0.3620964208820960
    0.0700    0.5608903992017920   0.6756111304642140   -0.0888630574702343   -4.5440277265296000   0.4259196396385800   17.3789888842896000   0.1654527289615880   0.1595821580576020   0.0862878935870393   -0.1000000000000000   -0.5266083805000000   20327.5550092185000000   -0.0514168691984178    0.1046974117607580   -0.0734422969499682   0.2799503455014960   0.2489917590200640   0.3746586339673930   0.2388079459053130   0.3679693342846950
    0.0800    0.7587338187758670   0.6654944435687730   -0.0879358050549968   -4.5316608540212100   0.4201856029560480   18.1449146140712000   0.2082896980661270   0.1963833825796230   0.1173006295737050   -0.1000000000000000   -0.6008677443000000   37180.0713615876000000   -0.0435691980080517    0.1025092491607880   -0.0752680566199420   0.2859658855181830   0.2463439179170010   0.3774411392189470   0.2367345282757910   0.3712407905364930
    0.0900    0.7846617840834190   0.6646697953340180   -0.0863250891406979   -4.4431082348763900   0.4096459868804770   17.7940601032257000   0.2058086667038910   0.2005503414374950   0.1239225551215690   -0.1000000000000000   -0.5999306542000000   36947.8787712051000000   -0.0386068513354472    0.1002656772083840   -0.0754602300716578   0.2879669013021380   0.2455100824949440   0.3784179393900740   0.2363877833587920   0.3725642499849370
    0.1000    0.6759351202982480   0.6701555114371590   -0.0842487133409999   -4.2989668796524600   0.3950150942756780   16.9499399186483000   0.1982256814552770   0.2004113577343430   0.1312995606293180   -0.1000000000000000   -0.5923724273999990   36597.5600451842000000   -0.0282989830364419    0.1005680531402610   -0.0799511338270659   0.2864871213393960   0.2433368095307030   0.3758825262843640   0.2343281863791130   0.3701142656330040
    0.1500    0.4060442725277190   0.6331119438041180   -0.0719075361587948   -3.8697055665013900   0.3527209416036150   13.7553799942078000   0.2489114740764020   0.3018776519585660   0.2531944049628660   -0.2273218226000000   -0.7549461000999990    5440.3520640067600000   -0.0299967463616727    0.0962871690940792   -0.0762552422410023   0.2888548904585920   0.2369264158171150   0.3735923905191180   0.2282717557908330   0.3681645586333300
    0.2000   -0.0964722849285886   0.6394154973768270   -0.0625617809919999   -3.4154331729506900   0.3010051287415760   11.4534631595437000   0.2821393446741870   0.4058929876762180   0.3690896542349010   -0.4384492105000000   -0.8944358918000000    1898.2857019816500000   -0.0036711341998802    0.0693398950336865   -0.0633905167734821   0.2932537473600140   0.2159716768759310   0.3641998428792800   0.2112464445303770   0.3614177924057090
    0.2500   -0.3647383426795800   0.6464747603497000   -0.0578247658996581   -3.1964135977713500   0.2821632069794590    9.3622805399371000   0.2473181283189340   0.4087140621326480   0.4091275508533770   -0.5647295578000000   -0.8869576743000000    1202.6438872651600000   -0.0230481052501677    0.0635985592512166   -0.0517929263524518   0.2957833156803400   0.2005738694875840   0.3573760581741280   0.1960766398604580   0.3548715521619380
    0.3000   -0.5066096034428090   0.6651087078884250   -0.0560927957029996   -3.0938863341046000   0.2735167424921670    8.2935173490201800   0.2096502844403170   0.4001251672249500   0.4335528490826920   -0.6554113151999990   -0.8545657526000000     967.9255537188150000   -0.0179995073864090    0.0508746350031907   -0.0417662944939413   0.2955895319193470   0.1990806049250280   0.3563793745962490   0.1961542457250350   0.3547529555849480
    0.3500   -0.6649758303983760   0.6953229230929930   -0.0557864397714597   -3.0349384847947700   0.2671679125064100    7.6913250281925500   0.2023161205394590   0.4213786462865890   0.4808046291639440   -0.7428322161999990   -0.8762700561999990     873.3816688525000000   -0.0055734214033697    0.0466580942068334   -0.0422068301406186   0.2982216260863880   0.2050086373684360   0.3618904249372630   0.2027190392808950   0.3605983182886170
    0.4000   -1.0245292872263400   0.7477063980675440   -0.0562692503039990   -2.8852884017037800   0.2536783288165920    6.2119923952061800   0.1858234788604210   0.4107333337437640   0.5209946684961200   -0.8100884283999990   -0.8530330874999990     794.5532205269830000    0.0004098177393169    0.0403886753646120   -0.0384916849988285   0.2987789982376510   0.2068578206071190   0.3633992951757350   0.2051740007093480   0.3624434581489560
    0.4500   -1.1767192815454400   0.7807854544753780   -0.0576268771260726   -2.8427465328831800   0.2513716341911510    5.6484108448966700   0.1887026468902940   0.4164505033363750   0.5807879191359790   -0.8751821459999990   -0.8642899934000000     759.8399393759310000    0.0142197714791134    0.0307891929593598   -0.0345417763728089   0.2980455169408630   0.2077050898342350   0.3632802423909040   0.2066635500615080   0.3626857497787470
    0.5000   -1.3699721103726600   0.8260335951960020   -0.0600971406249981   -2.7937608332104700   0.2483537730430820    5.2021129207912500   0.1912072326769610   0.4163804535941900   0.6163803445766500   -0.9166134093000000   -0.8761241065999990     747.1063408592170000    0.0086757291814616    0.0281315838618958   -0.0297855596041406   0.2988253255683690   0.2171380197001870   0.3693852931565640   0.2163378482566530   0.3689154913924880
    0.5500   -1.5532301153734600   0.8841219121467910   -0.0636142514828666   -2.7774746094333300   0.2461269298729300    5.2748945458277300   0.1922552801830820   0.4208041398937580   0.6473190646386160   -0.9489890487000000   -0.8873310084000000     734.2611220214530000    0.0079885694661098    0.0262605964386996   -0.0276217316684702   0.2988468611615510   0.2262163167887810   0.3748109769038170   0.2255424149235550   0.3744046305211260
    0.6000   -1.7510694008742700   0.9456512032228760   -0.0676680143359787   -2.7659990406711700   0.2459840593699260    5.2326502260400600   0.1856646121052040   0.4151367349613850   0.6685439427938120   -0.9849329436000000   -0.8818026484000000     707.0886231073480000    0.0074237017603430    0.0214264218419423   -0.0228007233673257   0.2969715874045810   0.2312084370851160   0.3763634747221610   0.2307626428710610   0.3760897779393020
    0.6500   -1.8812203651728900   0.9989155250987470   -0.0714815223937251   -2.7822077114825400   0.2481313717858710    5.1530276954427200   0.1694624830668270   0.4049886708458540   0.6814504542040380   -1.0153146583999900   -0.8594841094999990     681.4614028632290000    0.0196350915362537    0.0150526924795911   -0.0215740715394286   0.2954635012276740   0.2351953967474790   0.3776447473604260   0.2348427194465870   0.3774252024372300
    0.7000   -2.0011372675364000   1.0414823426186300   -0.0743785413870095   -2.8015816681340200   0.2515222494019510    5.1214455547007300   0.1675787426166320   0.4084745747915290   0.6970629308604510   -1.0382766045000000   -0.8628249864999990     673.9691869217830000    0.0259814084850892    0.0078403558745567   -0.0173402456257839   0.2960570452959170   0.2315424218284400   0.3758479309182480   0.2312259898442240   0.3756530745898280
    0.7500   -2.0314990724971100   1.0612712694853100   -0.0761444939880500   -2.8411938303340100   0.2583666046875160    5.0513970104537200   0.1680957619203300   0.4032102189640850   0.7080554315362520   -1.0313556675000000   -0.8579717241000000     675.6979435204260000    0.0239285639574445    0.0074392636316948   -0.0161682493304947   0.2951575001067500   0.2297001913951280   0.3740055184034380   0.2294277631124650   0.3738382649703680
    0.8000   -2.0713348083169800   1.0808387377773600   -0.0771826900479910   -2.8493436285427500   0.2594995074375930    5.0928181363661800   0.1544249375549970   0.3844974525043310   0.7007764887596430   -1.0231311698000000   -0.8263667377000000     657.2061757191460000    0.0214029074524496    0.0059539323088165   -0.0138202326157295   0.2924385987558100   0.2300565388177360   0.3720837877347760   0.2298456978854890   0.3719534633240150
    0.8500   -2.1262894469442700   1.1018607109753000   -0.0783302036376399   -2.8588995970631400   0.2606572708674180    5.0935967372020100   0.1504639323164540   0.3792467896733620   0.6969408900855260   -1.0144829571000000   -0.8137829287000000     653.8125318549530000    0.0211071089179620    0.0019008327668466   -0.0099488373254348   0.2928306239015010   0.2309607956817200   0.3729512882890680   0.2307840014488140   0.3728418292242290
    0.9000   -2.2073925801959200   1.1318798449661200   -0.0803255749490028   -2.8708066901482300   0.2616586646777760    5.1233267618852900   0.1544558393365580   0.3837285215652260   0.7012236860739990   -1.0093585997000000   -0.8210169819999990     662.2439869415740000    0.0289033402423253   -0.0019777781552522   -0.0093910224696682   0.2946269771092460   0.2326264459950670   0.3753932857908200   0.2322986984058910   0.3751902729570740
    0.9500   -2.3618782822744800   1.1773372149747300   -0.0831133136219246   -2.8589190798699000   0.2591714809709110    5.1245152218844200   0.1606585521223750   0.3919467205487660   0.7031862226528190   -1.0029855543999900   -0.8356389507999990     676.6634212470730000    0.0322919482685976   -0.0043031378218605   -0.0085106024606202   0.2959927052620510   0.2310434003198280   0.3754899924095130   0.2306211043763800   0.3752302964209160
    1.0000   -2.4878726728670000   1.2134822214871800   -0.0854280000001306   -2.8543804213112900   0.2593699405690310    4.9780849514747200   0.1576721986608320   0.3903326170865970   0.6932630179184690   -0.9891875651000000   -0.8248044635000000     678.6122701764790000    0.0344742509193550   -0.0054363568289916   -0.0082972422158440   0.2963176996743180   0.2307500744544380   0.3755659409491340   0.2302608656402460   0.3752655664801600
    1.0500   -2.6077715643895400   1.2585971052394800   -0.0891555463418651   -2.8698128821985400   0.2620344315381600    5.0106995281376200   0.1568834769642220   0.3926119120922580   0.6859752493739490   -0.9787207632000000   -0.8182870278999990     683.9014863863440000    0.0342851595332001   -0.0084217050339768   -0.0054727622220529   0.2956778853817590   0.2309807392831370   0.3752032966587760   0.2304571871861140   0.3748812172269560
    1.1000   -2.6283843967759800   1.2724304040636900   -0.0902954540190852   -2.8957258830060600   0.2658130031005880    5.0680549323866900   0.1552026620595820   0.3911807820188090   0.6792311574060850   -0.9656582968000000   -0.8078533354000000     685.9310034655940000    0.0343732130091374   -0.0110760496482985   -0.0031534311291972   0.2949665179443070   0.2350632307671690   0.3771736591636390   0.2344939964929200   0.3768191623304360
    1.1500   -2.5913200736133200   1.2648135583775700   -0.0894105129027280   -2.9212440688281100   0.2694982170772390    5.1323413380794200   0.1499754504722890   0.3853429932664650   0.6679284297037930   -0.9508205297999990   -0.7903242998999990     687.3803208241630000    0.0342476309187609   -0.0123918851956884   -0.0019473464552664   0.2934734744640030   0.2363129145260130   0.3767896943730720   0.2357190478967070   0.3764175205211680
    1.2000   -2.5460330051693400   1.2537233711043700   -0.0881090370310901   -2.9336359839233800   0.2706921021887200    5.2621167586743000   0.1467021411264490   0.3820119568800980   0.6548859781812440   -0.9329132548999990   -0.7727555701999990     688.2770061287130000    0.0394944822294414   -0.0140707722227885   -0.0025214068554055   0.2920371732072930   0.2373279847374510   0.3763114174117580   0.2365438371208830   0.3758173724228000
    1.2500   -2.5558035297311500   1.2598048987799600   -0.0884048124815144   -2.9436386794084200   0.2721280979853120    5.3334214341034300   0.1450122336761420   0.3799349477612370   0.6449782534536990   -0.9205079916999990   -0.7607625045000000     685.7605219483220000    0.0420775458086862   -0.0148688650392528   -0.0028152587595069   0.2920346061128800   0.2375158432220640   0.3764279306175260   0.2366264100045000   0.3758673556976230
    1.3000   -2.5821309228717900   1.2734202709970100   -0.0893057028854706   -2.9577833877387700   0.2735012756348860    5.4399529144861700   0.1430110207548210   0.3781448011175160   0.6368116110983800   -0.9099645055999990   -0.7537088990000000     685.3520122824150000    0.0426056513658025   -0.0158075818461505   -0.0021521561066715   0.2923782327275850   0.2374655849380110   0.3766628930527370   0.2365287430472170   0.3760729680000980
    1.3500   -2.6238708063339100   1.2893198984461000   -0.0902480861259522   -2.9650343169163600   0.2736430339455760    5.5472867869600800   0.1440167428698510   0.3830721447314410   0.6334483811371680   -0.9050477560999990   -0.7571984443000000     691.1287292276600000    0.0439642557930039   -0.0173565783718463   -0.0012399284693199   0.2928827158501950   0.2370525657940650   0.3767946445921650   0.2360197987997750   0.3761457572129090
    1.4000   -2.6897548934787500   1.3189732552765200   -0.0924700115191799   -2.9840040831474000   0.2753069977851920    5.7009743793087200   0.1400371608701880   0.3803881959613850   0.6237737813471980   -0.8974194347999990   -0.7469922270000000     689.8483314919400000    0.0427099254056566   -0.0177215105903934   -0.0003940284894682   0.2930820893807540   0.2366013546491680   0.3766660485597420   0.2355942585420310   0.3760342614360530
    1.4500   -2.7763781872992100   1.3562398947265200   -0.0951541121376767   -2.9979152445537700   0.2755039161660600    5.9406969906893200   0.1348384928791460   0.3751982848031280   0.6154277454749500   -0.8917076486999990   -0.7346528961999990     684.4934409269750000    0.0388657870247459   -0.0169632822391633    0.0004576792093976   0.2933475512926050   0.2355185949469470   0.3761938255941360   0.2346553895484990   0.3756540132802060
    1.5000   -2.8239409130152600   1.3703307095684200   -0.0953411922785113   -2.9872080730216500   0.2723397555246710    6.0993032129832700   0.1279790593354570   0.3695852395116420   0.6045492601465070   -0.8858640115000000   -0.7202632058000000     676.4648003906380000    0.0322281438599955   -0.0153728103886031    0.0016529590876036   0.2939574090335400   0.2337014624547160   0.3755360593593970   0.2330673335328770   0.3751417602531570
    1.5500   -2.9058807855376000   1.3972193664435300   -0.0965528808676893   -2.9789874939584600   0.2690506859058310    6.2819150839882200   0.1228703602752770   0.3636375877700690   0.5955044277942730   -0.8758306752000000   -0.7085584153999990     670.9736962567340000    0.0240867982150792   -0.0127750113990195    0.0024527440869498   0.2938025497825890   0.2327236235129170   0.3748069145569930   0.2323370172741810   0.3745669871393020
    1.6000   -2.9697452419663000   1.4242508352711900   -0.0985065989181234   -2.9934555813201100   0.2702162526361470    6.4510931722959500   0.1206729389832730   0.3580844544096510   0.5875273890730360   -0.8620284101999990   -0.6974626159000000     669.3314773407340000    0.0192500261080120   -0.0105936664057168    0.0023400276834960   0.2936737017486110   0.2314612069701580   0.3739231651433410   0.2312046955904600   0.3737644369944920
    1.6500   -3.0285947165296400   1.4530351338993800   -0.1008689095101130   -3.0132293343275000   0.2723711234336250    6.6565535109269300   0.1162008798427380   0.3511458083516420   0.5750237701071880   -0.8462288153000000   -0.6816303814999990     663.6394066644560000    0.0143926805023867   -0.0091046518034926    0.0028626080527576   0.2929173415026190   0.2298956351567980   0.3723608089167130   0.2297314730222000   0.3722594775823870
    1.7000   -3.1224529274310200   1.4897100201437400   -0.1036264897313260   -3.0220673024548200   0.2725628585490270    6.8649953588168000   0.1142843931807090   0.3484572719214820   0.5656123541095750   -0.8351013937000000   -0.6738748786999990     660.3768967697260000    0.0067847660806980   -0.0073685004154968    0.0042018087093833   0.2918984215972050   0.2293621923535740   0.3712299877598490   0.2292906358752220   0.3711857812888360
    1.7500   -3.1998736979744500   1.5236815285884400   -0.1062397204542160   -3.0323161841880300   0.2722705836451780    7.1493411712552800   0.1141550287191680   0.3478519469698870   0.5608204754496630   -0.8288853502999990   -0.6734694895999990     659.8985850058170000   -0.0007037557828340   -0.0052871357705642    0.0051757968361060   0.2915075344953930   0.2278550539152940   0.3699926597411420   0.2278274383496330   0.3699756537024280
    1.8000   -3.2607744218824900   1.5512981994209700   -0.1081734151231750   -3.0343691916594900   0.2705121679164970    7.4776984114104700   0.1133702591656230   0.3447622926767550   0.5549524280769710   -0.8208158142999990   -0.6689213870000000     657.3126819232200000   -0.0061777839005103   -0.0038681775835265    0.0059708804220087   0.2913477760316090   0.2268738188329180   0.3692631260637820   0.2268434551239150   0.3692444714970290
    1.8500   -3.2909776392730300   1.5682745232340400   -0.1091898494294350   -3.0363512632004200   0.2687979434647760    7.8329887072439300   0.1131215672250670   0.3413029900876950   0.5510641851374680   -0.8135836830999990   -0.6635266952999990     654.0100291172610000   -0.0083740926965916   -0.0038041437273774    0.0067550488580701   0.2914734387021230   0.2254861862925840   0.3685115814701270   0.2254430871194260   0.3684852113705210
    1.9000   -3.3343433075582200   1.5915991149446000   -0.1106936328673180   -3.0375507641043200   0.2666795724816280    8.2507975195835100   0.1098369163208500   0.3354068180823930   0.5449205911593960   -0.8059532870000000   -0.6520968804000000     647.4266150139670000   -0.0104189177845531   -0.0034604521911671    0.0072196635584641   0.2911665894857830   0.2248095256004810   0.3678550062640120   0.2247530230939940   0.3678204782535530
    1.9500   -3.4108504798713200   1.6252619686965300   -0.1131125547688830   -3.0397762621238200   0.2649863796637390    8.6493609142322000   0.1073862711270420   0.3299294214445370   0.5404962138620350   -0.7983612451999990   -0.6426525954999990     643.8446894795020000   -0.0134266785185502   -0.0027166457870933    0.0076790328462967   0.2910622141683770   0.2241991485105830   0.3673996063001000   0.2241174298617830   0.3673497446364290
    2.0000   -3.4835528377726100   1.6562992065143400   -0.1154147661327820   -3.0442154569996200   0.2641813503629670    8.9746507855722500   0.1058745154124260   0.3256670883462820   0.5351158123025720   -0.7909590998000000   -0.6342277109999990     641.1223369683100000   -0.0162396506874878   -0.0019971914576055    0.0080969196977610   0.2906422190364900   0.2257008081025670   0.3679860789005580   0.2255881657370820   0.3679170015194690
    2.0500   -3.5202975354726100   1.6731512260232200   -0.1165005271881150   -3.0477563883612600   0.2634846566131420    9.2432914325525100   0.1041709341573030   0.3232678877286640   0.5306570263136390   -0.7868670273000000   -0.6274791391999990     637.7141860181120000   -0.0191554489533734   -0.0012473047561400    0.0085218466106380   0.2907682358824010   0.2264100908539320   0.3685209576654350   0.2262572116427430   0.3684270522349200
    2.1000   -3.5458938885078200   1.6845777939319400   -0.1171181667917870   -3.0515383989493900   0.2632357138015600    9.4625130088972500   0.1012881822995030   0.3209059812135480   0.5252967722487980   -0.7841080493000000   -0.6199304787999990     633.7627542839510000   -0.0223249288146324   -0.0001124348910179    0.0086821556135342   0.2907756522847090   0.2265162799678280   0.3685920577712730   0.2263101781128020   0.3684654348497920
    2.1500   -3.5199825019468000   1.6792425435213300   -0.1165416774781870   -3.0597853386757800   0.2639617395075430    9.6775735981192400   0.1020760034791760   0.3212840907303280   0.5227146834108650   -0.7790012309999990   -0.6178458570000000     636.0483713824860000   -0.0231098302387173    0.0000496747344591    0.0088333141646443   0.2908183989406750   0.2266935312320810   0.3687347261445390   0.2264723574859560   0.3685987925477570
    2.2000   -3.5031419338245600   1.6764268436783600   -0.1160638437237370   -3.0623082143254100   0.2635468927546750    9.9304019444917400   0.1027788501234010   0.3209574693616680   0.5207400269584710   -0.7739187749999990   -0.6160142927999990     638.0998569495980000   -0.0241033865413230    0.0005503540992503    0.0087555682973175   0.2905090552219770   0.2270693384779830   0.3687221116814050   0.2268282830901860   0.3685737119974880
    2.2500   -3.5327494922274100   1.6854363336311600   -0.1163493747163140   -3.0524679397645100   0.2614107154446340   10.1609354498268000   0.1023983641001920   0.3195504091182840   0.5178463907777940   -0.7688815777999990   -0.6121204005000000     638.1598871439530000   -0.0255833627654846    0.0013614170963980    0.0085803694880260   0.2898419116195210   0.2274521055300810   0.3684328894674380   0.2271787748643290   0.3682642115114040
    2.3000   -3.6034185018030000   1.7053958462832100   -0.1174030046572510   -3.0333789777930400   0.2581517254351440   10.3141802331871000   0.0995632067023649   0.3157899687229460   0.5121518403979560   -0.7647115669999990   -0.6038968611999990     633.5439350350940000   -0.0279811702628499    0.0031092206565588    0.0078860845757750   0.2890951139293440   0.2275287420327450   0.3678930732547480   0.2271936822626840   0.3676859450100010
    2.3500   -3.6861910178165100   1.7288663571843100   -0.1187897511063700   -3.0143612007839700   0.2551674844371200   10.4285547711269000   0.0978867879029756   0.3121334367528690   0.5085425057358670   -0.7607104611999990   -0.5960672843000000     629.7683038672590000   -0.0295539708674842    0.0043637172325333    0.0073336397193709   0.2886267596273490   0.2278171020632830   0.3677037372199200   0.2274344803092100   0.3674668001418680
    2.4000   -3.7689037444354200   1.7529547023159300   -0.1201054792527720   -2.9947049781256600   0.2515135754997760   10.6050667019036000   0.0973966535081477   0.3083519663731510   0.5069691796615470   -0.7559279834000000   -0.5900635576999990     627.2428414098920000   -0.0323287785513523    0.0060804070705871    0.0068212119493280   0.2877838133878200   0.2282117716579920   0.3672875385461850   0.2277394870013780   0.3669942740529980
    2.4500   -3.8494938660404200   1.7791604255695100   -0.1216731863400040   -2.9797613343876500   0.2483373954345230   10.8043525744193000   0.0945259273969444   0.3020872016081880   0.5013501663390210   -0.7481718813000000   -0.5789829507000000     622.7702303382660000   -0.0367443606799345    0.0084730586414835    0.0063128366439689   0.2869916378459290   0.2286416561649510   0.3669348813173000   0.2280072580037770   0.3665399158289450
    2.5000   -3.9002572097670600   1.7946260101828300   -0.1224119618095850   -2.9677652725933600   0.2458454123655570   10.9716659441135000   0.0930720253586086   0.2982395942724850   0.4962479907705270   -0.7415534768000000   -0.5715721373000000     621.6507340579120000   -0.0405121691619917    0.0099957654390232    0.0063571979234923   0.2860344068541520   0.2285747614282030   0.3661449213991180   0.2277909442989510   0.3656561174232070
    2.5500   -3.8994528351623900   1.7936169784114500   -0.1218845821378960   -2.9624155764303500   0.2443985636796040   11.1866342824377000   0.0948409359988404   0.2981838265931670   0.4953720424609630   -0.7371349642000000   -0.5701902476000000     625.7864778923530000   -0.0426199802281462    0.0103070115694261    0.0068947513062576   0.2848050752929210   0.2284466171455710   0.3651051736114700   0.2275834232863720   0.3645656943094760
    2.6000   -3.8698179664311800   1.7840224381607500   -0.1208351845337930   -2.9646953103741100   0.2444133547013920   11.3517284451802000   0.0976131314287696   0.2985750376832070   0.4964706954572310   -0.7332587512999990   -0.5698742290999990     630.5587559238600000   -0.0443354219076693    0.0105699172352999    0.0073296590668999   0.2837864772009310   0.2285405252177750   0.3643700540781740   0.2276105550533470   0.3637874783631340
    2.6500   -3.8249518732679000   1.7684712223157600   -0.1193833065548650   -2.9705641575388300   0.2453170719053730   11.4521028059218000   0.1011266888525050   0.3010762903987990   0.4991181163848110   -0.7324219121999990   -0.5734599462999990     635.9633126441040000   -0.0453864025306060    0.0108989034436747    0.0074464663433345   0.2829203819718650   0.2286033460842100   0.3637353878522180   0.2276281517574240   0.3631232821062380
    2.7000   -3.8010918839736000   1.7594997949301300   -0.1185009660848080   -2.9742006247369300   0.2462015567627040   11.5219618174955000   0.1028753405002940   0.3014583372071570   0.5007677128293540   -0.7318206736000000   -0.5730980276999990     637.5152608571610000   -0.0446232983508252    0.0111165446787204    0.0069771242129164   0.2821061112580920   0.2289163333444540   0.3632995261227740   0.2279666078172250   0.3627018504072670
    2.7500   -3.7775495124959000   1.7514317483210300   -0.1178016223292640   -2.9802058749887600   0.2475132246720580   11.5846057470485000   0.1054994703480340   0.3022289610994910   0.5038724133576740   -0.7312017747000000   -0.5740378487999990     640.5276176995040000   -0.0438936164694749    0.0113697316347867    0.0064955165140587   0.2814840663440500   0.2296905340060150   0.3633056853636480   0.2287652609516670   0.3627214140685190
    2.8000   -3.7675173287034700   1.7465767133125000   -0.1172695006923400   -2.9814982243543100   0.2481000718768770   11.6493362971215000   0.1073867452979670   0.3022557568711900   0.5055866120385230   -0.7292501600999990   -0.5737987951999990     643.0684280352780000   -0.0431466781695960    0.0115605699278133    0.0060543782081399   0.2813433554348280   0.2304640669149770   0.3636863618371180   0.2295648522366750   0.3631172056372080
    2.8500   -3.7374236197664800   1.7365906358016800   -0.1164695577044140   -2.9875865198616200   0.2495350804118060   11.7176973800536000   0.1098461235064850   0.3031672861196750   0.5069985526411030   -0.7266300476999990   -0.5742307930000000     646.5573550149580000   -0.0420666298353129    0.0114599285177020    0.0057447283174137   0.2812416753501300   0.2309314935767940   0.3639041558973420   0.2300747016152490   0.3633610439729250
    2.9000   -3.7284955682732900   1.7312089848180300   -0.1158872648205700   -2.9851806834689800   0.2499292385372980   11.7417665563445000   0.1092801189078820   0.3013144405842280   0.5050333263388100   -0.7233565450000000   -0.5683084498999990     644.9932655652070000   -0.0419696695385640    0.0115189749272974    0.0056644789610511   0.2812125940523160   0.2314256826175710   0.3641955101708990   0.2305734454473940   0.3636545569618440
    2.9500   -3.7293477782697800   1.7305044988754600   -0.1157771908946510   -2.9864543650849900   0.2508656441106190   11.7706242218151000   0.1087673659644000   0.2992866185509130   0.5023343957707720   -0.7190851235999990   -0.5619106802000000     643.8430695390050000   -0.0418387212028657    0.0114385205458388    0.0057047398189249   0.2811940369283110   0.2320915251748830   0.3646046659904990   0.2312488305299750   0.3640688231990220
    3.0000   -3.7181242535349100   1.7276931468404900   -0.1155813789892250   -2.9923117217806700   0.2523813577739830   11.8386025178292000   0.1081958140488820   0.2968493026283440   0.4983702292088530   -0.7127096177999990   -0.5547507065000000     643.3004613025160000   -0.0434663752687055    0.0115341667660334    0.0062500594261708   0.2812581479022440   0.2324512752973370   0.3648831883613700   0.2315528559466120   0.3643115025063480
    3.0500   -3.6956340556871400   1.7206187941941100   -0.1150167598499700   -2.9988436649335800   0.2539419556951400   11.9194683915271000   0.1086737109732570   0.2959065975557770   0.4948486904189350   -0.7068463845000000   -0.5505036211999990     645.2135596868140000   -0.0452716433068448    0.0115064140929809    0.0069656892578927   0.2814701521390040   0.2324461162676770   0.3650433447045440   0.2314855797219940   0.3644324631044570
    3.1000   -3.6813160032904200   1.7155258202519800   -0.1145505404919180   -3.0038839370105000   0.2551858648729980   12.0039385111510000   0.1090050728960150   0.2949483545221940   0.4926409964218200   -0.7026153174000000   -0.5469183171999990     645.8935255249500000   -0.0468072038612910    0.0115728331592671    0.0074902920484155   0.2818331094534140   0.2325544222368890   0.3653922014577130   0.2315379655698930   0.3647461186694460
    3.1500   -3.6775783350675800   1.7155128319969200   -0.1146377916229110   -3.0118407676316600   0.2569358791900200   12.0742634904719000   0.1091443101736000   0.2938131649454180   0.4900236170672850   -0.6980489868000000   -0.5428919486000000     645.7804611139640000   -0.0479615510478418    0.0118381791218001    0.0077074163092398   0.2818751172675010   0.2332212849614790   0.3658493535509540   0.2321583978509210   0.3651727035626760
    3.2000   -3.6659294873541300   1.7136637460848000   -0.1147416240813500   -3.0252506083430600   0.2598508147657760   12.1048145398767000   0.1100889034892090   0.2931797204446210   0.4884942104824450   -0.6940035704000000   -0.5396923054000000     646.1919498532980000   -0.0483250927774362    0.0118108190612669    0.0078979662501019   0.2815906434174380   0.2338622314080920   0.3660393882349530   0.2327894798072230   0.3653549402555880
    3.2500   -3.6684173961240700   1.7163524483997700   -0.1151526871572930   -3.0362930264013800   0.2622711064006620   12.1519366896376000   0.1104048038719750   0.2916625146509380   0.4867114189241820   -0.6900107584000000   -0.5355403509999990     645.4950076960340000   -0.0490027472661637    0.0116646377564768    0.0083183567788232   0.2811251559606220   0.2343102657058470   0.3659681050706860   0.2332171398454860   0.3652691988542060
    3.3000   -3.6568737955112700   1.7149716752503300   -0.1152636829783470   -3.0497636006247000   0.2650378354531420   12.1895595447057000   0.1110253339198450   0.2899773735042430   0.4849558687508500   -0.6850873382000000   -0.5313441059000000     645.7166178580670000   -0.0499209970029787    0.0115349091863349    0.0088247114285518   0.2803604769175450   0.2348226217665170   0.3657098039576380   0.2336979833399140   0.3649886908310670
    3.3500   -3.6406199358627100   1.7128733853983800   -0.1153168220048140   -3.0638966368024300   0.2677208608308630   12.2532416189361000   0.1118343185619860   0.2885592102376240   0.4837750631605140   -0.6808719651000000   -0.5274345794999990     646.1798543932350000   -0.0500552073655930    0.0111990758754792    0.0092369004974032   0.2792772205267840   0.2357446163122870   0.3654740620418820   0.2346241531709040   0.3647523257723380
    3.4000   -3.6320648941606900   1.7114392949805400   -0.1153895399029740   -3.0748815506767300   0.2701811743391670   12.2787609002995000   0.1128839041023790   0.2877535026213520   0.4830963723597380   -0.6777430547999990   -0.5249056185999990     647.3687493714000000   -0.0499577842926745    0.0105809929270276    0.0098074986429933   0.2783073106557350   0.2363745948763410   0.3651409430170520   0.2352726722178970   0.3644285793636080
    3.4500   -3.6277066630409100   1.7091533463068000   -0.1153095002369290   -3.0812723228210000   0.2721527018996230   12.2622872679614000   0.1138093728359380   0.2872831464759160   0.4818112288488650   -0.6744795861000000   -0.5221834947000000     649.2429599627210000   -0.0504351208391289    0.0099509567203803    0.0105967275396663   0.2774355386167250   0.2366062639571390   0.3646272099436730   0.2354981680592570   0.3639091442198430
    3.5000   -3.6241027595330600   1.7068896302063800   -0.1150655998490320   -3.0837379022608600   0.2730220393316240   12.2869304307334000   0.1142131825843250   0.2866175318576930   0.4795045955336720   -0.6706088458000000   -0.5189584981000000     650.5789050172750000   -0.0511323930115620    0.0095002019685569    0.0113041119513092   0.2766478019814670   0.2368116062356760   0.3641616992341420   0.2356843042745170   0.3634296322846280
    3.5500   -3.6163104660176900   1.7025161261697900   -0.1146275000077160   -3.0860197184411800   0.2739062261324590   12.3113704759823000   0.1152999269843750   0.2870082252570050   0.4779339786419450   -0.6678414353999990   -0.5177923317999990     652.8997830155450000   -0.0521391963337313    0.0090472893991184    0.0121347825965561   0.2759258936515800   0.2371181856661000   0.3638133213077920   0.2359586338627690   0.3630586394534740
    3.6000   -3.6294370953839600   1.7039748933938100   -0.1145489644764370   -3.0848588842224500   0.2741705402890330   12.3291522318662000   0.1161949029066410   0.2871923549829870   0.4754508883528980   -0.6638451939999990   -0.5156973059999990     655.6257190513270000   -0.0530889942554273    0.0085503262271407    0.0129818556290183   0.2754505463399360   0.2376110451106150   0.3637746723419980   0.2364209201753270   0.3629984228278660
    3.6500   -3.6398070159613500   1.7046933492046600   -0.1143849777289760   -3.0828262088610200   0.2741731455121400   12.3513324285860000   0.1163309565722810   0.2869330249178970   0.4728280294042070   -0.6604666062999990   -0.5133079230999990     657.0825003541730000   -0.0541710773858037    0.0083370892618399    0.0136170496294720   0.2750766360704100   0.2380243282340750   0.3637619228878910   0.2367922101146580   0.3629568934223420
    3.7000   -3.6395887911470000   1.7030251061095400   -0.1140696835778000   -3.0836554111858300   0.2745206938143270   12.3920643864520000   0.1160718941814010   0.2868582868429140   0.4699969891834830   -0.6574088805000000   -0.5107877986000000     658.0496802990510000   -0.0546741847971717    0.0082239123751642    0.0139391476612276   0.2746501438549180   0.2385104771907240   0.3637580916615800   0.2372590206485420   0.3629387612237530
    3.7500   -3.6309771661119300   1.7002003479837400   -0.1137843826080070   -3.0894434037039000   0.2755984596399180   12.4575592774033000   0.1166973977575770   0.2880756904718440   0.4683832755008790   -0.6554482667000000   -0.5105194583000000     660.2308962025870000   -0.0554043103534579    0.0082315474301870    0.0142400890710062   0.2741289927948430   0.2389534739385580   0.3636556989764090   0.2376705246794200   0.3628139784960140
    3.8000   -3.6194797776890200   1.6975437359166500   -0.1135591503573950   -3.0972264066892100   0.2768921788324940   12.5449243242103000   0.1170505140277170   0.2887850057656030   0.4663049467855850   -0.6536567831999990   -0.5097442550999990     661.3661616220240000   -0.0564672989397864    0.0084191058722405    0.0145092777440548   0.2735460915160870   0.2394305500341310   0.3635305385705220   0.2380980406543120   0.3626542997775560
    3.8500   -3.6130956417243300   1.6956310138924200   -0.1133754935092480   -3.1031925330156500   0.2781073301923520   12.6072931769766000   0.1169401321297110   0.2883565071782320   0.4636065713814360   -0.6509617202999990   -0.5075026407999990     661.2732994013040000   -0.0573048498723883    0.0085669226763562    0.0147282391013579   0.2728482861716130   0.2398030957284080   0.3632515822232780   0.2384303122117890   0.3623467966578400
    3.9000   -3.6031295334224800   1.6926136454554600   -0.1131239080604600   -3.1091302237360700   0.2794311398184560   12.6635418358058000   0.1157517058625920   0.2867774016974020   0.4603476865536750   -0.6482912807999990   -0.5036479243000000     659.0190974997260000   -0.0576624600627754    0.0087120922512229    0.0147595402177523   0.2722817162405370   0.2401343769638430   0.3630452478668540   0.2387424963582330   0.3621261003659540
    3.9500   -3.5949772898376500   1.6903972174033100   -0.1129736011693220   -3.1156131624749000   0.2809162927380170   12.7032462910866000   0.1139693948531360   0.2844592700397060   0.4571252960058710   -0.6459389818999990   -0.4991603513000000     655.2656273981580000   -0.0579235389970591    0.0089757033110975    0.0146423073173128   0.2718274147529450   0.2404352609708810   0.3629039241030680   0.2390266191637310   0.3619721924127780
    4.0000   -3.5879174492308300   1.6890940080884600   -0.1128841877050190   -3.1224660460706600   0.2822792909686670   12.7543160967594000   0.1119890714433590   0.2817361688665730   0.4534378584333170   -0.6428214690000000   -0.4939849281000000     651.4850608555060000   -0.0583039796710700    0.0092717900996661    0.0145442858770594   0.2713069027359440   0.2407315750359230   0.3627108031082710   0.2392999325376520   0.3617622053016260
    4.0500   -3.5763924962613400   1.6868729155391200   -0.1127051537283960   -3.1284169355448200   0.2833667271003130   12.8327594914900000   0.1093760327290160   0.2782208210569330   0.4487487914174280   -0.6387628990999990   -0.4868682186000000     646.5514392717970000   -0.0589747287756861    0.0095363361233895    0.0145877595061200   0.2708009977914730   0.2410222502114330   0.3625257308134700   0.2395545293854020   0.3615515909990150
    4.1000   -3.5561194903530500   1.6812639519926600   -0.1121968733619400   -3.1330851527850800   0.2842102431833120   12.9076823134005000   0.1068429936799920   0.2744623345913440   0.4443139014535170   -0.6341213025999990   -0.4792011548999990     641.7741873892240000   -0.0597318761436417    0.0093612874116877    0.0150663062681507   0.2701503077125350   0.2412005187083920   0.3621586378679580   0.2396997884895970   0.3611608746239480
    4.1500   -3.5273416368016100   1.6717431535203000   -0.1112766657830570   -3.1347846590446600   0.2845296085525980   12.9896658546113000   0.1047409997800340   0.2708479635648800   0.4406340493541170   -0.6293715573999990   -0.4715921344999990     637.4753839329910000   -0.0600114329207680    0.0087858188645830    0.0157281379784937   0.2694071209823180   0.2415055212727570   0.3618081171577100   0.2400015261676900   0.3608059442398380
    4.2000   -3.4997489393881800   1.6612888810586800   -0.1102197981862750   -3.1330578644597700   0.2844459672515930   13.0469337176974000   0.1021887609388550   0.2670435135800740   0.4363958154716190   -0.6247797752999990   -0.4632166587000000     632.1805408261490000   -0.0603042109623019    0.0082529915238830    0.0163537926744507   0.2687544160784520   0.2418570502226690   0.3615574213096450   0.2403478757280660   0.3605496325454560
    4.2500   -3.4762523583709900   1.6517781498518600   -0.1092004902462710   -3.1295775096383100   0.2841087001940020   13.0946495078661000   0.0986921812472591   0.2624290493831400   0.4310497832921590   -0.6197863481000000   -0.4532882346000000     625.4956865090110000   -0.0610373573768943    0.0079986146894705    0.0168912066573881   0.2682015065286460   0.2420118826069770   0.3612503279267840   0.2404712011526090   0.3602199976236940
    4.3000   -3.4502259020163100   1.6417200086958400   -0.1081358423258660   -3.1265899446090600   0.2837061759044820   13.1774183714319000   0.0963275944264813   0.2592672839513340   0.4266815951068400   -0.6154905803000000   -0.4461362407000000     621.1206364187030000   -0.0615558132484956    0.0076499688237860    0.0174239878413989   0.2677449904034640   0.2420033090952430   0.3609057792543630   0.2404420019656070   0.3598607177720010
    4.3500   -3.4287606826597200   1.6331411187752600   -0.1072387615407900   -3.1245666107939400   0.2835171543271600   13.2534853841167000   0.0952601640698257   0.2573653050951480   0.4236644017539740   -0.6116777969000000   -0.4416495523999990     619.1938774723370000   -0.0620139065583387    0.0071835775899538    0.0180289535196418   0.2674890156803570   0.2418008761126110   0.3605801397726620   0.2402223676327360   0.3595235172012030
    4.4000   -3.4099449956929400   1.6254315569198900   -0.1064515943128010   -3.1229808572897600   0.2835008407896920   13.3251847656925000   0.0946785046599911   0.2562747101133940   0.4211033252084670   -0.6085565432999990   -0.4383699332000000     618.5642216339610000   -0.0622744971920122    0.0066728748651420    0.0185947018265968   0.2673344496700070   0.2416172893480460   0.3603423684390490   0.2400314702097480   0.3592809689802940
    4.4500   -3.4025087779085600   1.6215057124185900   -0.1059929760691070   -3.1217069816775200   0.2835989918763170   13.3938682591046000   0.0941887155321925   0.2552914230190840   0.4190168573291120   -0.6060118415999990   -0.4354505638000000     617.9165823293280000   -0.0627595139543909    0.0064342776331154    0.0190103057077894   0.2671464877881230   0.2417645808059260   0.3603017602923900   0.2401575322362350   0.3592253975254650
    4.5000   -3.3895810665514200   1.6168875161973000   -0.1055329740793120   -3.1226515068175700   0.2839737241859400   13.4803817912132000   0.0933517995954791   0.2539021166828280   0.4169151258660170   -0.6039710970999990   -0.4323361218999990     616.4154866619000000   -0.0634482566891636    0.0062871768529931    0.0194178289529625   0.2669272054584230   0.2417001613779250   0.3600959608548230   0.2400592172069230   0.3589966027413690
    4.5500   -3.3776358383447200   1.6133801470294400   -0.1052083549932070   -3.1255872747219300   0.2845760782909470   13.5671038138584000   0.0931392745768584   0.2529077620426640   0.4148793179070950   -0.6010468936999990   -0.4299401004999990     616.3523113461810000   -0.0635594443036693    0.0060265115317206    0.0197048130246428   0.2667622556651420   0.2417830076672180   0.3600293374770460   0.2401390301011270   0.3589273670611710
    4.6000   -3.3583685604454700   1.6083133769633200   -0.1047998760550490   -3.1300558401377100   0.2853083435964600   13.6736801584162000   0.0933685917047156   0.2521142005656470   0.4131192730965250   -0.5978044474999990   -0.4284097329000000     617.3414269325120000   -0.0636943547733742    0.0058339718548543    0.0199354759276697   0.2667099998649620   0.2418214152269550   0.3600164175288890   0.2401724232981810   0.3589108760415160
    4.6500   -3.3353714534085900   1.6020465369116700   -0.1042595150217670   -3.1340024245327900   0.2858501575677530   13.7902278657002000   0.0933366829798015   0.2509022073408270   0.4114378306377720   -0.5946546499000000   -0.4266634763000000     617.8788764453490000   -0.0641823557876443    0.0056401258444839    0.0203006243728890   0.2666786578690740   0.2416804899452340   0.3598985492927410   0.2400078954556290   0.3587775027003380
    4.7000   -3.3136458725658400   1.5949896014911400   -0.1036071318103250   -3.1358357904955600   0.2862177752772650   13.8958162399292000   0.0935174813792536   0.2499517181830500   0.4098611367770830   -0.5912620718000000   -0.4248152620000000     618.9635527893230000   -0.0645508342563654    0.0052024116022524    0.0208425501765327   0.2666008326333710   0.2416015149515580   0.3597878485825980   0.2399140758974160   0.3586568942241570
    4.7500   -3.3040382803331300   1.5902952660680300   -0.1030897885409430   -3.1342719172116100   0.2862092016309030   13.9608903577739000   0.0940578382876926   0.2492661566939460   0.4085820810084910   -0.5877118318999990   -0.4235087370999990     620.7913594409760000   -0.0647125764875177    0.0046796353471656    0.0213839057860384   0.2664860057614640   0.2416032613800280   0.3597039437873390   0.2399123309200220   0.3585703805310430
    4.8000   -3.2985848008716000   1.5861658982493000   -0.1026128924417360   -3.1309075682198600   0.2860592507098490   14.0001074844774000   0.0946579757261142   0.2488444805605260   0.4074648550053150   -0.5843745594000000   -0.4226452995000000     622.8605274135900000   -0.0645293053906076    0.0040890687329170    0.0218546991189860   0.2662906130884650   0.2416563982251820   0.3595949185155410   0.2399800143060320   0.3584704979288450
    4.8500   -3.2912524253793000   1.5823870637517600   -0.1021961369881520   -3.1290767836617900   0.2860837071713050   14.0515032314706000   0.0942725997279069   0.2476118482015540   0.4052024331089910   -0.5811900698999990   -0.4203603347000000     623.1374729120940000   -0.0648956836388662    0.0037688276259579    0.0222970136011541   0.2660523341853650   0.2416952580059350   0.3594446303229980   0.2400025117504130   0.3583085962853650
    4.9000   -3.2778155991912900   1.5768213355936100   -0.1017002840521340   -3.1295883701046900   0.2866265182215610   14.0914544600839000   0.0940177514688705   0.2467224202436840   0.4032268971460470   -0.5781619319999990   -0.4182912788000000     623.6375950679980000   -0.0649960409009332    0.0033079089281241    0.0227652721377919   0.2657564170316800   0.2417818894612210   0.3592839479644960   0.2400874427292060   0.3581458548549270
    4.9500   -3.2623270557302000   1.5714328821771800   -0.1012730276488560   -3.1321096015376000   0.2874184966285310   14.1473699300253000   0.0941279133445034   0.2463603081259310   0.4018059943434030   -0.5754450436000000   -0.4172301808000000     625.2114404563630000   -0.0650582976591393    0.0028169776659075    0.0232404977980566   0.2655209753509390   0.2417290569877320   0.3590742615998700   0.2400341922592250   0.3579354715655500
    5.0000   -3.2439985256051400   1.5653265813956600   -0.1007886129953940   -3.1355200449549300   0.2882419130281860   14.2096496423898000   0.0948155233149771   0.2464326163732380   0.4011116985104400   -0.5728648174999990   -0.4174608377999990     628.3041805684500000   -0.0651236304389814    0.0023153550799713    0.0237252376076290   0.2652891982301680   0.2416039640554840   0.3588186647109220   0.2399080230651890   0.3576789317651410
    5.0500   -3.2282739005574300   1.5596017026901200   -0.1003036350800760   -3.1378497708586000   0.2888721530190230   14.2619350839515000   0.0956601979536418   0.2467614631076690   0.4008522925389180   -0.5709478474000000   -0.4184400965000000     631.5172437869110000   -0.0655326653268977    0.0018006423569846    0.0243539408893790   0.2650786899267430   0.2413717952384460   0.3585067020153300   0.2396561255178630   0.3573538447414720
    5.1000   -3.2243867658544500   1.5564857507480300   -0.0999973141309094   -3.1373768450412900   0.2893046584289170   14.2828657424767000   0.0959995601802299   0.2466173695172760   0.3998498283188500   -0.5689155438999990   -0.4183917452000000     633.7247268828800000   -0.0662381016935691    0.0014981032701411    0.0249075536620884   0.2648545317494420   0.2411591661117440   0.3581978034381680   0.2394070376963200   0.3570205213804160
    5.1500   -3.2197529771889400   1.5538652975065300   -0.0998024263249297   -3.1393964441396100   0.2902159459845580   14.2915464222760000   0.0958182442165905   0.2457954168725990   0.3985704440733490   -0.5671424742999990   -0.4172787523000000     634.6240448717440000   -0.0671043271763086    0.0012618864667904    0.0254643241702727   0.2646469579064980   0.2409911469462960   0.3579312018190860   0.2391933333307340   0.3567232303047160
    5.2000   -3.2132897022320300   1.5516268742438700   -0.0996560108150757   -3.1427047265828000   0.2911988199806440   14.3288036344835000   0.0949993900100206   0.2444231084749430   0.3969117412299660   -0.5659360173999990   -0.4151676042999990     633.5605727404300000   -0.0679646599269399    0.0010639193511127    0.0259813249369123   0.2644878934360930   0.2407637124710500   0.3576604689046650   0.2389191814971840   0.3564214093759620
    5.2500   -3.2105955251598300   1.5493587777629000   -0.0994143867262801   -3.1421662528589800   0.2915490164532590   14.3574319745975000   0.0942909714320525   0.2432835948009390   0.3954834365715990   -0.5648892647000000   -0.4132132498000000     632.5899526168110000   -0.0684963742868849    0.0007791251847975    0.0264490262059553   0.2642518012521900   0.2405270911019320   0.3573265957341880   0.2386529852207160   0.3560677770028990
    5.3000   -3.2116867974075900   1.5482637512174400   -0.0992279515533292   -3.1405355109664400   0.2916128989645920   14.3929299328201000   0.0933739081378622   0.2419860072779600   0.3936817002125770   -0.5635265131000000   -0.4110132148999990     631.5136083764000000   -0.0689657476150063    0.0005828026467224    0.0268109377332723   0.2640491638926570   0.2402626993224960   0.3569987754014620   0.2383616613676300   0.3557221423560080
    5.3500   -3.2079731775380400   1.5448291037931200   -0.0988042628679176   -3.1377813906208300   0.2915409442241180   14.4155139528731000   0.0922698362660153   0.2403379702926210   0.3914979683399740   -0.5619057016999990   -0.4084197094000000     630.0930835761020000   -0.0694492227452831    0.0003919239209605    0.0271750721642349   0.2638191231085430   0.2400224090692160   0.3566669126414040   0.2380934049629340   0.3553716353405320
    5.4000   -3.2053662501303800   1.5415301734488600   -0.0983807892639273   -3.1345918981813300   0.2914055722005640   14.4339032319724000   0.0911374092195443   0.2389790396485020   0.3890972419850660   -0.5606843759999990   -0.4060920161000000     629.0014032056830000   -0.0696172801153399    0.0001219802672842    0.0274842445875781   0.2635750375895830   0.2396598859162980   0.3562424193688380   0.2377196640449070   0.3549400500281330
    5.4500   -3.2029513273544700   1.5387794452650400   -0.0980240149452041   -3.1322933778421600   0.2913307040042670   14.4567486187329000   0.0902770826099558   0.2377860433505480   0.3869686323326010   -0.5593609609999990   -0.4042931658000000     628.3306751930820000   -0.0694923969782872   -0.0001499966190543    0.0276812670123620   0.2632159641115550   0.2392420768049480   0.3556956776193940   0.2373065245847740   0.3543967132659090
    5.5000   -3.1989636471634800   1.5347501425112200   -0.0975405931058534   -3.1283403688001100   0.2910930301029990   14.4568145755095000   0.0893603709350110   0.2364649959981720   0.3851687055521690   -0.5583903971000000   -0.4023532407000000     627.0783562307410000   -0.0691237431311060   -0.0004757851618760    0.0278347570370638   0.2628058552846030   0.2389292563955870   0.3551817944850490   0.2370119514496280   0.3538948752127280
    5.5500   -3.1950105087392900   1.5308245655716300   -0.0970575480199530   -3.1242334871549400   0.2907680656479580   14.4635190920190000   0.0886835848408367   0.2355521603251950   0.3839920602463750   -0.5578242222000000   -0.4008960631999990     626.0464081853520000   -0.0681494055112516   -0.0009630556306487    0.0278970538026898   0.2624723883441180   0.2387362223739020   0.3548052120761850   0.2368703541741710   0.3535524279787410
    5.6000   -3.1900925489860500   1.5272748979939200   -0.0966305804087904   -3.1220956655624200   0.2906385657325940   14.4864287021763000   0.0884179759771170   0.2351561548533780   0.3831344563265770   -0.5573631157999990   -0.4002363782999990     625.8790007761530000   -0.0671829644747062   -0.0014731146899863    0.0279819736368724   0.2621521035872130   0.2385267150912190   0.3544273116273760   0.2367106715644230   0.3532076831662660
    5.6500   -3.1856900060357400   1.5236746577486200   -0.0961539938748360   -3.1190430811114500   0.2903035648334450   14.5212580531558000   0.0881889021236413   0.2345070476549380   0.3822540529573870   -0.5565331741000000   -0.3989210615000000     625.7228666867760000   -0.0663375595448081   -0.0021520997005004    0.0282694168221828   0.2617773726066300   0.2383857827206830   0.3540553264846940   0.2366116128797560   0.3528632145157530
    5.7000   -3.1858827866874900   1.5205194689806500   -0.0956456900557979   -3.1130381577607400   0.2895801493561650   14.5413292106297000   0.0874829929973436   0.2335817581143130   0.3810294359733970   -0.5560679938000000   -0.3965102751999990     624.1796810279920000   -0.0654028525131894   -0.0030084743577346    0.0286812797965481   0.2615005514221850   0.2383174327799060   0.3538046596088190   0.2365882440757950   0.3526422204288290
    5.7500   -3.1894148797533800   1.5180884928416400   -0.0951581700138832   -3.1049878911882200   0.2885181795227110   14.5592181875611000   0.0864371640644693   0.2324334147732050   0.3794136059453050   -0.5556492725999990   -0.3933479033999990     621.8628599920730000   -0.0644108886991400   -0.0037635963887194    0.0289783545749076   0.2612410352504270   0.2382638393071830   0.3535767747181850   0.2365814348395600   0.3524452493784820
    5.8000   -3.1890593549484300   1.5143458042825600   -0.0946106320024290   -3.0974694245743100   0.2876802804704590   14.5650814324923000   0.0856545717370460   0.2313381581324660   0.3777303310075440   -0.5546915571999990   -0.3903540170000000     620.3115856739680000   -0.0631208102145129   -0.0045022750470802    0.0291451087330944   0.2609292696903440   0.2382631112229630   0.3533459975021400   0.2366413740055110   0.3522544870861870
    5.8500   -3.1942531340577400   1.5121385575699600   -0.0941678554032399   -3.0884807371962400   0.2866139742504110   14.5677314812644000   0.0848964442816940   0.2300687980498170   0.3761438036226250   -0.5534541892000000   -0.3871408779000000     618.8360114641680000   -0.0620225405960140   -0.0051283270580084    0.0292869027991912   0.2605716067777900   0.2383205225657960   0.3531207070320200   0.2367487577545390   0.3520618362689670
    5.9000   -3.2037411677044200   1.5111114042845300   -0.0938086229062630   -3.0788323585361700   0.2854589179592360   14.5539728343649000   0.0839132221251302   0.2284080273116670   0.3746481334166860   -0.5521800067000000   -0.3835621270999990     616.9807148113480000   -0.0610331801953842   -0.0057019625609670    0.0294239815001999   0.2601879002059180   0.2383620508507730   0.3528657119916180   0.2368335504833190   0.3518350096964480
    5.9500   -3.2174377130075700   1.5111597362283400   -0.0935145920318069   -3.0678865947739800   0.2840863882499950   14.5510702433625000   0.0832105293392729   0.2271189127185690   0.3734728471876180   -0.5509146852999990   -0.3806635444000000     615.8522422150520000   -0.0598038363695909   -0.0062854956876494    0.0294748748073226   0.2598512413859530   0.2384684807103000   0.3526895007539910   0.2369936623410520   0.3516939914750400
    6.0000   -3.2419247458565700   1.5145128742596000   -0.0934857600323525   -3.0560261890040100   0.2826337771642030   14.5402347437832000   0.0822380513421082   0.2256878218919090   0.3721280121803770   -0.5498507915000000   -0.3772586841000000     614.0194500245730000   -0.0587964280772829   -0.0069458352704829    0.0296816489727441   0.2595209164493320   0.2385909250947910   0.3525290564085320   0.2371566321648210   0.3515599155968460
    6.0500   -3.2655990161226100   1.5174916555290800   -0.0934568091848268   -3.0444737680810300   0.2813549059228580   14.5069869553652000   0.0811310442622008   0.2240238698841760   0.3705161517087050   -0.5482883482999990   -0.3733244374999990     612.0234158541060000   -0.0575598348030240   -0.0077314448709843    0.0299099114610000   0.2591904789061310   0.2387216344335230   0.3523744075584980   0.2373349980758590   0.3514364888102230
    6.1000   -3.2885662925423500   1.5199955575209900   -0.0934044021221018   -3.0328045202890100   0.2801173771173870   14.4627849927047000   0.0804925796448099   0.2228565274959540   0.3693387961779840   -0.5468373896999990   -0.3703883190000000     611.1717144475960000   -0.0562202357666007   -0.0085460323465339    0.0301255368161442   0.2587601465019750   0.2389504424188960   0.3522131845202700   0.2376135881808930   0.3513076012640810
    6.1500   -3.3097319296510200   1.5226380136710300   -0.0934046811091601   -3.0228161966456500   0.2790767408416400   14.4299421085976000   0.0800507010547783   0.2218895550429010   0.3683033338838440   -0.5451023234999990   -0.3679156641999990     610.9853810017370000   -0.0549742752724836   -0.0093137613572728    0.0303350180632041   0.2583125222040380   0.2391939537803520   0.3520498638722780   0.2379009192550350   0.3511726164008270
    6.2000   -3.3299300216500500   1.5250645854976500   -0.0934124801801045   -3.0135939937824300   0.2781958347252780   14.3879798371209000   0.0796499818648570   0.2209483696597820   0.3670690067416890   -0.5432056664000000   -0.3656366285000000     610.8120932376750000   -0.0539244440291702   -0.0100198690672309    0.0305654406323642   0.2578875862264610   0.2394734967571550   0.3519283489274620   0.2382147036818410   0.3510730012119100
    6.2500   -3.3467591414732700   1.5270214734671800   -0.0934137190868939   -3.0061025209220200   0.2775339834818370   14.3589807534243000   0.0792927414762180   0.2200683415909280   0.3658199320118480   -0.5413255798999990   -0.3635192739000000     610.8270294748250000   -0.0530120544395716   -0.0107388276936095    0.0308603790289140   0.2574904296815300   0.2397468964939710   0.3518236713981450   0.2385144998176210   0.3509850253227750
    6.3000   -3.3632909282283800   1.5290157658892100   -0.0934460012820791   -2.9997267442064400   0.2770858415566860   14.3212372377562000   0.0791980610201015   0.2195071839950960   0.3646355817479010   -0.5392793299999990   -0.3618429435000000     611.7836025173550000   -0.0522314171325397   -0.0115278813834657    0.0312691394885233   0.2571180937124890   0.2400061487070890   0.3517281130810470   0.2387915949032420   0.3509004700919880
    6.3500   -3.3807920526636600   1.5316785989131400   -0.0935349429951796   -2.9942237873841300   0.2767051363803960   14.2925468825101000   0.0791097208165179   0.2189788809177080   0.3633124493032520   -0.5372452050999990   -0.3600581252000000     612.6442870780060000   -0.0515920917600824   -0.0123122175435842    0.0317286119881181   0.2567827277439290   0.2402929618759460   0.3516789399363110   0.2390890184196370   0.3508574183289820
    6.4000   -3.3998323040437600   1.5352129098970200   -0.0936982109118404   -2.9889787756498800   0.2762947014267130   14.2743478259463000   0.0789882537427104   0.2182472859348010   0.3615760935385190   -0.5345190881000000   -0.3578586651999990     613.5557514862320000   -0.0511099413496138   -0.0130453479085998    0.0322028289600271   0.2564919091260770   0.2405677433317490   0.3516545728109760   0.2393677194691160   0.3508347254349050
    6.4500   -3.4185821642075300   1.5379514844183500   -0.0937760410466936   -2.9823057705714900   0.2757544563638640   14.2401220472086000   0.0789480987443637   0.2175660774921150   0.3600749944211340   -0.5318259693000000   -0.3557117541000000     614.8543910318100000   -0.0505549767208206   -0.0137687962281953    0.0326378907380210   0.2562581598528310   0.2408324640042190   0.3516653525860950   0.2396384797827280   0.3508487501527340
    6.5000   -3.4386033491058300   1.5404150566950000   -0.0938129105012666   -2.9747808766417000   0.2751653454121600   14.1773482194363000   0.0789616498903425   0.2168635523896860   0.3589230179539440   -0.5292418432000000   -0.3537367120000000     616.3638428044000000   -0.0501320588936134   -0.0144682423909281    0.0331040779927825   0.2560141278719230   0.2411623777567160   0.3517136990725470   0.2399689012296320   0.3508964337626960
    6.5500   -3.4557420138509500   1.5423043325630000   -0.0938534293106528   -2.9691567740336800   0.2749097428020160   14.1064997335119000   0.0795008919237397   0.2165380252405830   0.3583740584197020   -0.5267151286000000   -0.3526545074999990     618.8954753562340000   -0.0495990533599464   -0.0152256955116513    0.0335776925252220   0.2557540384449370   0.2414857687601770   0.3517463641526210   0.2402946917213660   0.3509297181065760
    6.6000   -3.4694324469105000   1.5436534790582600   -0.0939060140555328   -2.9656960750539300   0.2750276536154800   14.0323428021090000   0.0801178683804650   0.2162797034235540   0.3580126170616010   -0.5243902594999990   -0.3515816135000000     621.2750934157910000   -0.0488580124449277   -0.0160380091363728    0.0340173722312017   0.2554575614059280   0.2419023039329310   0.3518171262567010   0.2407191867761410   0.3510046902273960
    6.6500   -3.4824267861218300   1.5446067819424700   -0.0939318167523680   -2.9616727104499300   0.2751044789402470   13.9518920031360000   0.0806078960435990   0.2161000708861830   0.3573620024221310   -0.5223502607000000   -0.3504987089000000     623.3292168195520000   -0.0479324153832062   -0.0168932546304068    0.0344207412621164   0.2551675941637210   0.2423097545772330   0.3518870817102840   0.2411389179141060   0.3510818691474790
    6.7000   -3.4959527235074300   1.5459817465347800   -0.0939965864810687   -2.9578211726514200   0.2751893349225890   13.8748592193211000   0.0804979969215772   0.2154531699439090   0.3562724228694260   -0.5207794963999990   -0.3487072198999990     623.7335298393420000   -0.0473062396225382   -0.0176218669083748    0.0348292527118081   0.2548735406215430   0.2427132296040500   0.3519520330013590   0.2415455524746930   0.3511477974148010
    6.7500   -3.5101774729237200   1.5477084019897600   -0.0940819209058811   -2.9536080061923200   0.2751719775614670   13.8052001379073000   0.0800714736356568   0.2146490040522440   0.3550852259878640   -0.5194563148999990   -0.3466574834000000     623.4945337565550000   -0.0467939651013536   -0.0182672216646854    0.0352077821906936   0.2546023417619600   0.2430999711658520   0.3520226532646900   0.2419327944706540   0.3512176383256590
    6.8000   -3.5257512984887000   1.5499669457751500   -0.0942090560376975   -2.9492823672534200   0.2751208903211000   13.7373947287364000   0.0794596799261875   0.2136913072784590   0.3537961950852710   -0.5181650002000000   -0.3444386314999990     623.0250957366770000   -0.0464763769738426   -0.0188325867283277    0.0355916609024285   0.2543234274928250   0.2434144892384370   0.3520383776563390   0.2422421750578490   0.3512288102483280
    6.8500   -3.5389657104198300   1.5516157732881400   -0.0942950491862606   -2.9458258252539600   0.2751868390303160   13.6760567785394000   0.0790131894515289   0.2129018538996340   0.3528051687311170   -0.5167241235999990   -0.3426130897999990     623.0813854294920000   -0.0463371476734471   -0.0193658239667276    0.0360176795352135   0.2540200942127030   0.2436959396914700   0.3520140896128160   0.2425123790224790   0.3511957605709010
    6.9000   -3.5506909760191700   1.5526602134815100   -0.0943304028901626   -2.9421247932955700   0.2752209545293070   13.6093211488898000   0.0786608676481704   0.2121795819441540   0.3518889574742230   -0.5153224914999990   -0.3409640199999990     623.2676857990690000   -0.0459961738575260   -0.0199803111683018    0.0364344976742414   0.2537856583934640   0.2439560722283800   0.3520251774849390   0.2427651386674630   0.3512009011355710
    6.9500   -3.5583126163775600   1.5526020149057900   -0.0942921123144045   -2.9389914328319900   0.2753515691454700   13.5428100674333000   0.0779421344638988   0.2110915623369280   0.3506842291517330   -0.5139248468999990   -0.3387569294000000     622.5609761540790000   -0.0457433233519513   -0.0205237734115054    0.0368221334566488   0.2536231934693360   0.2441748357317970   0.3520597600837600   0.2429750856894500   0.3512287239554590
    7.0000   -3.5680878816507200   1.5530133533842900   -0.0943017174593706   -2.9359832599815000   0.2755758818605460   13.4543805926777000   0.0771913361894477   0.2100365582608000   0.3495568458876840   -0.5127062714000000   -0.3366979507999990     621.8012625888410000   -0.0454864266269576   -0.0210057998307541    0.0371526784598116   0.2534691522210610   0.2443341397733820   0.3520593458303050   0.2431271874034460   0.3512227788489440
    7.0500   -3.5776626143312700   1.5533773732685500   -0.0943088609938562   -2.9329365972762200   0.2757690956839950   13.3708799811601000   0.0766655570019520   0.2092642206413030   0.3487929459992940   -0.5118054421000000   -0.3352774590000000     621.4061884641140000   -0.0452811490760098   -0.0214473671731834    0.0374671382968386   0.2533334126617360   0.2444291921463250   0.3520276238367480   0.2432140216482590   0.3511849630852110
    7.1000   -3.5870840525757700   1.5534892326707400   -0.0942901081580729   -2.9295695393616800   0.2759243921106440   13.2841516495929000   0.0762895235964840   0.2087795609276780   0.3481997595072660   -0.5111244534000000   -0.3342396328000000     621.3717593475940000   -0.0449178748046230   -0.0219472469582805    0.0377703840202655   0.2531253340632130   0.2445036229963120   0.3519296185360590   0.2432831684651350   0.3510828033428140
    7.1500   -3.5966734929365700   1.5537344428300800   -0.0942837674803623   -2.9260712264005500   0.2760434616291950   13.2020011142699000   0.0758866745225048   0.2084340681160120   0.3474959461310720   -0.5105446777000000   -0.3333214525000000     621.3783098397140000   -0.0442812152671488   -0.0225189147770388    0.0380279239857717   0.2528767229584040   0.2446042158565190   0.3518207774264680   0.2433847331763650   0.3509740237645980
    7.2000   -3.6061463737299900   1.5537330422971500   -0.0942595851547546   -2.9221647938300800   0.2761302865845200   13.1176720782210000   0.0758008933752110   0.2084372130762530   0.3468621750293620   -0.5098388379999990   -0.3328764595000000     622.1696511469980000   -0.0433772768079544   -0.0231507616081037    0.0382316803998185   0.2525829688825620   0.2447388338096720   0.3517033593016250   0.2435264622096880   0.3508607900092230
    7.2500   -3.6172285212370500   1.5539361153787500   -0.0942385965208437   -2.9173226222011400   0.2760859173465040   13.0279966221217000   0.0759354740654476   0.2087231082859950   0.3464118477648670   -0.5091423652000000   -0.3327886581999990     623.3541529341820000   -0.0425870151438778   -0.0238345696141753    0.0385263048756532   0.2522624999542180   0.2448122125186750   0.3515243779333680   0.2435993163084350   0.3506807605060320
    7.3000   -3.6248748146630100   1.5535920848078700   -0.0941807289019533   -2.9131826773407100   0.2760427728078190   12.9532362568259000   0.0758998258751642   0.2088417081220880   0.3457202369960790   -0.5082490224000000   -0.3323837189000000     624.3220993799350000   -0.0420499660782547   -0.0244853674778749    0.0388924029700660   0.2519739078616070   0.2448103793776000   0.3513160572676040   0.2435882780027160   0.3504655466994990
    7.3500   -3.6281669405652400   1.5523262979371500   -0.0940630335111695   -2.9102665861346000   0.2761137841299470   12.8877386078849000   0.0758424786074885   0.2089550398046970   0.3450958714357770   -0.5074440477999990   -0.3320487601000000     625.1833386417060000   -0.0416064468616196   -0.0250818510695210    0.0392472748097628   0.2517096047610700   0.2447519517379790   0.3510858057633840   0.2435187162189930   0.3502271980842160
    7.4000   -3.6319706880591800   1.5511886699821700   -0.0939601105141832   -2.9073390452757000   0.2761849671450480   12.8172679266047000   0.0760105700355551   0.2092355273234630   0.3448193153364580   -0.5067368934000000   -0.3321189018999990     626.3796940178580000   -0.0412164446505129   -0.0257239869171289    0.0396642065699308   0.2513997263890890   0.2446870478399860   0.3508184342493370   0.2434380849145930   0.3499484585126690
    7.4500   -3.6361300810207100   1.5498775696587200   -0.0938399735985484   -2.9041987041568400   0.2762649513311770   12.7416534271347000   0.0764459303547961   0.2097004047591710   0.3448892107568300   -0.5062637049999990   -0.3327043315000000     627.9013104260920000   -0.0407394622046592   -0.0263742527017879    0.0400532195685200   0.2510204999239190   0.2446311629080130   0.3505077705954340   0.2433681435770070   0.3496274369813860
    7.5000   -3.6413102012857400   1.5491171333455400   -0.0937587514789519   -2.9012215966351200   0.2763076044049140   12.6707894567118000   0.0767901211095901   0.2100833062723400   0.3447637097493530   -0.5057133681000000   -0.3330766739000000     629.5302123329360000   -0.0403836189997975   -0.0269838470708551    0.0404542322963499   0.2506287256684990   0.2446426897946910   0.3502353548689760   0.2433630058366370   0.3493426838221440
    7.5500   -3.6429284488614700   1.5472340860723100   -0.0936049990730804   -2.8990387013541600   0.2765143775157050   12.5937415785727000   0.0772302595078572   0.2105468594982560   0.3446852640912320   -0.5050045587000000   -0.3335881740000000     631.5814772452800000   -0.0400887665235369   -0.0275760060943420    0.0408637468976041   0.2502417920607550   0.2446381447032350   0.3499553919253410   0.2433398507511170   0.3490490473520220
    7.6000   -3.6415774515680200   1.5444666963072000   -0.0934015236998554   -2.8976316342720700   0.2768826720989120   12.5104123820138000   0.0776774162094018   0.2110915921486320   0.3447759010623180   -0.5046486038999990   -0.3342638751000000     633.3374527663820000   -0.0397017752646344   -0.0281806885544526    0.0412467532813875   0.2498600992629510   0.2446521684639170   0.3496923687154020   0.2433368717663730   0.3487734255426240
    7.6500   -3.6419947524743700   1.5420399062619400   -0.0932029831167112   -2.8954361095929200   0.2771096841121290   12.4234165801757000   0.0780495846720744   0.2115340908470660   0.3450116058977520   -0.5046280654999990   -0.3349774332000000     634.6225604383500000   -0.0393592770614283   -0.0287377246520244    0.0416050433569227   0.2494851374746500   0.2446490685073900   0.3494223812841400   0.2433169638991360   0.3484910023829000
    7.7000   -3.6422330835287400   1.5395934346237800   -0.0929910677802022   -2.8929903074804600   0.2772576228028580   12.3371230813471000   0.0781866605569601   0.2117340205515500   0.3449858720910160   -0.5045600138000000   -0.3353413833000000     635.3326384740720000   -0.0390425778722102   -0.0292489804183969    0.0419328743898065   0.2490845698180290   0.2446240654285120   0.3491189715672950   0.2432759856711960   0.3481757144398850
    7.7500   -3.6438986966077400   1.5376127594417200   -0.0928073232238873   -2.8903915354311800   0.2773744123432290   12.2538015453754000   0.0780306639892205   0.2117081512529020   0.3449503371750670   -0.5047459638000000   -0.3352890304999990     635.2447512488620000   -0.0387947121974889   -0.0296794196962101    0.0422160614005424   0.2486515686145710   0.2446319038622260   0.3488156690312870   0.2432695703877130   0.3478615909397150
    7.8000   -3.6481827775040700   1.5360905094615900   -0.0926483895423095   -2.8868680521687300   0.2774129682750040   12.1597681505752000   0.0777181274604425   0.2115007549417290   0.3446709196698810   -0.5046993069999990   -0.3349040955000000     634.9864986148490000   -0.0386004400689410   -0.0300376930883555    0.0424564367208103   0.2482160154995170   0.2446364579965110   0.3485085177316830   0.2432616333538440   0.3475448353988830
    7.8500   -3.6557757008641600   1.5353916708912900   -0.0925383104555221   -2.8821392173347900   0.2772523800622020   12.0649424725226000   0.0775105083361645   0.2113933913355480   0.3445263646282500   -0.5047488967000000   -0.3346783183000000     634.9128246717880000   -0.0385699920096820   -0.0304157590366838    0.0427802888576706   0.2477921136332770   0.2446271259074000   0.3482001756297120   0.2432334601924370   0.3472224758509060
    7.9000   -3.6632034711458500   1.5347254368934500   -0.0924419695549686   -2.8776425917172200   0.2771375218202710   11.9705526038295000   0.0774081926186780   0.2113776158573320   0.3443588789734790   -0.5045465473000000   -0.3345664011000000     635.2152488962340000   -0.0384687670241981   -0.0308083933707864    0.0430883058191606   0.2473742410582440   0.2446420474947590   0.3479134181683280   0.2432308432286430   0.3469225536583910
    7.9500   -3.6698673383425100   1.5334963335897000   -0.0922844516618693   -2.8725052041219200   0.2769343738926880   11.8744592291191000   0.0775371791020591   0.2114756524010330   0.3444688068872050   -0.5042771957999990   -0.3347089710000000     635.8778131908850000   -0.0382191565454950   -0.0312503847090833    0.0433799675986459   0.2469849106291560   0.2446614408897570   0.3476503512679720   0.2432342919113000   0.3466474676672020
    8.0000   -3.6774135888345300   1.5324789148036200   -0.0921425322846974   -2.8672333084406100   0.2767245242664380   11.7729031662293000   0.0775754969696484   0.2113970628998590   0.3445112567971350   -0.5038173933999990   -0.3346483116000000     636.3811194857250000   -0.0379819332375004   -0.0316473692599367    0.0436365679050326   0.2466042993802610   0.2446590026396580   0.3473783356017780   0.2432175434245510   0.3463646256509210
    8.0500   -3.6855554352083100   1.5319649244927000   -0.0920540429911193   -2.8626463464767000   0.2765899427582910   11.6756720656539000   0.0777005465795477   0.2112470118926920   0.3446655193972410   -0.5033389660000000   -0.3345851240000000     636.8833368798910000   -0.0377010876025194   -0.0319968014186889    0.0438329432059766   0.2462264167786110   0.2447118825705460   0.3471474525196100   0.2432600910050690   0.3461255844277170
    8.1000   -3.6927334987750700   1.5313124580586700   -0.0919578455452731   -2.8584185594468000   0.2764841551697780   11.5829624417984000   0.0778035309165148   0.2110723995451890   0.3448255874466090   -0.5028718873999990   -0.3344576074999990     637.5257849777690000   -0.0372734534812204   -0.0323857426199125    0.0440046638370211   0.2458710609512640   0.2447826622586140   0.3469454284980780   0.2433224686200130   0.3459167563860380
    8.1500   -3.7002705667796600   1.5311286258476400   -0.0919109341906256   -2.8548290584590400   0.2764260281569180   11.5009269270098000   0.0779567027719054   0.2110965360497900   0.3449256407390320   -0.5025462993999990   -0.3345776003000000     638.2418504853890000   -0.0367761351456194   -0.0328272685014731    0.0441942872226325   0.2454902370351010   0.2448753979242120   0.3467411382978540   0.2434055742204370   0.3457046861717830
    8.2000   -3.7091010038253400   1.5312673481383500   -0.0918862965054688   -2.8508482234771100   0.2763244992585120   11.4148390380886000   0.0779760537966993   0.2109360050660020   0.3448192958920780   -0.5021499412000000   -0.3343735365000000     638.5630011453020000   -0.0363681846938404   -0.0332557896552353    0.0444082659361202   0.2451524227069410   0.2449718113488760   0.3465701930556520   0.2434900800102650   0.3455244266654500
    8.2500   -3.7140482820193100   1.5305317115142300   -0.0918158149482178   -2.8480698950144800   0.2763999431000050   11.3343110744414000   0.0780064504187941   0.2107368894432190   0.3445666552133530   -0.5015042252000000   -0.3340769270000000     638.9034966050130000   -0.0360907014386575   -0.0336524800343376    0.0446466917958435   0.2448189271964060   0.2450031080625840   0.3463565071915430   0.2435069335560830   0.3452997738248400
    8.3000   -3.7140139243887200   1.5285902294647400   -0.0916779845660045   -2.8467811070681300   0.2766892530696140   11.2573986014576000   0.0781985317930050   0.2107337676178980   0.3444622323405860   -0.5008686902000000   -0.3341552841000000     639.6290848801480000   -0.0358195739828726   -0.0340345242237886    0.0448749166518763   0.2445171652524710   0.2449860888618540   0.3461312292165100   0.2434756123382860   0.3450637881705460
    8.3500   -3.7114703015462300   1.5256421709616300   -0.0914478130455275   -2.8450614511057700   0.2769155718638500   11.1747698165946000   0.0783600584936493   0.2106851317846600   0.3444622836932940   -0.5004233551000000   -0.3342017217000000     640.0856651287340000   -0.0356130336449998   -0.0343941624961300    0.0451093146140605   0.2442123253962000   0.2449325267116400   0.3458780168162870   0.2434066754880090   0.3447991727767690
    8.4000   -3.7082830391194300   1.5225063853583300   -0.0911956646746712   -2.8432693606855900   0.2771079772940120   11.0945831208501000   0.0784305453740230   0.2104849876224390   0.3442831869400420   -0.4997264807999990   -0.3340303557000000     640.4303074276040000   -0.0355736135510081   -0.0347009071062986    0.0453645507523860   0.2439038980736560   0.2448559115163290   0.3456060313420710   0.2433128201324720   0.3445144988768120
    8.4500   -3.7049206213043400   1.5194919019909000   -0.0909545664626330   -2.8417743096472500   0.2773015338814990   11.0161770842056000   0.0784825426125472   0.2101966937645880   0.3440762258392920   -0.4988840573000000   -0.3338855561999990     640.7795652736930000   -0.0356648562663716   -0.0349304222421780    0.0456043383323840   0.2435984707253290   0.2447591965319550   0.3453219935461500   0.2431995541106700   0.3442183000064750
    8.5000   -3.7010141717514300   1.5165278670288200   -0.0907291419197284   -2.8408449452909800   0.2775584946858110   10.9410849981403000   0.0786514163539941   0.2099524799482810   0.3440355609566650   -0.4980665561000000   -0.3338796224000000     641.2327259270890000   -0.0358684490934794   -0.0351637024852086    0.0458926622913311   0.2432914097851960   0.2446919251334120   0.3450577463277180   0.2431123386884320   0.3439394122484760
    8.5500   -3.6965465396070300   1.5136121575212800   -0.0905075655861555   -2.8401997946353300   0.2777992933813600   10.8770376816639000   0.0788550522753079   0.2097731590292650   0.3441366843628410   -0.4973828305999990   -0.3340424564999990     641.7620008326820000   -0.0361401647683447   -0.0353635019716198    0.0461789582399479   0.2429902198287920   0.2446232704701370   0.3447967392362500   0.2430237198383110   0.3436637533061860
    8.6000   -3.6933147418121300   1.5112356832881900   -0.0903308986158630   -2.8395517844872900   0.2780107886648850   10.8173564892716000   0.0789683970963707   0.2096043108247190   0.3443663363448500   -0.4970214427999990   -0.3342115869000000     641.9999459809560000   -0.0364169614142221   -0.0355574392394532    0.0464625037327467   0.2427341708969990   0.2445391879291560   0.3445566602956690   0.2429196936748670   0.3434091660048470
    8.6500   -3.6896393554115200   1.5087338522090600   -0.0901367618072944   -2.8385530340471700   0.2781364008711620   10.7613312805042000   0.0790664029439188   0.2094140861765260   0.3445386090207450   -0.4965476999999990   -0.3343345579000000     642.2473643871060000   -0.0367315795879219   -0.0357436716518592    0.0467547452450055   0.2424760563379760   0.2444512767909280   0.3443124520285790   0.2428111087498260   0.3431499270428840
    8.7000   -3.6849350348832800   1.5061538479566000   -0.0899492419902969   -2.8380852740855600   0.2783361012691960   10.7058026852738000   0.0789195923697523   0.2090151408808270   0.3445056929825270   -0.4960329186999990   -0.3341103154999990     641.9365079798560000   -0.0370231034371418   -0.0359049928884039    0.0470157062695128   0.2422199529833410   0.2443810987307190   0.3440822969001540   0.2427224927976460   0.3429062760189080
    8.7500   -3.6790176168598900   1.5033115365872300   -0.0897477312456605   -2.8379169252133800   0.2785657620076420   10.6529959972119000   0.0788491074387327   0.2086915772411140   0.3444249446879340   -0.4953892791000000   -0.3340215921999990     641.8245059457760000   -0.0373014864081716   -0.0360682453393203    0.0472730727805588   0.2419608430937880   0.2443156864241200   0.3438534633002060   0.2426388198060910   0.3426640431494810
    8.8000   -3.6738638629016700   1.5005752426618300   -0.0895389610294034   -2.8370822916640400   0.2786732111668410   10.6034331780535000   0.0787450198680787   0.2084641387818590   0.3443237825756450   -0.4949087479000000   -0.3340055597000000     641.5943364644630000   -0.0376519917932829   -0.0361794899204876    0.0475134113735110   0.2416428157709140   0.2442382524293530   0.3435746998302410   0.2425440240236250   0.3423723908309030
    8.8500   -3.6702332422763000   1.4984195948599000   -0.0893823706528591   -2.8363338046513800   0.2787999153849470   10.5533518304072000   0.0785393402202086   0.2081341212511240   0.3441242633173210   -0.4944914154000000   -0.3337494375999990     641.1162318786930000   -0.0379432681198694   -0.0362594075696263    0.0477018216567978   0.2413042152812710   0.2441638323753070   0.3432837038845670   0.2424558828033130   0.3420710151685520
    8.9000   -3.6663944114543400   1.4962504748400600   -0.0892217554885038   -2.8353984876229800   0.2788780725768000   10.5044046073523000   0.0782402339553087   0.2076708036514600   0.3437943041050670   -0.4940152870000000   -0.3332411676000000     640.3159874397300000   -0.0381582257847086   -0.0363660312376523    0.0478829004673776   0.2409994790330260   0.2441117374348100   0.3430324900758400   0.2423906463349560   0.3418098511232050
    8.9500   -3.6613281712120900   1.4936288345380800   -0.0890333651278736   -2.8346373306257200   0.2790256921152320   10.4459490713722000   0.0778622211942156   0.2070550511873120   0.3433784112286330   -0.4934402694999990   -0.3325334254000000     639.2004886201900000   -0.0383181297689560   -0.0364803705096302    0.0480485893390548   0.2407037171519900   0.2440568159279740   0.3427856602188690   0.2423236484754020   0.3415538465033230
    9.0000   -3.6576638824254300   1.4913881139328600   -0.0888677819789125   -2.8335273400823000   0.2791016661893740   10.3906172753920000   0.0776542335480457   0.2065488901682770   0.3429822661716800   -0.4926009827000000   -0.3320362795999990     638.6262762005310000   -0.0383816466094626   -0.0365876325510213    0.0481686836697394   0.2403924106709280   0.2440212363566270   0.3425417856279680   0.2422794769173910   0.3413031732104830
    9.0500   -3.6523883244329100   1.4886711182327600   -0.0886707904896492   -2.8328950694503100   0.2792585696986440   10.3353348528525000   0.0775583447524663   0.2061542595263360   0.3425805660930600   -0.4917003675000000   -0.3316515037000000     638.3332092742640000   -0.0383608251593725   -0.0367138459456531    0.0482711161385959   0.2400550874567110   0.2439649473303090   0.3422650150683380   0.2422157729378000   0.3410204182651020
    9.1000   -3.6469034310848400   1.4857579151255000   -0.0884544607030383   -2.8319513328365200   0.2793852639411170   10.2783570187741000   0.0775185193680708   0.2057931150008280   0.3421355420399910   -0.4907236253999990   -0.3312501501999990     638.2089094791710000   -0.0383289185945528   -0.0368598866361707    0.0483864441436829   0.2397048448197640   0.2439143073829300   0.3419833358164720   0.2421567899794590   0.3407320407053370
    9.1500   -3.6415684868140000   1.4832496750290600   -0.0882801776254173   -2.8316984719199600   0.2795706157440860   10.2292965524120000   0.0774127740445665   0.2054313832171680   0.3414856316940290   -0.4895528027000000   -0.3307385072999990     638.1316644967620000   -0.0383259229823866   -0.0369360409269914    0.0484516171928708   0.2393922545039400   0.2438834516974920   0.3417422852506880   0.2421211728061710   0.3404868776289550
    9.2000   -3.6358618030125400   1.4806503766183400   -0.0880951372082101   -2.8313584177433000   0.2797124921374630   10.1831224953033000   0.0774343593906956   0.2051768377633560   0.3409281682670690   -0.4883293713999990   -0.3303773998999990     638.3054168111270000   -0.0382826188349576   -0.0370625470105356    0.0485450115802021   0.2391193626346570   0.2438527968513160   0.3415292902797860   0.2420836969158860   0.3402684027341760
    9.2500   -3.6306578720600800   1.4782969192214200   -0.0879297532833681   -2.8310842221739700   0.2798305283546610   10.1386633260520000   0.0775605524907102   0.2049956139727700   0.3405222790283220   -0.4872095866999990   -0.3302475446000000     638.6315712994390000   -0.0381114855795094   -0.0372012360607918    0.0485973441461446   0.2388731886871020   0.2438147533507740   0.3413298027202470   0.2420418117766810   0.3400656391370380
    9.3000   -3.6285131734070000   1.4767666345109300   -0.0878221281613636   -2.8302203227721400   0.2798692303064070   10.0895660088383000   0.0776395140424447   0.2048346537755090   0.3400668564641560   -0.4862067884999990   -0.3301435334999990     638.7415782733240000   -0.0378660975262286   -0.0373251358359453    0.0486064193354909   0.2386364124246430   0.2437692904447910   0.3411316524435410   0.2419955524202410   0.3398664218867200
    9.3500   -3.6256188748134900   1.4751848179024900   -0.0877223903085246   -2.8298248359189200   0.2799692849142380   10.0438013343963000   0.0778209538599002   0.2047301823165290   0.3397316640919970   -0.4852416660000000   -0.3302405800000000     639.1501627649190000   -0.0374677809083872   -0.0374536957406505    0.0485573801786802   0.2384073926144210   0.2437168969682260   0.3409340269334050   0.2419464858804530   0.3396707035985100
    9.4000   -3.6243625074887200   1.4739965848952700   -0.0876414297747248   -2.8287909257367600   0.2799609771275360    9.9997380408693000   0.0779686763238270   0.2046643268601000   0.3394323035667090   -0.4845466159999990   -0.3303758259999990     639.3680439436500000   -0.0369612331478237   -0.0375900670503463    0.0484711915534354   0.2381863894127830   0.2436683681926270   0.3407448161882370   0.2419038545583080   0.3394852440853140
    9.4500   -3.6252296417523400   1.4734590078938600   -0.0876033217895421   -2.8272924974842200   0.2798661246911100    9.9568486646732500   0.0779673334390103   0.2045033623618200   0.3389628023810330   -0.4839306506000000   -0.3302964102000000     639.2690767054230000   -0.0364576834275160   -0.0377048505377346    0.0483671689776835   0.2379847663807150   0.2436345469237430   0.3405797138468780   0.2418771381549220   0.3393247691980330
    9.5000   -3.6291890753648300   1.4740755286446200   -0.0876536115044512   -2.8257298517925400   0.2797180741694120    9.9176348490441000   0.0778646135692573   0.2042469463890010   0.3383567831442550   -0.4833098198999990   -0.3299349122999990     638.9393968115410000   -0.0360094055217468   -0.0378268634922470    0.0482916609998345   0.2377938320589950   0.2436370915964900   0.3404481443140760   0.2418846731002860   0.3391962583020840
    9.5500   -3.6337178261922700   1.4747344076730900   -0.0876994165450255   -2.8238293717526200   0.2795154439863610    9.8762867582252200   0.0778935614381431   0.2040196173013390   0.3378878217732090   -0.4826367426000000   -0.3297270065999990     638.8379816021620000   -0.0355937969740863   -0.0379432680192283    0.0482241132657414   0.2375845091330840   0.2436639206360030   0.3403211794756780   0.2419159062574670   0.3390718282021970
    9.6000   -3.6372920186273100   1.4750027616986600   -0.0877156829119312   -2.8220326243203100   0.2793515953660900    9.8329346847442500   0.0780269918113046   0.2038442541730860   0.3375404661522670   -0.4820215192000000   -0.3296752041000000     638.9863066040760000   -0.0351487918627335   -0.0380594977772842    0.0481446815238407   0.2373929478567240   0.2436564015824890   0.3401820890702980   0.2419132110620430   0.3389357068508040
    9.6500   -3.6367948684118700   1.4742636589705600   -0.0876736231252098   -2.8214265714996900   0.2793685132820200    9.7924789130203500   0.0782053274252555   0.2037778824268180   0.3372390668492160   -0.4814860798999990   -0.3297891102000000     639.3612080601100000   -0.0346375357671435   -0.0381940456533048    0.0480544639188358   0.2371821835733990   0.2436178284835270   0.3400074036836180   0.2418796687918110   0.3387641692674740
    9.7000   -3.6334548250800900   1.4726632353229400   -0.0875733072615184   -2.8215704529991900   0.2795224877396360    9.7505101759597900   0.0784877222346124   0.2037844236399530   0.3369955057265060   -0.4808777315000000   -0.3300690017999990     640.0415310612710000   -0.0341481526578314   -0.0383016311484454    0.0479496318233520   0.2369603650516700   0.2435020991748550   0.3397697557287610   0.2417693290206470   0.3385300918094500
    9.7500   -3.6278862713021700   1.4705904486300700   -0.0874562913467633   -2.8226770880444000   0.2798372238118730    9.7090645858519900   0.0787597819413020   0.2037594985799430   0.3367005457769100   -0.4802290482000000   -0.3303649006000000     640.7337323417640000   -0.0337597796762549   -0.0383814737720275    0.0478613048148427   0.2367264661639160   0.2433884787054420   0.3395252145997480   0.2416599902436130   0.3382882951965650
    9.8000   -3.6219089832604400   1.4683874874746500   -0.0873327347322782   -2.8239198402529100   0.2801860702763330    9.6657189764286600   0.0790526515238753   0.2037099468178050   0.3364818228030130   -0.4796036269999990   -0.3306238406000000     641.2944362338770000   -0.0332430776964065   -0.0384847678284701    0.0477405688480982   0.2364998862831830   0.2433323484556730   0.3393270222321110   0.2416101447882870   0.3380941559337810
    9.8500   -3.6141066294235500   1.4657340654784900   -0.0871812501686083   -2.8256844419057100   0.2806116869267060    9.6262800923935100   0.0794070061916134   0.2037263971456380   0.3362471697536570   -0.4789140136999990   -0.3308819422000000     641.9790142517690000   -0.0326936020777195   -0.0386379526119478    0.0476506466152012   0.2363060001928540   0.2432864522815560   0.3391589945599130   0.2415679302597400   0.3379283809583320
    9.9000   -3.6046110470587000   1.4625537483932500   -0.0869909066222028   -2.8275892122588000   0.2810579452208420    9.5888786116330800   0.0798387153955572   0.2038659047951920   0.3361687670176440   -0.4783942108000000   -0.3313416531000000     642.7801845789930000   -0.0320028415491399   -0.0388303475596501    0.0475378734149687   0.2361403438202960   0.2432438818702880   0.3390130499654760   0.2415298062245280   0.3377853005600240
    9.9500   -3.5947824200752900   1.4592781904090000   -0.0867955772967101   -2.8295373839007700   0.2815176089652910    9.5508587467113700   0.0802364187522683   0.2039584653400550   0.3360978547766230   -0.4778683249999990   -0.3317447512999990     643.6959838189770000   -0.0312406042225414   -0.0390147022607702    0.0473891211033721   0.2360009544070160   0.2431903585115490   0.3388775604167350   0.2414825704864920   0.3376540867956230
    10.000   -3.5847384302061900   1.4558870236650600   -0.0865869101137698   -2.8314179611693300   0.2819614748639540    9.5125955757678000   0.0806523182496776   0.2040163247484210   0.3361158872481720   -0.4773499893000000   -0.3321858028999990     644.7099922203470000   -0.0304518098784769   -0.0391893111651246    0.0472212949777613   0.2358942564213900   0.2431253208902690   0.3387565820328770   0.2414245587622760   0.3375380242078810
    """)


# NB: the implementation is really ugly and should be changed by using
# parametric GMPEs: we will do that if the need to add subclasses will arise

class CauzziEtAl2014NoSOF(CauzziEtAl2014):
    """
    Returns the Cauzzi et al (GMPE) for the case when no style-of-faulting is
    input. This modifies both the expected ground motion as well as the
    inter-event (and thus total) standard deviations.
    """
    sof = False  # no style of faulting term

    #: Required rupture parameters are magnitude
    REQUIRES_RUPTURE_PARAMETERS = {'mag'}


class CauzziEtAl2014FixedVs30(CauzziEtAl2014):
    """
    Implements the Cauzzi et al (2014) model for the case in which
    the reference Vs30 in the site amplification term is fixed at 800 m/s
    """


class CauzziEtAl2014FixedVs30NoSOF(CauzziEtAl2014NoSOF):
    """
    Implements the Cauzzi et al (2014) model for the case in which
    the reference Vs30 in the site amplification term is fixed at 800 m/s
    """


class CauzziEtAl2014Eurocode8(CauzziEtAl2014):
    """
    Implemements the Cauzzi et al. (2014) GMPE for the case in which
    the Eurocode 8 site classification is preferred
    """


class CauzziEtAl2014Eurocode8NoSOF(CauzziEtAl2014NoSOF):
    """
    Implemements the Cauzzi et al. (2014) GMPE for the case in which
    the Eurocode 8 site classification is preferred and style of faulting
    is not specified.
    """


def rhypo_to_rrup(rhypo, mag):
    """
    Converts hypocentral distance to an equivalent rupture distance
    dependent on the magnitude
    """
    rrup = rhypo - (0.7108 + 2.496E-6 * (mag ** 7.982))
    rrup[rrup < 3.0] = 3.0
    return rrup


# Cauzzi et al. 2014 - Converted from Rhypo
class CauzziEtAl2014RhypoGermany(CauzziEtAl2014):
    """
    Implements the Cauzzi et al. (2015) GMPE applying the rhypo to rrup
    adjustment factor adopted for Germany
    """
    REQUIRES_DISTANCES = {"rhypo", "rrup"}
    REQUIRES_RUPTURE_PARAMETERS = {"rake", "mag", "width"}


class CauzziEtAl2014Eurocode8scaled(CauzziEtAl2014Eurocode8):
    """
    Implements GMPE developed by Carlo Cauzzi et al (2014) and published
    as C.Cauzzi, E. Faccioli, M. Vanini and A. Bianchini (2014) "Updated
    predictive equations for broadband (0.0 - 10.0 s) horizontal response
    spectra and peak ground motions, based on a global dataset of digital
    acceleration records", Bulletin of Earthquake Engineering, In Press

    Spectral acceleration (SA) values are obtained from displacement response
    spectrum  (DSR) values (as provided by the original equations) using the
    following formula ::

        SA = DSR * (2 * π / T) ** 2
    """
    #: Coefficient table constructed from the electronic suplements of the
    #: original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    imt                       c1                   m1                    m2                    r1                   r2                    r3                   sB                   sC                   sD                    bV                 bV800                       VA                    fN                    fR                   fSS                    f                    t                    s                   tM                   sM
    pgv       0.483292714   0.5482239378818140   -0.0319470258028777   -2.8457788432226700   0.2406737047414070   6.51696666287798000   0.1919277313361110   0.3706196443240230   0.4978018189249720   -0.6909580227999990   -0.7596804404000000     883.9565406477700000   -0.1433313027208760    0.0184633160924233    0.0049897699311183   0.2398935826956730   0.2213004346208430   0.3263783286033860   0.2149404298227920   0.3220998664588330
    pga      -2.153913989   0.5237450060972680   -0.0609447663010394   -3.8019035608295600   0.3550808121411740   11.6415555876916000   0.2106985279596590   0.2825106921224770   0.2828846140789600   -0.3100704816000000   -0.7024376883999990    2319.1859784562300000   -0.0241122431339601    0.0724633666482452   -0.0563165754085399   0.2589229720745850   0.2214506097241730   0.3407073201666570   0.2162221044760760   0.3373323345485870
    0.0500   -0.133056941   0.7038280147755070   -0.0875507720994941   -4.4155335972196800   0.4140306595178590   15.9634085011888000   0.1678531023791990   0.1863342553869610   0.1392855607298930   -0.1000000000000000   -0.5639501998999990   30552.3271673005000000   -0.0389955855635790    0.0960304396771990   -0.0706779148995298   0.2702569776994650   0.2423624069839720   0.3630129065395790   0.2337736891265200   0.3573359367920780
    0.1000    0.720631637   0.6701555114371590   -0.0842487133409999   -4.2989668796524600   0.3950150942756780   16.9499399186483000   0.1982256814552770   0.2004113577343430   0.1312995606293180   -0.1000000000000000   -0.5923724273999990   36597.5600451842000000   -0.0282989830364419    0.1005680531402610   -0.0799511338270659   0.2864871213393960   0.2433368095307030   0.3758825262843640   0.2343281863791130   0.3701142656330040
    0.1500    0.457351183   0.6331119438041180   -0.0719075361587948   -3.8697055665013900   0.3527209416036150   13.7553799942078000   0.2489114740764020   0.3018776519585660   0.2531944049628660   -0.2273218226000000   -0.7549461000999990    5440.3520640067600000   -0.0299967463616727    0.0962871690940792   -0.0762552422410023   0.2888548904585920   0.2369264158171150   0.3735923905191180   0.2282717557908330   0.3681645586333300
    0.2000   -0.041706067   0.6394154973768270   -0.0625617809919999   -3.4154331729506900   0.3010051287415760   11.4534631595437000   0.2821393446741870   0.4058929876762180   0.3690896542349010   -0.4384492105000000   -0.8944358918000000    1898.2857019816500000   -0.0036711341998802    0.0693398950336865   -0.0633905167734821   0.2932537473600140   0.2159716768759310   0.3641998428792800   0.2112464445303770   0.3614177924057090
    0.3000   -0.44693261   0.6651087078884250   -0.0560927957029996   -3.0938863341046000   0.2735167424921670    8.2935173490201800   0.2096502844403170   0.4001251672249500   0.4335528490826920   -0.6554113151999990   -0.8545657526000000     967.9255537188150000   -0.0179995073864090    0.0508746350031907   -0.0417662944939413   0.2955895319193470   0.1990806049250280   0.3563793745962490   0.1961542457250350   0.3547529555849480
    0.4000   -0.9682344   0.7477063980675440   -0.0562692503039990   -2.8852884017037800   0.2536783288165920    6.2119923952061800   0.1858234788604210   0.4107333337437640   0.5209946684961200   -0.8100884283999990   -0.8530330874999990     794.5532205269830000    0.0004098177393169    0.0403886753646120   -0.0384916849988285   0.2987789982376510   0.2068578206071190   0.3633992951757350   0.2051740007093480   0.3624434581489560
    0.5000   -1.314364459   0.8260335951960020   -0.0600971406249981   -2.7937608332104700   0.2483537730430820    5.2021129207912500   0.1912072326769610   0.4163804535941900   0.6163803445766500   -0.9166134093000000   -0.8761241065999990     747.1063408592170000    0.0086757291814616    0.0281315838618958   -0.0297855596041406   0.2988253255683690   0.2171380197001870   0.3693852931565640   0.2163378482566530   0.3689154913924880
    0.7500   -1.972769452   1.0612712694853100   -0.0761444939880500   -2.8411938303340100   0.2583666046875160    5.0513970104537200   0.1680957619203300   0.4032102189640850   0.7080554315362520   -1.0313556675000000   -0.8579717241000000     675.6979435204260000    0.0239285639574445    0.0074392636316948   -0.0161682493304947   0.2951575001067500   0.2297001913951280   0.3740055184034380   0.2294277631124650   0.3738382649703680
    1.0000   -2.432723783   1.2134822214871800   -0.0854280000001306   -2.8543804213112900   0.2593699405690310    4.9780849514747200   0.1576721986608320   0.3903326170865970   0.6932630179184690   -0.9891875651000000   -0.8248044635000000     678.6122701764790000    0.0344742509193550   -0.0054363568289916   -0.0082972422158440   0.2963176996743180   0.2307500744544380   0.3755659409491340   0.2302608656402460   0.3752655664801600
    2.0000   -3.428939783   1.6562992065143400   -0.1154147661327820   -3.0442154569996200   0.2641813503629670    8.9746507855722500   0.1058745154124260   0.3256670883462820   0.5351158123025720   -0.7909590998000000   -0.6342277109999990     641.1223369683100000   -0.0162396506874878   -0.0019971914576055    0.0080969196977610   0.2906422190364900   0.2257008081025670   0.3679860789005580   0.2255881657370820   0.3679170015194690
    3.0000   -3.663741045   1.7276931468404900   -0.1155813789892250   -2.9923117217806700   0.2523813577739830   11.8386025178292000   0.1081958140488820   0.2968493026283440   0.4983702292088530   -0.7127096177999990   -0.5547507065000000     643.3004613025160000   -0.0434663752687055    0.0115341667660334    0.0062500594261708   0.2812581479022440   0.2324512752973370   0.3648831883613700   0.2315528559466120   0.3643115025063480
    4.0000   -3.529301652   1.6890940080884600   -0.1128841877050190   -3.1224660460706600   0.2822792909686670   12.7543160967594000   0.1119890714433590   0.2817361688665730   0.4534378584333170   -0.6428214690000000   -0.4939849281000000     651.4850608555060000   -0.0583039796710700    0.0092717900996661    0.0145442858770594   0.2713069027359440   0.2407315750359230   0.3627108031082710   0.2392999325376520   0.3617622053016260
    """)
