# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2015-2021 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.


"""
Module exports :class:`BoraEtAl2019`, :class:`BoraEtAl2019Drvt`
"""

import numpy as np
from openquake.hazardlib import const
from openquake.hazardlib.imt import FAS, DRVT
from openquake.hazardlib.gsim.base import GMPE, CoeffsTable


CONSTANTS = {"r0": 1,
             "r1": 50,
             "Mh": 5.0,
             "Vc": 1100.,
             "Vref": 800.,
             "Mref": 4.5}

CONST_DURATION = {"M": 5.3,
                  "Vs30": 450}


def _get_source_term_duration(C, ctx):
    above = ctx.mag > CONST_DURATION['M']
    below = ctx.mag <= CONST_DURATION['M']
    fsource = np.zeros_like(ctx.mag)
    if np.any(below):
        fsource[below] = C['d1'] * ctx.mag[below]
    if np.any(above):
        fsource[above] =  (C['d1'] * CONST_DURATION['M'] +
                           C['d2'] * (ctx.mag[above] - CONST_DURATION['M']))
    return fsource


def _get_path_term_duration(C, ctx):
    return (C['d3'] + C['d4'] * (ctx.mag - 6)) * np.log(ctx.rrup)


def _get_site_term_duration(C, ctx):
    fsite = np.ones_like(ctx.vs30)
    below = ctx.vs30 <= CONST_DURATION['Vs30']
    above = ctx.vs30 > CONST_DURATION['Vs30']
    fsite[below] = C['d5'] * np.log(ctx.vs30[below])
    fsite[above] = C['d5'] * np.log(CONST_DURATION['Vs30'])
    return fsite


def _get_source_term(C, ctx):
    above = ctx.mag > CONSTANTS['Mh']
    below = ctx.mag <= CONSTANTS['Mh']
    fsource = np.zeros_like(ctx.mag)
    if np.any(above):
        fsource[above] = (C['c3'] * (ctx.mag[above] - CONSTANTS['Mh']) +
                          C['c2'] * (8.5 - ctx.mag[above])**2)
    if np.any(below):
        fsource[below] = (C['c1'] * (ctx.mag[below] - CONSTANTS['Mh']) +
                          C['c2'] * (8.5 - ctx.mag[below])**2)
    return fsource


def _get_finite_fault_factor(C, ctx):
    h = np.ones_like(ctx.mag)
    h[ctx.mag <= 4] = 2
    selec = (ctx.mag > 4) & (ctx.mag <= 5)
    h[selec] = C['c4'] - (C['c4'] - 1)*(5 - ctx.mag[selec])
    h[ctx.mag > 5] = C['c4']
    return h


def _get_path_term(C, ctx):
    fff = _get_finite_fault_factor(C, ctx)
    t1 = np.sqrt(ctx.rrup**2 + fff**2)
    t2 = np.sqrt(CONSTANTS['r1']**2 + fff**2)
    m = ctx.mag - CONSTANTS['Mref']
    g = np.ones_like(ctx.rrup)
    dshort = ctx.rrup <= 50
    dlong = ctx.rrup > 50
    g[dshort] = ((C['b1'] + C['c7']*m[dshort]) *
                 np.log(t1[dshort] / CONSTANTS['r0']))
    g[dlong] = ((C['b1'] + C['c7']*m[dlong]) *
                np.log(t2[dlong] / CONSTANTS['r0']) +
                (C['b2'] + C['c7']*m[dlong]) *
                np.log(t1[dlong] / t2[dlong]))
    fpath = g + C['c5'] * (t1 - CONSTANTS['r0'])
    return fpath


def _get_site_term(C, ctx):
    fsite = np.ones_like(ctx.vs30)
    below = ctx.vs30 < CONSTANTS['Vc']
    above = ctx.vs30 >= CONSTANTS['Vc']
    fsite[below] = C['c6']*np.log(ctx.vs30[below]/CONSTANTS['Vref'])
    fsite[above] = C['c6']*np.log(CONSTANTS['Vc']/CONSTANTS['Vref'])
    return fsite


def _get_mean_stddevs(C, ctx):
    mean = C['c0'] \
           + _get_source_term(C, ctx) \
           + _get_path_term(C, ctx) \
           + _get_site_term(C, ctx)
    phi = np.sqrt(C['phiss']**2 + C['phis2s']**2)
    tau = C['tau']
    sigma = np.sqrt(C['phiss']**2 + C['tau']**2 + C['phis2s']**2)
    return mean, sigma, tau, phi


class BoraEtAl2019(GMPE):
    """
    Implements the Fourier amplitude spectra model proposed by Bora et al.,
    2019 as described in Bora, S.S., Cotton, F., & Scherbaum, F. (2019).
    NGA-West2 empirical Fourier and duration models to generate adjustable
    response spectra.  Earthquake Spectra, 35(1), 61-93.
    """

    #: Supported tectonic region type is active shallow crust, see title!
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    #: Supported intensity measure types
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {FAS}

    #: Supported intensity measure component
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.HORIZONTAL

    #: Supported standard deviation types
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL,
        const.StdDev.INTER_EVENT,
        const.StdDev.INTRA_EVENT}

    #: Required site parameters
    REQUIRES_SITES_PARAMETERS = {'vs30'}

    #: Required rupture parameters
    REQUIRES_RUPTURE_PARAMETERS = {'mag'}

    #: Required distance measures
    REQUIRES_DISTANCES = {'rrup'}

    def compute(self, ctx: np.recarray, imts, mean, sigma, tau, phi):
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            mean[m], sigma[m], tau[m], phi[m] = _get_mean_stddevs(C, ctx)

    TMP = """
IMT	c0	c1	c2	c3	c5	c6	c7	b1	b2	c4	tau	phis2s	phiss
FAS(0.100000)	2.502721022594540212	-3.019871021031180103	-0.699489878261237874	-0.753382658510122871	-0.000663526369472840	-0.447088569310536033	0.022658004530799799	-0.600521004853297069	-0.616534271818211077	4.695512756528509968	0.588480822628952960	0.433800904045682001	0.618731304409973037
FAS(0.106365)	2.493093234671220237	-2.950991457891970260	-0.698331013803817902	-0.702345053547144937	-0.000575375766529017	-0.473010336996122993	0.008472958415733959	-0.558480811812944955	-0.633469461477807028	4.957987757942290408	0.599289476605178950	0.448806612314632980	0.602762279580551064
FAS(0.113136)	2.200097814344990077	-2.752070777717450234	-0.665213553374781053	-0.538500924250570057	-0.000351531830222749	-0.512475440993337972	0.001339078393139250	-0.555657634468791017	-0.668379380917146992	5.483681114674491042	0.566032844137735913	0.466240117754409988	0.598172081119088928
FAS(0.120337)	2.034254128263019989	-2.515671395969700352	-0.635100535813052924	-0.399333072210403983	0.000067971644099580	-0.559914435537282928	-0.005286554502698210	-0.580985931071911987	-0.720196134071792060	5.849424106305820104	0.590205620260948027	0.480868159478248958	0.599358224410653007
FAS(0.127997)	1.141053806149149752	-1.603872458039000026	-0.533125263623092871	-0.093856302508275896	0.000567573193981721	-0.601643463294283976	-0.001066708448104160	-0.599125945606298949	-0.807875445247379087	5.479585767835110310	0.619351025773987929	0.513743008213592978	0.595123266923756056
FAS(0.136145)	1.097930883567810101	-1.443321597298969916	-0.513488251279912045	0.000711116989402148	0.000471478220459774	-0.649213142597603099	-0.007345942610395500	-0.621231011766171015	-0.806566765858632961	5.655956204038489865	0.609747368343163898	0.547180999785885969	0.579869067743471023
FAS(0.144811)	1.483404727375449816	-1.566081523484779803	-0.524719921624065000	-0.039643110238059401	0.000094656593009214	-0.685792316316777062	-0.014700859926340499	-0.668480204675433898	-0.770633955866142029	6.277617893464900511	0.577565513334653891	0.571829774674226021	0.574022902559107040
FAS(0.154028)	1.722276945857439845	-1.532084517949950042	-0.529629551808791010	-0.082756418078219401	-0.000249112362031415	-0.729455198663002125	-0.020041529947657601	-0.687414128008660974	-0.743207187104985012	7.372562584721709555	0.568067224541579985	0.586442004940022965	0.566853276929819905
FAS(0.163833)	1.460061832200230114	-1.045958970972379998	-0.486671789668866983	0.050119627190902502	0.000079904794452668	-0.757136513057866001	-0.019793028587677799	-0.721625679328829928	-0.805301862987097938	8.475856774799581217	0.567493377501720020	0.591049931388512140	0.560264230585062029
FAS(0.174261)	1.578707195963670040	-0.976147429486504903	-0.482629149981250005	0.020786800183539698	0.000335719591923292	-0.788054689949609033	-0.017829642992326700	-0.735112011550997035	-0.880703399477094084	9.400855804550859673	0.558559224687700029	0.595600285094823989	0.552429955919354954
FAS(0.185354)	1.182895724732189846	-0.585629210490318020	-0.437617651848761002	0.178991073840818971	-0.000098018019830675	-0.811825974939115902	-0.014908985772967599	-0.732046838351504969	-0.856809026037520072	9.634206805427080411	0.551310605088821015	0.595140813656870993	0.542029860790392948
FAS(0.197152)	1.028944878879419988	-0.401910184286409011	-0.418764009782740976	0.249816920751483956	-0.000102009340222675	-0.831471942021133059	-0.017980135641335201	-0.723225281340862058	-0.882838238408401921	9.488215067692301830	0.549276058939023981	0.589499687309888998	0.534847559293664010
FAS(0.209701)	1.008199243530689904	-0.281530366031195023	-0.403386697603251021	0.274326485622819971	0.000282642262354618	-0.857238397358372040	-0.013728189407098801	-0.748505165535349959	-0.954770187899388945	9.074101982989120074	0.537859463486816014	0.583163852557681994	0.526973023366866022
FAS(0.223050)	0.952214083535355083	-0.100524685664820007	-0.381329592135126072	0.292816061786009996	0.000465364877644078	-0.878500388719588021	-0.004891352642382350	-0.779028662494515967	-0.986529146516370026	8.908124929344319654	0.536849168508442998	0.592284479109781992	0.513297762497532117
FAS(0.237248)	1.178575027637770001	-0.160341960573721992	-0.390391125691087915	0.173209062877783038	0.000122741299879163	-0.904451559537362004	0.002093170920431040	-0.781467095295025005	-0.961359950176325029	8.723395906403499822	0.513502594537827983	0.602816924830573031	0.507707468277024954
FAS(0.252349)	0.708780135345348006	0.248824882663123970	-0.345601421855248003	0.325379468791085036	-0.000450021263163198	-0.912880947534275955	0.000957617117364880	-0.754971372196290069	-0.922064611722420047	8.629659190302980676	0.493815493038287034	0.613268572027439895	0.502845184828948022
FAS(0.268412)	1.004978293941600098	0.122070719138283004	-0.364488055989922954	0.230865142691424968	-0.000863719788451891	-0.911890525934737961	-0.005947828907500821	-0.734667926891354073	-0.891611724916564929	8.583764776695399945	0.481502175277732092	0.621305501892941092	0.496708426688614035
FAS(0.285497)	1.261065086886659969	0.032291382824867002	-0.375802863118586017	0.152242049392573991	-0.001050919809684140	-0.915325187417972086	-0.005031269658473920	-0.744518105837879030	-0.886561792470585974	8.283252334204579626	0.467006407858811035	0.631639083264063927	0.486605406036418997
FAS(0.303670)	1.297380503118820139	0.092338844851424198	-0.368150181936367948	0.123975140360036998	-0.000851208585344156	-0.932075059721235943	0.000869549927931463	-0.759432872154474081	-0.907780153703750980	7.922853183533959509	0.465368669782417033	0.637448521710694993	0.481833131186140018
FAS(0.323000)	1.562149231571200003	-0.059743320080940993	-0.379727531053058953	-0.017064169449521199	-0.000373385303942172	-0.952464846372410912	0.011775101901495701	-0.781885072640022050	-0.954932005583784993	7.194634306341050056	0.464928949482427023	0.644505686093211017	0.477795924429797958
FAS(0.343560)	1.769220056094679805	-0.211335436633630025	-0.393160326784775027	-0.168477891788259004	-0.000611414891522460	-0.975668984549608931	0.020777065383791701	-0.769297708576439865	-0.936470115592560881	6.413037582106531076	0.475511785777099960	0.650128494282875002	0.471975520949308947
FAS(0.365429)	1.890658457580830065	-0.250479609268315973	-0.396678244501707911	-0.237744597416399972	-0.001480812495483090	-0.999728230130709994	0.022317840721777799	-0.759166148298237919	-0.833776534892909926	6.521672344910610875	0.472457945049025996	0.648981901767755054	0.469485760759561055
FAS(0.388689)	2.130683801531799926	-0.129224100574196976	-0.389769945851931032	-0.134336064221033002	-0.001105609259912660	-1.020442296384320047	-0.001383633794928650	-0.828360818749292904	-0.856438740963645029	8.918801729001259559	0.459026976490558958	0.643222753515050094	0.474264953191637917
FAS(0.413431)	2.361291117517209770	-0.266048169011476987	-0.398039368389469961	-0.235029873463643985	-0.000864282977971807	-1.029346821105540100	0.005013360103172870	-0.849435354935159093	-0.892611916330538935	8.863510249344338732	0.443919409274569021	0.641078776738296940	0.466337543427628953
FAS(0.439747)	2.526447974390619855	-0.353975835222572033	-0.400915885530191085	-0.318870876163908024	-0.001224570944282310	-1.034310491376940089	0.016244390866307402	-0.864688750589019950	-0.866882599537526999	8.738874989287820583	0.427872055437270027	0.641955716229769102	0.459823324936523947
FAS(0.467739)	2.712478114489979664	-0.424865222634677964	-0.406352496140182984	-0.392185019107090971	-0.001676935073627110	-1.045025388091409990	0.019552411280142599	-0.868453459478694012	-0.821506139803228064	9.011041861994778301	0.407666800569367049	0.644818749877216102	0.449622751562542000
FAS(0.497512)	2.612817885317419986	-0.300817838380562974	-0.389129948300667994	-0.356463682507150048	-0.001903834554617320	-1.059898022662359995	0.020111037809693699	-0.866729659196408986	-0.804309308380740995	9.330571830482599083	0.398756355757518999	0.646515930274845996	0.444090430549561987
FAS(0.529180)	2.319697295967570128	-0.114681771567168009	-0.357296547665871989	-0.283039180205739038	-0.002275824888184850	-1.080105908553409932	0.028976951643349701	-0.873671516399949000	-0.776718444834420874	8.836028170859311359	0.410077002109618949	0.646757617417074937	0.443034930494605017
FAS(0.562864)	2.334835881340250108	-0.049313740379160899	-0.346319346860950950	-0.296059101732377983	-0.002658851865653740	-1.078860296535770091	0.034576760290465497	-0.882866419973704075	-0.757249023460896953	8.566387509598150984	0.413933747417838005	0.647490109191039998	0.441455359253865975
FAS(0.598692)	2.397870433313529670	0.036464778620993801	-0.337456912147471033	-0.290048361734939975	-0.003081197957075320	-1.065418022810560084	0.035405459405754500	-0.897131778136861069	-0.731072144316156969	8.022384131279400421	0.419604168869344984	0.641890170333777044	0.443352869728790011
FAS(0.636801)	2.647084206513470317	-0.114577619003008002	-0.351190994109601973	-0.353195532951641022	-0.003369797210788650	-1.074561613480760069	0.032901720359310299	-0.908839038316000991	-0.701560400979207022	7.623680651728860447	0.418915052516563025	0.637685203207327023	0.441449931279451979
FAS(0.677336)	2.848014506048390349	-0.236295421340618983	-0.360716790531181997	-0.424204132852936011	-0.003631787740873220	-1.085634757141869855	0.033817073659735797	-0.918273929680213041	-0.672746429172529026	7.153077243850329836	0.417074079153188004	0.636161990923952980	0.436686605477567991
FAS(0.720450)	2.943613830263649866	-0.323764928081191006	-0.363214400127210013	-0.492163282281458947	-0.004122136278680030	-1.072912856768720058	0.042180685736825499	-0.915610098340040035	-0.636083370189713948	6.597247041429250558	0.409386910756943001	0.633402620664181959	0.435368977526132017
FAS(0.766309)	2.912696421358130117	-0.334542473972756049	-0.356047934271072952	-0.502258620691582003	-0.004653549324892010	-1.065518668249830103	0.048036106590697701	-0.913443241917395010	-0.596759536142490998	6.467297784875010613	0.403887670901603990	0.631698797055604944	0.435412517771909979
FAS(0.815088)	2.835456438076859875	-0.262488866521023045	-0.342249150602167984	-0.479608629209481052	-0.005298496912870950	-1.067293072914740071	0.050056509299579402	-0.915190368437207913	-0.539470281562923981	6.199514964925490368	0.400224214996387939	0.632177321314288942	0.433291755768260012
FAS(0.866971)	2.652585072909509734	-0.090136467423734698	-0.315409372553206002	-0.394593019481305984	-0.005770312971332490	-1.062613777828480099	0.053941843560772702	-0.934394957191602038	-0.508114904395530975	6.263060281637430116	0.408389268483085044	0.626971944616180998	0.432250367441257999
FAS(0.922156)	2.631964613958559962	-0.020550023283345500	-0.301797999431991038	-0.366646074526873034	-0.005996303574383100	-1.058997149531550219	0.056845965974488502	-0.956149528534425031	-0.501033408218898968	6.867407293350820119	0.403314713896149979	0.620558747856974957	0.431108801487243898
FAS(0.980855)	2.616111224985009809	-0.001736303354988020	-0.293826745566638015	-0.366779374517169965	-0.006359333893512661	-1.054700436683519982	0.061407562294999198	-0.962545827687454114	-0.473457507902477093	7.125266173481990428	0.397364797092185029	0.617919560226579012	0.428981857938473987
FAS(1.043289)	2.316386014099700130	0.198838918139166015	-0.263258910599950047	-0.267676866006804015	-0.007021153329969000	-1.040763493717610011	0.067259446338629000	-0.966125669257194120	-0.407852698280496029	6.944847976813269597	0.407264417147701008	0.613527900523678005	0.427656304267155007
FAS(1.109698)	2.241021726599170094	0.264156617591023990	-0.250976389936497013	-0.253382750500576959	-0.007861745870259800	-1.026116412623020002	0.070970230498249395	-0.964207513518117953	-0.319211322174382028	6.582391068275000201	0.395521004368949991	0.604894397041107923	0.425915016712619943
FAS(1.180334)	2.269478141622939926	0.211849500641538985	-0.251291064562774002	-0.290285189647782993	-0.008348216606951959	-1.022461411457650016	0.075176125899384605	-0.957412289454017129	-0.282657040255269998	6.078867653557109563	0.391499703789027931	0.599433990457188925	0.427714577468203960
FAS(1.255467)	2.499807522966320139	0.013730240546671999	-0.266120662122381002	-0.375131779135912946	-0.008403282628155970	-1.015689690162180137	0.080435208788969800	-0.972081487238992015	-0.295962977587782006	5.964107775291210167	0.396226912533207964	0.598449629277272965	0.426733497722626975
FAS(1.335381)	2.491481604594099952	0.057561163806657600	-0.254909943611511980	-0.362719017315936032	-0.008297249007011500	-1.004641085099819975	0.086904203676444802	-0.998517906331295113	-0.332486107534317010	6.147740396043769096	0.399183127739278998	0.595224371038884037	0.425148343953605012
FAS(1.420383)	2.432962776108370306	0.088293289412819706	-0.244572901062636017	-0.345552942175220978	-0.008630784142967309	-0.990512391009102044	0.091881732044760203	-1.005692810762609968	-0.326305097077344008	6.434795859690701114	0.397523011982742980	0.594756658096475044	0.425026964671159024
FAS(1.510795)	2.334814612375049947	0.191931337077721031	-0.230556192183095005	-0.294290394814178957	-0.009149978715968871	-0.970915214420055017	0.090699067514448603	-1.004291021307609943	-0.288785210873021947	7.105800862259740391	0.400002590524112966	0.592036237807510024	0.425695078614066968
FAS(1.606962)	2.273794678788979784	0.265673649877563034	-0.219752701528971972	-0.260308173647532004	-0.009355230552916920	-0.945864065122142161	0.089310519387134704	-1.003019218482720021	-0.288781722541468056	7.708343893470799735	0.399659010117432045	0.587274674716987910	0.427449117835926984
FAS(1.709251)	2.277332165499180050	0.276810682734645985	-0.216431959536553031	-0.258176394290102973	-0.009664868622764101	-0.930580309184950938	0.089909123757779194	-1.003295454395800146	-0.280762545193573021	7.951065086885110489	0.396070018269900959	0.579031521968740970	0.431992595809263968
FAS(1.818050)	2.194919495457579917	0.353264266800144999	-0.206047087803720025	-0.220656169335018026	-0.009916555974403409	-0.929167986275826974	0.090119064335879198	-1.007090121924719917	-0.277264129041976004	7.688198686949349714	0.396330928367786051	0.571162762381795086	0.435440672418478003
FAS(1.933776)	2.073758830039129641	0.445146624826465998	-0.192704575790575011	-0.161338785683713987	-0.009929148391346810	-0.930662318709149949	0.087186552547937904	-1.009811169460989921	-0.303536183841271956	7.640644506706550487	0.400646712150686035	0.562000754297667937	0.435190044587877023
FAS(10.233153)	-0.209618874964315000	1.072217726525839909	-0.021111212928080200	0.305755484205412986	-0.016239185823753201	-0.219503328034676981	0.130771093919370002	-1.074730665765680104	-0.632261339399334932	8.615700840265480664	0.530334753083881116	0.745723877648974076	0.483583218381053948
FAS(10.884528)	-0.590813144797872947	1.327519913603480095	0.006089767274142720	0.433989954148591039	-0.016484099498620501	-0.190698748810083002	0.126989384907950975	-1.078255016537269961	-0.631858436619976094	8.344938367083040376	0.539616529137784018	0.759351094474527977	0.492577410701642016
FAS(11.577365)	-0.794289152882065919	1.411859872513180125	0.016993546395580501	0.497851007068987994	-0.016694434378872601	-0.155731563268681011	0.123957969892848005	-1.083776567205450014	-0.640712226731398937	8.048460033050789164	0.548349363976569970	0.775077463117370025	0.501089942379123965
FAS(12.314304)	-0.875127985353424065	1.419564050503040020	0.019291055509810001	0.533434500351142926	-0.016656578341860500	-0.119546010049038004	0.119100078670731982	-1.098543460408959893	-0.677906883374374991	8.063852598825111073	0.556697560464489039	0.794163806704076936	0.509974710811784915
FAS(13.098151)	-1.087532125643119851	1.533419947098699776	0.030969148759252899	0.611051619983936978	-0.016441609927984199	-0.088442212083798100	0.112347249988790990	-1.108904749015310109	-0.722907376583707073	7.969872804552700529	0.566131049094776961	0.812443629180207960	0.519800329213585988
FAS(13.931892)	-1.328812116376870023	1.679332691854070081	0.042959612086331200	0.688804250047505895	-0.016329234248289901	-0.074177429849918305	0.104556983532634998	-1.113136498722489875	-0.751793408609464198	7.799690617169139806	0.575603092083382117	0.826440992860562051	0.531994163925623931
FAS(14.818705)	-1.498357229967989923	1.763035332891299989	0.047271137863715201	0.750287993606142067	-0.016054700258360300	-0.063893044821328807	0.093011833628343393	-1.115266059006990051	-0.803599172647122884	7.731392197476780304	0.585761293310917974	0.838267548638435001	0.547850520386518935
FAS(15.761965)	-1.594774667646130073	1.804876190413480019	0.045408820120141004	0.793233774035929096	-0.015567446106718900	-0.047881533808477497	0.078529725455174906	-1.119252549728720014	-0.880983114193309103	7.798978584321860197	0.595311352205806998	0.850113054804297996	0.566532224586875910
FAS(16.765268)	-1.687225506547299858	1.841602935993169821	0.043180587610235702	0.831448737585770981	-0.015078100553371001	-0.037132595724561797	0.064705653768284896	-1.127958602513710051	-0.950957118974364968	7.664151524003190552	0.600039702825464927	0.859327998873569920	0.587097377509165019
FAS(17.832434)	-1.733393004486639688	1.820598432366769881	0.036407902361524497	0.864677454018279001	-0.014747051451567499	-0.043133792107731599	0.049738027247880401	-1.139191127771830070	-0.993590369578250976	7.772944111619620067	0.605521617784765986	0.868930080941780925	0.606246369879168023
FAS(18.967528)	-1.766743304734269993	1.827727954687589973	0.033442106618153097	0.903538852565525019	-0.014566534928056299	-0.047681704108470203	0.037749670840227498	-1.165310763269870042	-1.028395372245849870	8.260292817908389651	0.616296042289264046	0.880365454307502016	0.624572915614326041
FAS(2.056867)	1.950722612327440153	0.511381460872607052	-0.179350220924403003	-0.115822263640146986	-0.010062189742789499	-0.920702654294535017	0.088088214089653602	-1.013533028067000075	-0.316144700721367022	7.760724978037929489	0.398097673741692970	0.556703819991699933	0.435871218180977016
FAS(2.187793)	1.913675651395550181	0.506036875923698060	-0.173174548834662989	-0.118144324676435000	-0.010082673521074600	-0.903578160772396055	0.094098924173753706	-1.022749537871780134	-0.339283181739019002	7.758897394048039686	0.398616600352099004	0.557406254395655054	0.435780503604420000
FAS(2.327054)	1.796000830486730093	0.595645475396295088	-0.158644277765936009	-0.080674184950737707	-0.010314557622370100	-0.878033312603445970	0.097394492001415700	-1.027177210608150038	-0.337959770546094995	7.731090853909089233	0.393594615710041007	0.556826008840942022	0.435258326438241028
FAS(2.475179)	1.620285315109859958	0.764311583099434033	-0.138454670808604019	0.021460795291475900	-0.010597250221211300	-0.852861995867525091	0.093325134571863894	-1.029557194082360017	-0.339022071180713036	8.044114812487190846	0.392600076763116035	0.556979419274640009	0.437653234898232046
FAS(2.632732)	1.627093988262349855	0.753860085431024007	-0.137407698064522993	0.053171107906888899	-0.010704325805571300	-0.844250435795420806	0.087954033289816297	-1.031592920041620065	-0.369310288644840057	8.595810742031900986	0.396190183346697977	0.554588363884636060	0.442587061837257989
FAS(2.800315)	1.668307768602830254	0.681330290587479959	-0.141010196705142005	0.020270803484819199	-0.010972920268386300	-0.838743654918777004	0.090585462666275304	-1.030957393876759998	-0.379072131007765989	8.649041543828460021	0.399244353238977945	0.553042276809841082	0.448083200434988005
FAS(2.978564)	1.714209671191959883	0.587653237403796047	-0.144912477566619013	-0.044386768188871599	-0.011447552494321998	-0.828820680646170871	0.098000713719713195	-1.028281228090670130	-0.361646804134289013	8.304531155893540628	0.399522541986926050	0.553551162154163001	0.452000371810782009
FAS(20.174876)	-1.780579369021959790	1.859323680771189924	0.032220225177018703	0.937434847747877997	-0.014284164811124199	-0.041120504172542399	0.027635379777555501	-1.200551139169780068	-1.085972103999800042	8.711637901265200767	0.628991368431012887	0.891405651368297081	0.643730684455237068
FAS(21.459075)	-1.893706013752879835	1.902726065724679838	0.033557414021277800	0.984457812496492890	-0.013870740952588501	-0.043093359634551201	0.016119095506882598	-1.220211306333570001	-1.151852067609800079	8.893815429810759454	0.638458139233130995	0.903752779232336989	0.663891271767931057
FAS(22.825018)	-2.164515155469469931	1.998536469815849870	0.042546109726455499	1.067264456408159923	-0.013358365830870999	-0.060096929197271803	0.004079208448600950	-1.229291703207469943	-1.213587685605250099	9.025439023043249520	0.645357922901107051	0.916123024745528980	0.686061638871032886
FAS(24.277907)	-2.569154958647499765	2.144212467117970089	0.058187991506292398	1.190592941159939855	-0.012875627957906701	-0.078526214114103196	-0.009653602056897449	-1.228808143146729925	-1.259127515264110153	8.974260000777279700	0.656469133279297901	0.930473846943656979	0.711645880437019041
FAS(25.823278)	-2.858158209061369970	2.272352081808190150	0.067389354480938898	1.314532935697520166	-0.012052374773800100	-0.040597519390977603	-0.030689086450752599	-1.225112873399570201	-1.340043035701590002	8.435883915713880299	0.680050243512336006	0.937978461209225900	0.739612295451622881
FAS(27.467017)	-2.910308141551120276	2.158616052469560120	0.052507107017089202	1.283832909691330126	-0.011617812944882400	-0.032747934573783402	-0.046468701997043900	-1.212824253679460007	-1.364189445660429989	8.268536095173249834	0.696824914776556059	0.950489799301430849	0.766761441786662945
FAS(29.215386)	-2.921080998581390364	1.994125646206420077	0.033580068523296001	1.195666906642639971	-0.011217944404457600	-0.039379828469870898	-0.057356525722447503	-1.205176580135870035	-1.378796062866769923	8.145032362615509669	0.722729184814978054	0.966892164718562097	0.796592626847953067
FAS(3.168160)	1.888433507706319903	0.370606182705787046	-0.160801288175799001	-0.156172430808219992	-0.011917512217839399	-0.815109960378089138	0.106644498213048999	-1.029103869473640076	-0.348420547392885971	7.956270330170849725	0.394332017356339048	0.553379621135952959	0.453768939057046017
FAS(3.369824)	1.841222752415719865	0.329693756488679990	-0.158176867594909010	-0.159449384176328007	-0.012333494853759600	-0.797525243622735025	0.111079243337794994	-1.030632797089819919	-0.335553850462763981	7.829499317441589668	0.389562688246129996	0.555494391737813897	0.453567078201757956
FAS(3.584325)	1.452247627236940009	0.578018944713293958	-0.126024456633502008	-0.015422924884020601	-0.012795492649099200	-0.773373246994338071	0.111379135421456998	-1.024433113326719935	-0.314192549632272999	8.106266862807130380	0.391960938264515002	0.562392887063643987	0.454009138926969036
FAS(3.812479)	1.002015855198829852	0.940987286789070954	-0.084818453495162205	0.165048589564936982	-0.013348366157634800	-0.734463509252414104	0.112160283592876997	-1.022283112187329923	-0.285849455672411989	8.441606280163268750	0.401075084682757987	0.570410581788017068	0.454121314416447042
FAS(31.075044)	-2.894233900103700297	1.829165784680850226	0.014356133518117500	1.114364279186790041	-0.010572939891332400	-0.042152376906846901	-0.065475945941614794	-1.218698529547789899	-1.432312890868570276	8.090635938332709998	0.741953597581202051	0.982743874930970929	0.825808258299242093
FAS(33.053076)	-3.068559902453870336	1.795816657011199968	0.011578034265810300	1.138235653354350063	-0.010037538483602400	-0.052038685774478600	-0.072716713784271705	-1.233328649793550058	-1.481914458423039926	8.077779394913418898	0.744801127075814917	0.999597153592035936	0.855189361344300902
FAS(35.157016)	-3.218202896077470232	1.749713327516639882	0.006997797768027960	1.145362552115209986	-0.009505419414836241	-0.071321507614524701	-0.081633939932982805	-1.243946248185560099	-1.525713179909889972	8.321422892269300320	0.756290342490030132	1.021409589339489887	0.883427222871264961
FAS(37.394879)	-3.396429115260180431	1.766780796362610095	0.006144283621304010	1.175931242171909918	-0.008732470332479940	-0.083625361598639994	-0.094992770060148099	-1.257295476639199938	-1.591637896781980199	8.777748002416378981	0.773898263230579087	1.049722529098920054	0.910205047947699075
FAS(39.775190)	-3.806548591852600083	1.959988264090740007	0.022933572202815701	1.299714418243639980	-0.008065144650801751	-0.089600985576852907	-0.108512471296916990	-1.264876753615969962	-1.641371472816439914	9.212935617545118916	0.792458414222642071	1.081230389407660031	0.937564372298899995
FAS(4.055156)	0.934088259292449097	0.998121570453324169	-0.076779455715246206	0.181126480562366993	-0.013588408608514299	-0.694315144083230029	0.114045382926054983	-1.025449803901649970	-0.299310951430069971	8.457871044629440505	0.404918962664182036	0.573885777472056047	0.453376758460161011
FAS(4.313281)	0.978760406803861005	0.944300556564814930	-0.080406103879342106	0.157708059803183032	-0.013626612644736399	-0.674449810882857048	0.111770781184714008	-1.028179326764060209	-0.327427778457447971	8.584397665757411033	0.408823719332159996	0.580042985445601067	0.455730131759205959
FAS(4.587835)	0.895981393222339939	1.012460973618890092	-0.071841167383121504	0.196697420944581036	-0.013706397999668600	-0.659240069510770965	0.109685602366838011	-1.033924917894909923	-0.355237016580385034	9.058279148994449415	0.416913895421275049	0.591075266732089988	0.456808770427917010
FAS(4.879867)	0.738707275790305928	1.128370574526009973	-0.057437198519106399	0.247990626713151002	-0.013775879458194900	-0.633937608800921981	0.111991999577119999	-1.039127168724969907	-0.398739394157647919	9.436983487799690096	0.425449979113499965	0.601325984830880955	0.456186507917868966
FAS(42.307015)	-4.201142708407290094	2.173735129324219972	0.047307966593178603	1.433504745135740110	-0.006942993760169260	-0.038901771903109401	-0.117509975242933010	-1.296483425631010045	-1.705301910655889763	9.900439036123160008	0.825831078608603053	1.075435867011240054	0.920789968890337041
FAS(45.000000)	-4.200408689725469991	1.863641624224169790	0.013109204074609700	1.186467380905149804	-0.006308701600106889	-0.121036344599585996	-0.088168798607184601	-1.331613336822620175	-1.723189435363529975	9.846608307466429011	0.864049412982307974	1.123788784529600182	0.962902228210263944
FAS(5.190487)	0.843314389290270028	1.001349962258019977	-0.067764834378238700	0.165621654106080984	-0.013975500818050700	-0.601121602932303878	0.119192144456181010	-1.041614043345880036	-0.426150084912609028	9.285792700438250336	0.432090546511866014	0.611021687575231010	0.454473356710271004
FAS(5.520879)	0.954444967642319897	0.849786951904609067	-0.079489827151022804	0.080688770845849495	-0.014205594516716401	-0.555095707191162968	0.125357590192648999	-1.041757437726909874	-0.445741100631778964	8.920317060707850132	0.441700459389111011	0.624089519701367057	0.452538125490065002
FAS(5.872301)	0.847067662426540857	0.866019416057661973	-0.072214327030276207	0.105088918849696983	-0.014342708154161500	-0.510112620136737971	0.128363959293771007	-1.044514130917840067	-0.480477527189185016	9.016769423465689215	0.450335225434803976	0.636131175052333964	0.451592809370327974
FAS(6.246093)	0.650458261049530972	0.942741561980227072	-0.058738344539385902	0.160967323647353999	-0.014527202684471500	-0.467603672269010007	0.129486528446309007	-1.042372794359069932	-0.512960980074621986	9.064821419645399558	0.455356291702607052	0.642466466889722976	0.451601564288474033
FAS(6.643678)	0.395932850305472039	1.084612707831839939	-0.040378881007953699	0.247560917697082022	-0.014769481962669999	-0.427797662772937071	0.126353265439903023	-1.039602468192160067	-0.524215863865547083	8.990409692864618663	0.459310900598808969	0.650192412275129028	0.452811397209231026
FAS(7.066571)	0.258101618019613011	1.154304212105679905	-0.030547245654296500	0.298291601653825011	-0.014914961290852600	-0.401966205919412067	0.124262805105551988	-1.047411909426099852	-0.542402798186386037	9.245328756109129742	0.460633520582332034	0.660970849937567051	0.453915163490114959
FAS(7.516382)	0.236972748176986991	1.109398575627750017	-0.032099976753252699	0.283023971353994985	-0.015151093150525700	-0.374580256716730009	0.127293177099638000	-1.055105903779919974	-0.556395664503702925	9.163345433029750353	0.470246985027484021	0.671505349846401978	0.457201898038130050
FAS(7.994825)	0.201426860516765005	1.043349742653210122	-0.035220078202162802	0.262327932749071013	-0.015407810615977600	-0.339505956903788986	0.130053321859984000	-1.055342827154410079	-0.571700561687920938	8.829506517700810520	0.486575054952254005	0.682328569575974098	0.460281623401607054
FAS(8.503723)	0.207898405263681030	0.937583605845900947	-0.042630308272016797	0.219399073574211040	-0.015778014679178300	-0.301438490165143957	0.132443410386299004	-1.052848961641990044	-0.576775046319193962	8.526529512369179997	0.494631043029450990	0.695280185841289033	0.463065391201623988
FAS(9.045014)	0.210092314951341980	0.858955393522459976	-0.047779697256899199	0.192755085526255004	-0.016059750743852699	-0.265419090972488947	0.132554906269495010	-1.057961341972339930	-0.582931980295925078	8.401484033071481505	0.503862159243763985	0.711257639806089137	0.469515846994462971
FAS(9.620759)	0.099842361640679500	0.875281258376868987	-0.042770492873562903	0.206747585550481006	-0.016143294093816003	-0.241093370336476015	0.133255376127781988	-1.068192253488890131	-0.611051323052614004	8.493482559735490156	0.517668051643455973	0.729219849154350896	0.476089175945325072
"""
    COEFFS = CoeffsTable(sa_damping=5, table=TMP, opt=1)


def _get_mean_stddevs_dur(C, ctx):
    mean = (C['d0'] + _get_source_term_duration(C, ctx) +
            _get_path_term_duration(C, ctx) + _get_site_term_duration(C, ctx))
    phi = np.sqrt(C['phi']**2 + C['phis2s']**2)
    tau = C['tau']
    sigma = np.sqrt(C['phi']**2 + C['tau']**2 + C['phis2s']**2)
    return mean, sigma, tau, phi


class BoraEtAl2019Drvt(BoraEtAl2019):
    """
    Implements the duration model proposed by Bora et al.,
    2019 as described in Bora, S.S., Cotton, F., & Scherbaum, F. (2019).
    NGA-West2 empirical Fourier and duration models to generate adjustable
    response spectra.  Earthquake Spectra, 35(1), 61-93.
    """

    #: Supported intensity measure types
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {DRVT}

    def compute(self, ctx: np.recarray, imts, mean, sigma, tau, phi):
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            mean[m], sigma[m], tau[m], phi[m] = _get_mean_stddevs_dur(C, ctx)

    TMP = """
IMT	d0	d1	d2	d3	d4	d5	tau	phis2s	phi
DRVT(0.100000)	-12.453125867808900296	2.265070750964019997	2.343787976452410327	0.624482385734875911	-0.403026387873397007	-0.027247547171696900	0.754093856676959051	0.241163041780326021	0.493592034082187991
DRVT(0.200000)	-6.520143974112809993	1.689999610718379897	1.535317175191810213	0.482067029770475997	-0.309724383145583038	-0.286619826784507981	0.612559040651896924	0.243883341321730007	0.522043296092606024
DRVT(0.250000)	-5.008938342780889563	1.480040538635470027	1.321562332056499978	0.454445581097838947	-0.279235652516796995	-0.309525589489802966	0.473919393735442007	0.270380233178010021	0.525324691843597980
DRVT(0.333000)	-2.608286562814499998	1.185726020745870191	1.045334285987550071	0.393819834267916002	-0.230404711185352995	-0.387909999169650022	0.391346221024220009	0.282685706609304976	0.524179906342668978
DRVT(0.400000)	-1.543230159925520040	1.036039791553069911	0.895364868473363940	0.356791268591267030	-0.202116198735990016	-0.403690671736557982	0.327018121954518959	0.298188804728176993	0.517337792658840057
DRVT(0.500000)	0.333960534472892978	0.722860333982584202	0.733305991973792870	0.328221749966453036	-0.159773539307610984	-0.424222945989308931	0.264576109960628025	0.299068836403101002	0.503190176931284983
DRVT(1.000000)	3.352207867910570016	0.196309672573529992	0.314485461712703973	0.321713861639480014	-0.061798494046040597	-0.462225253665662028	0.203465392995559990	0.266091309811819998	0.455356026751957965
DRVT(10.000000)	-0.684120430793722978	0.335392352348517997	1.032043892977950073	0.470537730997860004	-0.127868332410535024	-0.247554258451504999	0.254629155926244022	0.296960230511646994	0.478635897020813994
DRVT(100.000000)	-0.471808081684987002	0.478525378447060001	1.080285308770210007	0.493213457226542973	-0.144555092594704004	-0.436570591288972965	0.296931016199210973	0.333051970648100992	0.512956304082603953
DRVT(14.925000)	-0.901557775920589166	0.412972908394066029	1.102590275535660025	0.484578608523549947	-0.146593980713971006	-0.293387253757383004	0.272449977974769997	0.309184126581008967	0.494911071620452947
DRVT(2.000000)	2.543344627776029654	0.109596919035278018	0.422668874081128021	0.333078387918121055	-0.059626589682534301	-0.341829991086883000	0.193235659031861989	0.260454902093160023	0.452489730270711987
DRVT(2.941000)	1.764707467445299827	0.134039162728899014	0.576623181821514064	0.347315102127133957	-0.072957138509880004	-0.297266562741991980	0.199737655346333975	0.246197734300676985	0.456325405716896004
DRVT(20.000000)	-0.596865801412305053	0.389895192833306037	1.078990399007730083	0.478029151111236006	-0.140924276632781009	-0.315509916737803986	0.275404883110626053	0.324688008507653003	0.506501496777110982
DRVT(25.000000)	-0.432884819448578928	0.383148149256407000	1.069735583604789930	0.470847471400426987	-0.140021138364184006	-0.332489973387724014	0.282555299866926013	0.327653626888446070	0.513584429580520974
DRVT(3.333000)	1.513555474497090048	0.144582459928320001	0.633301021194999980	0.351207788584513014	-0.079471722424804506	-0.284643140317568988	0.204775345292148997	0.243523400288019998	0.458870385544763948
DRVT(33.333000)	-0.308867220286403010	0.393466618358936038	1.072528094496000062	0.476950908870955037	-0.142529832241635984	-0.369331981663112041	0.285192523677043008	0.327557647415606001	0.521909774846307961
DRVT(4.000000)	1.123971405402959922	0.171747194907011985	0.700166035410864973	0.364605512924223996	-0.084683859552102600	-0.272200417248687987	0.208262645821426035	0.249021947295400975	0.462945254205767043
DRVT(5.000000)	0.746117088434967046	0.170509958557413993	0.756681058958569075	0.379255760724008018	-0.088942144435340001	-0.237698798828760022	0.233768556448271020	0.261602925140040021	0.462929057754140061
DRVT(50.000000)	-0.256358613475116071	0.414853860589568024	1.066389043467179887	0.485214931491419976	-0.139202751158666982	-0.409158387336240992	0.294321989323505007	0.331609598511373949	0.514039076789606031
DRVT(7.519000)	-0.298594366137973022	0.287009958260428966	0.918043828258243932	0.432754898896367035	-0.112681829368921002	-0.228576818740991977	0.248548859910108971	0.284093931942531963	0.470221366005818042
"""
    COEFFS = CoeffsTable(sa_damping=5, table=TMP, opt=1)
