# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2021 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports :class:`BaylessAbrahamson2018`
"""
import numpy as np
from openquake.hazardlib.gsim.base import GMPE, CoeffsTable
from openquake.hazardlib import const
from openquake.hazardlib.imt import EAS


def _magnitude_scaling(C, ctx):
    """ Compute the magnitude scaling term """
    t1 = C['c1']
    t2 = C['c2'] * (ctx.mag - 6.)
    tmp = np.log(1.0 + np.exp(C['cn']*(C['cM']-ctx.mag)))
    t3 = ((C['c2'] - C['c3']) / C['cn']) * tmp
    return t1 + t2 + t3


def _path_scaling(C, ctx):
    """ Compute path scaling term """
    tmp = np.maximum(ctx.mag-C['chm'], 0.)
    t1 = C['c4'] * np.log(ctx.rrup + C['c5'] * np.cosh(C['c6'] * tmp))
    tmp = (ctx.rrup**2 + 50.**2)**0.5
    t2 = (-0.5-C['c4'])*np.log(tmp)
    t3 = C['c7'] * ctx.rrup
    return t1 + t2 + t3


def _normal_fault_effect(C, ctx):
    """ Compute the correction coefficient for the normal faulting """
    idx = (ctx.rake > -150) & (ctx.rake < -30)
    fnm = np.zeros_like(ctx.rake)
    fnm[idx] = 1.0
    return C['c10'] * fnm


def _ztor_scaling(C, ctx):
    """ Compute top of rupture term """
    return C['c9'] * np.minimum(ctx.ztor, 20.)


def _get_ln_ir_outcrop(self, ctx):
    """
    Compute the natural logarithm of peak PGA at rock outcrop, Ir.
    See eq.10e page 2092.
    """
    # Set imt and cofficients. The value of frequency is consistent with
    # the one used in the matlab script - line 156
    im = EAS(5.0118725)
    tc = self.COEFFS[im]
    # Get mean
    mean = (_magnitude_scaling(tc, ctx) +
            _path_scaling(tc, ctx) +
            _ztor_scaling(tc, ctx) +
            _normal_fault_effect(tc, ctx))
    return 1.238 + 0.846 * mean


def _linear_site_response(C, ctx):
    """ Compute the site response term """

    tmp = np.minimum(ctx.vs30, 1000.)
    fsl = C['c8'] * np.log(tmp / 1000.)
    return fsl


def _soil_depth_scaling(C, ctx):
    """ Compute the soil depth scaling term """
    # Set the c11 coefficient - See eq.13b at page 2093
    c11 = np.ones_like(ctx.vs30) * C['c11a']
    c11[(ctx.vs30 <= 300) & (ctx.vs30 > 200)] = C['c11b']
    c11[(ctx.vs30 <= 500) & (ctx.vs30 > 300)] = C['c11c']
    c11[(ctx.vs30 > 500)] = C['c11d']
    # Compute the Z1ref parameter
    tmp = (ctx.vs30**4 + 610**4) / (1360**4 + 610**4)
    z1ref = 1/1000. * np.exp(-7.67/4*np.log(tmp))
    # Return the fz1 parameter. The z1pt0 is converted from m (standard in OQ)
    # to km as indicated in the paper
    tmp = np.minimum(ctx.z1pt0/1000, np.ones_like(ctx.z1pt0)*2.0) + 0.01
    return c11 * np.log(tmp / (z1ref + 0.01))


def _get_stddevs(C, ctx):
    """
    Compute the standard deviations
    """
    # Set components of std
    tau = np.ones_like(ctx.mag) * C['s1']
    phi_s2s = np.ones_like(ctx.mag) * C['s3']
    phi_ss = np.ones_like(ctx.mag) * C['s5']
    above = ctx.mag > 6
    tau[above] = C['s2']
    phi_s2s[above] = C['s4']
    phi_ss[above] = C['s6']
    below = (ctx.mag > 4) & (ctx.mag <= 6)
    tau[below] = C['s1']+(C['s2']-C['s1'])/2.*(ctx.mag[below]-4)
    phi_s2s[below] = C['s3']+(C['s4']-C['s3'])/2.*(ctx.mag[below]-4)
    phi_ss[below] = C['s5']+(C['s6']-C['s5'])/2.*(ctx.mag[below]-4)

    # Collect the requested stds
    sigma = np.sqrt(tau**2+phi_s2s**2+phi_ss**2+C['c1a']**2)
    phi = np.sqrt(phi_s2s**2+phi_ss**2)

    sigma = np.ones_like(ctx.vs30) * sigma
    tau = np.ones_like(ctx.vs30) * tau
    phi = np.ones_like(ctx.vs30) * phi

    return sigma, tau, phi


def _get_nl_site_response(C, ctx, imt, ln_ir_outcrop, freq_nl, coeff_nl):
    """
    :param ln_ir_outcrop:
        The peak ground acceleration (PGA [g]) at rock outcrop
    :param freq_nl:
        Frequencies for the coefficients f3, f4 and f5 used to compute the
        non-linear term
    :param coeff_nl:
        A :class:`numpy.ndarray` instance of shape [# freq, 3] with the values
        of the coefficients f3, f4 and f5 which are used to compute the
        non-linear term
    """

    vsref = 760.0
    NSITES = len(ctx.vs30)
    NFREQS = coeff_nl.shape[0]

    # Updating the matrix with the coefficients. This has shape (number of
    # frequencies) x (number of coeffs) x (number of sites)
    coeff_nl = np.expand_dims(coeff_nl, 2)
    coeff_nl = np.repeat(coeff_nl, NSITES, 2)

    # Updating the matrix with Vs30 values. This has shape (number of
    # frequencies) x (number of sites)
    #vs30 = np.matlib.repmat(ctx.vs30, NFREQS, 1)
    #ln_ir_outcrop = np.matlib.repmat(ln_ir_outcrop, NFREQS, 1)
    vs30 = np.tile(ctx.vs30, (NFREQS, 1))
    ln_ir_outcrop = np.tile(ln_ir_outcrop, (NFREQS, 1))

    # Computing
    t1 = np.exp(coeff_nl[:, 2] * (np.minimum(vs30, vsref)-360.))
    t2 = np.exp(coeff_nl[:, 2] * (vsref-360.))
    f2 = coeff_nl[:, 1] * (t1 - t2)
    f3 = coeff_nl[:, 0]

    fnl_0 = f2 * np.log((np.exp(ln_ir_outcrop) + f3) / f3)
    idxs = np.argmin(fnl_0, axis=0)

    # Applying correction as described at page 2093 in BA18
    fnl = []
    for i, j in enumerate(idxs):
        fnl_0[j:, i] = min(fnl_0[:, i])
        tmp = np.interp(imt.frequency, freq_nl, fnl_0[:, i])
        fnl.append(tmp)
    return np.array(fnl)


def _get_dimunition_factor(ctx, imt):
    max_freq = 23.988321
    kappa = np.exp(-0.4*np.log(ctx.vs30/760)-3.5)
    D = np.exp(-np.pi * kappa * (imt.frequency - max_freq))
    return D


def _get_mean_linear(C, ctx):
    mean = (_magnitude_scaling(C, ctx) +
            _path_scaling(C, ctx) +
            _linear_site_response(C, ctx) +
            _ztor_scaling(C, ctx) +
            _normal_fault_effect(C, ctx) +
            _soil_depth_scaling(C, ctx))
    return mean


def _get_mean(self, C, ctx, imt):
    if imt.frequency >= 24.:
        im = EAS(23.988321)
        C = self.COEFFS[im]
        t1 = np.exp(_get_mean_linear(C, ctx))
        mean = np.log(_get_dimunition_factor(ctx, imt) * t1)
    else:
        mean = _get_mean_linear(C, ctx)
    return mean


class BaylessAbrahamson2018(GMPE):
    """
    Implements the Bayless and Abrahamson (2018, 2019) model. References:
    - Bayless, J., and N. A. Abrahamson (2018b). An empirical model for Fourier
    amplitude spectra using the NGA-West2 database, PEER Rept. No. 2018/07,
    Pacific Earthquake Engineering Research Center, University of California,
    Berkeley, California.
    - Bayless, J. and N.A. Abrahamson (2019). Summary of the BA18
    Ground-Motion Model for Fourier Amplitude Spectra for Crustal Earthquakes
    in California.  Bull. Seism. Soc. Am., 109(5): 2088–2105

    Disclaimer: The authors describe a smoothing technique that needs to be
    applied to the non linear component of the site response. We did not
    implement these smoothing functions in this initial versions since the
    match with the values in the verification tables is good even without it.
    """

    #: Supported tectonic region type is active shallow crust, see title!
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    #: Supported intensity measure types
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {EAS}

    #: Supported intensity measure component
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.HORIZONTAL

    #: Supported standard deviation types
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL,
        const.StdDev.INTER_EVENT,
        const.StdDev.INTRA_EVENT
    }

    #: Required site parameters
    REQUIRES_SITES_PARAMETERS = {'vs30', 'z1pt0'}

    #: Required rupture parameters
    REQUIRES_RUPTURE_PARAMETERS = {'mag', 'rake', 'ztor'}

    #: Required distance measures
    REQUIRES_DISTANCES = {'rrup'}

    def compute(self, ctx: np.recarray, imts, mean, sigma, tau, phi):
        freq_nl, coeff_nl = self.COEFFS.get_coeffs(['f3', 'f4', 'f5'])
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            ln_ir_outcrop = _get_ln_ir_outcrop(self, ctx)
            lin_component = _get_mean(self, C, ctx, imt)
            nl_component = _get_nl_site_response(C, ctx, imt, ln_ir_outcrop,
                                                 freq_nl, coeff_nl)
            mean[m] = (lin_component + nl_component)
            sigma[m], tau[m], phi[m] = _get_stddevs(C, ctx)

    TMP = """
IMT	c1	c2	c3	cn	cM	c4	c5	c6	chm	c7	c8	c9	c10	c11a	c11b	c11c	c11d	c1a	s1	s2	s3	s4	s5	s6	f3	f4	f5
EAS(0.100000)	 -3.650000	  1.270000	  3.485707	  1.552290	  6.798373	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000000	 -0.743232	 -0.031205	 -0.200000	  0.435485	  0.264594	  0.206825	  0.076926	  0.162700	  0.557087	  0.428744	  0.350000	  0.350000	  0.403571	  0.403571	  1.000000	 -0.000100	 -0.020000
EAS(0.102329)	 -3.647746	  1.270000	  3.481357	  1.562805	  6.786185	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000026	 -0.749511	 -0.031217	 -0.200000	  0.436075	  0.265782	  0.207919	  0.079101	  0.171104	  0.557087	  0.428744	  0.350891	  0.350891	  0.403859	  0.403859	  1.000000	 -0.000100	 -0.020000
EAS(0.104713)	 -3.645493	  1.270000	  3.476973	  1.573470	  6.774008	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000053	 -0.755784	 -0.031229	 -0.200000	  0.436648	  0.266948	  0.209011	  0.081277	  0.178404	  0.557087	  0.428744	  0.351782	  0.351782	  0.404147	  0.404147	  1.000000	 -0.000100	 -0.020000
EAS(0.107152)	 -3.643239	  1.270000	  3.472554	  1.584291	  6.761841	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000079	 -0.762052	 -0.031241	 -0.200000	  0.437203	  0.268091	  0.210101	  0.083452	  0.184616	  0.557087	  0.428744	  0.352673	  0.352673	  0.404435	  0.404435	  1.000000	 -0.000100	 -0.020000
EAS(0.109648)	 -3.640985	  1.270000	  3.468100	  1.595270	  6.749684	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000105	 -0.768316	 -0.031254	 -0.200000	  0.437741	  0.269211	  0.211187	  0.085626	  0.189759	  0.557087	  0.428744	  0.353564	  0.353564	  0.404723	  0.404723	  1.000000	 -0.000100	 -0.020000
EAS(0.112202)	 -3.638731	  1.270000	  3.463612	  1.606411	  6.737537	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000132	 -0.774576	 -0.031266	 -0.200000	  0.438261	  0.270308	  0.212270	  0.087800	  0.193858	  0.557087	  0.428744	  0.354455	  0.354455	  0.405011	  0.405011	  1.000000	 -0.000100	 -0.020000
EAS(0.114815)	 -3.636478	  1.270000	  3.459088	  1.617719	  6.725397	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000158	 -0.780835	 -0.031278	 -0.200000	  0.438765	  0.271382	  0.213350	  0.089971	  0.196938	  0.557087	  0.428744	  0.355346	  0.355346	  0.405299	  0.405299	  1.000000	 -0.000100	 -0.020000
EAS(0.117490)	 -3.634224	  1.270000	  3.454531	  1.629200	  6.713264	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000184	 -0.787094	 -0.031290	 -0.200000	  0.439252	  0.272433	  0.214427	  0.092141	  0.199027	  0.557087	  0.428744	  0.356237	  0.356237	  0.405587	  0.405587	  1.000000	 -0.000100	 -0.020000
EAS(0.120226)	 -3.631970	  1.270000	  3.449940	  1.640858	  6.701136	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000210	 -0.793355	 -0.031302	 -0.200000	  0.439722	  0.273462	  0.215501	  0.094308	  0.200151	  0.557087	  0.428744	  0.357128	  0.357128	  0.405875	  0.405875	  1.000000	 -0.000100	 -0.020000
EAS(0.123027)	 -3.629717	  1.270000	  3.445317	  1.652700	  6.689011	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000237	 -0.799619	 -0.031314	 -0.200000	  0.440175	  0.274469	  0.216571	  0.096471	  0.200337	  0.557087	  0.428744	  0.358019	  0.358019	  0.406163	  0.406163	  1.000000	 -0.000100	 -0.020000
EAS(0.125893)	 -3.627463	  1.270000	  3.440663	  1.664730	  6.676886	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000263	 -0.805887	 -0.031326	 -0.200000	  0.440613	  0.275455	  0.217638	  0.098631	  0.199607	  0.557087	  0.428744	  0.358910	  0.358910	  0.406451	  0.406451	  1.000000	 -0.000100	 -0.020000
EAS(0.128825)	 -3.625209	  1.270000	  3.435979	  1.676957	  6.664758	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000289	 -0.812160	 -0.031338	 -0.200000	  0.441035	  0.276420	  0.218701	  0.100786	  0.197976	  0.557087	  0.428744	  0.359801	  0.359801	  0.406739	  0.406739	  1.000000	 -0.000100	 -0.020000
EAS(0.131826)	 -3.622956	  1.270000	  3.431266	  1.689386	  6.652626	 -1.860000	  7.581800	  0.450000	  3.838000	 -0.000316	 -0.818440	 -0.031350	 -0.200000	  0.441442	  0.277365	  0.219760	  0.102937	  0.195453	  0.557087	  0.428744	  0.360692	  0.360692	  0.407027	  0.407027	  1.000000	 -0.000100	 -0.020000
EAS(0.134896)	 -3.620702	  1.270000	  3.426526	  1.702022	  6.640488	 -1.860000	  7.581800	  0.450000	  3.837989	 -0.000342	 -0.824727	 -0.031362	 -0.200000	  0.441834	  0.278292	  0.220816	  0.105082	  0.192038	  0.557087	  0.428744	  0.361583	  0.361583	  0.407315	  0.407315	  1.000000	 -0.000100	 -0.020000
EAS(0.138038)	 -3.618448	  1.270000	  3.421758	  1.714851	  6.628341	 -1.860000	  7.581800	  0.450000	  3.837966	 -0.000368	 -0.831022	 -0.031374	 -0.200000	  0.442211	  0.279199	  0.221868	  0.107220	  0.187709	  0.557087	  0.428744	  0.362474	  0.362474	  0.407603	  0.407603	  1.000000	 -0.000100	 -0.020000
EAS(0.141254)	 -3.616194	  1.270000	  3.416961	  1.727530	  6.616185	 -1.860000	  7.581800	  0.450000	  3.837943	 -0.000395	 -0.837326	 -0.031386	 -0.200000	  0.442574	  0.280089	  0.222916	  0.109353	  0.182415	  0.557087	  0.428744	  0.363365	  0.363365	  0.407891	  0.407891	  1.000000	 -0.000100	 -0.020000
EAS(0.144544)	 -3.613941	  1.270000	  3.412138	  1.740292	  6.604018	 -1.860000	  7.581800	  0.450000	  3.837920	 -0.000421	 -0.843641	 -0.031398	 -0.200000	  0.442923	  0.280962	  0.223959	  0.111478	  0.176050	  0.557087	  0.428744	  0.364256	  0.364256	  0.408179	  0.408179	  1.000000	 -0.000100	 -0.020000
EAS(0.147911)	 -3.611687	  1.270000	  3.407288	  1.753135	  6.591841	 -1.860000	  7.581800	  0.450000	  3.837898	 -0.000447	 -0.849965	 -0.031410	 -0.200000	  0.443259	  0.281818	  0.224998	  0.113596	  0.168417	  0.557087	  0.428744	  0.365147	  0.365147	  0.408466	  0.408466	  1.000000	 -0.000100	 -0.020000
EAS(0.151356)	 -3.609433	  1.270000	  3.402410	  1.766056	  6.579653	 -1.860000	  7.581800	  0.450000	  3.837875	 -0.000473	 -0.856302	 -0.031422	 -0.200000	  0.443581	  0.282658	  0.226032	  0.115706	  0.159204	  0.557087	  0.428744	  0.366038	  0.366038	  0.408754	  0.408754	  1.000000	 -0.000100	 -0.020000
EAS(0.154882)	 -3.607180	  1.270000	  3.397505	  1.779054	  6.567454	 -1.860000	  7.581800	  0.450000	  3.837852	 -0.000500	 -0.862649	 -0.031434	 -0.200000	  0.443892	  0.283483	  0.227060	  0.117808	  0.148232	  0.557087	  0.428744	  0.366929	  0.366929	  0.409042	  0.409042	  1.000000	 -0.000100	 -0.020000
EAS(0.158489)	 -3.604926	  1.270000	  3.392571	  1.792128	  6.555244	 -1.860000	  7.581800	  0.450000	  3.837829	 -0.000526	 -0.869009	 -0.031446	 -0.200000	  0.444190	  0.284293	  0.228083	  0.119903	  0.137220	  0.557087	  0.428744	  0.367820	  0.367820	  0.409330	  0.409330	  1.000000	 -0.000100	 -0.020000
EAS(0.162181)	 -3.602672	  1.270000	  3.387609	  1.805279	  6.543025	 -1.860000	  7.581800	  0.450000	  3.837807	 -0.000552	 -0.875381	 -0.031458	 -0.200000	  0.444476	  0.285089	  0.229099	  0.121989	  0.126377	  0.557087	  0.428744	  0.368711	  0.368711	  0.409618	  0.409618	  1.000000	 -0.000100	 -0.020000
EAS(0.165959)	 -3.600419	  1.270000	  3.382616	  1.818506	  6.530798	 -1.860000	  7.581800	  0.450000	  3.837784	 -0.000579	 -0.881766	 -0.031470	 -0.200000	  0.444750	  0.285873	  0.230109	  0.124068	  0.115998	  0.557087	  0.428744	  0.369602	  0.369602	  0.409906	  0.409906	  1.000000	 -0.000100	 -0.020000
EAS(0.169824)	 -3.598165	  1.270000	  3.377593	  1.831813	  6.518563	 -1.860000	  7.581800	  0.450000	  3.837761	 -0.000605	 -0.888161	 -0.031482	 -0.200000	  0.445013	  0.286645	  0.231112	  0.126139	  0.106305	  0.557087	  0.428744	  0.370493	  0.370493	  0.410194	  0.410194	  1.000000	 -0.000100	 -0.020000
EAS(0.173780)	 -3.595911	  1.270000	  3.372537	  1.845204	  6.506323	 -1.860000	  7.581800	  0.450000	  3.837739	 -0.000631	 -0.894568	 -0.031494	 -0.200000	  0.445264	  0.287406	  0.232107	  0.128202	  0.097432	  0.557087	  0.428744	  0.371384	  0.371384	  0.410482	  0.410482	  1.000000	 -0.000100	 -0.020000
EAS(0.177828)	 -3.593657	  1.270000	  3.367447	  1.858689	  6.494080	 -1.860000	  7.581800	  0.450000	  3.837716	 -0.000658	 -0.900984	 -0.031506	 -0.200000	  0.445506	  0.288157	  0.233095	  0.130259	  0.089408	  0.557087	  0.428744	  0.372275	  0.372275	  0.410770	  0.410770	  1.000000	 -0.000100	 -0.020000
EAS(0.181970)	 -3.591404	  1.270000	  3.362321	  1.872279	  6.481837	 -1.860000	  7.581800	  0.450000	  3.837693	 -0.000684	 -0.907408	 -0.031518	 -0.200000	  0.445736	  0.288899	  0.234075	  0.132310	  0.082191	  0.557087	  0.428744	  0.373166	  0.373166	  0.411058	  0.411058	  1.000000	 -0.000100	 -0.020000
EAS(0.186209)	 -3.589150	  1.270000	  3.357157	  1.885993	  6.469595	 -1.860000	  7.581800	  0.450000	  3.837670	 -0.000710	 -0.913838	 -0.031530	 -0.200000	  0.445958	  0.289633	  0.235045	  0.134355	  0.075719	  0.557087	  0.428744	  0.374057	  0.374057	  0.411346	  0.411346	  1.000000	 -0.000100	 -0.020000
EAS(0.190546)	 -3.586896	  1.270000	  3.351952	  1.899852	  6.457360	 -1.860000	  7.581800	  0.450000	  3.837648	 -0.000736	 -0.920273	 -0.031542	 -0.200000	  0.446170	  0.290360	  0.236007	  0.136396	  0.069960	  0.557087	  0.428744	  0.374948	  0.374948	  0.411634	  0.411634	  1.000000	 -0.000100	 -0.020000
EAS(0.194984)	 -3.584643	  1.270000	  3.346704	  1.913881	  6.445135	 -1.860000	  7.581800	  0.450000	  3.837625	 -0.000763	 -0.926707	 -0.031554	 -0.200000	  0.446373	  0.291079	  0.236960	  0.138434	  0.064894	  0.557087	  0.428744	  0.375839	  0.375839	  0.411922	  0.411922	  1.000000	 -0.000100	 -0.020000
EAS(0.199526)	 -3.582389	  1.270000	  3.341409	  1.928109	  6.432925	 -1.860000	  7.581800	  0.450000	  3.837602	 -0.000789	 -0.933139	 -0.031565	 -0.200000	  0.446567	  0.291792	  0.237904	  0.140470	  0.060432	  0.557087	  0.428744	  0.376730	  0.376730	  0.412210	  0.412210	  1.000000	 -0.000100	 -0.020000
EAS(0.204174)	 -3.580135	  1.270000	  3.336062	  1.942562	  6.420737	 -1.860000	  7.581800	  0.450009	  3.837535	 -0.000815	 -0.939562	 -0.031576	 -0.200000	  0.446753	  0.292498	  0.238839	  0.142506	  0.056355	  0.557087	  0.428744	  0.377621	  0.377621	  0.412498	  0.412498	  1.000000	 -0.000100	 -0.020000
EAS(0.208930)	 -3.577882	  1.270000	  3.330660	  1.957265	  6.408579	 -1.860000	  7.581800	  0.450020	  3.837463	 -0.000842	 -0.945971	 -0.031584	 -0.200000	  0.446929	  0.293196	  0.239765	  0.144546	  0.052425	  0.557087	  0.428744	  0.378512	  0.378512	  0.412786	  0.412786	  1.000000	 -0.000100	 -0.020000
EAS(0.213796)	 -3.575628	  1.270000	  3.325194	  1.972236	  6.396460	 -1.860000	  7.581800	  0.450030	  3.837391	 -0.000868	 -0.952357	 -0.031591	 -0.200000	  0.447093	  0.293886	  0.240683	  0.146591	  0.048448	  0.557087	  0.428744	  0.379403	  0.379403	  0.413074	  0.413074	  1.000000	 -0.000100	 -0.020000
EAS(0.218776)	 -3.573374	  1.270000	  3.319656	  1.987482	  6.384395	 -1.860000	  7.581800	  0.450040	  3.837319	 -0.000895	 -0.958710	 -0.031594	 -0.200000	  0.447242	  0.294562	  0.241593	  0.148647	  0.044314	  0.557087	  0.428744	  0.380294	  0.380294	  0.413362	  0.413362	  1.000000	 -0.000100	 -0.020000
EAS(0.223872)	 -3.571120	  1.270000	  3.314036	  2.003003	  6.372400	 -1.860000	  7.581800	  0.450051	  3.837246	 -0.000921	 -0.965017	 -0.031592	 -0.200000	  0.447369	  0.295218	  0.242495	  0.150717	  0.040002	  0.557087	  0.428744	  0.381185	  0.381185	  0.413650	  0.413650	  1.000000	 -0.000100	 -0.020000
EAS(0.229087)	 -3.568867	  1.270000	  3.308317	  2.018786	  6.360501	 -1.860000	  7.581800	  0.450061	  3.837174	 -0.000948	 -0.971260	 -0.031584	 -0.200000	  0.447464	  0.295840	  0.243388	  0.152807	  0.035561	  0.557087	  0.428744	  0.382076	  0.382076	  0.413938	  0.413938	  1.000000	 -0.000100	 -0.020000
EAS(0.234423)	 -3.566613	  1.270000	  3.302481	  2.034811	  6.348728	 -1.860000	  7.581800	  0.450071	  3.837102	 -0.000975	 -0.977419	 -0.031569	 -0.200000	  0.447509	  0.296404	  0.244272	  0.154923	  0.031081	  0.557087	  0.428744	  0.382967	  0.382967	  0.414226	  0.414226	  1.000000	 -0.000100	 -0.020000
EAS(0.239883)	 -3.564359	  1.270000	  3.296505	  2.051047	  6.337118	 -1.860000	  7.581800	  0.450081	  3.837030	 -0.001002	 -0.983476	 -0.031544	 -0.200000	  0.447477	  0.296875	  0.245143	  0.157071	  0.026665	  0.557087	  0.428744	  0.383858	  0.383858	  0.414514	  0.414514	  1.000000	 -0.000100	 -0.020000
EAS(0.245471)	 -3.562106	  1.270000	  3.290368	  2.067466	  6.325703	 -1.860000	  7.581800	  0.450092	  3.836957	 -0.001029	 -0.989424	 -0.031508	 -0.200000	  0.447337	  0.297205	  0.245996	  0.159251	  0.022398	  0.557087	  0.428744	  0.384749	  0.384749	  0.414802	  0.414802	  1.000000	 -0.000100	 -0.020000
EAS(0.251189)	 -3.559852	  1.270000	  3.284245	  2.084234	  6.314195	 -1.860000	  7.581800	  0.450100	  3.836887	 -0.001057	 -0.995535	 -0.031459	 -0.200000	  0.446390	  0.296806	  0.246440	  0.161109	  0.018400	  0.557087	  0.428744	  0.385640	  0.385640	  0.415090	  0.415090	  1.000000	 -0.000100	 -0.020000
EAS(0.257040)	 -3.557598	  1.270000	  3.278072	  2.101274	  6.302698	 -1.860000	  7.581800	  0.450100	  3.836823	 -0.001085	 -1.001593	 -0.031395	 -0.200000	  0.445297	  0.296295	  0.246824	  0.162928	  0.014828	  0.557087	  0.428744	  0.386531	  0.386531	  0.415377	  0.415377	  1.000000	 -0.000100	 -0.020000
EAS(0.263027)	 -3.555345	  1.270000	  3.271860	  2.118586	  6.291195	 -1.860000	  7.581800	  0.450100	  3.836759	 -0.001114	 -1.007597	 -0.031314	 -0.200000	  0.444055	  0.295672	  0.247148	  0.164701	  0.011802	  0.557087	  0.428744	  0.387422	  0.387423	  0.415665	  0.415665	  1.000000	 -0.000100	 -0.020000
EAS(0.269153)	 -3.553091	  1.270000	  3.265620	  2.136168	  6.279663	 -1.860000	  7.581800	  0.450100	  3.836695	 -0.001143	 -1.013548	 -0.031216	 -0.200000	  0.442664	  0.294939	  0.247410	  0.166419	  0.009332	  0.557087	  0.428744	  0.388313	  0.388315	  0.415953	  0.415953	  1.000000	 -0.000100	 -0.020000
EAS(0.275423)	 -3.550837	  1.270000	  3.259367	  2.154022	  6.268081	 -1.860000	  7.581800	  0.450100	  3.836631	 -0.001173	 -1.019444	 -0.031097	 -0.200000	  0.441120	  0.294096	  0.247610	  0.168075	  0.007374	  0.557087	  0.428744	  0.389204	  0.389210	  0.416241	  0.416241	  1.000000	 -0.000101	 -0.020000
EAS(0.281838)	 -3.548583	  1.270000	  3.253112	  2.172148	  6.256428	 -1.860000	  7.581800	  0.450100	  3.836567	 -0.001204	 -1.025283	 -0.030956	 -0.200000	  0.439421	  0.293142	  0.247747	  0.169662	  0.005851	  0.557087	  0.428744	  0.390095	  0.390106	  0.416529	  0.416529	  1.000000	 -0.000104	 -0.020000
EAS(0.288403)	 -3.546330	  1.270000	  3.246867	  2.190547	  6.244686	 -1.860000	  7.581800	  0.450100	  3.836503	 -0.001235	 -1.031063	 -0.030793	 -0.200000	  0.437563	  0.292078	  0.247821	  0.171173	  0.004670	  0.557087	  0.428744	  0.390986	  0.391006	  0.416817	  0.416817	  1.000000	 -0.000108	 -0.020000
EAS(0.295121)	 -3.544076	  1.270000	  3.240639	  2.209219	  6.232840	 -1.860000	  7.581800	  0.450100	  3.836439	 -0.001267	 -1.036779	 -0.030606	 -0.200000	  0.435544	  0.290904	  0.247830	  0.172604	  0.003770	  0.557087	  0.428744	  0.391877	  0.391909	  0.417105	  0.417105	  1.000000	 -0.000115	 -0.020000
EAS(0.301995)	 -3.541822	  1.270000	  3.234433	  2.228162	  6.220881	 -1.860000	  7.581800	  0.450100	  3.836375	 -0.001300	 -1.042426	 -0.030393	 -0.200000	  0.433360	  0.289619	  0.247774	  0.173950	  0.003132	  0.557087	  0.428744	  0.392768	  0.392817	  0.417393	  0.417393	  1.000000	 -0.000125	 -0.020000
EAS(0.309030)	 -3.539569	  1.270000	  3.228251	  2.247373	  6.208807	 -1.860000	  7.581800	  0.450100	  3.836311	 -0.001334	 -1.047999	 -0.030153	 -0.200000	  0.431011	  0.288223	  0.247650	  0.175206	  0.002769	  0.557087	  0.428744	  0.393659	  0.393730	  0.417681	  0.417681	  1.000000	 -0.000140	 -0.020000
EAS(0.316228)	 -3.537315	  1.270000	  3.222090	  2.266844	  6.196622	 -1.860000	  7.581800	  0.450100	  3.836246	 -0.001368	 -1.053496	 -0.029886	 -0.200000	  0.428495	  0.286715	  0.247457	  0.176371	  0.002709	  0.557087	  0.428744	  0.394550	  0.394648	  0.417969	  0.417969	  1.000000	 -0.000160	 -0.020000
EAS(0.323594)	 -3.535061	  1.270000	  3.215942	  2.286567	  6.184337	 -1.860000	  7.581800	  0.450100	  3.836182	 -0.001404	 -1.058916	 -0.029591	 -0.200000	  0.425813	  0.285093	  0.247191	  0.177442	  0.002939	  0.557087	  0.428744	  0.395440	  0.395570	  0.418257	  0.418257	  1.000000	 -0.000185	 -0.020000
EAS(0.331131)	 -3.532807	  1.270000	  3.209797	  2.306530	  6.171970	 -1.860000	  7.581800	  0.450100	  3.836118	 -0.001440	 -1.064256	 -0.029267	 -0.200000	  0.422970	  0.283359	  0.246850	  0.178416	  0.003388	  0.557087	  0.428744	  0.396331	  0.396497	  0.418545	  0.418545	  1.000000	 -0.000216	 -0.020000
EAS(0.338844)	 -3.530554	  1.270000	  3.203643	  2.326725	  6.159537	 -1.860000	  7.581800	  0.450112	  3.835975	 -0.001477	 -1.069512	 -0.028914	 -0.200000	  0.419972	  0.281515	  0.246433	  0.179294	  0.003931	  0.557087	  0.428744	  0.397220	  0.397429	  0.418833	  0.418833	  1.000000	 -0.000253	 -0.020000
EAS(0.346737)	 -3.528300	  1.270000	  3.197469	  2.347145	  6.147058	 -1.860000	  7.581800	  0.450129	  3.835799	 -0.001516	 -1.074678	 -0.028530	 -0.200000	  0.416828	  0.279565	  0.245937	  0.180076	  0.004412	  0.557087	  0.428744	  0.398108	  0.398363	  0.419121	  0.419121	  1.000000	 -0.000298	 -0.020000
EAS(0.354813)	 -3.526046	  1.270000	  3.191262	  2.367787	  6.134551	 -1.860000	  7.581800	  0.450146	  3.835623	 -0.001555	 -1.079748	 -0.028117	 -0.200000	  0.413544	  0.277516	  0.245360	  0.180761	  0.004666	  0.557087	  0.428744	  0.398993	  0.399300	  0.419409	  0.419409	  1.000000	 -0.000352	 -0.020000
EAS(0.363078)	 -3.523793	  1.270000	  3.185011	  2.388649	  6.122035	 -1.860000	  7.581800	  0.450163	  3.835446	 -0.001595	 -1.084716	 -0.027674	 -0.200000	  0.410128	  0.275372	  0.244700	  0.181351	  0.004536	  0.557087	  0.428744	  0.399874	  0.400238	  0.419697	  0.419697	  1.000000	 -0.000414	 -0.020000
EAS(0.371535)	 -3.521539	  1.270000	  3.178704	  2.409733	  6.109527	 -1.860000	  7.581800	  0.450180	  3.835270	 -0.001636	 -1.089577	 -0.027201	 -0.200000	  0.406586	  0.273139	  0.243957	  0.181850	  0.003910	  0.557087	  0.428744	  0.400750	  0.401174	  0.419985	  0.419985	  1.000000	 -0.000485	 -0.020000
EAS(0.380189)	 -3.519285	  1.270000	  3.172331	  2.431039	  6.097043	 -1.860000	  7.581800	  0.450197	  3.835094	 -0.001678	 -1.094325	 -0.026697	 -0.200000	  0.402925	  0.270823	  0.243130	  0.182260	  0.002756	  0.557086	  0.428744	  0.401620	  0.402109	  0.420273	  0.420273	  1.000000	 -0.000567	 -0.020000
EAS(0.389045)	 -3.517032	  1.270000	  3.165884	  2.452573	  6.084595	 -1.860000	  7.581800	  0.450214	  3.834918	 -0.001721	 -1.098950	 -0.026163	 -0.200000	  0.399154	  0.268430	  0.242219	  0.182586	  0.001114	  0.557084	  0.428742	  0.402481	  0.403039	  0.420561	  0.420561	  1.000000	 -0.000658	 -0.020000
EAS(0.398107)	 -3.514778	  1.270000	  3.159356	  2.474337	  6.072195	 -1.860000	  7.581800	  0.450231	  3.834742	 -0.001764	 -1.103447	 -0.025599	 -0.200000	  0.395277	  0.265964	  0.241226	  0.182831	 -0.000939	  0.557080	  0.428737	  0.403333	  0.403962	  0.420849	  0.420849	  1.000000	 -0.000760	 -0.020000
EAS(0.407380)	 -3.512524	  1.270000	  3.152739	  2.496332	  6.059850	 -1.860000	  7.581800	  0.450248	  3.834566	 -0.001809	 -1.107805	 -0.025006	 -0.200000	  0.391301	  0.263432	  0.240151	  0.183001	 -0.003337	  0.557070	  0.428726	  0.404175	  0.404877	  0.421137	  0.421137	  0.999999	 -0.000872	 -0.020000
EAS(0.416869)	 -3.510270	  1.270000	  3.146030	  2.518558	  6.047570	 -1.860000	  7.581800	  0.450265	  3.834390	 -0.001854	 -1.112013	 -0.024384	 -0.200000	  0.387230	  0.260835	  0.238995	  0.183100	 -0.005990	  0.557053	  0.428707	  0.405003	  0.405781	  0.421425	  0.421425	  0.999990	 -0.000994	 -0.020000
EAS(0.426580)	 -3.508017	  1.270000	  3.139224	  2.541013	  6.035358	 -1.860000	  7.581800	  0.450282	  3.834214	 -0.001901	 -1.116063	 -0.023734	 -0.200000	  0.383067	  0.258174	  0.237759	  0.183132	 -0.008784	  0.557025	  0.428676	  0.405818	  0.406673	  0.421713	  0.421713	  0.999959	 -0.001125	 -0.019999
EAS(0.436516)	 -3.505763	  1.270000	  3.132318	  2.563694	  6.023217	 -1.860000	  7.581800	  0.450300	  3.834038	 -0.001948	 -1.119944	 -0.023056	 -0.200000	  0.378815	  0.255450	  0.236445	  0.183099	 -0.011564	  0.556981	  0.428627	  0.406617	  0.407549	  0.422001	  0.422001	  0.999882	 -0.001266	 -0.019999
EAS(0.446684)	 -3.503509	  1.270000	  3.125313	  2.586595	  6.011148	 -1.860000	  7.581800	  0.450317	  3.833862	 -0.001996	 -1.123646	 -0.022352	 -0.200000	  0.374473	  0.252660	  0.235055	  0.183007	 -0.014176	  0.556917	  0.428557	  0.407399	  0.408409	  0.422289	  0.422289	  0.999729	 -0.001416	 -0.019997
EAS(0.457088)	 -3.501256	  1.270000	  3.118208	  2.609709	  5.999150	 -1.860000	  7.581800	  0.450334	  3.833686	 -0.002045	 -1.127159	 -0.021623	 -0.200000	  0.370042	  0.249806	  0.233591	  0.182859	 -0.016508	  0.556829	  0.428459	  0.408164	  0.409250	  0.422576	  0.422576	  0.999464	 -0.001576	 -0.019996
EAS(0.467735)	 -3.499002	  1.270000	  3.111000	  2.633025	  5.987224	 -1.860000	  7.581800	  0.450351	  3.833510	 -0.002094	 -1.130475	 -0.020870	 -0.200000	  0.365519	  0.246887	  0.232054	  0.182658	 -0.018519	  0.556712	  0.428329	  0.408910	  0.410070	  0.422864	  0.422864	  0.999045	 -0.001749	 -0.019993
EAS(0.478630)	 -3.496748	  1.270000	  3.103688	  2.656528	  5.975375	 -1.860000	  7.581800	  0.450368	  3.833334	 -0.002145	 -1.133585	 -0.020095	 -0.200000	  0.360903	  0.243903	  0.230449	  0.182408	 -0.020199	  0.556559	  0.428160	  0.409636	  0.410867	  0.423152	  0.423152	  0.998431	 -0.001937	 -0.019989
EAS(0.489779)	 -3.494495	  1.270000	  3.096267	  2.680208	  5.963605	 -1.860000	  7.581800	  0.450385	  3.833158	 -0.002196	 -1.136482	 -0.019301	 -0.200000	  0.356194	  0.240857	  0.228778	  0.182110	 -0.021532	  0.556366	  0.427947	  0.410342	  0.411640	  0.423440	  0.423440	  0.997579	 -0.002141	 -0.019983
EAS(0.501187)	 -3.492241	  1.270000	  3.088737	  2.704052	  5.951918	 -1.860000	  7.581799	  0.450402	  3.832962	 -0.002247	 -1.139158	 -0.018488	 -0.200000	  0.351394	  0.237750	  0.227042	  0.181768	 -0.022485	  0.556126	  0.427684	  0.411026	  0.412387	  0.423728	  0.423728	  0.996449	 -0.002363	 -0.019975
EAS(0.512861)	 -3.489987	  1.270000	  3.081094	  2.728054	  5.940316	 -1.860000	  7.581791	  0.450426	  3.832594	 -0.002300	 -1.141606	 -0.017661	 -0.200000	  0.346506	  0.234587	  0.225245	  0.181380	 -0.023031	  0.555834	  0.427366	  0.411689	  0.413108	  0.424016	  0.424016	  0.995004	 -0.002605	 -0.019964
EAS(0.524807)	 -3.487733	  1.270000	  3.073335	  2.752211	  5.928802	 -1.860000	  7.581783	  0.450450	  3.832226	 -0.002353	 -1.143817	 -0.016820	 -0.200000	  0.341534	  0.231371	  0.223386	  0.180949	 -0.023139	  0.555485	  0.426987	  0.412330	  0.413800	  0.424304	  0.424304	  0.993210	 -0.002868	 -0.019951
EAS(0.537032)	 -3.485480	  1.270000	  3.065459	  2.776524	  5.917380	 -1.860000	  7.581775	  0.450475	  3.831858	 -0.002407	 -1.145788	 -0.015969	 -0.200000	  0.336483	  0.228106	  0.221469	  0.180474	 -0.022801	  0.555072	  0.426543	  0.412948	  0.414463	  0.424590	  0.424590	  0.991035	 -0.003154	 -0.019935
EAS(0.549541)	 -3.483226	  1.270000	  3.057461	  2.801000	  5.906056	 -1.860000	  7.581767	  0.450499	  3.831489	 -0.002461	 -1.147513	 -0.015111	 -0.200000	  0.331353	  0.224794	  0.219493	  0.179957	 -0.022069	  0.554591	  0.426030	  0.413544	  0.415096	  0.424872	  0.424872	  0.988456	 -0.003463	 -0.019915
EAS(0.562341)	 -3.480972	  1.270000	  3.049337	  2.825649	  5.894835	 -1.860000	  7.581759	  0.450523	  3.831121	 -0.002517	 -1.148988	 -0.014247	 -0.200000	  0.326144	  0.221435	  0.217458	  0.179400	 -0.021014	  0.554037	  0.425444	  0.414117	  0.415699	  0.425147	  0.425148	  0.985448	 -0.003797	 -0.019892
EAS(0.575440)	 -3.478719	  1.270000	  3.041083	  2.850490	  5.883720	 -1.860000	  7.581751	  0.450547	  3.830753	 -0.002573	 -1.150213	 -0.013382	 -0.200000	  0.320855	  0.218030	  0.215365	  0.178803	 -0.019679	  0.553404	  0.424781	  0.414668	  0.416271	  0.425412	  0.425412	  0.981995	 -0.004156	 -0.019865
EAS(0.588844)	 -3.476465	  1.270000	  3.032699	  2.875550	  5.872714	 -1.860000	  7.581743	  0.450571	  3.830385	 -0.002630	 -1.151189	 -0.012518	 -0.200000	  0.315485	  0.214577	  0.213212	  0.178167	 -0.018094	  0.552687	  0.424038	  0.415195	  0.416818	  0.425665	  0.425663	  0.978080	 -0.004541	 -0.019834
EAS(0.602560)	 -3.474211	  1.270000	  3.024187	  2.900864	  5.861812	 -1.860000	  7.581735	  0.450595	  3.830017	 -0.002688	 -1.151922	 -0.011657	 -0.200000	  0.310033	  0.211073	  0.210999	  0.177491	 -0.016316	  0.551884	  0.423213	  0.415699	  0.417344	  0.425903	  0.425895	  0.973690	 -0.004951	 -0.019800
EAS(0.616595)	 -3.471958	  1.270000	  3.015549	  2.926474	  5.851009	 -1.860000	  7.581727	  0.450619	  3.829648	 -0.002747	 -1.152420	 -0.010803	 -0.200000	  0.304498	  0.207515	  0.208727	  0.176775	 -0.014459	  0.550988	  0.422305	  0.416181	  0.417857	  0.426125	  0.426104	  0.968812	 -0.005386	 -0.019762
EAS(0.630957)	 -3.469704	  1.270000	  3.006791	  2.952431	  5.840294	 -1.860000	  7.581719	  0.450643	  3.829280	 -0.002806	 -1.152693	 -0.009956	 -0.200000	  0.298882	  0.203901	  0.206396	  0.176017	 -0.012667	  0.549997	  0.421311	  0.416639	  0.418362	  0.426329	  0.426288	  0.963436	 -0.005845	 -0.019720
EAS(0.645654)	 -3.467450	  1.270000	  2.997920	  2.978786	  5.829656	 -1.860000	  7.581711	  0.450667	  3.828912	 -0.002867	 -1.152749	 -0.009121	 -0.200000	  0.293191	  0.200230	  0.204010	  0.175220	 -0.011068	  0.548905	  0.420231	  0.417075	  0.418864	  0.426515	  0.426442	  0.957549	 -0.006328	 -0.019674
EAS(0.660693)	 -3.465196	  1.270000	  2.988941	  3.005598	  5.819085	 -1.860000	  7.581703	  0.450691	  3.828544	 -0.002928	 -1.152596	 -0.008297	 -0.200000	  0.287429	  0.196499	  0.201569	  0.174382	 -0.009741	  0.547711	  0.419064	  0.417487	  0.419365	  0.426683	  0.426564	  0.951141	 -0.006836	 -0.019625
EAS(0.676083)	 -3.462943	  1.270000	  2.979861	  3.032922	  5.808568	 -1.860000	  7.581690	  0.450735	  3.827916	 -0.002990	 -1.152244	 -0.007488	 -0.200000	  0.281605	  0.192710	  0.199078	  0.173505	 -0.008732	  0.546410	  0.417809	  0.417877	  0.419865	  0.426833	  0.426651	  0.944203	 -0.007370	 -0.019572
EAS(0.691831)	 -3.460689	  1.270000	  2.970685	  3.060815	  5.798097	 -1.860000	  7.581673	  0.450791	  3.827121	 -0.003053	 -1.151699	 -0.006694	 -0.200000	  0.275728	  0.188865	  0.196538	  0.172590	 -0.008075	  0.545000	  0.416468	  0.418245	  0.420362	  0.426967	  0.426700	  0.936727	 -0.007929	 -0.019515
EAS(0.707946)	 -3.458435	  1.270000	  2.961417	  3.089331	  5.787665	 -1.860000	  7.581656	  0.450848	  3.826326	 -0.003116	 -1.150967	 -0.005915	 -0.200000	  0.269809	  0.184968	  0.193956	  0.171638	 -0.007797	  0.543476	  0.415041	  0.418589	  0.420852	  0.427085	  0.426710	  0.928709	 -0.008515	 -0.019454
EAS(0.724436)	 -3.456182	  1.270000	  2.952060	  3.118525	  5.777264	 -1.860000	  7.581639	  0.450905	  3.825531	 -0.003180	 -1.150051	 -0.005154	 -0.200000	  0.263857	  0.181024	  0.191332	  0.170650	 -0.007920	  0.541836	  0.413528	  0.418911	  0.421331	  0.427190	  0.426679	  0.920150	 -0.009131	 -0.019389
EAS(0.741310)	 -3.453928	  1.270000	  2.942615	  3.148451	  5.766891	 -1.860000	  7.581621	  0.450962	  3.824736	 -0.003245	 -1.148952	 -0.004409	 -0.200000	  0.257886	  0.177039	  0.188669	  0.169625	 -0.008437	  0.540076	  0.411930	  0.419202	  0.421791	  0.427285	  0.426606	  0.911053	 -0.009779	 -0.019319
EAS(0.758578)	 -3.451674	  1.270000	  2.933083	  3.179161	  5.756545	 -1.860000	  7.581604	  0.451019	  3.823941	 -0.003311	 -1.147671	 -0.003681	 -0.200000	  0.251912	  0.173019	  0.185970	  0.168562	 -0.009305	  0.538195	  0.410250	  0.419453	  0.422225	  0.427370	  0.426490	  0.901429	 -0.010465	 -0.019246
EAS(0.776247)	 -3.449421	  1.270000	  2.923464	  3.210705	  5.746224	 -1.860000	  7.581587	  0.451075	  3.823145	 -0.003377	 -1.146210	 -0.002970	 -0.200000	  0.245958	  0.168973	  0.183238	  0.167461	 -0.010456	  0.536189	  0.408489	  0.419651	  0.422624	  0.427450	  0.426333	  0.891292	 -0.011196	 -0.019167
EAS(0.794328)	 -3.447167	  1.270000	  2.913755	  3.243128	  5.735933	 -1.860000	  7.581570	  0.451132	  3.822350	 -0.003443	 -1.144570	 -0.002275	 -0.200000	  0.240047	  0.164906	  0.180475	  0.166322	 -0.011800	  0.534055	  0.406649	  0.419786	  0.422981	  0.427525	  0.426132	  0.880660	 -0.011978	 -0.019082
EAS(0.812830)	 -3.444913	  1.270000	  2.903952	  3.276473	  5.725675	 -1.860000	  7.581553	  0.451189	  3.821555	 -0.003510	 -1.142751	 -0.001597	 -0.200000	  0.234207	  0.160828	  0.177683	  0.165146	 -0.013254	  0.531793	  0.404732	  0.419849	  0.423286	  0.427600	  0.425889	  0.869558	 -0.012818	 -0.018993
EAS(0.831764)	 -3.442659	  1.270000	  2.894051	  3.310782	  5.715457	 -1.860000	  7.581536	  0.451246	  3.820760	 -0.003578	 -1.140753	 -0.000935	 -0.200000	  0.228463	  0.156743	  0.174864	  0.163935	 -0.014752	  0.529400	  0.402742	  0.419834	  0.423531	  0.427676	  0.425604	  0.858011	 -0.013725	 -0.018897
EAS(0.851138)	 -3.440406	  1.270000	  2.884047	  3.346091	  5.705286	 -1.860000	  7.581519	  0.451302	  3.819965	 -0.003646	 -1.138577	 -0.000288	 -0.200000	  0.222838	  0.152658	  0.172021	  0.162691	 -0.016237	  0.526875	  0.400683	  0.419739	  0.423710	  0.427756	  0.425278	  0.846048	 -0.014705	 -0.018794
EAS(0.870964)	 -3.438152	  1.270000	  2.873937	  3.382435	  5.695167	 -1.860000	  7.581502	  0.451359	  3.819170	 -0.003714	 -1.136224	  0.000344	 -0.200000	  0.217350	  0.148575	  0.169155	  0.161415	 -0.017662	  0.524218	  0.398557	  0.419564	  0.423814	  0.427841	  0.424911	  0.833698	 -0.015763	 -0.018685
EAS(0.891251)	 -3.435898	  1.270000	  2.863715	  3.419845	  5.685108	 -1.860000	  7.581485	  0.451416	  3.818375	 -0.003783	 -1.133698	  0.000962	 -0.200000	  0.212013	  0.144499	  0.166269	  0.160112	 -0.018975	  0.521430	  0.396370	  0.419312	  0.423837	  0.427935	  0.424505	  0.820985	 -0.016904	 -0.018568
EAS(0.912011)	 -3.433645	  1.270000	  2.853378	  3.458355	  5.675113	 -1.860000	  7.581468	  0.451473	  3.817580	 -0.003852	 -1.130996	  0.001567	 -0.200000	  0.206838	  0.140432	  0.163366	  0.158781	 -0.020140	  0.518510	  0.394127	  0.418991	  0.423775	  0.428039	  0.424060	  0.807931	 -0.018131	 -0.018445
EAS(0.933254)	 -3.431391	  1.270000	  2.842925	  3.497994	  5.665183	 -1.860000	  7.581451	  0.451530	  3.816785	 -0.003921	 -1.128118	  0.002160	 -0.200000	  0.201834	  0.136378	  0.160446	  0.157425	 -0.021147	  0.515461	  0.391832	  0.418607	  0.423622	  0.428155	  0.423577	  0.794551	 -0.019446	 -0.018313
EAS(0.954992)	 -3.429137	  1.270000	  2.832354	  3.538793	  5.655321	 -1.860000	  7.581434	  0.451586	  3.815990	 -0.003990	 -1.125059	  0.002741	 -0.200000	  0.197005	  0.132342	  0.157510	  0.156046	 -0.021995	  0.512285	  0.389493	  0.418172	  0.423375	  0.428284	  0.423057	  0.780852	 -0.020853	 -0.018173
EAS(0.977237)	 -3.426883	  1.270000	  2.821664	  3.580782	  5.645526	 -1.860000	  7.581417	  0.451643	  3.815195	 -0.004060	 -1.121815	  0.003312	 -0.200000	  0.192355	  0.128327	  0.154558	  0.154643	 -0.022669	  0.508984	  0.387115	  0.417695	  0.423031	  0.428428	  0.422501	  0.766833	 -0.022354	 -0.018026
EAS(1.000000)	 -3.424630	  1.270000	  2.810855	  3.623988	  5.635799	 -1.860000	  7.581400	  0.451700	  3.814400	 -0.004130	 -1.118379	  0.003873	 -0.200000	  0.187886	  0.124336	  0.151591	  0.153219	 -0.023136	  0.505564	  0.384704	  0.417189	  0.422589	  0.428586	  0.421910	  0.752487	 -0.023958	 -0.017870
EAS(1.023293)	 -3.422376	  1.270000	  2.799927	  3.668438	  5.626139	 -1.860000	  7.581352	  0.451812	  3.812775	 -0.004200	 -1.114744	  0.004426	 -0.200000	  0.183598	  0.120373	  0.148607	  0.151774	 -0.023389	  0.502029	  0.382269	  0.416663	  0.422048	  0.428761	  0.421284	  0.737807	 -0.025672	 -0.017706
EAS(1.047129)	 -3.420122	  1.270000	  2.788879	  3.714156	  5.616547	 -1.860000	  7.581304	  0.451924	  3.811150	 -0.004270	 -1.110901	  0.004972	 -0.200000	  0.179490	  0.116440	  0.145606	  0.150307	 -0.023457	  0.498384	  0.379816	  0.416128	  0.421406	  0.428952	  0.420623	  0.722785	 -0.027509	 -0.017533
EAS(1.071519)	 -3.417869	  1.270000	  2.777711	  3.761170	  5.607021	 -1.860000	  7.581256	  0.452036	  3.809526	 -0.004341	 -1.106843	  0.005512	 -0.200000	  0.175557	  0.112541	  0.142588	  0.148818	 -0.023398	  0.494636	  0.377352	  0.415594	  0.420664	  0.429161	  0.419928	  0.707413	 -0.029480	 -0.017352
EAS(1.096478)	 -3.415615	  1.270000	  2.766424	  3.809507	  5.597561	 -1.860000	  7.581208	  0.452148	  3.807901	 -0.004412	 -1.102564	  0.006048	 -0.200000	  0.171794	  0.108682	  0.139553	  0.147307	 -0.023277	  0.490791	  0.374886	  0.415069	  0.419823	  0.429388	  0.419200	  0.691683	 -0.031596	 -0.017163
EAS(1.122018)	 -3.413361	  1.270000	  2.754283	  3.857288	  5.588162	 -1.860000	  7.581160	  0.452260	  3.806276	 -0.004484	 -1.098060	  0.006580	 -0.200000	  0.168197	  0.104868	  0.136500	  0.145772	 -0.023137	  0.486858	  0.372423	  0.414559	  0.418883	  0.429634	  0.418441	  0.675587	 -0.033867	 -0.016965
EAS(1.148153)	 -3.411108	  1.270000	  2.741957	  3.906198	  5.578820	 -1.860000	  7.581112	  0.452372	  3.804651	 -0.004556	 -1.093329	  0.007110	 -0.200000	  0.164764	  0.101106	  0.133429	  0.144211	 -0.023009	  0.482843	  0.369972	  0.414068	  0.417846	  0.429900	  0.417651	  0.659116	 -0.036301	 -0.016759
EAS(1.174897)	 -3.408854	  1.270000	  2.729479	  3.956348	  5.569529	 -1.860000	  7.581064	  0.452484	  3.803026	 -0.004628	 -1.088372	  0.007639	 -0.200000	  0.161493	  0.097405	  0.130341	  0.142625	 -0.022930	  0.478757	  0.367540	  0.413600	  0.416714	  0.430188	  0.416832	  0.642261	 -0.038902	 -0.016545
EAS(1.202264)	 -3.406600	  1.270000	  2.716894	  4.007886	  5.560283	 -1.860000	  7.581016	  0.452596	  3.801402	 -0.004701	 -1.083189	  0.008169	 -0.200000	  0.158380	  0.093768	  0.127233	  0.141011	 -0.022945	  0.474607	  0.365132	  0.413155	  0.415487	  0.430499	  0.415986	  0.625013	 -0.041675	 -0.016324
EAS(1.230269)	 -3.404346	  1.270000	  2.704257	  4.061001	  5.551076	 -1.860000	  7.580968	  0.452708	  3.799777	 -0.004774	 -1.077786	  0.008700	 -0.200000	  0.155421	  0.090201	  0.124107	  0.139368	 -0.023088	  0.470402	  0.362755	  0.412732	  0.414167	  0.430836	  0.415115	  0.607364	 -0.044621	 -0.016096
EAS(1.258926)	 -3.402093	  1.270000	  2.691618	  4.115883	  5.541904	 -1.860000	  7.580920	  0.452821	  3.798152	 -0.004848	 -1.072164	  0.009233	 -0.200000	  0.152613	  0.086706	  0.120961	  0.137694	 -0.023369	  0.466152	  0.360413	  0.412330	  0.412757	  0.431200	  0.414223	  0.589307	 -0.047741	 -0.015860
EAS(1.288250)	 -3.399839	  1.270000	  2.679019	  4.172706	  5.532763	 -1.860000	  7.580872	  0.452933	  3.796527	 -0.004923	 -1.066330	  0.009770	 -0.200000	  0.149951	  0.083284	  0.117795	  0.135987	 -0.023804	  0.461865	  0.358110	  0.411943	  0.411257	  0.431593	  0.413312	  0.570853	 -0.051031	 -0.015619
EAS(1.318257)	 -3.397585	  1.270000	  2.666497	  4.231632	  5.523652	 -1.860000	  7.580824	  0.453045	  3.794902	 -0.004999	 -1.060287	  0.010312	 -0.200000	  0.147432	  0.079936	  0.114609	  0.134245	 -0.024402	  0.457550	  0.355850	  0.411566	  0.409671	  0.432018	  0.412384	  0.552038	 -0.054488	 -0.015371
EAS(1.348963)	 -3.395332	  1.270000	  2.654081	  4.292814	  5.514568	 -1.860000	  7.580714	  0.453215	  3.792548	 -0.005075	 -1.054039	  0.010858	 -0.200000	  0.145048	  0.076660	  0.111402	  0.132465	 -0.025116	  0.453216	  0.353635	  0.411193	  0.407998	  0.432475	  0.411445	  0.532923	 -0.058104	 -0.015118
EAS(1.380384)	 -3.393078	  1.270000	  2.641791	  4.356362	  5.505512	 -1.860000	  7.580543	  0.453442	  3.789481	 -0.005152	 -1.047589	  0.011411	 -0.200000	  0.142791	  0.073456	  0.108173	  0.130646	 -0.025880	  0.448871	  0.351465	  0.410816	  0.406243	  0.432969	  0.410497	  0.513592	 -0.061872	 -0.014861
EAS(1.412537)	 -3.390824	  1.270000	  2.629627	  4.422333	  5.496481	 -1.860000	  7.580373	  0.453669	  3.786415	 -0.005230	 -1.040942	  0.011970	 -0.200000	  0.140653	  0.070320	  0.104921	  0.128784	 -0.026644	  0.444523	  0.349342	  0.410428	  0.404407	  0.433500	  0.409544	  0.494149	 -0.065782	 -0.014599
EAS(1.445440)	 -3.388571	  1.270000	  2.617572	  4.490719	  5.487474	 -1.860000	  7.580203	  0.453896	  3.783348	 -0.005309	 -1.034106	  0.012537	 -0.200000	  0.138625	  0.067251	  0.101645	  0.126880	 -0.027376	  0.440181	  0.347265	  0.410019	  0.402493	  0.434070	  0.408590	  0.474707	 -0.069821	 -0.014334
EAS(1.479108)	 -3.386317	  1.270000	  2.605611	  4.561496	  5.478492	 -1.860000	  7.580032	  0.454124	  3.780282	 -0.005390	 -1.027088	  0.013110	 -0.200000	  0.136695	  0.064245	  0.098347	  0.124931	 -0.028047	  0.435853	  0.345234	  0.409582	  0.400505	  0.434682	  0.407639	  0.455387	 -0.073976	 -0.014066
EAS(1.513561)	 -3.384063	  1.270000	  2.593733	  4.634658	  5.469534	 -1.860000	  7.579862	  0.454351	  3.777215	 -0.005471	 -1.019897	  0.013690	 -0.200000	  0.134859	  0.061301	  0.095026	  0.122938	 -0.028636	  0.431547	  0.343248	  0.409112	  0.398447	  0.435336	  0.406695	  0.436305	 -0.078232	 -0.013796
EAS(1.548816)	 -3.381809	  1.270000	  2.581943	  4.710229	  5.460603	 -1.860000	  7.579692	  0.454578	  3.774148	 -0.005553	 -1.012544	  0.014278	 -0.200000	  0.133112	  0.058414	  0.091684	  0.120902	 -0.029129	  0.427273	  0.341308	  0.408605	  0.396327	  0.436034	  0.405761	  0.417577	 -0.082574	 -0.013525
EAS(1.584893)	 -3.379556	  1.270000	  2.570248	  4.788262	  5.451703	 -1.860000	  7.579521	  0.454805	  3.771082	 -0.005636	 -1.005035	  0.014872	 -0.200000	  0.131452	  0.055581	  0.088322	  0.118822	 -0.029497	  0.423039	  0.339414	  0.408061	  0.394149	  0.436776	  0.404840	  0.399306	 -0.086988	 -0.013253
EAS(1.621810)	 -3.377302	  1.270000	  2.558665	  4.868828	  5.442839	 -1.860000	  7.579351	  0.455032	  3.768015	 -0.005721	 -0.997378	  0.015473	 -0.200000	  0.129874	  0.052802	  0.084941	  0.116698	 -0.029732	  0.418855	  0.337564	  0.407478	  0.391923	  0.437563	  0.403936	  0.381586	 -0.091462	 -0.012981
EAS(1.659587)	 -3.375048	  1.270000	  2.547212	  4.952031	  5.434016	 -1.860000	  7.579180	  0.455259	  3.764949	 -0.005807	 -0.989576	  0.016081	 -0.200000	  0.128372	  0.050073	  0.081545	  0.114528	 -0.029852	  0.414729	  0.335761	  0.406857	  0.389655	  0.438394	  0.403051	  0.364496	 -0.095987	 -0.012709
EAS(1.698244)	 -3.372795	  1.270000	  2.535915	  5.038010	  5.425239	 -1.860000	  7.579010	  0.455487	  3.761882	 -0.005893	 -0.981634	  0.016694	 -0.200000	  0.126944	  0.047393	  0.078136	  0.112311	 -0.029864	  0.410670	  0.334003	  0.406203	  0.387357	  0.439268	  0.402185	  0.348106	 -0.100560	 -0.012438
EAS(1.737801)	 -3.370541	  1.270000	  2.524802	  5.126924	  5.416512	 -1.860000	  7.578840	  0.455714	  3.758815	 -0.005981	 -0.973553	  0.017312	 -0.200000	  0.125588	  0.044758	  0.074716	  0.110045	 -0.029753	  0.406688	  0.332293	  0.405518	  0.385037	  0.440184	  0.401342	  0.332469	 -0.105180	 -0.012169
EAS(1.778279)	 -3.368287	  1.270000	  2.513889	  5.218912	  5.407842	 -1.860000	  7.578669	  0.455941	  3.755749	 -0.006071	 -0.965337	  0.017934	 -0.200000	  0.124306	  0.042166	  0.071287	  0.107727	 -0.029498	  0.402792	  0.330632	  0.404809	  0.382706	  0.441140	  0.400519	  0.317629	 -0.109855	 -0.011902
EAS(1.819701)	 -3.366034	  1.270000	  2.503185	  5.314098	  5.399234	 -1.860000	  7.578499	  0.456168	  3.752682	 -0.006161	 -0.956987	  0.018560	 -0.200000	  0.123098	  0.039614	  0.067853	  0.105355	 -0.029082	  0.398990	  0.329021	  0.404081	  0.380375	  0.442133	  0.399718	  0.303623	 -0.114597	 -0.011636
EAS(1.862087)	 -3.363780	  1.270000	  2.492692	  5.412573	  5.390695	 -1.860000	  7.578329	  0.456395	  3.749616	 -0.006253	 -0.948502	  0.019189	 -0.200000	  0.121968	  0.037099	  0.064416	  0.102930	 -0.028504	  0.395290	  0.327462	  0.403340	  0.378055	  0.443161	  0.398938	  0.290477	 -0.119423	 -0.011374
EAS(1.905461)	 -3.361526	  1.270000	  2.482405	  5.514432	  5.382232	 -1.860000	  7.578158	  0.456622	  3.746549	 -0.006345	 -0.939879	  0.019820	 -0.200000	  0.120914	  0.034619	  0.060980	  0.100449	 -0.027782	  0.391701	  0.325959	  0.402596	  0.375757	  0.444220	  0.398176	  0.278214	 -0.124354	 -0.011114
EAS(1.949844)	 -3.359272	  1.270000	  2.472319	  5.619762	  5.373850	 -1.860000	  7.577988	  0.456849	  3.743482	 -0.006439	 -0.931117	  0.020452	 -0.200000	  0.119936	  0.032169	  0.057545	  0.097912	 -0.026948	  0.388231	  0.324513	  0.401853	  0.373494	  0.445308	  0.397431	  0.266857	 -0.129409	 -0.010857
EAS(1.995262)	 -3.357020	  1.270000	  2.462429	  5.728672	  5.365555	 -1.860000	  7.577818	  0.457077	  3.740416	 -0.006534	 -0.922213	  0.021084	 -0.200000	  0.119032	  0.029749	  0.054113	  0.095318	 -0.026040	  0.384887	  0.323128	  0.401120	  0.371276	  0.446422	  0.396700	  0.256425	 -0.134605	 -0.010604
EAS(10.000000)	 -3.863724	  1.270000	  1.941007	 18.860347	  5.069512	 -1.860000	  6.830500	  0.485400	  3.238100	 -0.014363	 -0.082902	  0.050635	 -0.200000	  0.073981	 -0.071018	 -0.124373	 -0.128303	  0.000000	  0.475950	  0.485413	  0.565725	  0.466998	  0.514103	  0.379036	  0.200000	 -0.880079	 -0.001539
EAS(10.232930)	 -3.890285	  1.270000	  1.936070	 18.827301	  5.068441	 -1.860000	  6.800453	  0.485544	  3.227799	 -0.014442	 -0.067578	  0.050880	 -0.200000	  0.070904	 -0.072371	 -0.125861	 -0.129865	  0.000000	  0.479689	  0.488721	  0.573014	  0.471243	  0.515830	  0.380144	  0.200000	 -0.889228	 -0.001468
EAS(10.471284)	 -3.917270	  1.270000	  1.931178	 18.783867	  5.067677	 -1.860000	  6.770407	  0.485688	  3.217498	 -0.014518	 -0.052279	  0.051116	 -0.200000	  0.067826	 -0.073690	 -0.127323	 -0.131399	  0.000000	  0.483422	  0.491946	  0.580437	  0.475473	  0.517655	  0.381338	  0.200000	 -0.898083	 -0.001398
EAS(10.715192)	 -3.944761	  1.270000	  1.926383	 18.731758	  5.067092	 -1.860000	  6.740360	  0.485832	  3.207197	 -0.014589	 -0.037009	  0.051343	 -0.200000	  0.064749	 -0.074988	 -0.128771	 -0.132912	  0.000000	  0.487146	  0.495085	  0.587985	  0.479685	  0.519577	  0.382618	  0.200000	 -0.906630	 -0.001330
EAS(10.964780)	 -3.972849	  1.270000	  1.921751	 18.673080	  5.066606	 -1.860000	  6.710313	  0.485976	  3.196896	 -0.014658	 -0.021764	  0.051559	 -0.200000	  0.061674	 -0.076273	 -0.130211	 -0.134411	  0.000000	  0.490858	  0.498135	  0.595648	  0.483874	  0.521598	  0.383987	  0.200000	 -0.914855	 -0.001264
EAS(100.000000)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.591565	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(11.220183)	 -4.001646	  1.270000	  1.917340	 18.609728	  5.066171	 -1.860000	  6.680267	  0.486120	  3.186595	 -0.014722	 -0.006539	  0.051766	 -0.200000	  0.058599	 -0.077551	 -0.131648	 -0.135898	  0.000000	  0.494557	  0.501096	  0.603417	  0.488036	  0.523719	  0.385446	  0.200000	 -0.922742	 -0.001200
EAS(11.481534)	 -4.031261	  1.270000	  1.913193	 18.543001	  5.065757	 -1.860000	  6.650220	  0.486264	  3.176294	 -0.014782	  0.008674	  0.051962	 -0.200000	  0.055523	 -0.078826	 -0.133084	 -0.137377	  0.000000	  0.498238	  0.503968	  0.611280	  0.492168	  0.525938	  0.386999	  0.200000	 -0.930276	 -0.001137
EAS(11.748973)	 -4.061794	  1.270000	  1.909339	 18.473757	  5.065352	 -1.860000	  6.620173	  0.486408	  3.165993	 -0.014839	  0.023879	  0.052148	 -0.200000	  0.052446	 -0.080101	 -0.134521	 -0.138850	  0.000000	  0.501900	  0.506756	  0.619224	  0.496268	  0.528257	  0.388645	  0.200000	 -0.937447	 -0.001076
EAS(12.022642)	 -4.093295	  1.270000	  1.905802	 18.402698	  5.064948	 -1.860000	  6.590126	  0.486553	  3.155692	 -0.014891	  0.039084	  0.052323	 -0.200000	  0.049367	 -0.081378	 -0.135959	 -0.140318	  0.000000	  0.505540	  0.509464	  0.627234	  0.500330	  0.530675	  0.390388	  0.200000	 -0.944250	 -0.001017
EAS(12.302684)	 -4.125757	  1.270000	  1.902597	 18.330269	  5.064542	 -1.860000	  6.560080	  0.486697	  3.145391	 -0.014940	  0.054291	  0.052488	 -0.200000	  0.046284	 -0.082657	 -0.137399	 -0.141781	  0.000000	  0.509156	  0.512100	  0.635296	  0.504353	  0.533191	  0.392229	  0.200000	 -0.950685	 -0.000960
EAS(12.589251)	 -4.159102	  1.270000	  1.899725	 18.256562	  5.064134	 -1.860000	  6.530033	  0.486841	  3.135090	 -0.014984	  0.069504	  0.052595	 -0.200000	  0.043198	 -0.083939	 -0.138840	 -0.143240	  0.000000	  0.512746	  0.514670	  0.643394	  0.508332	  0.535804	  0.394169	  0.200000	 -0.956753	 -0.000905
EAS(12.882492)	 -4.193201	  1.270000	  1.897181	 18.181584	  5.063724	 -1.860000	  6.499986	  0.486985	  3.124789	 -0.015025	  0.084726	  0.052695	 -0.200000	  0.040108	 -0.085226	 -0.140281	 -0.144697	  0.000000	  0.516306	  0.517178	  0.651509	  0.512265	  0.538512	  0.396210	  0.200000	 -0.962449	 -0.000852
EAS(13.182563)	 -4.227902	  1.270000	  1.894960	 18.105536	  5.063314	 -1.860000	  6.469940	  0.487129	  3.114488	 -0.015061	  0.099959	  0.052791	 -0.200000	  0.037015	 -0.086517	 -0.141724	 -0.146150	  0.000000	  0.519835	  0.519629	  0.659624	  0.516146	  0.541313	  0.398350	  0.200000	 -0.967766	 -0.000801
EAS(13.489624)	 -4.263059	  1.270000	  1.893059	 18.028773	  5.062907	 -1.860000	  6.455100	  0.487234	  3.109161	 -0.015094	  0.115205	  0.052886	 -0.200000	  0.033918	 -0.087812	 -0.143167	 -0.147602	  0.000000	  0.523328	  0.522024	  0.667717	  0.519971	  0.544201	  0.400588	  0.200000	 -0.972686	 -0.000753
EAS(13.803840)	 -4.298557	  1.270000	  1.891477	 17.951796	  5.062505	 -1.860000	  6.455100	  0.487303	  3.108690	 -0.015123	  0.130465	  0.052980	 -0.200000	  0.030817	 -0.089112	 -0.144609	 -0.149052	  0.000000	  0.526782	  0.524364	  0.675767	  0.523737	  0.547171	  0.402919	  0.200000	 -0.977190	 -0.000707
EAS(14.125370)	 -4.334309	  1.270000	  1.890218	 17.875271	  5.062113	 -1.860000	  6.455100	  0.487371	  3.108219	 -0.015147	  0.145739	  0.053073	 -0.200000	  0.027713	 -0.090415	 -0.146051	 -0.150501	  0.000000	  0.530193	  0.526647	  0.683751	  0.527439	  0.550214	  0.405339	  0.200000	 -0.981260	 -0.000663
EAS(14.454392)	 -4.370258	  1.270000	  1.889294	 17.800171	  5.061734	 -1.860000	  6.455100	  0.487439	  3.107747	 -0.015168	  0.161030	  0.053165	 -0.200000	  0.024604	 -0.091723	 -0.147493	 -0.151948	  0.000000	  0.533555	  0.528871	  0.691648	  0.531072	  0.553322	  0.407841	  0.200000	 -0.984882	 -0.000622
EAS(14.791080)	 -4.406372	  1.270000	  1.888713	 17.727310	  5.061372	 -1.860000	  6.455100	  0.487507	  3.107276	 -0.015185	  0.176336	  0.053257	 -0.200000	  0.021491	 -0.093034	 -0.148932	 -0.153394	  0.000000	  0.536864	  0.531032	  0.699435	  0.534631	  0.556485	  0.410416	  0.200000	 -0.988049	 -0.000583
EAS(15.135614)	 -4.442632	  1.270000	  1.888461	 17.656819	  5.061031	 -1.860000	  6.455100	  0.487575	  3.106805	 -0.015200	  0.191660	  0.053347	 -0.200000	  0.018373	 -0.094347	 -0.150370	 -0.154839	  0.000000	  0.540116	  0.533129	  0.707089	  0.538111	  0.559692	  0.413056	  0.200000	 -0.990765	 -0.000546
EAS(15.488170)	 -4.479012	  1.270000	  1.888502	 17.588196	  5.060715	 -1.860000	  6.455100	  0.487643	  3.106333	 -0.015211	  0.207000	  0.053437	 -0.200000	  0.015251	 -0.095663	 -0.151806	 -0.156283	  0.000000	  0.543305	  0.535157	  0.714590	  0.541510	  0.562930	  0.415751	  0.200000	 -0.993047	 -0.000512
EAS(15.848933)	 -4.514684	  1.270000	  1.888953	 17.525244	  5.060429	 -1.860000	  6.455100	  0.487712	  3.105862	 -0.015220	  0.222356	  0.053526	 -0.200000	  0.012124	 -0.096981	 -0.153238	 -0.157726	  0.000000	  0.546427	  0.537114	  0.721918	  0.544821	  0.566190	  0.418489	  0.200000	 -0.994919	 -0.000480
EAS(16.218101)	 -4.550644	  1.270000	  1.889450	 17.458058	  5.060176	 -1.860000	  6.455100	  0.487780	  3.105391	 -0.015226	  0.237730	  0.053614	 -0.200000	  0.008992	 -0.098301	 -0.154668	 -0.159168	  0.000000	  0.549478	  0.538999	  0.729057	  0.548041	  0.569459	  0.421261	  0.200000	 -0.996414	 -0.000451
EAS(16.595870)	 -4.586930	  1.270000	  1.890040	 17.388522	  5.059960	 -1.860000	  6.455100	  0.487848	  3.104919	 -0.015231	  0.253120	  0.053700	 -0.200000	  0.005855	 -0.099622	 -0.156094	 -0.160608	  0.000000	  0.552452	  0.540808	  0.735991	  0.551166	  0.572724	  0.424055	  0.200000	 -0.997573	 -0.000423
EAS(16.982440)	 -4.623513	  1.270000	  1.890745	 17.317793	  5.059786	 -1.860000	  6.455100	  0.487916	  3.104448	 -0.015234	  0.268526	  0.053786	 -0.200000	  0.002713	 -0.100943	 -0.157516	 -0.162046	  0.000000	  0.555348	  0.542540	  0.742707	  0.554193	  0.575976	  0.426860	  0.200000	 -0.998439	 -0.000397
EAS(17.378010)	 -4.660355	  1.270000	  1.891565	 17.246427	  5.059658	 -1.860000	  6.455100	  0.487984	  3.103977	 -0.015236	  0.283948	  0.053870	 -0.200000	 -0.000433	 -0.102264	 -0.158934	 -0.163482	  0.000000	  0.558160	  0.544193	  0.749194	  0.557118	  0.579204	  0.429666	  0.200000	 -0.999058	 -0.000374
EAS(17.782793)	 -4.697431	  1.270000	  1.892696	 17.174719	  5.059579	 -1.860000	  6.455100	  0.488052	  3.103505	 -0.015237	  0.299385	  0.053953	 -0.200000	 -0.003584	 -0.103584	 -0.160347	 -0.164916	  0.000000	  0.560887	  0.545765	  0.755447	  0.559938	  0.582397	  0.432462	  0.200000	 -0.999476	 -0.000351
EAS(18.197010)	 -4.734720	  1.270000	  1.893811	 17.102855	  5.059552	 -1.860000	  6.455100	  0.488120	  3.103034	 -0.015238	  0.314838	  0.054034	 -0.200000	 -0.006741	 -0.104903	 -0.161755	 -0.166348	  0.000000	  0.563526	  0.547255	  0.761458	  0.562651	  0.585547	  0.435239	  0.200000	 -0.999739	 -0.000331
EAS(18.620870)	 -4.772211	  1.270000	  1.894904	 17.030954	  5.059582	 -1.860000	  6.455100	  0.488189	  3.102563	 -0.015238	  0.330305	  0.054114	 -0.200000	 -0.009902	 -0.106220	 -0.163158	 -0.167777	  0.000000	  0.566074	  0.548665	  0.767225	  0.565255	  0.588645	  0.437988	  0.200000	 -0.999888	 -0.000312
EAS(19.054610)	 -4.809901	  1.270000	  1.895976	 16.959084	  5.059672	 -1.860000	  6.455100	  0.488257	  3.102091	 -0.015238	  0.345786	  0.054193	 -0.200000	 -0.013068	 -0.107534	 -0.164555	 -0.169203	  0.000000	  0.568530	  0.549994	  0.772748	  0.567746	  0.591683	  0.440700	  0.200000	 -0.999961	 -0.000294
EAS(19.498443)	 -4.847789	  1.270000	  1.897028	 16.887282	  5.059824	 -1.860000	  6.455100	  0.488325	  3.101620	 -0.015238	  0.361280	  0.054270	 -0.200000	 -0.016239	 -0.108846	 -0.165947	 -0.170626	  0.000000	  0.570893	  0.551246	  0.778026	  0.570124	  0.594654	  0.443367	  0.200000	 -0.999990	 -0.000277
EAS(19.952621)	 -4.885880	  1.270000	  1.898060	 16.815565	  5.060041	 -1.860000	  6.455100	  0.488393	  3.101149	 -0.015238	  0.376787	  0.054346	 -0.200000	 -0.019414	 -0.110154	 -0.167333	 -0.172046	  0.000000	  0.573162	  0.552424	  0.783061	  0.572386	  0.597554	  0.445983	  0.200000	 -0.999999	 -0.000261
EAS(2.041738)	 -3.354776	  1.270000	  2.452741	  5.841332	  5.357355	 -1.860000	  7.577402	  0.457433	  3.735879	 -0.006631	 -0.913168	  0.021716	 -0.200000	  0.118202	  0.027357	  0.050688	  0.092668	 -0.025095	  0.381676	  0.321807	  0.400403	  0.369113	  0.447561	  0.395983	  0.246936	 -0.139950	 -0.010353
EAS(2.089296)	 -3.352541	  1.270000	  2.443259	  5.957932	  5.349255	 -1.860000	  7.576958	  0.457805	  3.731174	 -0.006728	 -0.903980	  0.022346	 -0.200000	  0.117446	  0.024995	  0.047269	  0.089961	 -0.024144	  0.378603	  0.320554	  0.399708	  0.367016	  0.448723	  0.395279	  0.238405	 -0.145446	 -0.010106
EAS(2.137962)	 -3.350317	  1.270000	  2.433972	  6.078616	  5.341264	 -1.860000	  7.576515	  0.458176	  3.726468	 -0.006827	 -0.894648	  0.022974	 -0.200000	  0.116764	  0.022665	  0.043858	  0.087199	 -0.023209	  0.375676	  0.319374	  0.399039	  0.364994	  0.449907	  0.394587	  0.230838	 -0.151093	 -0.009863
EAS(2.187761)	 -3.348097	  1.270000	  2.424854	  6.203464	  5.333392	 -1.860000	  7.576071	  0.458548	  3.721763	 -0.006927	 -0.885171	  0.023599	 -0.200000	  0.116157	  0.020369	  0.040455	  0.084380	 -0.022307	  0.372899	  0.318269	  0.398401	  0.363054	  0.451115	  0.393907	  0.224229	 -0.156894	 -0.009623
EAS(2.238721)	 -3.345855	  1.270000	  2.415874	  6.332545	  5.325653	 -1.860000	  7.575627	  0.458919	  3.717058	 -0.007028	 -0.875548	  0.024220	 -0.200000	  0.115624	  0.018111	  0.037062	  0.081505	 -0.021442	  0.370277	  0.317245	  0.397795	  0.361205	  0.452344	  0.393239	  0.218564	 -0.162853	 -0.009387
EAS(2.290868)	 -3.343572	  1.270000	  2.407008	  6.465977	  5.318060	 -1.860000	  7.575183	  0.459291	  3.712352	 -0.007130	 -0.865780	  0.024837	 -0.200000	  0.115169	  0.015895	  0.033678	  0.078573	 -0.020592	  0.367813	  0.316306	  0.397222	  0.359452	  0.453593	  0.392582	  0.213810	 -0.168986	 -0.009155
EAS(2.344229)	 -3.341242	  1.270000	  2.398240	  6.603930	  5.310629	 -1.860000	  7.574740	  0.459662	  3.707647	 -0.007233	 -0.855869	  0.025448	 -0.200000	  0.114794	  0.013724	  0.030304	  0.075580	 -0.019727	  0.365511	  0.315456	  0.396682	  0.357798	  0.454862	  0.391937	  0.209921	 -0.175310	 -0.008926
EAS(2.398833)	 -3.338864	  1.270000	  2.389570	  6.746692	  5.303374	 -1.860000	  7.574296	  0.460034	  3.702941	 -0.007337	 -0.845816	  0.026052	 -0.200000	  0.114499	  0.011602	  0.026942	  0.072527	 -0.018819	  0.363374	  0.314702	  0.396173	  0.356248	  0.456148	  0.391301	  0.206836	 -0.181848	 -0.008701
EAS(2.454709)	 -3.336444	  1.270000	  2.381020	  6.894687	  5.296310	 -1.860000	  7.573852	  0.460405	  3.698236	 -0.007442	 -0.835623	  0.026650	 -0.200000	  0.114285	  0.009532	  0.023593	  0.069411	 -0.017862	  0.361403	  0.314047	  0.395692	  0.354803	  0.457450	  0.390676	  0.204476	 -0.188619	 -0.008480
EAS(2.511886)	 -3.333989	  1.270000	  2.372600	  7.048311	  5.289448	 -1.860000	  7.573277	  0.460796	  3.693324	 -0.007549	 -0.825295	  0.027241	 -0.200000	  0.114148	  0.007517	  0.020259	  0.066232	 -0.016845	  0.359601	  0.313498	  0.395236	  0.353463	  0.458767	  0.390061	  0.202751	 -0.195640	 -0.008263
EAS(2.570396)	 -3.331508	  1.270000	  2.364299	  7.207827	  5.282800	 -1.860000	  7.572197	  0.461260	  3.687618	 -0.007656	 -0.814837	  0.027823	 -0.200000	  0.114084	  0.005558	  0.016941	  0.062988	 -0.015749	  0.357968	  0.313060	  0.394802	  0.352226	  0.460095	  0.389454	  0.201559	 -0.202920	 -0.008049
EAS(2.630268)	 -3.329012	  1.270000	  2.356091	  7.373382	  5.276374	 -1.860000	  7.571116	  0.461724	  3.681911	 -0.007765	 -0.804255	  0.028397	 -0.200000	  0.114085	  0.003659	  0.013640	  0.059680	 -0.014556	  0.356506	  0.312738	  0.394387	  0.351093	  0.461433	  0.388857	  0.200794	 -0.210459	 -0.007840
EAS(2.691535)	 -3.326511	  1.270000	  2.347943	  7.545036	  5.270176	 -1.860000	  7.570036	  0.462188	  3.676204	 -0.007874	 -0.793552	  0.028961	 -0.200000	  0.114141	  0.001818	  0.010357	  0.056307	 -0.013266	  0.355213	  0.312539	  0.393992	  0.350059	  0.462778	  0.388267	  0.200348	 -0.218249	 -0.007636
EAS(2.754228)	 -3.324028	  1.270000	  2.339823	  7.722791	  5.264210	 -1.860000	  7.568955	  0.462653	  3.670497	 -0.007984	 -0.782735	  0.029517	 -0.200000	  0.114241	  0.000036	  0.007094	  0.052869	 -0.011901	  0.354091	  0.312467	  0.393621	  0.349121	  0.464128	  0.387684	  0.200123	 -0.226273	 -0.007436
EAS(2.818383)	 -3.321602	  1.270000	  2.331703	  7.906617	  5.258478	 -1.860000	  7.567875	  0.463117	  3.664790	 -0.008095	 -0.771810	  0.030062	 -0.200000	  0.114374	 -0.001688	  0.003849	  0.049371	 -0.010494	  0.353138	  0.312530	  0.393282	  0.348278	  0.465479	  0.387109	  0.200030	 -0.234508	 -0.007241
EAS(2.884031)	 -3.319282	  1.270000	  2.323557	  8.096427	  5.252978	 -1.860000	  7.566794	  0.463581	  3.659084	 -0.008208	 -0.760782	  0.030598	 -0.200000	  0.114527	 -0.003358	  0.000623	  0.045813	 -0.009073	  0.352353	  0.312734	  0.392991	  0.347526	  0.466829	  0.386539	  0.200003	 -0.242930	 -0.007052
EAS(2.951209)	 -3.317119	  1.270000	  2.315355	  8.292032	  5.247706	 -1.860000	  7.565714	  0.464045	  3.653377	 -0.008320	 -0.749663	  0.031125	 -0.200000	  0.114690	 -0.004977	 -0.002584	  0.042201	 -0.007661	  0.351734	  0.313084	  0.392762	  0.346864	  0.468174	  0.385974	  0.200000	 -0.251511	 -0.006870
EAS(20.417380)	 -4.924177	  1.270000	  1.899074	 16.743944	  5.060327	 -1.860000	  6.455100	  0.488437	  3.100887	 -0.015238	  0.392305	  0.054420	 -0.200000	 -0.022593	 -0.111458	 -0.168712	 -0.173463	  0.000000	  0.575335	  0.553531	  0.787856	  0.574532	  0.600375	  0.448542	  0.200000	 -1.000000	 -0.000246
EAS(20.892960)	 -4.962688	  1.270000	  1.900066	 16.672403	  5.060682	 -1.860000	  6.455100	  0.488478	  3.100650	 -0.015238	  0.407834	  0.054493	 -0.200000	 -0.025775	 -0.112758	 -0.170085	 -0.174876	  0.000000	  0.577413	  0.554570	  0.792412	  0.576560	  0.603114	  0.451037	  0.200000	 -1.000000	 -0.000232
EAS(21.379620)	 -5.001419	  1.270000	  1.901037	 16.600904	  5.061110	 -1.860000	  6.455100	  0.488520	  3.100412	 -0.015238	  0.423373	  0.054564	 -0.200000	 -0.028960	 -0.114054	 -0.171452	 -0.176286	  0.000000	  0.579395	  0.555547	  0.796735	  0.578469	  0.605766	  0.453464	  0.200000	 -1.000000	 -0.000219
EAS(21.877611)	 -5.040377	  1.270000	  1.901982	 16.529380	  5.061609	 -1.860000	  6.455100	  0.488561	  3.100175	 -0.015238	  0.438922	  0.054634	 -0.200000	 -0.032149	 -0.115346	 -0.172812	 -0.177692	  0.000000	  0.581281	  0.556467	  0.800825	  0.580259	  0.608328	  0.455819	  0.200000	 -1.000000	 -0.000206
EAS(22.387210)	 -5.079571	  1.270000	  1.902898	 16.457730	  5.062182	 -1.860000	  6.455100	  0.488602	  3.099938	 -0.015238	  0.454477	  0.054702	 -0.200000	 -0.035339	 -0.116632	 -0.174164	 -0.179095	  0.000000	  0.583073	  0.557333	  0.804687	  0.581930	  0.610796	  0.458097	  0.200000	 -1.000000	 -0.000195
EAS(22.908672)	 -5.119006	  1.270000	  1.903777	 16.385804	  5.062827	 -1.860000	  6.455100	  0.488643	  3.099700	 -0.015238	  0.470040	  0.054769	 -0.200000	 -0.038532	 -0.117914	 -0.175510	 -0.180494	  0.000000	  0.584769	  0.558152	  0.808325	  0.583482	  0.613167	  0.460294	  0.200000	 -1.000000	 -0.000184
EAS(23.442283)	 -5.158691	  1.270000	  1.904615	 16.313449	  5.063544	 -1.860000	  6.455100	  0.488685	  3.099463	 -0.015238	  0.485609	  0.054834	 -0.200000	 -0.041727	 -0.119190	 -0.176850	 -0.181889	  0.000000	  0.586372	  0.558929	  0.811741	  0.584915	  0.615439	  0.462408	  0.200000	 -1.000000	 -0.000173
EAS(23.988321)	 -5.198632	  1.270000	  1.905405	 16.240566	  5.064330	 -1.860000	  6.455100	  0.488726	  3.099226	 -0.015238	  0.501183	  0.054898	 -0.200000	 -0.044924	 -0.120462	 -0.178182	 -0.183280	  0.000000	  0.587882	  0.559669	  0.814941	  0.586230	  0.617609	  0.464436	  0.200000	 -1.000000	 -0.000164
EAS(24.547080)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.589300	  0.560376	  0.817928	  0.587430	  0.619676	  0.466373	  0.200000	 -1.000000	 -0.000155
EAS(25.118860)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.590627	  0.561055	  0.820705	  0.588516	  0.621636	  0.468217	  0.200000	 -1.000000	 -0.000147
EAS(25.703950)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.591863	  0.561708	  0.823277	  0.589491	  0.623487	  0.469966	  0.200000	 -1.000000	 -0.000140
EAS(26.302670)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.593011	  0.562337	  0.825649	  0.590357	  0.625227	  0.471615	  0.200000	 -1.000000	 -0.000133
EAS(26.915340)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.594071	  0.562942	  0.827825	  0.591118	  0.626856	  0.473164	  0.200000	 -1.000000	 -0.000128
EAS(27.542291)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.595045	  0.563523	  0.829810	  0.591778	  0.628370	  0.474609	  0.200000	 -1.000000	 -0.000122
EAS(28.183832)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.595934	  0.564077	  0.831610	  0.592343	  0.629770	  0.475949	  0.200000	 -1.000000	 -0.000118
EAS(28.840320)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.596740	  0.564601	  0.833231	  0.592815	  0.631056	  0.477183	  0.200000	 -1.000000	 -0.000114
EAS(29.512094)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.597467	  0.565094	  0.834680	  0.593202	  0.632227	  0.478311	  0.200000	 -1.000000	 -0.000110
EAS(3.019952)	 -3.315156	  1.270000	  2.307060	  8.493113	  5.242655	 -1.860000	  7.564633	  0.464509	  3.647670	 -0.008434	 -0.738463	  0.031641	 -0.200000	  0.114851	 -0.006549	 -0.005772	  0.038539	 -0.006280	  0.351281	  0.313586	  0.392612	  0.346290	  0.469512	  0.385415	  0.200000	 -0.260227	 -0.006693
EAS(3.090296)	 -3.313417	  1.270000	  2.298642	  8.699322	  5.237818	 -1.860000	  7.563553	  0.464974	  3.641963	 -0.008549	 -0.727195	  0.032148	 -0.200000	  0.115003	 -0.008078	 -0.008941	  0.034831	 -0.004954	  0.350992	  0.314248	  0.392554	  0.345807	  0.470839	  0.384859	  0.200000	 -0.269054	 -0.006524
EAS(3.162278)	 -3.311904	  1.270000	  2.290093	  8.910444	  5.233185	 -1.860000	  7.562472	  0.465438	  3.636256	 -0.008664	 -0.715874	  0.032644	 -0.200000	  0.115137	 -0.009571	 -0.012090	  0.031082	 -0.003696	  0.350863	  0.315074	  0.392598	  0.345417	  0.472151	  0.384307	  0.200000	 -0.277969	 -0.006361
EAS(3.235937)	 -3.310615	  1.270000	  2.281416	  9.126353	  5.228744	 -1.860000	  7.561392	  0.465902	  3.630550	 -0.008780	 -0.704510	  0.033131	 -0.200000	  0.115247	 -0.011031	 -0.015216	  0.027299	 -0.002522	  0.350893	  0.316071	  0.392757	  0.345124	  0.473446	  0.383759	  0.200000	 -0.286956	 -0.006205
EAS(3.311311)	 -3.309540	  1.270000	  2.272621	  9.346953	  5.224481	 -1.860000	  7.560311	  0.466366	  3.624843	 -0.008897	 -0.693114	  0.033609	 -0.200000	  0.115330	 -0.012464	 -0.018318	  0.023485	 -0.001464	  0.351078	  0.317242	  0.393038	  0.344936	  0.474721	  0.383213	  0.200000	 -0.296005	 -0.006056
EAS(3.388441)	 -3.308676	  1.270000	  2.263724	  9.572236	  5.220384	 -1.860000	  7.558345	  0.466869	  3.618829	 -0.009014	 -0.681696	  0.034077	 -0.200000	  0.115382	 -0.013873	 -0.021390	  0.019646	 -0.000561	  0.351416	  0.318590	  0.393450	  0.344858	  0.475972	  0.382670	  0.200000	 -0.305120	 -0.005914
EAS(3.467368)	 -3.308028	  1.270000	  2.254767	  9.802432	  5.216440	 -1.860000	  7.556021	  0.467387	  3.612691	 -0.009132	 -0.670263	  0.034536	 -0.200000	  0.115404	 -0.015262	 -0.024429	  0.015787	  0.000160	  0.351904	  0.320118	  0.393997	  0.344898	  0.477196	  0.382130	  0.200000	 -0.314312	 -0.005779
EAS(3.548134)	 -3.307615	  1.270000	  2.245799	 10.037894	  5.212639	 -1.860000	  7.553698	  0.467904	  3.606553	 -0.009250	 -0.658819	  0.034986	 -0.200000	  0.115393	 -0.016635	 -0.027431	  0.011912	  0.000679	  0.352537	  0.321824	  0.394684	  0.345061	  0.478390	  0.381594	  0.200000	 -0.323603	 -0.005650
EAS(3.630780)	 -3.307458	  1.270000	  2.236857	 10.278908	  5.208977	 -1.860000	  7.551374	  0.468422	  3.600416	 -0.009369	 -0.647366	  0.035428	 -0.200000	  0.115347	 -0.017992	 -0.030391	  0.008027	  0.000990	  0.353313	  0.323710	  0.395514	  0.345348	  0.479552	  0.381060	  0.200000	 -0.333021	 -0.005527
EAS(3.715352)	 -3.307591	  1.270000	  2.227956	 10.525507	  5.205449	 -1.860000	  7.549050	  0.468940	  3.594278	 -0.009489	 -0.635904	  0.035862	 -0.200000	  0.115267	 -0.019334	 -0.033307	  0.004135	  0.001103	  0.354228	  0.325774	  0.396487	  0.345760	  0.480680	  0.380530	  0.200000	 -0.342596	 -0.005410
EAS(3.801893)	 -3.308062	  1.270000	  2.219095	 10.777541	  5.202048	 -1.860000	  7.546726	  0.469458	  3.588140	 -0.009608	 -0.624434	  0.036287	 -0.200000	  0.115149	 -0.020660	 -0.036176	  0.000241	  0.001055	  0.355278	  0.328013	  0.397605	  0.346296	  0.481770	  0.380005	  0.200000	 -0.352362	 -0.005297
EAS(3.890451)	 -3.308911	  1.270000	  2.210263	 11.034672	  5.198771	 -1.860000	  7.544402	  0.469976	  3.582002	 -0.009729	 -0.612956	  0.036705	 -0.200000	  0.114993	 -0.021969	 -0.038997	 -0.003650	  0.000890	  0.356458	  0.330424	  0.398865	  0.346956	  0.482821	  0.379484	  0.200000	 -0.362348	 -0.005189
EAS(3.981071)	 -3.310168	  1.270000	  2.201444	 11.296383	  5.195610	 -1.860000	  7.542079	  0.470493	  3.575864	 -0.009849	 -0.601466	  0.037116	 -0.200000	  0.114800	 -0.023261	 -0.041770	 -0.007537	  0.000663	  0.357764	  0.333003	  0.400267	  0.347739	  0.483831	  0.378970	  0.200000	 -0.372583	 -0.005085
EAS(30.199520)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.598116	  0.565551	  0.835965	  0.593508	  0.633284	  0.479333	  0.200000	 -1.000000	 -0.000108
EAS(30.902954)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.598690	  0.565969	  0.837093	  0.593739	  0.634231	  0.480250	  0.200000	 -1.000000	 -0.000106
EAS(31.622780)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.599193	  0.566345	  0.838074	  0.593902	  0.635070	  0.481065	  0.200000	 -1.000000	 -0.000104
EAS(32.359363)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.599629	  0.566676	  0.838917	  0.594004	  0.635803	  0.481780	  0.200000	 -1.000000	 -0.000103
EAS(33.113110)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.600003	  0.566960	  0.839633	  0.594050	  0.636438	  0.482399	  0.200000	 -1.000000	 -0.000102
EAS(33.884414)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.600317	  0.567194	  0.840230	  0.594048	  0.636977	  0.482927	  0.200000	 -1.000000	 -0.000101
EAS(34.673683)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.600578	  0.567379	  0.840722	  0.594005	  0.637429	  0.483371	  0.200000	 -1.000000	 -0.000101
EAS(35.481334)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.600790	  0.567514	  0.841118	  0.593928	  0.637800	  0.483736	  0.200000	 -1.000000	 -0.000100
EAS(36.307800)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.600959	  0.567603	  0.841431	  0.593824	  0.638098	  0.484029	  0.200000	 -1.000000	 -0.000100
EAS(37.153514)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601090	  0.567653	  0.841672	  0.593698	  0.638330	  0.484260	  0.200000	 -1.000000	 -0.000100
EAS(38.018932)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601188	  0.567672	  0.841851	  0.593558	  0.638507	  0.484435	  0.200000	 -1.000000	 -0.000100
EAS(38.904510)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601260	  0.567667	  0.841980	  0.593409	  0.638636	  0.484563	  0.200000	 -1.000000	 -0.000100
EAS(39.810710)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601309	  0.567647	  0.842068	  0.593256	  0.638726	  0.484652	  0.200000	 -1.000000	 -0.000100
EAS(4.073803)	 -3.311841	  1.270000	  2.192626	 11.562117	  5.192558	 -1.860000	  7.537962	  0.471001	  3.569684	 -0.009970	 -0.589961	  0.037520	 -0.200000	  0.114573	 -0.024534	 -0.044494	 -0.011414	  0.000431	  0.359193	  0.335744	  0.401807	  0.348645	  0.484798	  0.378462	  0.200000	 -0.383094	 -0.004984
EAS(4.168694)	 -3.313927	  1.270000	  2.183806	 11.831337	  5.189606	 -1.860000	  7.533381	  0.471507	  3.563493	 -0.010091	 -0.578437	  0.037917	 -0.200000	  0.114317	 -0.025789	 -0.047169	 -0.015278	  0.000239	  0.360740	  0.338641	  0.403482	  0.349673	  0.485722	  0.377962	  0.200000	 -0.393904	 -0.004886
EAS(4.265795)	 -3.316431	  1.270000	  2.174990	 12.103519	  5.186742	 -1.860000	  7.528799	  0.472013	  3.557301	 -0.010213	 -0.566889	  0.038309	 -0.200000	  0.114039	 -0.027027	 -0.049796	 -0.019125	  0.000108	  0.362401	  0.341683	  0.405288	  0.350822	  0.486601	  0.377471	  0.200000	 -0.405027	 -0.004790
EAS(4.365158)	 -3.319359	  1.270000	  2.166191	 12.378190	  5.183957	 -1.860000	  7.524218	  0.472518	  3.551110	 -0.010334	 -0.555313	  0.038695	 -0.200000	  0.113744	 -0.028251	 -0.052375	 -0.022952	  0.000037	  0.364171	  0.344862	  0.407220	  0.352091	  0.487435	  0.376989	  0.200000	 -0.416478	 -0.004696
EAS(4.466835)	 -3.322724	  1.270000	  2.157418	 12.654775	  5.181238	 -1.860000	  7.519636	  0.473024	  3.544919	 -0.010456	 -0.543704	  0.039076	 -0.200000	  0.113440	 -0.029460	 -0.054907	 -0.026753	  0.000009	  0.366047	  0.348169	  0.409276	  0.353480	  0.488224	  0.376518	  0.200000	 -0.428259	 -0.004603
EAS(4.570881)	 -3.326542	  1.270000	  2.148667	 12.932471	  5.178573	 -1.860000	  7.515054	  0.473530	  3.538728	 -0.010578	 -0.532056	  0.039453	 -0.200000	  0.113127	 -0.030658	 -0.057393	 -0.030524	  0.000001	  0.368025	  0.351595	  0.411450	  0.354987	  0.488968	  0.376059	  0.200000	 -0.440370	 -0.004512
EAS(4.677351)	 -3.330835	  1.270000	  2.139935	 13.210426	  5.175948	 -1.860000	  7.510473	  0.474035	  3.532536	 -0.010700	 -0.520364	  0.039825	 -0.200000	  0.112805	 -0.031844	 -0.059834	 -0.034261	  0.000000	  0.370102	  0.355130	  0.413738	  0.356614	  0.489668	  0.375613	  0.200000	 -0.452804	 -0.004421
EAS(4.786300)	 -3.335635	  1.270000	  2.131236	 13.487983	  5.173348	 -1.860000	  7.505891	  0.474541	  3.526345	 -0.010823	 -0.508620	  0.040194	 -0.200000	  0.112473	 -0.033019	 -0.062231	 -0.037962	  0.000000	  0.372273	  0.358766	  0.416139	  0.358359	  0.490325	  0.375180	  0.200000	 -0.465550	 -0.004330
EAS(4.897787)	 -3.340980	  1.270000	  2.122600	 13.764688	  5.170758	 -1.860000	  7.501310	  0.475046	  3.520154	 -0.010945	 -0.496818	  0.040559	 -0.200000	  0.112126	 -0.034185	 -0.064587	 -0.041624	  0.000000	  0.374536	  0.362493	  0.418648	  0.360223	  0.490940	  0.374762	  0.200000	 -0.478593	 -0.004240
EAS(40.738020)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601342	  0.567620	  0.842126	  0.593106	  0.638786	  0.484712	  0.200000	 -1.000000	 -0.000100
EAS(41.686930)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601362	  0.567594	  0.842161	  0.593069	  0.638822	  0.484748	  0.200000	 -1.000000	 -0.000100
EAS(42.657940)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601372	  0.567573	  0.842180	  0.593039	  0.638843	  0.484769	  0.200000	 -1.000000	 -0.000100
EAS(43.651570)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601378	  0.567559	  0.842189	  0.593017	  0.638853	  0.484779	  0.200000	 -1.000000	 -0.000100
EAS(44.668342)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567553	  0.842192	  0.593003	  0.638857	  0.484782	  0.200000	 -1.000000	 -0.000100
EAS(45.708810)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592994	  0.638858	  0.484783	  0.200000	 -1.000000	 -0.000100
EAS(46.773500)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592988	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(47.862991)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592983	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(48.977890)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592978	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(5.011872)	 -3.346900	  1.270000	  2.114069	 14.040277	  5.168165	 -1.860000	  7.496314	  0.475547	  3.513932	 -0.011067	 -0.484947	  0.040921	 -0.200000	  0.111759	 -0.035338	 -0.066902	 -0.045245	  0.000000	  0.376889	  0.366301	  0.421265	  0.362206	  0.491516	  0.374360	  0.200000	 -0.491916	 -0.004150
EAS(5.128613)	 -3.353425	  1.270000	  2.105702	 14.314734	  5.165553	 -1.860000	  7.487714	  0.476000	  3.507443	 -0.011190	 -0.472998	  0.041280	 -0.200000	  0.111370	 -0.036480	 -0.069179	 -0.048823	  0.000000	  0.379327	  0.370181	  0.423988	  0.364307	  0.492056	  0.373975	  0.200000	 -0.505500	 -0.004060
EAS(5.248074)	 -3.360575	  1.270000	  2.097560	 14.588167	  5.162910	 -1.860000	  7.479114	  0.476453	  3.500954	 -0.011312	 -0.460960	  0.041637	 -0.200000	  0.110954	 -0.037607	 -0.071420	 -0.052355	  0.000000	  0.381849	  0.374125	  0.426818	  0.366528	  0.492562	  0.373608	  0.200000	 -0.519328	 -0.003970
EAS(5.370318)	 -3.368376	  1.270000	  2.089699	 14.860615	  5.160221	 -1.860000	  7.470514	  0.476907	  3.494465	 -0.011434	 -0.448822	  0.041992	 -0.200000	  0.110512	 -0.038721	 -0.073627	 -0.055840	  0.000000	  0.384453	  0.378125	  0.429757	  0.368867	  0.493038	  0.373261	  0.200000	 -0.533380	 -0.003880
EAS(5.495409)	 -3.376863	  1.270000	  2.082153	 15.131833	  5.157473	 -1.860000	  7.461914	  0.477360	  3.487976	 -0.011556	 -0.436577	  0.042344	 -0.200000	  0.110043	 -0.039819	 -0.075803	 -0.059277	  0.000000	  0.387137	  0.382173	  0.432808	  0.371325	  0.493488	  0.372934	  0.200000	 -0.547633	 -0.003788
EAS(5.623413)	 -3.386065	  1.270000	  2.074940	 15.401292	  5.154653	 -1.860000	  7.453314	  0.477814	  3.481487	 -0.011677	 -0.424220	  0.042695	 -0.200000	  0.109548	 -0.040904	 -0.077950	 -0.062667	  0.000000	  0.389898	  0.386263	  0.435976	  0.373900	  0.493918	  0.372630	  0.200000	 -0.562063	 -0.003697
EAS(5.754399)	 -3.395991	  1.270000	  2.068059	 15.668216	  5.151750	 -1.860000	  7.444714	  0.478267	  3.474998	 -0.011799	 -0.411744	  0.043044	 -0.200000	  0.109022	 -0.041976	 -0.080068	 -0.066007	  0.000000	  0.392734	  0.390389	  0.439266	  0.376592	  0.494332	  0.372350	  0.200000	 -0.576637	 -0.003605
EAS(5.888436)	 -3.406653	  1.270000	  2.061493	 15.931535	  5.148752	 -1.860000	  7.435886	  0.478717	  3.468480	 -0.011919	 -0.399150	  0.043392	 -0.200000	  0.108464	 -0.043038	 -0.082161	 -0.069297	  0.000000	  0.395644	  0.394545	  0.442685	  0.379400	  0.494736	  0.372095	  0.200000	 -0.591317	 -0.003513
EAS(50.118730)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592972	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(51.286144)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592963	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(52.480751)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592953	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(53.703182)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592939	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(54.954090)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592923	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(56.234130)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592904	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(57.543991)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592881	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(58.884361)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592856	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(6.025596)	 -3.418063	  1.270000	  2.055209	 16.189860	  5.145652	 -1.860000	  7.422198	  0.479104	  3.461342	 -0.012040	 -0.386435	  0.043738	 -0.200000	  0.107869	 -0.044093	 -0.084229	 -0.072536	  0.000000	  0.398624	  0.398727	  0.446240	  0.382323	  0.495135	  0.371868	  0.200000	 -0.606063	 -0.003420
EAS(6.165949)	 -3.430236	  1.270000	  2.049163	 16.441663	  5.142445	 -1.860000	  7.408511	  0.479490	  3.454204	 -0.012160	 -0.373601	  0.044083	 -0.200000	  0.107234	 -0.045144	 -0.086274	 -0.075725	  0.000000	  0.401673	  0.402930	  0.449936	  0.385356	  0.495535	  0.371671	  0.200000	 -0.620827	 -0.003327
EAS(6.309573)	 -3.443203	  1.270000	  2.043311	 16.685347	  5.139128	 -1.860000	  7.394824	  0.479876	  3.447066	 -0.012279	 -0.360652	  0.044427	 -0.200000	  0.106555	 -0.046195	 -0.088296	 -0.078862	  0.000000	  0.404788	  0.407145	  0.453781	  0.388500	  0.495943	  0.371505	  0.200000	 -0.635562	 -0.003234
EAS(6.456542)	 -3.456996	  1.270000	  2.037611	 16.919430	  5.135704	 -1.860000	  7.381137	  0.480263	  3.439928	 -0.012397	 -0.347594	  0.044769	 -0.200000	  0.105826	 -0.047250	 -0.090297	 -0.081946	  0.000000	  0.407966	  0.411367	  0.457780	  0.391749	  0.496365	  0.371375	  0.200000	 -0.650220	 -0.003141
EAS(6.606934)	 -3.471660	  1.270000	  2.032036	 17.142756	  5.132178	 -1.860000	  7.367450	  0.480649	  3.432790	 -0.012515	 -0.334432	  0.045110	 -0.200000	  0.105040	 -0.048314	 -0.092277	 -0.084977	  0.000000	  0.411205	  0.415588	  0.461938	  0.395101	  0.496806	  0.371282	  0.200000	 -0.664757	 -0.003048
EAS(6.760828)	 -3.487224	  1.270000	  2.026561	 17.354308	  5.128560	 -1.860000	  7.347719	  0.480982	  3.424387	 -0.012632	 -0.321172	  0.045449	 -0.200000	  0.104193	 -0.049393	 -0.094237	 -0.087954	  0.000000	  0.414501	  0.419800	  0.466260	  0.398552	  0.497273	  0.371228	  0.200000	 -0.679133	 -0.002955
EAS(6.918308)	 -3.503697	  1.270000	  2.021165	 17.553213	  5.124864	 -1.860000	  7.324110	  0.481282	  3.415172	 -0.012748	 -0.307820	  0.045787	 -0.200000	  0.103277	 -0.050491	 -0.096177	 -0.090875	  0.000000	  0.417850	  0.423998	  0.470750	  0.402097	  0.497772	  0.371217	  0.200000	 -0.693309	 -0.002862
EAS(60.255954)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592828	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(61.659500)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592796	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(63.095730)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592762	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(64.565414)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592725	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(66.069330)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592685	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(67.608283)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592642	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(69.183082)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592596	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(7.079456)	 -3.521060	  1.270000	  2.015840	 17.738926	  5.121107	 -1.860000	  7.300500	  0.481581	  3.405958	 -0.012862	 -0.294379	  0.046122	 -0.200000	  0.102286	 -0.051613	 -0.098098	 -0.093738	  0.000000	  0.421251	  0.428174	  0.475410	  0.405732	  0.498308	  0.371252	  0.200000	 -0.707255	 -0.002771
EAS(7.244360)	 -3.539274	  1.270000	  2.010586	 17.911234	  5.117305	 -1.860000	  7.276890	  0.481880	  3.396743	 -0.012976	 -0.280853	  0.046455	 -0.200000	  0.101214	 -0.052760	 -0.099999	 -0.096540	  0.000000	  0.424699	  0.432324	  0.480241	  0.409452	  0.498888	  0.371335	  0.200000	 -0.720940	 -0.002680
EAS(7.413103)	 -3.558302	  1.270000	  2.005400	 18.069916	  5.113473	 -1.860000	  7.253281	  0.482179	  3.387528	 -0.013088	 -0.267248	  0.046786	 -0.200000	  0.100052	 -0.053935	 -0.101879	 -0.099279	  0.000000	  0.428190	  0.436443	  0.485246	  0.413250	  0.499516	  0.371469	  0.200000	 -0.734340	 -0.002590
EAS(7.585776)	 -3.578107	  1.270000	  2.000271	 18.214661	  5.109629	 -1.860000	  7.229671	  0.482479	  3.378313	 -0.013198	 -0.253565	  0.047113	 -0.200000	  0.098790	 -0.055138	 -0.103739	 -0.101952	  0.000000	  0.431722	  0.440526	  0.490425	  0.417122	  0.500199	  0.371658	  0.200000	 -0.747437	 -0.002501
EAS(7.762471)	 -3.598662	  1.270000	  1.995192	 18.345257	  5.105786	 -1.860000	  7.206062	  0.482778	  3.369099	 -0.013307	 -0.239809	  0.047437	 -0.200000	  0.097421	 -0.056368	 -0.105577	 -0.104557	  0.000000	  0.435290	  0.444569	  0.495777	  0.421061	  0.500940	  0.371903	  0.200000	 -0.760216	 -0.002413
EAS(7.943282)	 -3.619950	  1.270000	  1.990166	 18.461851	  5.101958	 -1.860000	  7.182452	  0.483077	  3.359884	 -0.013415	 -0.225982	  0.047757	 -0.200000	  0.095934	 -0.057623	 -0.107395	 -0.107091	  0.000000	  0.438891	  0.448567	  0.501303	  0.425061	  0.501745	  0.372207	  0.200000	 -0.772670	 -0.002326
EAS(70.794560)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592548	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(72.443572)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592497	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(74.131004)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592444	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(75.857734)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592389	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(77.624690)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592331	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(79.432792)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592271	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(8.128304)	 -3.641945	  1.270000	  1.985195	 18.564622	  5.098160	 -1.860000	  7.158843	  0.483376	  3.350669	 -0.013520	 -0.212087	  0.048072	 -0.200000	  0.094326	 -0.058902	 -0.109192	 -0.109555	  0.000000	  0.442522	  0.452518	  0.507002	  0.429116	  0.502618	  0.372573	  0.200000	 -0.784796	 -0.002241
EAS(8.317636)	 -3.664607	  1.270000	  1.980271	 18.653585	  5.094408	 -1.860000	  7.135233	  0.483675	  3.341455	 -0.013624	 -0.198122	  0.048383	 -0.200000	  0.092593	 -0.060201	 -0.110969	 -0.111947	  0.000000	  0.446178	  0.456417	  0.512871	  0.433219	  0.503563	  0.373004	  0.200000	 -0.796596	 -0.002157
EAS(8.511379)	 -3.687890	  1.270000	  1.975380	 18.728546	  5.090719	 -1.860000	  7.098190	  0.483897	  3.328803	 -0.013725	 -0.184086	  0.048688	 -0.200000	  0.090735	 -0.061519	 -0.112725	 -0.114266	  0.000000	  0.449857	  0.460261	  0.518911	  0.437364	  0.504583	  0.373501	  0.200000	 -0.808075	 -0.002075
EAS(8.709635)	 -3.711741	  1.270000	  1.970504	 18.789334	  5.087114	 -1.860000	  7.059948	  0.484112	  3.315846	 -0.013825	 -0.169977	  0.048988	 -0.200000	  0.088751	 -0.062851	 -0.114462	 -0.116513	  0.000000	  0.453555	  0.464047	  0.525118	  0.441544	  0.505683	  0.374068	  0.200000	 -0.819240	 -0.001994
EAS(8.912507)	 -3.736105	  1.270000	  1.965629	 18.835901	  5.083613	 -1.860000	  7.021707	  0.484327	  3.302888	 -0.013921	 -0.155790	  0.049281	 -0.200000	  0.086642	 -0.064194	 -0.116178	 -0.118689	  0.000000	  0.457267	  0.467772	  0.531491	  0.445753	  0.506865	  0.374706	  0.200000	 -0.830100	 -0.001915
EAS(81.283030)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592209	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(83.176350)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592145	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(85.113770)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592078	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(87.096321)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.592010	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(89.125100)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.591940	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(9.120107)	 -3.760927	  1.270000	  1.960746	 18.868287	  5.080238	 -1.860000	  6.983466	  0.484541	  3.289931	 -0.014015	 -0.141523	  0.049567	 -0.200000	  0.084412	 -0.065545	 -0.117875	 -0.120794	  0.000000	  0.460992	  0.471435	  0.538027	  0.449984	  0.508132	  0.375417	  0.200000	 -0.840663	 -0.001837
EAS(9.332541)	 -3.786139	  1.270000	  1.955842	 18.886540	  5.077009	 -1.860000	  6.945224	  0.484756	  3.276973	 -0.014107	 -0.127175	  0.049846	 -0.200000	  0.082065	 -0.066901	 -0.119551	 -0.122831	  0.000000	  0.464726	  0.475034	  0.544722	  0.454230	  0.509488	  0.376204	  0.200000	 -0.850938	 -0.001760
EAS(9.549923)	 -3.811684	  1.270000	  1.950912	 18.890865	  5.073942	 -1.860000	  6.906983	  0.484971	  3.264015	 -0.014195	 -0.112744	  0.050117	 -0.200000	  0.079609	 -0.068260	 -0.121208	 -0.124799	  0.000000	  0.468466	  0.478566	  0.551573	  0.458485	  0.510933	  0.377068	  0.200000	 -0.860931	 -0.001685
EAS(9.772372)	 -3.837538	  1.270000	  1.945961	 18.881848	  5.071053	 -1.860000	  6.868741	  0.485185	  3.251058	 -0.014281	 -0.098234	  0.050380	 -0.200000	  0.077051	 -0.069619	 -0.122845	 -0.126702	  0.000000	  0.472208	  0.482027	  0.558576	  0.462743	  0.512471	  0.378012	  0.200000	 -0.870645	 -0.001611
EAS(91.201100)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.591869	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(93.325440)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.591795	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(95.499260)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.591720	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100
EAS(97.723724)	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	-9999.999900	  0.601380	  0.567551	  0.842193	  0.591643	  0.638858	  0.484784	  0.200000	 -1.000000	 -0.000100"""
    COEFFS = CoeffsTable(table=TMP)
