# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['vban_cmd']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'vban-cmd',
    'version': '1.0.0',
    'description': 'Python interface for the VBAN RT Packet Service (Sendtext)',
    'long_description': "[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://github.com/onyx-and-iris/vban-cmd-python/blob/dev/LICENSE)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n![Tests Status](./tests/basic.svg?dummy=8484744)\n![Tests Status](./tests/banana.svg?dummy=8484744)\n![Tests Status](./tests/potato.svg?dummy=8484744)\n\n# VBAN CMD\n\nThis package offers a Python interface for the Voicemeeter RT Packet Service as well as Voicemeeter VBAN-TEXT.\n\nThis allows a user to get (rt packets) and set (vban-text) parameters over a local network. Consider the Streamer View app over VBAN, for example.\n\nIt may be used standalone or to extend the [Voicemeeter Remote Python API](https://github.com/onyx-and-iris/voicemeeter-api-python)\n\nFor sending audio across a network with VBAN you will need to look elsewhere.\n\nFor an outline of past/future changes refer to: [CHANGELOG](CHANGELOG.md)\n\n## Tested against\n\n-   Basic 1.0.8.2\n-   Banana 2.0.6.2\n-   Potato 3.0.2.2\n\n## Prerequisites\n\n-   [Voicemeeter](https://voicemeeter.com/)\n-   Python 3.9+\n\n## Installation\n\n```\ngit clone https://github.com/onyx-and-iris/vban-cmd-python\ncd vban-cmd-python\n```\n\nJust the interface:\n\n```\npip install .\n```\n\nWith development dependencies:\n\n```\npip install -e .['development']\n```\n\n## Usage\n\n#### Use with a context manager:\n\nParameter coverage is not as extensive for this interface as with the Remote API.\n\n### Example 1\n\n```python\nimport vbancmd\n\nclass ManyThings:\n    def __init__(self, vban):\n        self.vban = vban\n\n    def things(self):\n        # Set the mapping of the second input strip\n        self.vban.strip[1].A3 = True\n        print(f'Output A3 of Strip {self.vban.strip[1].label}: {self.vban.strip[1].A3}')\n\n    def other_things(self):\n        # Toggle mute for the leftmost output bus\n        self.vban.bus[0].mute = not self.vban.bus[0].mute\n\n\ndef main():\n    with vbancmd.connect(kind_id, ip=ip) as vban:\n        do = ManyThings(vban)\n        do.things()\n        do.other_things()\n\nif __name__ == '__main__':\n    kind_id = 'potato'\n    ip = '<ip address>'\n\n    main()\n```\n\n#### Or perform setup/teardown independently:\n\n### Example 2\n\n```python\nimport vbancmd\n\nkind_id = 'potato'\nip = '<ip address>'\n\nvban = vbancmd.connect(kind_id, ip=ip)\n\n# call login() at the start of your code\nvban.login()\n\n# Toggle mute for leftmost input strip\nvban.strip[0].mute = not vban.strip[0].mute\n\n# Toggle eq for leftmost output bus\nvban.bus[0].eq = not vban.bus[0].eq\n\n# call logout() at the end of your code\nvban.logout()\n```\n\n## Profiles\n\nProfiles through config files are supported.\n\nThree example profiles are provided with the package, one for each kind of Voicemeeter.\nTo test one first rename \\_profiles directory to profiles.\nThey will be loaded into memory but not applied. To apply one you may do:\n`vmr.apply_profile('config')`, but remember to save your current settings first.\n\nprofiles directory can be safely deleted if you don't wish to load them each time.\n\nA config can contain any key that `connect.apply()` would accept. Additionally, `extends` can be provided to inherit from another profile. Two profiles are available by default:\n\n-   `blank`, all strip off and all sliders to `0.0`. mono, solo, mute, eq all disabled.\n-   `base`, all physical strip to `A1`, all virtual strip to `B1`, all sliders to `0.0`.\n\nSample `mySetup.toml`\n\n```toml\nextends = 'base'\n[strip-0]\nmute = 1\n\n[strip-5]\nA1 = 0\nA2 = 1\nA4 = 1\ngain = 0.0\n\n[strip-6]\nA1 = 0\nA2 = 1\nA4 = 1\ngain = 0.0\n```\n\n## API\n\n### Kinds\n\nA _kind_ specifies a major Voicemeeter version. Currently this encompasses\n\n-   `basic`\n-   `banana`\n-   `potato`\n\n#### `vbancmd.connect(kind_id, **kwargs) -> '(VbanCmd)'`\n\nFactory function for remotes. Keyword arguments include:\n\n-   `ip`: remote pc you wish to send requests to.\n-   `streamname`: default 'Command1'\n-   `port`: default 6990\n-   `channel`: from 0 to 255\n-   `bps`: bitrate of stream, default 0 should be safe for most cases.\n\n### `VbanCmd` (higher level)\n\n#### `vban.type`\n\nThe kind of the Voicemeeter instance.\n\n#### `vban.version`\n\nA tuple of the form `(v1, v2, v3, v4)`.\n\n#### `vban.strip`\n\nAn `InputStrip` tuple, containing both physical and virtual.\n\n#### `vban.bus`\n\nAn `OutputBus` tuple, containing both physical and virtual.\n\n#### `vban.show()`\n\nShows Voicemeeter if it's hidden. No effect otherwise.\n\n#### `vban.hide()`\n\nHides Voicemeeter if it's shown. No effect otherwise.\n\n#### `vban.shutdown()`\n\nCloses Voicemeeter.\n\n#### `vban.restart()`\n\nRestarts Voicemeeter's audio engine.\n\n#### `vban.apply(mapping)`\n\nUpdates values through a dict.  \nExample:\n\n```python\nvban.apply({\n    'strip-2': dict(A1=True, B1=True, gain=-6.0),\n    'bus-2': dict(mute=True),\n})\n```\n\n### `Strip`\n\nThe following properties are gettable and settable:\n\n-   `mono`: boolean\n-   `solo`: boolean\n-   `mute`: boolean\n-   `label`: string\n-   `gain`: float, -60 to 12\n-   Output mapping (e.g. `A1`, `B3`, etc.): boolean, depends on the Voicemeeter kind\n\nThe following properties are settable:\n\n-   `comp`: float, from 0.0 to 10.0\n-   `gate`: float, from 0.0 to 10.0\n-   `limit`: int, from -40 to 12\n\n#### `gainlayer`\n\n-   `gainlayer[j].gain`: float, -60 to 12\n\nfor example:\n\n```python\n# set and get the value of the second input strip, fourth gainlayer\nvban.strip[1].gainlayer[3].gain = -6.3\nprint(vban.strip[1].gainlayer[3].gain)\n```\n\nGainlayers defined for Potato version only.\n\n### `Bus`\n\nThe following properties are gettable and settable:\n\n-   `mute`: boolean\n-   `mono`: boolean\n-   `eq`: boolean\n-   `eq_ab`: boolean\n-   `label`: string\n-   `gain`: float, -60 to 12\n\n#### `mode`\n\nBus modes are gettable and settable\n\n-   `normal`, `amix`, `bmix`, `repeat`, `composite`, `tvmix`, `upmix21`,\n-   `upmix41`, `upmix61`, `centeronly`, `lfeonly`, `rearonly`\n\nfor example:\n\n```python\n# set leftmost bus mode to tvmix\nvban.bus[0].mode.tvmix = True\n```\n\n### `VbanCmd` (lower level)\n\n#### `vban.pdirty`\n\nTrue iff a parameter has been changed. Typically this is checked periodically to update states.\n\n#### `vban.set_rt(id_, param, val)`\n\nSends a string request RT Packet where the command would take the form:\n\n```python\nf'{id_}.{param}={val}'\n```\n\n#### `vban.public_packet`\n\nReturns a Voicemeeter rt data packet. Designed to be used internally by the interface but available for parsing through this read only property object. States may or may not be current, use the polling parameter pdirty to be sure.\n\n### `Errors`\n\n-   `errors.VMCMDErrors`: Base VMCMD error class.\n\n### `Tests`\n\nFirst make sure you installed the [development dependencies](https://github.com/onyx-and-iris/vban-cmd-python#installation)\n\nThen from tests directory:\n\n`pytest -v`\n\n## Resources\n\n-   [Voicemeeter VBAN TEXT](https://vb-audio.com/Voicemeeter/VBANProtocol_Specifications.pdf#page=19)\n\n-   [Voicemeeter RT Packet Service](https://vb-audio.com/Voicemeeter/VBANProtocol_Specifications.pdf#page=27)\n",
    'author': 'onyx-and-iris',
    'author_email': 'code@onyxandiris.online',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/onyx-and-iris/vban-cmd-python',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.11,<4.0',
}


setup(**setup_kwargs)
