[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://github.com/onyx-and-iris/vban-cmd-python/blob/dev/LICENSE)
[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)
![Tests Status](./tests/basic.svg?dummy=8484744)
![Tests Status](./tests/banana.svg?dummy=8484744)
![Tests Status](./tests/potato.svg?dummy=8484744)

# VBAN CMD

This package offers a Python interface for the Voicemeeter RT Packet Service as well as Voicemeeter VBAN-TEXT.

This allows a user to get (rt packets) and set (vban-text) parameters over a local network. Consider the Streamer View app over VBAN, for example.

It may be used standalone or to extend the [Voicemeeter Remote Python API](https://github.com/onyx-and-iris/voicemeeter-api-python)

For sending audio across a network with VBAN you will need to look elsewhere.

For an outline of past/future changes refer to: [CHANGELOG](CHANGELOG.md)

## Tested against

-   Basic 1.0.8.2
-   Banana 2.0.6.2
-   Potato 3.0.2.2

## Prerequisites

-   [Voicemeeter](https://voicemeeter.com/)
-   Python 3.9+

## Installation

```
git clone https://github.com/onyx-and-iris/vban-cmd-python
cd vban-cmd-python
```

Just the interface:

```
pip install .
```

With development dependencies:

```
pip install -e .['development']
```

## Usage

#### Use with a context manager:

Parameter coverage is not as extensive for this interface as with the Remote API.

### Example 1

```python
import vbancmd

class ManyThings:
    def __init__(self, vban):
        self.vban = vban

    def things(self):
        # Set the mapping of the second input strip
        self.vban.strip[1].A3 = True
        print(f'Output A3 of Strip {self.vban.strip[1].label}: {self.vban.strip[1].A3}')

    def other_things(self):
        # Toggle mute for the leftmost output bus
        self.vban.bus[0].mute = not self.vban.bus[0].mute


def main():
    with vbancmd.connect(kind_id, ip=ip) as vban:
        do = ManyThings(vban)
        do.things()
        do.other_things()

if __name__ == '__main__':
    kind_id = 'potato'
    ip = '<ip address>'

    main()
```

#### Or perform setup/teardown independently:

### Example 2

```python
import vbancmd

kind_id = 'potato'
ip = '<ip address>'

vban = vbancmd.connect(kind_id, ip=ip)

# call login() at the start of your code
vban.login()

# Toggle mute for leftmost input strip
vban.strip[0].mute = not vban.strip[0].mute

# Toggle eq for leftmost output bus
vban.bus[0].eq = not vban.bus[0].eq

# call logout() at the end of your code
vban.logout()
```

## Profiles

Profiles through config files are supported.

Three example profiles are provided with the package, one for each kind of Voicemeeter.
To test one first rename \_profiles directory to profiles.
They will be loaded into memory but not applied. To apply one you may do:
`vmr.apply_profile('config')`, but remember to save your current settings first.

profiles directory can be safely deleted if you don't wish to load them each time.

A config can contain any key that `connect.apply()` would accept. Additionally, `extends` can be provided to inherit from another profile. Two profiles are available by default:

-   `blank`, all strip off and all sliders to `0.0`. mono, solo, mute, eq all disabled.
-   `base`, all physical strip to `A1`, all virtual strip to `B1`, all sliders to `0.0`.

Sample `mySetup.toml`

```toml
extends = 'base'
[strip-0]
mute = 1

[strip-5]
A1 = 0
A2 = 1
A4 = 1
gain = 0.0

[strip-6]
A1 = 0
A2 = 1
A4 = 1
gain = 0.0
```

## API

### Kinds

A _kind_ specifies a major Voicemeeter version. Currently this encompasses

-   `basic`
-   `banana`
-   `potato`

#### `vbancmd.connect(kind_id, **kwargs) -> '(VbanCmd)'`

Factory function for remotes. Keyword arguments include:

-   `ip`: remote pc you wish to send requests to.
-   `streamname`: default 'Command1'
-   `port`: default 6990
-   `channel`: from 0 to 255
-   `bps`: bitrate of stream, default 0 should be safe for most cases.

### `VbanCmd` (higher level)

#### `vban.type`

The kind of the Voicemeeter instance.

#### `vban.version`

A tuple of the form `(v1, v2, v3, v4)`.

#### `vban.strip`

An `InputStrip` tuple, containing both physical and virtual.

#### `vban.bus`

An `OutputBus` tuple, containing both physical and virtual.

#### `vban.show()`

Shows Voicemeeter if it's hidden. No effect otherwise.

#### `vban.hide()`

Hides Voicemeeter if it's shown. No effect otherwise.

#### `vban.shutdown()`

Closes Voicemeeter.

#### `vban.restart()`

Restarts Voicemeeter's audio engine.

#### `vban.apply(mapping)`

Updates values through a dict.  
Example:

```python
vban.apply({
    'strip-2': dict(A1=True, B1=True, gain=-6.0),
    'bus-2': dict(mute=True),
})
```

### `Strip`

The following properties are gettable and settable:

-   `mono`: boolean
-   `solo`: boolean
-   `mute`: boolean
-   `label`: string
-   `gain`: float, -60 to 12
-   Output mapping (e.g. `A1`, `B3`, etc.): boolean, depends on the Voicemeeter kind

The following properties are settable:

-   `comp`: float, from 0.0 to 10.0
-   `gate`: float, from 0.0 to 10.0
-   `limit`: int, from -40 to 12

#### `gainlayer`

-   `gainlayer[j].gain`: float, -60 to 12

for example:

```python
# set and get the value of the second input strip, fourth gainlayer
vban.strip[1].gainlayer[3].gain = -6.3
print(vban.strip[1].gainlayer[3].gain)
```

Gainlayers defined for Potato version only.

### `Bus`

The following properties are gettable and settable:

-   `mute`: boolean
-   `mono`: boolean
-   `eq`: boolean
-   `eq_ab`: boolean
-   `label`: string
-   `gain`: float, -60 to 12

#### `mode`

Bus modes are gettable and settable

-   `normal`, `amix`, `bmix`, `repeat`, `composite`, `tvmix`, `upmix21`,
-   `upmix41`, `upmix61`, `centeronly`, `lfeonly`, `rearonly`

for example:

```python
# set leftmost bus mode to tvmix
vban.bus[0].mode.tvmix = True
```

### `VbanCmd` (lower level)

#### `vban.pdirty`

True iff a parameter has been changed. Typically this is checked periodically to update states.

#### `vban.set_rt(id_, param, val)`

Sends a string request RT Packet where the command would take the form:

```python
f'{id_}.{param}={val}'
```

#### `vban.public_packet`

Returns a Voicemeeter rt data packet. Designed to be used internally by the interface but available for parsing through this read only property object. States may or may not be current, use the polling parameter pdirty to be sure.

### `Errors`

-   `errors.VMCMDErrors`: Base VMCMD error class.

### `Tests`

First make sure you installed the [development dependencies](https://github.com/onyx-and-iris/vban-cmd-python#installation)

Then from tests directory:

`pytest -v`

## Resources

-   [Voicemeeter VBAN TEXT](https://vb-audio.com/Voicemeeter/VBANProtocol_Specifications.pdf#page=19)

-   [Voicemeeter RT Packet Service](https://vb-audio.com/Voicemeeter/VBANProtocol_Specifications.pdf#page=27)
