"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Operator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdk8s_1 = require("cdk8s");
/**
 * A CDK8s app which allows implementing Kubernetes operators using CDK8s constructs.
 */
class Operator extends cdk8s_1.App {
    constructor(props = {}) {
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdk8s'));
        super({ outdir });
        this.providers = [];
        this.inputFile = props.inputFile ?? process.argv[2] ?? '/dev/stdin';
        this.outputFile = props.outputFile;
    }
    /**
     * Adds a custom resource provider to this operator.
     * @param provider The provider to add
     */
    addProvider(provider) {
        this.providers.push(provider);
    }
    /**
     * Reads a Kubernetes manifest in JSON format from STDIN or the file specified
     * as the first positional command-line argument. This manifest is expected to
     * include a single Kubernetes resource. Then, we match `apiVersion` and
     * `kind` to one of the registered providers and if we do, we invoke
     * `apply()`, passing it the `spec` of the input manifest and a chart as a
     * scope. The chart is then synthesized and the output manifest is written to
     * STDOUT.
     */
    synth() {
        const input = JSON.parse(fs.readFileSync(this.inputFile, 'utf-8'));
        let write;
        if (this.outputFile) {
            const outfile = this.outputFile;
            write = (data) => fs.writeFileSync(outfile, data);
        }
        else {
            write = (data) => process.stdout.write(data);
        }
        if (typeof (input) !== 'object') {
            throw new Error('input must be a single kubernetes resource');
        }
        const provider = this.findProvider(input);
        const name = input.metadata?.name;
        if (!name) {
            throw new Error('"metadata.name" must be defined');
        }
        const namespace = input.metadata?.namespace;
        // TODO: namespace
        const spec = input.spec ?? {};
        const chart = new cdk8s_1.Chart(this, name, { namespace });
        console.error(`Synthesizing ${input.kind}.${input.apiVersion}`);
        provider.handler.apply(chart, name, spec);
        super.synth();
        for (const file of fs.readdirSync(this.outdir)) {
            const filepath = path.join(this.outdir, file);
            const manifest = fs.readFileSync(filepath);
            write(manifest);
        }
    }
    findProvider(input) {
        const { apiVersion, kind } = input;
        if (!apiVersion) {
            throw new Error('"apiVersion" is required');
        }
        if (!kind) {
            throw new Error('"kind" is required');
        }
        for (const p of this.providers) {
            if (p.apiVersion === apiVersion && p.kind === kind) {
                return p;
            }
        }
        throw new Error(`No custom resource provider found for ${kind}.${apiVersion}`);
    }
}
exports.Operator = Operator;
_a = JSII_RTTI_SYMBOL_1;
Operator[_a] = { fqn: "cdk8s-operator.Operator", version: "0.1.42" };
//# sourceMappingURL=data:application/json;base64,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