"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NatInstanceImage = exports.NatInstanceProvider = exports.NatProvider = void 0;
const iam = require("@aws-cdk/aws-iam");
const connections_1 = require("./connections");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * NAT providers
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 * @experimental
 */
class NatProvider {
    /**
     * Use NAT Gateways to provide NAT services for your VPC
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     */
    static gateway() {
        return new NatGatewayProvider();
    }
    /**
     * Use NAT instances to provide NAT services for your VPC
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     */
    static instance(props) {
        return new NatInstanceProvider(props);
    }
}
exports.NatProvider = NatProvider;
/**
 * Provider for NAT Gateways
 */
class NatGatewayProvider extends NatProvider {
    constructor() {
        super(...arguments);
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        // Create the NAT gateways
        for (const sub of options.natSubnets) {
            const gateway = sub.addNatGateway();
            this.gateways.add(sub.availabilityZone, gateway.ref);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az);
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.NAT_GATEWAY,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1] }));
    }
}
/**
 * NAT provider which uses NAT Instances
 */
class NatInstanceProvider extends NatProvider {
    constructor(props) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        var _a, _b;
        // Create the NAT instances. They can share a security group and a Role.
        const machineImage = this.props.machineImage || new NatInstanceImage();
        this._securityGroup = (_a = this.props.securityGroup) !== null && _a !== void 0 ? _a : new security_group_1.SecurityGroup(options.vpc, 'NatSecurityGroup', {
            vpc: options.vpc,
            description: 'Security Group for NAT instances',
        });
        this._connections = new connections_1.Connections({ securityGroups: [this._securityGroup] });
        if ((_b = this.props.allowAllTraffic) !== null && _b !== void 0 ? _b : true) {
            this.connections.allowFromAnyIpv4(port_1.Port.allTraffic());
        }
        // FIXME: Ideally, NAT instances don't have a role at all, but
        // 'Instance' does not allow that right now.
        const role = new iam.Role(options.vpc, 'NatRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        for (const sub of options.natSubnets) {
            const natInstance = new instance_1.Instance(sub, 'NatInstance', {
                instanceType: this.props.instanceType,
                machineImage,
                sourceDestCheck: false,
                vpc: options.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: this._securityGroup,
                role,
                keyName: this.props.keyName,
            });
            // NAT instance routes all traffic, both ways
            this.gateways.add(sub.availabilityZone, natInstance);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    /**
     * The Security Group associated with the NAT instances
     */
    get securityGroup() {
        if (!this._securityGroup) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'securityGroup\'');
        }
        return this._securityGroup;
    }
    /**
     * Manage the Security Groups associated with the NAT instances
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'connections\'');
        }
        return this._connections;
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1].instanceId }));
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az).instanceId;
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.INSTANCE,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
}
exports.NatInstanceProvider = NatInstanceProvider;
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push([pref, value]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length][1];
    }
    values() {
        return this.vals;
    }
}
/**
 * Machine image representing the latest NAT instance image
 *
 * @experimental
 */
class NatInstanceImage extends machine_image_1.LookupMachineImage {
    constructor() {
        super({
            name: 'amzn-ami-vpc-nat-*',
            owners: ['amazon'],
        });
    }
}
exports.NatInstanceImage = NatInstanceImage;
//# sourceMappingURL=data:application/json;base64,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