"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImageBuilder = exports.ImageBuilderComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const constructs_1 = require("constructs");
const utils_1 = require("../../utils");
const common_1 = require("../common");
const dockerfileTemplate = `FROM {{{ imagebuilder:parentImage }}}
ENV RUNNER_VERSION=___RUNNER_VERSION___
{{{ imagebuilder:environments }}}
{{{ imagebuilder:components }}}`;
function uniqueName(scope) {
    return cdk.Names.uniqueResourceName(scope, { maxLength: 126, separator: '-', allowedSpecialCharacters: '_-' });
}
class ImageBuilderObjectBase extends cdk.Resource {
    constructor(scope, id) {
        super(scope, id);
    }
    version(type, name, data) {
        return new aws_cdk_lib_1.CustomResource(this, 'Version', {
            serviceToken: this.versionFunction().functionArn,
            resourceType: `Custom::ImageBuilder-${type}-Version`,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            properties: {
                ObjectType: type,
                ObjectName: name,
                VersionedData: data,
            },
        }).ref;
    }
    versionFunction() {
        return utils_1.BundledNodejsFunction.singleton(this, 'aws-image-builder-versioner', {
            description: 'Custom resource handler that bumps up Image Builder versions',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'imagebuilder:ListComponents',
                        'imagebuilder:ListContainerRecipes',
                        'imagebuilder:ListImageRecipes',
                    ],
                    resources: ['*'],
                }),
            ],
        });
    }
}
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 */
class ImageBuilderComponent extends ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.assets = [];
        this.platform = props.platform;
        let steps = [];
        if (props.assets) {
            let inputs = [];
            let extractCommands = [];
            for (const asset of props.assets) {
                this.assets.push(asset.asset);
                if (asset.asset.isFile) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: asset.path,
                    });
                }
                else if (asset.asset.isZipArchive) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: `${asset.path}.zip`,
                    });
                    if (props.platform === 'Windows') {
                        extractCommands.push('$ErrorActionPreference = \'Stop\'');
                        extractCommands.push(`Expand-Archive "${asset.path}.zip" -DestinationPath "${asset.path}"`);
                        extractCommands.push(`del "${asset.path}.zip"`);
                    }
                    else {
                        extractCommands.push(`unzip "${asset.path}.zip" -d "${asset.path}"`);
                        extractCommands.push(`rm "${asset.path}.zip"`);
                    }
                }
                else {
                    throw new Error(`Unknown asset type: ${asset.asset}`);
                }
            }
            steps.push({
                name: 'Download',
                action: 'S3Download',
                inputs,
            });
            if (extractCommands.length > 0) {
                steps.push({
                    name: 'Extract',
                    action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                    inputs: {
                        commands: extractCommands,
                    },
                });
            }
        }
        steps.push({
            name: 'Run',
            action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
            inputs: {
                commands: props.commands,
            },
        });
        const data = {
            name: props.displayName,
            description: props.description,
            schemaVersion: '1.0',
            phases: [
                {
                    name: 'build',
                    steps,
                },
            ],
        };
        const name = uniqueName(this);
        const component = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, 'Component', {
            name: name,
            platform: props.platform,
            version: this.version('Component', name, {
                platform: props.platform,
                data,
            }),
            data: JSON.stringify(data),
        });
        this.arn = component.attrArn;
    }
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee) {
        for (const asset of this.assets) {
            asset.grantRead(grantee);
        }
    }
}
exports.ImageBuilderComponent = ImageBuilderComponent;
_a = JSII_RTTI_SYMBOL_1;
ImageBuilderComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ImageBuilderComponent", version: "0.5.3" };
/**
 * Image builder recipe for a Docker container image.
 */
class ContainerRecipe extends ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = uniqueName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnContainerRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ContainerRecipe', name, {
                platform: props.platform,
                components,
                dockerfileTemplate,
            }),
            // TODO mcr.microsoft.com/windows/servercore:ltsc2019
            parentImage: 'arn:aws:imagebuilder:us-east-1:aws:image/windows-server-2019-x86-core-ltsc2019-amd64/2020.12.8',
            components,
            containerType: 'DOCKER',
            targetRepository: {
                service: 'ECR',
                repositoryName: props.targetRepository.repositoryName,
            },
            dockerfileTemplateData: props.dockerfileTemplate,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An image builder that uses Image Builder to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * The CodeBuild builder is better and faster. Only use this one if you have no choice. For example, if you need Windows containers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation. The default components work with CodeBuild and Fargate.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new ContainerImageBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * new CodeBuildRunner(this, 'CodeBuild provider', {
 *     label: 'windows-codebuild',
 *     imageBuilder: builder,
 * });
 * ```
 */
class ContainerImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.components = [];
        // set platform
        this.architecture = props?.architecture ?? common_1.Architecture.X86_64;
        if (!this.architecture.is(common_1.Architecture.X86_64)) {
            throw new Error(`Unsupported architecture: ${this.architecture}. Consider CodeBuild for faster image builds.`);
        }
        this.os = props?.os ?? common_1.Os.LINUX;
        if (this.os.is(common_1.Os.WINDOWS)) {
            this.platform = 'Windows';
        }
        else {
            throw new Error(`Unsupported OS: ${this.os}. Consider CodeBuild for faster image builds.`);
        }
        // set builder options
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (props?.vpc && props?.subnetSelection) {
            this.subnetId = props.vpc.selectSubnets(props.subnetSelection).subnetIds[0];
        }
        if (props?.securityGroup) {
            this.securityGroupIds = [props.securityGroup.securityGroupId];
        }
        this.instanceTypes = [props?.instanceType?.toString() ?? 'm5.large'];
        this.description = `Build image for GitHub Actions runner ${this.node.path} (${this.os.name}/${this.architecture.name})`;
        this.logRetention = props?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        // runner version
        this.runnerVersion = props?.runnerVersion ?? common_1.RunnerVersion.latest();
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove all but the latest image',
                    tagStatus: aws_ecr_1.TagStatus.ANY,
                    maxImageCount: 1,
                },
            ],
        });
        // add all basic components
        this.addBaseWindowsComponents();
    }
    addBaseWindowsComponents() {
        this.addComponent(new ImageBuilderComponent(this, 'AWS CLI', {
            platform: 'Windows',
            displayName: 'AWS CLI',
            description: 'Install latest version of AWS CLI',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
            ],
        }));
        this.addComponent(new ImageBuilderComponent(this, 'GitHub CLI', {
            platform: 'Windows',
            displayName: 'GitHub CLI',
            description: 'Install latest version of gh',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/cli/cli/releases/latest > $Env:TEMP\\latest-gh',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-gh',
                '$GH_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                '$ProgressPreference = \'SilentlyContinue\'',
                'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/cli/cli/releases/download/v${GH_VERSION}/gh_${GH_VERSION}_windows_amd64.msi" -OutFile gh.msi',
                'Start-Process msiexec.exe -Wait -ArgumentList \'/i gh.msi /qn\'',
                'del gh.msi',
            ],
        }));
        this.addComponent(new ImageBuilderComponent(this, 'git', {
            platform: 'Windows',
            displayName: 'Git',
            description: 'Install latest version of git',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                '$ProgressPreference = \'SilentlyContinue\'',
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/git-for-windows/git/releases/latest > $Env:TEMP\\latest-git',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-git',
                '$GIT_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                '$GIT_VERSION_SHORT = ($GIT_VERSION -Split \'.windows.\')[0]',
                'Invoke-WebRequest -UseBasicParsing -Uri https://github.com/git-for-windows/git/releases/download/v${GIT_VERSION}/Git-${GIT_VERSION_SHORT}-64-bit.exe -OutFile git-setup.exe',
                'Start-Process git-setup.exe -Wait -ArgumentList \'/VERYSILENT\'',
                'del git-setup.exe',
            ],
        }));
        let runnerCommands;
        if (this.runnerVersion.version == common_1.RunnerVersion.latest().version) {
            runnerCommands = [
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest > $Env:TEMP\\latest-gha',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-gha',
                '$RUNNER_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
            ];
        }
        else {
            runnerCommands = [`$RUNNER_VERSION = '${this.runnerVersion.version}'`];
        }
        this.addComponent(new ImageBuilderComponent(this, 'GitHub Actions Runner', {
            platform: 'Windows',
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                '$ProgressPreference = \'SilentlyContinue\'',
            ].concat(runnerCommands, [
                'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/actions/runner/releases/download/v${RUNNER_VERSION}/actions-runner-win-x64-${RUNNER_VERSION}.zip" -OutFile actions.zip',
                'Expand-Archive actions.zip -DestinationPath C:\\actions',
                'del actions.zip',
            ]),
        }));
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        this.prependComponent(new ImageBuilderComponent(this, 'Extra Certs', {
            platform: this.platform,
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'Import-Certificate -FilePath certs\\certs.pem -CertStoreLocation Cert:\\LocalMachine\\Root',
            ],
            assets: [
                {
                    path: 'certs',
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Extra Certs Asset', { path }),
                },
            ],
        }));
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        const infra = this.infrastructure();
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: uniqueName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    containerDistributionConfiguration: {
                        ContainerTags: ['latest'],
                        TargetRepository: {
                            Service: 'ECR',
                            RepositoryName: this.repository.repositoryName,
                        },
                    },
                },
            ],
        });
        const recipe = new ContainerRecipe(this, 'Container Recipe', {
            platform: this.platform,
            components: this.components,
            targetRepository: this.repository,
            dockerfileTemplate: dockerfileTemplate.replace('___RUNNER_VERSION___', this.runnerVersion.version),
        });
        const log = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Log', {
            logGroupName: `/aws/imagebuilder/${recipe.name}`,
            retention: this.logRetention,
            removalPolicy: this.logRemovalPolicy,
        });
        const image = new aws_cdk_lib_1.aws_imagebuilder.CfnImage(this, 'Image', {
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            containerRecipeArn: recipe.arn,
        });
        image.node.addDependency(log);
        this.imageCleaner(image, recipe.name);
        let scheduleOptions;
        if (this.rebuildInterval.toDays() > 0) {
            scheduleOptions = {
                scheduleExpression: aws_cdk_lib_1.aws_events.Schedule.rate(this.rebuildInterval).expressionString,
                pipelineExecutionStartCondition: 'EXPRESSION_MATCH_ONLY',
            };
        }
        const pipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, 'Pipeline', {
            name: uniqueName(this),
            description: this.description,
            containerRecipeArn: recipe.arn,
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            schedule: scheduleOptions,
        });
        pipeline.node.addDependency(log);
        this.boundImage = {
            // There are simpler ways to get the ARN, but we want an image object that depends on the newly built image.
            // We want whoever is using this image to automatically wait for Image Builder to finish building before using the image.
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryName(this, 'Dependable Image', 
            // we can't use image.attrName because it comes up with upper case
            cdk.Fn.split(':', cdk.Fn.split('/', image.attrImageUri, 2)[1], 2)[0]),
            imageTag: 'latest',
            os: this.os,
            architecture: this.architecture,
            logGroup: log,
        };
        return this.boundImage;
    }
    infrastructure() {
        let role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
            ],
        });
        for (const component of this.components) {
            component.grantAssetsRead(role);
        }
        return new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: uniqueName(this),
            description: this.description,
            subnetId: this.subnetId,
            securityGroupIds: this.securityGroupIds,
            instanceTypes: this.instanceTypes,
            instanceProfileName: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                roles: [
                    role.roleName,
                ],
            }).ref,
        });
    }
    imageCleaner(image, recipeName) {
        const crHandler = utils_1.BundledNodejsFunction.singleton(this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['imagebuilder:ListImages', 'imagebuilder:ListImageBuildVersions', 'imagebuilder:DeleteImage'],
                    resources: ['*'],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Deleter', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageDeleter',
            properties: {
                RepoName: this.repository.repositoryName,
                ImageBuilderName: recipeName,
                DeleteOnly: true,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(image);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
}
exports.ContainerImageBuilder = ContainerImageBuilder;
_b = JSII_RTTI_SYMBOL_1;
ContainerImageBuilder[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.ContainerImageBuilder", version: "0.5.3" };
//# sourceMappingURL=data:application/json;base64,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