"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScheduleTargetBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const helpers_internal_1 = require("aws-cdk-lib/core/lib/helpers-internal");
/**
 * Base class for Schedule Targets
 */
class ScheduleTargetBase {
    constructor(baseProps, targetArn) {
        this.baseProps = baseProps;
        this.targetArn = targetArn;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_scheduler_targets_alpha_ScheduleTargetBaseProps(baseProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ScheduleTargetBase);
            }
            throw error;
        }
    }
    bindBaseTargetConfig(_schedule) {
        const role = this.baseProps.role ?? this.createOrGetScheduleTargetRole(_schedule, this.targetArn);
        this.addTargetActionToRole(role);
        if (this.baseProps.deadLetterQueue) {
            this.addDeadLetterQueueActionToRole(role, this.baseProps.deadLetterQueue);
        }
        return {
            arn: this.targetArn,
            role,
            deadLetterConfig: this.baseProps.deadLetterQueue ? {
                arn: this.baseProps.deadLetterQueue.queueArn,
            } : undefined,
            retryPolicy: this.renderRetryPolicy(this.baseProps.maxEventAge, this.baseProps.retryAttempts),
            input: this.baseProps.input,
        };
    }
    /**
     * Create a return a Schedule Target Configuration for the given schedule
     * @param schedule
     * @returns a Schedule Target Configuration
     */
    bind(schedule) {
        return this.bindBaseTargetConfig(schedule);
    }
    /**
     * Get or create the Role for the EventBridge Scheduler event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * schedules have the same target, they will share a role.
     */
    createOrGetScheduleTargetRole(schedule, targetArn) {
        const stack = aws_cdk_lib_1.Stack.of(schedule);
        const arn = aws_cdk_lib_1.Token.isUnresolved(targetArn) ? JSON.stringify(stack.resolve(targetArn)) : targetArn;
        const hash = (0, helpers_internal_1.md5hash)(arn).slice(0, 6);
        const id = 'SchedulerRoleForTarget-' + hash;
        const existingRole = stack.node.tryFindChild(id);
        const principal = new iam.ServicePrincipal('scheduler.amazonaws.com', {
            conditions: {
                StringEquals: {
                    'aws:SourceAccount': schedule.env.account,
                    'aws:SourceArn': schedule.group?.groupArn ?? aws_cdk_lib_1.Stack.of(schedule).formatArn({
                        service: 'scheduler',
                        resource: 'schedule-group',
                        region: schedule.env.region,
                        account: schedule.env.account,
                        resourceName: schedule.group?.groupName ?? 'default',
                    }),
                },
            },
        });
        if (existingRole) {
            existingRole.assumeRolePolicy?.addStatements(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [principal],
                actions: ['sts:AssumeRole'],
            }));
            return existingRole;
        }
        const role = new iam.Role(stack, id, {
            roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: principal,
        });
        return role;
    }
    /**
     * Allow schedule to send events with failed invocation to an Amazon SQS queue.
     */
    addDeadLetterQueueActionToRole(role, queue) {
        role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [queue.queueArn],
        }));
    }
    renderRetryPolicy(maximumEventAge, maximumRetryAttempts) {
        const maxMaxAge = aws_cdk_lib_1.Duration.days(1).toSeconds();
        const minMaxAge = aws_cdk_lib_1.Duration.minutes(1).toSeconds();
        let maxAge = maxMaxAge;
        if (maximumEventAge) {
            maxAge = maximumEventAge.toSeconds({ integral: true });
            if (maxAge > maxMaxAge) {
                throw new Error('Maximum event age is 1 day');
            }
            if (maxAge < minMaxAge) {
                throw new Error('Minimum event age is 1 minute');
            }
        }
        ;
        let maxAttempts = 185;
        if (typeof maximumRetryAttempts != 'undefined') {
            if (maximumRetryAttempts < 0) {
                throw Error('Number of retry attempts should be greater or equal than 0');
            }
            if (maximumRetryAttempts > 185) {
                throw Error('Number of retry attempts should be less or equal than 185');
            }
            maxAttempts = maximumRetryAttempts;
        }
        return {
            maximumEventAgeInSeconds: maxAge,
            maximumRetryAttempts: maxAttempts,
        };
    }
}
exports.ScheduleTargetBase = ScheduleTargetBase;
_a = JSII_RTTI_SYMBOL_1;
ScheduleTargetBase[_a] = { fqn: "@aws-cdk/aws-scheduler-targets-alpha.ScheduleTargetBase", version: "2.173.4-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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