"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseScalableAttribute = void 0;
const scalable_target_1 = require("./scalable-target");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Represent an attribute for which autoscaling can be configured.
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *    Step Scaling, so the Dynamo subclass won't expose this method).
 *
 * @stability stable
 */
class BaseScalableAttribute extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity,
        });
    }
    /**
     * Scale out or in based on time.
     *
     * @stability stable
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value.
     *
     * @stability stable
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value.
     *
     * @stability stable
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,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