"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const codedeploy = require("../../lib");
function mockFunction(stack, id) {
    return new lambda.Function(stack, id, {
        code: lambda.Code.fromInline('mock'),
        handler: 'index.handler',
        runtime: lambda.Runtime.NODEJS_10_X,
    });
}
function mockAlias(stack) {
    return new lambda.Alias(stack, 'Alias', {
        aliasName: 'my-alias',
        version: new lambda.Version(stack, 'Version', {
            lambda: mockFunction(stack, 'Function'),
        }),
    });
}
module.exports = {
    'CodeDeploy Lambda DeploymentGroup': {
        'can be created with default AllAtOnce IN_PLACE configuration'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: 'MyApp3CE31C26',
                },
                ServiceRoleArn: {
                    'Fn::GetAtt': [
                        'MyDGServiceRole5E94FD88',
                        'Arn',
                    ],
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE',
                    ],
                },
                DeploymentConfigName: 'CodeDeployDefault.LambdaAllAtOnce',
                DeploymentStyle: {
                    DeploymentOption: 'WITH_TRAFFIC_CONTROL',
                    DeploymentType: 'BLUE_GREEN',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
                Type: 'AWS::Lambda::Alias',
                Properties: {
                    FunctionName: {
                        Ref: 'Function76856677',
                    },
                    FunctionVersion: {
                        'Fn::GetAtt': [
                            'Version6A868472',
                            'Version',
                        ],
                    },
                    Name: 'my-alias',
                },
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26',
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F',
                        },
                    },
                },
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [{
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: { 'Fn::Join': ['', ['codedeploy.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }]] },
                            },
                        }],
                    Version: '2012-10-17',
                },
                ManagedPolicyArns: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':iam::aws:policy/service-role/AWSCodeDeployRoleForLambdaLimited',
                            ],
                        ],
                    },
                ],
            }));
            test.done();
        },
        'can be created with explicit name'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                deploymentGroupName: 'test',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                DeploymentGroupName: 'test',
            }));
            test.done();
        },
        'can be created with explicit role'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const serviceRole = new iam.Role(stack, 'MyRole', {
                assumedBy: new iam.ServicePrincipal('not-codedeploy.test'),
            });
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                role: serviceRole,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [{
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'not-codedeploy.test',
                            },
                        }],
                    Version: '2012-10-17',
                },
                ManagedPolicyArns: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':iam::aws:policy/service-role/AWSCodeDeployRoleForLambdaLimited',
                            ],
                        ],
                    },
                ],
            }));
            test.done();
        },
        'can configure blue/green traffic shifting'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: 'MyApp3CE31C26',
                },
                ServiceRoleArn: {
                    'Fn::GetAtt': [
                        'MyDGServiceRole5E94FD88',
                        'Arn',
                    ],
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE',
                    ],
                },
                DeploymentConfigName: 'CodeDeployDefault.LambdaLinear10PercentEvery1Minute',
                DeploymentStyle: {
                    DeploymentOption: 'WITH_TRAFFIC_CONTROL',
                    DeploymentType: 'BLUE_GREEN',
                },
            }));
            test.done();
        },
        'can rollback on alarm'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1,
                    })],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AlarmConfiguration: {
                    Alarms: [{
                            Name: {
                                Ref: 'Failures8A3E1A2F',
                            },
                        }],
                    Enabled: true,
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE',
                        'DEPLOYMENT_STOP_ON_ALARM',
                    ],
                },
            }));
            test.done();
        },
        'onPreHook throws error if pre-hook already defined'(test) {
            const stack = new cdk.Stack();
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                alias,
                preHook: mockFunction(stack, 'PreHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
            });
            test.throws(() => group.addPreHook(mockFunction(stack, 'PreHook2')));
            test.done();
        },
        'onPostHook throws error if post-hook already defined'(test) {
            const stack = new cdk.Stack();
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
            });
            test.throws(() => group.addPostHook(mockFunction(stack, 'PostHook2')));
            test.done();
        },
        'can run pre hook lambda function before deployment'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                preHook: mockFunction(stack, 'PreHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26',
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F',
                        },
                        BeforeAllowTrafficHook: {
                            Ref: 'PreHook8B53F672',
                        },
                    },
                },
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: 'MyDGServiceRoleDefaultPolicy65E8E1EA',
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88',
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                'Fn::GetAtt': [
                                    'PreHook8B53F672',
                                    'Arn',
                                ],
                            },
                            Effect: 'Allow',
                        }],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'can add pre hook lambda function after creating the deployment group'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
            });
            group.addPreHook(mockFunction(stack, 'PreHook'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26',
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F',
                        },
                        BeforeAllowTrafficHook: {
                            Ref: 'PreHook8B53F672',
                        },
                    },
                },
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: 'MyDGServiceRoleDefaultPolicy65E8E1EA',
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88',
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                'Fn::GetAtt': [
                                    'PreHook8B53F672',
                                    'Arn',
                                ],
                            },
                            Effect: 'Allow',
                        }],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'can run post hook lambda function before deployment'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26',
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F',
                        },
                        AfterAllowTrafficHook: {
                            Ref: 'PostHookF2E49B30',
                        },
                    },
                },
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: 'MyDGServiceRoleDefaultPolicy65E8E1EA',
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88',
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                'Fn::GetAtt': [
                                    'PostHookF2E49B30',
                                    'Arn',
                                ],
                            },
                            Effect: 'Allow',
                        }],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'can add post hook lambda function after creating the deployment group'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
            });
            group.addPostHook(mockFunction(stack, 'PostHook'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26',
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F',
                        },
                        AfterAllowTrafficHook: {
                            Ref: 'PostHookF2E49B30',
                        },
                    },
                },
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: 'MyDGServiceRoleDefaultPolicy65E8E1EA',
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88',
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                'Fn::GetAtt': [
                                    'PostHookF2E49B30',
                                    'Arn',
                                ],
                            },
                            Effect: 'Allow',
                        }],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'can disable rollback when alarm polling fails'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                ignorePollAlarmsFailure: true,
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1,
                    })],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AlarmConfiguration: {
                    Alarms: [{
                            Name: {
                                Ref: 'Failures8A3E1A2F',
                            },
                        }],
                    Enabled: true,
                    IgnorePollAlarmFailure: true,
                },
            }));
            test.done();
        },
        'can disable rollback when deployment fails'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    failedDeployment: false,
                },
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: 'MyApp3CE31C26',
                },
                ServiceRoleArn: {
                    'Fn::GetAtt': [
                        'MyDGServiceRole5E94FD88',
                        'Arn',
                    ],
                },
                DeploymentConfigName: 'CodeDeployDefault.LambdaAllAtOnce',
                DeploymentStyle: {
                    DeploymentOption: 'WITH_TRAFFIC_CONTROL',
                    DeploymentType: 'BLUE_GREEN',
                },
            }));
            test.done();
        },
        'can enable rollback when deployment stops'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    stoppedDeployment: true,
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE',
                        'DEPLOYMENT_STOP_ON_REQUEST',
                    ],
                },
            }));
            test.done();
        },
        'can disable rollback when alarm in failure state'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    deploymentInAlarm: false,
                },
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1,
                    })],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE',
                    ],
                },
            }));
            test.done();
        },
        'imported with fromLambdaDeploymentGroupAttributes': {
            'defaults the Deployment Config to Canary10Percent5Minutes'(test) {
                const stack = new cdk.Stack();
                const lambdaApp = codedeploy.LambdaApplication.fromLambdaApplicationName(stack, 'LA', 'LambdaApplication');
                const importedGroup = codedeploy.LambdaDeploymentGroup.fromLambdaDeploymentGroupAttributes(stack, 'LDG', {
                    application: lambdaApp,
                    deploymentGroupName: 'LambdaDeploymentGroup',
                });
                test.equal(importedGroup.deploymentConfig, codedeploy.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES);
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5kZXBsb3ltZW50LWdyb3VwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5kZXBsb3ltZW50LWdyb3VwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBdUY7QUFDdkYsc0RBQXNEO0FBQ3RELHdDQUF3QztBQUN4Qyw4Q0FBOEM7QUFDOUMscUNBQXFDO0FBRXJDLHdDQUF3QztBQUV4QyxTQUFTLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVU7SUFDaEQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtRQUNwQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDO1FBQ3BDLE9BQU8sRUFBRSxlQUFlO1FBQ3hCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7S0FDcEMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUNELFNBQVMsU0FBUyxDQUFDLEtBQWdCO0lBQ2pDLE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7UUFDdEMsU0FBUyxFQUFFLFVBQVU7UUFDckIsT0FBTyxFQUFFLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQzVDLE1BQU0sRUFBRSxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQztTQUN4QyxDQUFDO0tBQ0gsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELGlCQUFTO0lBQ1AsbUNBQW1DLEVBQUU7UUFDbkMsOERBQThELENBQUMsSUFBVTtZQUN2RSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFdBQVcsR0FBRyxJQUFJLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDckUsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQy9CLElBQUksVUFBVSxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ2xELFdBQVc7Z0JBQ1gsS0FBSztnQkFDTCxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsc0JBQXNCLENBQUMsV0FBVzthQUNoRSxDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ2hFLGVBQWUsRUFBRTtvQkFDZixHQUFHLEVBQUUsZUFBZTtpQkFDckI7Z0JBQ0QsY0FBYyxFQUFFO29CQUNkLFlBQVksRUFBRTt3QkFDWix5QkFBeUI7d0JBQ3pCLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QseUJBQXlCLEVBQUU7b0JBQ3pCLE9BQU8sRUFBRSxJQUFJO29CQUNiLE1BQU0sRUFBRTt3QkFDTixvQkFBb0I7cUJBQ3JCO2lCQUNGO2dCQUNELG9CQUFvQixFQUFFLG1DQUFtQztnQkFDekQsZUFBZSxFQUFFO29CQUNmLGdCQUFnQixFQUFFLHNCQUFzQjtvQkFDeEMsY0FBYyxFQUFFLFlBQVk7aUJBQzdCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsb0JBQW9CLEVBQUU7Z0JBQ2xELElBQUksRUFBRSxvQkFBb0I7Z0JBQzFCLFVBQVUsRUFBRTtvQkFDVixZQUFZLEVBQUU7d0JBQ1osR0FBRyxFQUFFLGtCQUFrQjtxQkFDeEI7b0JBQ0QsZUFBZSxFQUFFO3dCQUNmLFlBQVksRUFBRTs0QkFDWixpQkFBaUI7NEJBQ2pCLFNBQVM7eUJBQ1Y7cUJBQ0Y7b0JBQ0QsSUFBSSxFQUFFLFVBQVU7aUJBQ2pCO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkIsRUFBRTt3QkFDM0IsZUFBZSxFQUFFOzRCQUNmLEdBQUcsRUFBRSxlQUFlO3lCQUNyQjt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsR0FBRyxFQUFFLGNBQWM7eUJBQ3BCO3FCQUNGO2lCQUNGO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUVyQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQzlDLHdCQUF3QixFQUFFO29CQUN4QixTQUFTLEVBQUUsQ0FBQzs0QkFDVixNQUFNLEVBQUUsZ0JBQWdCOzRCQUN4QixNQUFNLEVBQUUsT0FBTzs0QkFDZixTQUFTLEVBQUU7Z0NBQ1QsT0FBTyxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsYUFBYSxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsRUFBRTs2QkFDdkc7eUJBQ0YsQ0FBQztvQkFDRixPQUFPLEVBQUUsWUFBWTtpQkFDdEI7Z0JBQ0QsaUJBQWlCLEVBQUU7b0JBQ2pCO3dCQUNFLFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNFLE1BQU07Z0NBQ04sRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7Z0NBQ3pCLGlFQUFpRTs2QkFDbEU7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxtQ0FBbUMsQ0FBQyxJQUFVO1lBQzVDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxXQUFXO2dCQUMvRCxtQkFBbUIsRUFBRSxNQUFNO2FBQzVCLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ3BFLG1CQUFtQixFQUFFLE1BQU07YUFDNUIsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsbUNBQW1DLENBQUMsSUFBVTtZQUM1QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFdBQVcsR0FBRyxJQUFJLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDckUsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQy9CLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNoRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMscUJBQXFCLENBQUM7YUFDM0QsQ0FBQyxDQUFDO1lBRUgsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxXQUFXO2dCQUMvRCxJQUFJLEVBQUUsV0FBVzthQUNsQixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQzlDLHdCQUF3QixFQUFFO29CQUN4QixTQUFTLEVBQUUsQ0FBQzs0QkFDVixNQUFNLEVBQUUsZ0JBQWdCOzRCQUN4QixNQUFNLEVBQUUsT0FBTzs0QkFDZixTQUFTLEVBQUU7Z0NBQ1QsT0FBTyxFQUFFLHFCQUFxQjs2QkFDL0I7eUJBQ0YsQ0FBQztvQkFDRixPQUFPLEVBQUUsWUFBWTtpQkFDdEI7Z0JBQ0QsaUJBQWlCLEVBQUU7b0JBQ2pCO3dCQUNFLFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNFLE1BQU07Z0NBQ04sRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7Z0NBQ3pCLGlFQUFpRTs2QkFDbEU7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCwyQ0FBMkMsQ0FBQyxJQUFVO1lBQ3BELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyw4QkFBOEI7YUFDbkYsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtDQUFrQyxFQUFFO2dCQUNoRSxlQUFlLEVBQUU7b0JBQ2YsR0FBRyxFQUFFLGVBQWU7aUJBQ3JCO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxZQUFZLEVBQUU7d0JBQ1oseUJBQXlCO3dCQUN6QixLQUFLO3FCQUNOO2lCQUNGO2dCQUNELHlCQUF5QixFQUFFO29CQUN6QixPQUFPLEVBQUUsSUFBSTtvQkFDYixNQUFNLEVBQUU7d0JBQ04sb0JBQW9CO3FCQUNyQjtpQkFDRjtnQkFDRCxvQkFBb0IsRUFBRSxxREFBcUQ7Z0JBQzNFLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRSxzQkFBc0I7b0JBQ3hDLGNBQWMsRUFBRSxZQUFZO2lCQUM3QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELHVCQUF1QixDQUFDLElBQVU7WUFDaEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsZ0JBQWdCLEVBQUUsVUFBVSxDQUFDLHNCQUFzQixDQUFDLFdBQVc7Z0JBQy9ELE1BQU0sRUFBRSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO3dCQUMvQyxNQUFNLEVBQUUsS0FBSyxDQUFDLFlBQVksRUFBRTt3QkFDNUIsa0JBQWtCLEVBQUUsVUFBVSxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQjt3QkFDeEUsU0FBUyxFQUFFLENBQUM7d0JBQ1osaUJBQWlCLEVBQUUsQ0FBQztxQkFDckIsQ0FBQyxDQUFDO2FBQ0osQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDcEUsa0JBQWtCLEVBQUU7b0JBQ2xCLE1BQU0sRUFBRSxDQUFDOzRCQUNQLElBQUksRUFBRTtnQ0FDSixHQUFHLEVBQUUsa0JBQWtCOzZCQUN4Qjt5QkFDRixDQUFDO29CQUNGLE9BQU8sRUFBRSxJQUFJO2lCQUNkO2dCQUNELHlCQUF5QixFQUFFO29CQUN6QixPQUFPLEVBQUUsSUFBSTtvQkFDYixNQUFNLEVBQUU7d0JBQ04sb0JBQW9CO3dCQUNwQiwwQkFBMEI7cUJBQzNCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0Qsb0RBQW9ELENBQUMsSUFBVTtZQUM3RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDaEUsS0FBSztnQkFDTCxPQUFPLEVBQUUsWUFBWSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUM7Z0JBQ3ZDLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxXQUFXO2FBQ2hFLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNyRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0Qsc0RBQXNELENBQUMsSUFBVTtZQUMvRCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDaEUsS0FBSztnQkFDTCxRQUFRLEVBQUUsWUFBWSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUM7Z0JBQ3pDLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxXQUFXO2FBQ2hFLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN2RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0Qsb0RBQW9ELENBQUMsSUFBVTtZQUM3RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFdBQVcsR0FBRyxJQUFJLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDckUsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQy9CLElBQUksVUFBVSxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ2xELFdBQVc7Z0JBQ1gsS0FBSztnQkFDTCxPQUFPLEVBQUUsWUFBWSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUM7Z0JBQ3ZDLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxXQUFXO2FBQ2hFLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0JBQW9CLEVBQUU7Z0JBQ3RELFlBQVksRUFBRTtvQkFDWiwyQkFBMkIsRUFBRTt3QkFDM0IsZUFBZSxFQUFFOzRCQUNmLEdBQUcsRUFBRSxlQUFlO3lCQUNyQjt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsR0FBRyxFQUFFLGNBQWM7eUJBQ3BCO3dCQUNELHNCQUFzQixFQUFFOzRCQUN0QixHQUFHLEVBQUUsaUJBQWlCO3lCQUN2QjtxQkFDRjtpQkFDRjthQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7WUFFckMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO2dCQUNoRCxVQUFVLEVBQUUsc0NBQXNDO2dCQUNsRCxLQUFLLEVBQUUsQ0FBQzt3QkFDTixHQUFHLEVBQUUseUJBQXlCO3FCQUMvQixDQUFDO2dCQUNGLGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUUsQ0FBQzs0QkFDVixNQUFNLEVBQUUsdUJBQXVCOzRCQUMvQixRQUFRLEVBQUU7Z0NBQ1IsWUFBWSxFQUFFO29DQUNaLGlCQUFpQjtvQ0FDakIsS0FBSztpQ0FDTjs2QkFDRjs0QkFDRCxNQUFNLEVBQUUsT0FBTzt5QkFDaEIsQ0FBQztvQkFDRixPQUFPLEVBQUUsWUFBWTtpQkFDdEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxzRUFBc0UsQ0FBQyxJQUFVO1lBQy9FLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDaEUsV0FBVztnQkFDWCxLQUFLO2dCQUNMLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxXQUFXO2FBQ2hFLENBQUMsQ0FBQztZQUNILEtBQUssQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO1lBRWpELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0JBQW9CLEVBQUU7Z0JBQ3RELFlBQVksRUFBRTtvQkFDWiwyQkFBMkIsRUFBRTt3QkFDM0IsZUFBZSxFQUFFOzRCQUNmLEdBQUcsRUFBRSxlQUFlO3lCQUNyQjt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsR0FBRyxFQUFFLGNBQWM7eUJBQ3BCO3dCQUNELHNCQUFzQixFQUFFOzRCQUN0QixHQUFHLEVBQUUsaUJBQWlCO3lCQUN2QjtxQkFDRjtpQkFDRjthQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7WUFFckMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO2dCQUNoRCxVQUFVLEVBQUUsc0NBQXNDO2dCQUNsRCxLQUFLLEVBQUUsQ0FBQzt3QkFDTixHQUFHLEVBQUUseUJBQXlCO3FCQUMvQixDQUFDO2dCQUNGLGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUUsQ0FBQzs0QkFDVixNQUFNLEVBQUUsdUJBQXVCOzRCQUMvQixRQUFRLEVBQUU7Z0NBQ1IsWUFBWSxFQUFFO29DQUNaLGlCQUFpQjtvQ0FDakIsS0FBSztpQ0FDTjs2QkFDRjs0QkFDRCxNQUFNLEVBQUUsT0FBTzt5QkFDaEIsQ0FBQztvQkFDRixPQUFPLEVBQUUsWUFBWTtpQkFDdEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxxREFBcUQsQ0FBQyxJQUFVO1lBQzlELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLFFBQVEsRUFBRSxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQztnQkFDekMsZ0JBQWdCLEVBQUUsVUFBVSxDQUFDLHNCQUFzQixDQUFDLFdBQVc7YUFDaEUsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQkFBb0IsRUFBRTtnQkFDdEQsWUFBWSxFQUFFO29CQUNaLDJCQUEyQixFQUFFO3dCQUMzQixlQUFlLEVBQUU7NEJBQ2YsR0FBRyxFQUFFLGVBQWU7eUJBQ3JCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNuQixHQUFHLEVBQUUsY0FBYzt5QkFDcEI7d0JBQ0QscUJBQXFCLEVBQUU7NEJBQ3JCLEdBQUcsRUFBRSxrQkFBa0I7eUJBQ3hCO3FCQUNGO2lCQUNGO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUVyQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELFVBQVUsRUFBRSxzQ0FBc0M7Z0JBQ2xELEtBQUssRUFBRSxDQUFDO3dCQUNOLEdBQUcsRUFBRSx5QkFBeUI7cUJBQy9CLENBQUM7Z0JBQ0YsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRSxDQUFDOzRCQUNWLE1BQU0sRUFBRSx1QkFBdUI7NEJBQy9CLFFBQVEsRUFBRTtnQ0FDUixZQUFZLEVBQUU7b0NBQ1osa0JBQWtCO29DQUNsQixLQUFLO2lDQUNOOzZCQUNGOzRCQUNELE1BQU0sRUFBRSxPQUFPO3lCQUNoQixDQUFDO29CQUNGLE9BQU8sRUFBRSxZQUFZO2lCQUN0QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELHVFQUF1RSxDQUFDLElBQVU7WUFDaEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNoRSxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsZ0JBQWdCLEVBQUUsVUFBVSxDQUFDLHNCQUFzQixDQUFDLFdBQVc7YUFDaEUsQ0FBQyxDQUFDO1lBQ0gsS0FBSyxDQUFDLFdBQVcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFFbkQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQkFBb0IsRUFBRTtnQkFDdEQsWUFBWSxFQUFFO29CQUNaLDJCQUEyQixFQUFFO3dCQUMzQixlQUFlLEVBQUU7NEJBQ2YsR0FBRyxFQUFFLGVBQWU7eUJBQ3JCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNuQixHQUFHLEVBQUUsY0FBYzt5QkFDcEI7d0JBQ0QscUJBQXFCLEVBQUU7NEJBQ3JCLEdBQUcsRUFBRSxrQkFBa0I7eUJBQ3hCO3FCQUNGO2lCQUNGO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUVyQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELFVBQVUsRUFBRSxzQ0FBc0M7Z0JBQ2xELEtBQUssRUFBRSxDQUFDO3dCQUNOLEdBQUcsRUFBRSx5QkFBeUI7cUJBQy9CLENBQUM7Z0JBQ0YsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRSxDQUFDOzRCQUNWLE1BQU0sRUFBRSx1QkFBdUI7NEJBQy9CLFFBQVEsRUFBRTtnQ0FDUixZQUFZLEVBQUU7b0NBQ1osa0JBQWtCO29DQUNsQixLQUFLO2lDQUNOOzZCQUNGOzRCQUNELE1BQU0sRUFBRSxPQUFPO3lCQUNoQixDQUFDO29CQUNGLE9BQU8sRUFBRSxZQUFZO2lCQUN0QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELCtDQUErQyxDQUFDLElBQVU7WUFDeEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsUUFBUSxFQUFFLFlBQVksQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDO2dCQUN6QyxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsc0JBQXNCLENBQUMsV0FBVztnQkFDL0QsdUJBQXVCLEVBQUUsSUFBSTtnQkFDN0IsTUFBTSxFQUFFLENBQUMsSUFBSSxVQUFVLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7d0JBQy9DLE1BQU0sRUFBRSxLQUFLLENBQUMsWUFBWSxFQUFFO3dCQUM1QixrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCO3dCQUN4RSxTQUFTLEVBQUUsQ0FBQzt3QkFDWixpQkFBaUIsRUFBRSxDQUFDO3FCQUNyQixDQUFDLENBQUM7YUFDSixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGtDQUFrQyxFQUFFO2dCQUNwRSxrQkFBa0IsRUFBRTtvQkFDbEIsTUFBTSxFQUFFLENBQUM7NEJBQ1AsSUFBSSxFQUFFO2dDQUNKLEdBQUcsRUFBRSxrQkFBa0I7NkJBQ3hCO3lCQUNGLENBQUM7b0JBQ0YsT0FBTyxFQUFFLElBQUk7b0JBQ2Isc0JBQXNCLEVBQUUsSUFBSTtpQkFDN0I7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCw0Q0FBNEMsQ0FBQyxJQUFVO1lBQ3JELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLFFBQVEsRUFBRSxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQztnQkFDekMsZ0JBQWdCLEVBQUUsVUFBVSxDQUFDLHNCQUFzQixDQUFDLFdBQVc7Z0JBQy9ELFlBQVksRUFBRTtvQkFDWixnQkFBZ0IsRUFBRSxLQUFLO2lCQUN4QjthQUNGLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDaEUsZUFBZSxFQUFFO29CQUNmLEdBQUcsRUFBRSxlQUFlO2lCQUNyQjtnQkFDRCxjQUFjLEVBQUU7b0JBQ2QsWUFBWSxFQUFFO3dCQUNaLHlCQUF5Qjt3QkFDekIsS0FBSztxQkFDTjtpQkFDRjtnQkFDRCxvQkFBb0IsRUFBRSxtQ0FBbUM7Z0JBQ3pELGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRSxzQkFBc0I7b0JBQ3hDLGNBQWMsRUFBRSxZQUFZO2lCQUM3QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELDJDQUEyQyxDQUFDLElBQVU7WUFDcEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsUUFBUSxFQUFFLFlBQVksQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDO2dCQUN6QyxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsc0JBQXNCLENBQUMsV0FBVztnQkFDL0QsWUFBWSxFQUFFO29CQUNaLGlCQUFpQixFQUFFLElBQUk7aUJBQ3hCO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDcEUseUJBQXlCLEVBQUU7b0JBQ3pCLE9BQU8sRUFBRSxJQUFJO29CQUNiLE1BQU0sRUFBRTt3QkFDTixvQkFBb0I7d0JBQ3BCLDRCQUE0QjtxQkFDN0I7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxrREFBa0QsQ0FBQyxJQUFVO1lBQzNELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLFFBQVEsRUFBRSxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQztnQkFDekMsZ0JBQWdCLEVBQUUsVUFBVSxDQUFDLHNCQUFzQixDQUFDLFdBQVc7Z0JBQy9ELFlBQVksRUFBRTtvQkFDWixpQkFBaUIsRUFBRSxLQUFLO2lCQUN6QjtnQkFDRCxNQUFNLEVBQUUsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTt3QkFDL0MsTUFBTSxFQUFFLEtBQUssQ0FBQyxZQUFZLEVBQUU7d0JBQzVCLGtCQUFrQixFQUFFLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxzQkFBc0I7d0JBQ3hFLFNBQVMsRUFBRSxDQUFDO3dCQUNaLGlCQUFpQixFQUFFLENBQUM7cUJBQ3JCLENBQUMsQ0FBQzthQUNKLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ3BFLHlCQUF5QixFQUFFO29CQUN6QixPQUFPLEVBQUUsSUFBSTtvQkFDYixNQUFNLEVBQUU7d0JBQ04sb0JBQW9CO3FCQUNyQjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG1EQUFtRCxFQUFFO1lBQ25ELDJEQUEyRCxDQUFDLElBQVU7Z0JBQ3BFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUU5QixNQUFNLFNBQVMsR0FBRyxVQUFVLENBQUMsaUJBQWlCLENBQUMseUJBQXlCLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO2dCQUMzRyxNQUFNLGFBQWEsR0FBRyxVQUFVLENBQUMscUJBQXFCLENBQUMsbUNBQW1DLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtvQkFDdkcsV0FBVyxFQUFFLFNBQVM7b0JBQ3RCLG1CQUFtQixFQUFFLHVCQUF1QjtpQkFDN0MsQ0FBQyxDQUFDO2dCQUVILElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO2dCQUV4RyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1NBQ0Y7S0FDRjtDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QsIGhhdmVSZXNvdXJjZSwgaGF2ZVJlc291cmNlTGlrZSwgUmVzb3VyY2VQYXJ0IH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCAqIGFzIGNvZGVkZXBsb3kgZnJvbSAnLi4vLi4vbGliJztcblxuZnVuY3Rpb24gbW9ja0Z1bmN0aW9uKHN0YWNrOiBjZGsuU3RhY2ssIGlkOiBzdHJpbmcpIHtcbiAgcmV0dXJuIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssIGlkLCB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUlubGluZSgnbW9jaycpLFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgfSk7XG59XG5mdW5jdGlvbiBtb2NrQWxpYXMoc3RhY2s6IGNkay5TdGFjaykge1xuICByZXR1cm4gbmV3IGxhbWJkYS5BbGlhcyhzdGFjaywgJ0FsaWFzJywge1xuICAgIGFsaWFzTmFtZTogJ215LWFsaWFzJyxcbiAgICB2ZXJzaW9uOiBuZXcgbGFtYmRhLlZlcnNpb24oc3RhY2ssICdWZXJzaW9uJywge1xuICAgICAgbGFtYmRhOiBtb2NrRnVuY3Rpb24oc3RhY2ssICdGdW5jdGlvbicpLFxuICAgIH0pLFxuICB9KTtcbn1cblxuZXhwb3J0ID0ge1xuICAnQ29kZURlcGxveSBMYW1iZGEgRGVwbG95bWVudEdyb3VwJzoge1xuICAgICdjYW4gYmUgY3JlYXRlZCB3aXRoIGRlZmF1bHQgQWxsQXRPbmNlIElOX1BMQUNFIGNvbmZpZ3VyYXRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBuZXcgY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdNeURHJywge1xuICAgICAgICBhcHBsaWNhdGlvbixcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIEFwcGxpY2F0aW9uTmFtZToge1xuICAgICAgICAgIFJlZjogJ015QXBwM0NFMzFDMjYnLFxuICAgICAgICB9LFxuICAgICAgICBTZXJ2aWNlUm9sZUFybjoge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgJ015REdTZXJ2aWNlUm9sZTVFOTRGRDg4JyxcbiAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIEF1dG9Sb2xsYmFja0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgICAgIEV2ZW50czogW1xuICAgICAgICAgICAgJ0RFUExPWU1FTlRfRkFJTFVSRScsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICAgRGVwbG95bWVudENvbmZpZ05hbWU6ICdDb2RlRGVwbG95RGVmYXVsdC5MYW1iZGFBbGxBdE9uY2UnLFxuICAgICAgICBEZXBsb3ltZW50U3R5bGU6IHtcbiAgICAgICAgICBEZXBsb3ltZW50T3B0aW9uOiAnV0lUSF9UUkFGRklDX0NPTlRST0wnLFxuICAgICAgICAgIERlcGxveW1lbnRUeXBlOiAnQkxVRV9HUkVFTicsXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6QWxpYXMnLCB7XG4gICAgICAgIFR5cGU6ICdBV1M6OkxhbWJkYTo6QWxpYXMnLFxuICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgRnVuY3Rpb25OYW1lOiB7XG4gICAgICAgICAgICBSZWY6ICdGdW5jdGlvbjc2ODU2Njc3JyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIEZ1bmN0aW9uVmVyc2lvbjoge1xuICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICdWZXJzaW9uNkE4Njg0NzInLFxuICAgICAgICAgICAgICAnVmVyc2lvbicsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgTmFtZTogJ215LWFsaWFzJyxcbiAgICAgICAgfSxcbiAgICAgICAgVXBkYXRlUG9saWN5OiB7XG4gICAgICAgICAgQ29kZURlcGxveUxhbWJkYUFsaWFzVXBkYXRlOiB7XG4gICAgICAgICAgICBBcHBsaWNhdGlvbk5hbWU6IHtcbiAgICAgICAgICAgICAgUmVmOiAnTXlBcHAzQ0UzMUMyNicsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRGVwbG95bWVudEdyb3VwTmFtZToge1xuICAgICAgICAgICAgICBSZWY6ICdNeURHQzM1MEJEM0YnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogW3tcbiAgICAgICAgICAgIEFjdGlvbjogJ3N0czpBc3N1bWVSb2xlJyxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgICBTZXJ2aWNlOiB7ICdGbjo6Sm9pbic6IFsnJywgWydjb2RlZGVwbG95LicsIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sICcuJywgeyBSZWY6ICdBV1M6OlVSTFN1ZmZpeCcgfV1dIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH1dLFxuICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgfSxcbiAgICAgICAgTWFuYWdlZFBvbGljeUFybnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgIHsgUmVmOiAnQVdTOjpQYXJ0aXRpb24nIH0sXG4gICAgICAgICAgICAgICAgJzppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FXU0NvZGVEZXBsb3lSb2xlRm9yTGFtYmRhTGltaXRlZCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgJ2NhbiBiZSBjcmVhdGVkIHdpdGggZXhwbGljaXQgbmFtZScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50Q29uZmlnLkFMTF9BVF9PTkNFLFxuICAgICAgICBkZXBsb3ltZW50R3JvdXBOYW1lOiAndGVzdCcsXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgRGVwbG95bWVudEdyb3VwTmFtZTogJ3Rlc3QnLFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdjYW4gYmUgY3JlYXRlZCB3aXRoIGV4cGxpY2l0IHJvbGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBjb25zdCBzZXJ2aWNlUm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ015Um9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ25vdC1jb2RlZGVwbG95LnRlc3QnKSxcbiAgICAgIH0pO1xuXG4gICAgICBuZXcgY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdNeURHJywge1xuICAgICAgICBhcHBsaWNhdGlvbixcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgICAgcm9sZTogc2VydmljZVJvbGUsXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpSb2xlJywge1xuICAgICAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgICBTdGF0ZW1lbnQ6IFt7XG4gICAgICAgICAgICBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgU2VydmljZTogJ25vdC1jb2RlZGVwbG95LnRlc3QnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9XSxcbiAgICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgIH0sXG4gICAgICAgIE1hbmFnZWRQb2xpY3lBcm5zOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICB7IFJlZjogJ0FXUzo6UGFydGl0aW9uJyB9LFxuICAgICAgICAgICAgICAgICc6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NDb2RlRGVwbG95Um9sZUZvckxhbWJkYUxpbWl0ZWQnLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdjYW4gY29uZmlndXJlIGJsdWUvZ3JlZW4gdHJhZmZpYyBzaGlmdGluZycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50Q29uZmlnLkxJTkVBUl8xMFBFUkNFTlRfRVZFUllfMU1JTlVURSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIEFwcGxpY2F0aW9uTmFtZToge1xuICAgICAgICAgIFJlZjogJ015QXBwM0NFMzFDMjYnLFxuICAgICAgICB9LFxuICAgICAgICBTZXJ2aWNlUm9sZUFybjoge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgJ015REdTZXJ2aWNlUm9sZTVFOTRGRDg4JyxcbiAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIEF1dG9Sb2xsYmFja0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgICAgIEV2ZW50czogW1xuICAgICAgICAgICAgJ0RFUExPWU1FTlRfRkFJTFVSRScsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICAgRGVwbG95bWVudENvbmZpZ05hbWU6ICdDb2RlRGVwbG95RGVmYXVsdC5MYW1iZGFMaW5lYXIxMFBlcmNlbnRFdmVyeTFNaW51dGUnLFxuICAgICAgICBEZXBsb3ltZW50U3R5bGU6IHtcbiAgICAgICAgICBEZXBsb3ltZW50T3B0aW9uOiAnV0lUSF9UUkFGRklDX0NPTlRST0wnLFxuICAgICAgICAgIERlcGxveW1lbnRUeXBlOiAnQkxVRV9HUkVFTicsXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgJ2NhbiByb2xsYmFjayBvbiBhbGFybScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50Q29uZmlnLkFMTF9BVF9PTkNFLFxuICAgICAgICBhbGFybXM6IFtuZXcgY2xvdWR3YXRjaC5BbGFybShzdGFjaywgJ0ZhaWx1cmVzJywge1xuICAgICAgICAgIG1ldHJpYzogYWxpYXMubWV0cmljRXJyb3JzKCksXG4gICAgICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBjbG91ZHdhdGNoLkNvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fVEhSRVNIT0xELFxuICAgICAgICAgIHRocmVzaG9sZDogMSxcbiAgICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgICAgfSldLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIEFsYXJtQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIEFsYXJtczogW3tcbiAgICAgICAgICAgIE5hbWU6IHtcbiAgICAgICAgICAgICAgUmVmOiAnRmFpbHVyZXM4QTNFMUEyRicsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH1dLFxuICAgICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICAgIEF1dG9Sb2xsYmFja0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgICAgIEV2ZW50czogW1xuICAgICAgICAgICAgJ0RFUExPWU1FTlRfRkFJTFVSRScsXG4gICAgICAgICAgICAnREVQTE9ZTUVOVF9TVE9QX09OX0FMQVJNJyxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdvblByZUhvb2sgdGhyb3dzIGVycm9yIGlmIHByZS1ob29rIGFscmVhZHkgZGVmaW5lZCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBjb25zdCBncm91cCA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFsaWFzLFxuICAgICAgICBwcmVIb29rOiBtb2NrRnVuY3Rpb24oc3RhY2ssICdQcmVIb29rJyksXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgIH0pO1xuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gZ3JvdXAuYWRkUHJlSG9vayhtb2NrRnVuY3Rpb24oc3RhY2ssICdQcmVIb29rMicpKSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdvblBvc3RIb29rIHRocm93cyBlcnJvciBpZiBwb3N0LWhvb2sgYWxyZWFkeSBkZWZpbmVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIGNvbnN0IGdyb3VwID0gbmV3IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnTXlERycsIHtcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIHBvc3RIb29rOiBtb2NrRnVuY3Rpb24oc3RhY2ssICdQb3N0SG9vaycpLFxuICAgICAgICBkZXBsb3ltZW50Q29uZmlnOiBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRDb25maWcuQUxMX0FUX09OQ0UsXG4gICAgICB9KTtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IGdyb3VwLmFkZFBvc3RIb29rKG1vY2tGdW5jdGlvbihzdGFjaywgJ1Bvc3RIb29rMicpKSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdjYW4gcnVuIHByZSBob29rIGxhbWJkYSBmdW5jdGlvbiBiZWZvcmUgZGVwbG95bWVudCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgcHJlSG9vazogbW9ja0Z1bmN0aW9uKHN0YWNrLCAnUHJlSG9vaycpLFxuICAgICAgICBkZXBsb3ltZW50Q29uZmlnOiBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRDb25maWcuQUxMX0FUX09OQ0UsXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkxhbWJkYTo6QWxpYXMnLCB7XG4gICAgICAgIFVwZGF0ZVBvbGljeToge1xuICAgICAgICAgIENvZGVEZXBsb3lMYW1iZGFBbGlhc1VwZGF0ZToge1xuICAgICAgICAgICAgQXBwbGljYXRpb25OYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogJ015QXBwM0NFMzFDMjYnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIERlcGxveW1lbnRHcm91cE5hbWU6IHtcbiAgICAgICAgICAgICAgUmVmOiAnTXlER0MzNTBCRDNGJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBCZWZvcmVBbGxvd1RyYWZmaWNIb29rOiB7XG4gICAgICAgICAgICAgIFJlZjogJ1ByZUhvb2s4QjUzRjY3MicsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgICBQb2xpY3lOYW1lOiAnTXlER1NlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTY1RThFMUVBJyxcbiAgICAgICAgUm9sZXM6IFt7XG4gICAgICAgICAgUmVmOiAnTXlER1NlcnZpY2VSb2xlNUU5NEZEODgnLFxuICAgICAgICB9XSxcbiAgICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgICBTdGF0ZW1lbnQ6IFt7XG4gICAgICAgICAgICBBY3Rpb246ICdsYW1iZGE6SW52b2tlRnVuY3Rpb24nLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgJ1ByZUhvb2s4QjUzRjY3MicsXG4gICAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIH1dLFxuICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnY2FuIGFkZCBwcmUgaG9vayBsYW1iZGEgZnVuY3Rpb24gYWZ0ZXIgY3JlYXRpbmcgdGhlIGRlcGxveW1lbnQgZ3JvdXAnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBjb25zdCBncm91cCA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50Q29uZmlnLkFMTF9BVF9PTkNFLFxuICAgICAgfSk7XG4gICAgICBncm91cC5hZGRQcmVIb29rKG1vY2tGdW5jdGlvbihzdGFjaywgJ1ByZUhvb2snKSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpMYW1iZGE6OkFsaWFzJywge1xuICAgICAgICBVcGRhdGVQb2xpY3k6IHtcbiAgICAgICAgICBDb2RlRGVwbG95TGFtYmRhQWxpYXNVcGRhdGU6IHtcbiAgICAgICAgICAgIEFwcGxpY2F0aW9uTmFtZToge1xuICAgICAgICAgICAgICBSZWY6ICdNeUFwcDNDRTMxQzI2JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBEZXBsb3ltZW50R3JvdXBOYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogJ015REdDMzUwQkQzRicsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgQmVmb3JlQWxsb3dUcmFmZmljSG9vazoge1xuICAgICAgICAgICAgICBSZWY6ICdQcmVIb29rOEI1M0Y2NzInLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgUG9saWN5TmFtZTogJ015REdTZXJ2aWNlUm9sZURlZmF1bHRQb2xpY3k2NUU4RTFFQScsXG4gICAgICAgIFJvbGVzOiBbe1xuICAgICAgICAgIFJlZjogJ015REdTZXJ2aWNlUm9sZTVFOTRGRDg4JyxcbiAgICAgICAgfV0sXG4gICAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICAgICAgQWN0aW9uOiAnbGFtYmRhOkludm9rZUZ1bmN0aW9uJyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICdQcmVIb29rOEI1M0Y2NzInLFxuICAgICAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICB9XSxcbiAgICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgJ2NhbiBydW4gcG9zdCBob29rIGxhbWJkYSBmdW5jdGlvbiBiZWZvcmUgZGVwbG95bWVudCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgcG9zdEhvb2s6IG1vY2tGdW5jdGlvbihzdGFjaywgJ1Bvc3RIb29rJyksXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6TGFtYmRhOjpBbGlhcycsIHtcbiAgICAgICAgVXBkYXRlUG9saWN5OiB7XG4gICAgICAgICAgQ29kZURlcGxveUxhbWJkYUFsaWFzVXBkYXRlOiB7XG4gICAgICAgICAgICBBcHBsaWNhdGlvbk5hbWU6IHtcbiAgICAgICAgICAgICAgUmVmOiAnTXlBcHAzQ0UzMUMyNicsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRGVwbG95bWVudEdyb3VwTmFtZToge1xuICAgICAgICAgICAgICBSZWY6ICdNeURHQzM1MEJEM0YnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIEFmdGVyQWxsb3dUcmFmZmljSG9vazoge1xuICAgICAgICAgICAgICBSZWY6ICdQb3N0SG9va0YyRTQ5QjMwJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICAgIFBvbGljeU5hbWU6ICdNeURHU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NjVFOEUxRUEnLFxuICAgICAgICBSb2xlczogW3tcbiAgICAgICAgICBSZWY6ICdNeURHU2VydmljZVJvbGU1RTk0RkQ4OCcsXG4gICAgICAgIH1dLFxuICAgICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogW3tcbiAgICAgICAgICAgIEFjdGlvbjogJ2xhbWJkYTpJbnZva2VGdW5jdGlvbicsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAnUG9zdEhvb2tGMkU0OUIzMCcsXG4gICAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIH1dLFxuICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnY2FuIGFkZCBwb3N0IGhvb2sgbGFtYmRhIGZ1bmN0aW9uIGFmdGVyIGNyZWF0aW5nIHRoZSBkZXBsb3ltZW50IGdyb3VwJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGFwcGxpY2F0aW9uID0gbmV3IGNvZGVkZXBsb3kuTGFtYmRhQXBwbGljYXRpb24oc3RhY2ssICdNeUFwcCcpO1xuICAgICAgY29uc3QgYWxpYXMgPSBtb2NrQWxpYXMoc3RhY2spO1xuICAgICAgY29uc3QgZ3JvdXAgPSBuZXcgY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdNeURHJywge1xuICAgICAgICBhcHBsaWNhdGlvbixcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgIH0pO1xuICAgICAgZ3JvdXAuYWRkUG9zdEhvb2sobW9ja0Z1bmN0aW9uKHN0YWNrLCAnUG9zdEhvb2snKSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpMYW1iZGE6OkFsaWFzJywge1xuICAgICAgICBVcGRhdGVQb2xpY3k6IHtcbiAgICAgICAgICBDb2RlRGVwbG95TGFtYmRhQWxpYXNVcGRhdGU6IHtcbiAgICAgICAgICAgIEFwcGxpY2F0aW9uTmFtZToge1xuICAgICAgICAgICAgICBSZWY6ICdNeUFwcDNDRTMxQzI2JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBEZXBsb3ltZW50R3JvdXBOYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogJ015REdDMzUwQkQzRicsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgQWZ0ZXJBbGxvd1RyYWZmaWNIb29rOiB7XG4gICAgICAgICAgICAgIFJlZjogJ1Bvc3RIb29rRjJFNDlCMzAnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgUG9saWN5TmFtZTogJ015REdTZXJ2aWNlUm9sZURlZmF1bHRQb2xpY3k2NUU4RTFFQScsXG4gICAgICAgIFJvbGVzOiBbe1xuICAgICAgICAgIFJlZjogJ015REdTZXJ2aWNlUm9sZTVFOTRGRDg4JyxcbiAgICAgICAgfV0sXG4gICAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICAgICAgQWN0aW9uOiAnbGFtYmRhOkludm9rZUZ1bmN0aW9uJyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICdQb3N0SG9va0YyRTQ5QjMwJyxcbiAgICAgICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgfV0sXG4gICAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdjYW4gZGlzYWJsZSByb2xsYmFjayB3aGVuIGFsYXJtIHBvbGxpbmcgZmFpbHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBuZXcgY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdNeURHJywge1xuICAgICAgICBhcHBsaWNhdGlvbixcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIHBvc3RIb29rOiBtb2NrRnVuY3Rpb24oc3RhY2ssICdQb3N0SG9vaycpLFxuICAgICAgICBkZXBsb3ltZW50Q29uZmlnOiBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRDb25maWcuQUxMX0FUX09OQ0UsXG4gICAgICAgIGlnbm9yZVBvbGxBbGFybXNGYWlsdXJlOiB0cnVlLFxuICAgICAgICBhbGFybXM6IFtuZXcgY2xvdWR3YXRjaC5BbGFybShzdGFjaywgJ0ZhaWx1cmVzJywge1xuICAgICAgICAgIG1ldHJpYzogYWxpYXMubWV0cmljRXJyb3JzKCksXG4gICAgICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBjbG91ZHdhdGNoLkNvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fVEhSRVNIT0xELFxuICAgICAgICAgIHRocmVzaG9sZDogMSxcbiAgICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgICAgfSldLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIEFsYXJtQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIEFsYXJtczogW3tcbiAgICAgICAgICAgIE5hbWU6IHtcbiAgICAgICAgICAgICAgUmVmOiAnRmFpbHVyZXM4QTNFMUEyRicsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH1dLFxuICAgICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgSWdub3JlUG9sbEFsYXJtRmFpbHVyZTogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnY2FuIGRpc2FibGUgcm9sbGJhY2sgd2hlbiBkZXBsb3ltZW50IGZhaWxzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGFwcGxpY2F0aW9uID0gbmV3IGNvZGVkZXBsb3kuTGFtYmRhQXBwbGljYXRpb24oc3RhY2ssICdNeUFwcCcpO1xuICAgICAgY29uc3QgYWxpYXMgPSBtb2NrQWxpYXMoc3RhY2spO1xuICAgICAgbmV3IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnTXlERycsIHtcbiAgICAgICAgYXBwbGljYXRpb24sXG4gICAgICAgIGFsaWFzLFxuICAgICAgICBwb3N0SG9vazogbW9ja0Z1bmN0aW9uKHN0YWNrLCAnUG9zdEhvb2snKSxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50Q29uZmlnLkFMTF9BVF9PTkNFLFxuICAgICAgICBhdXRvUm9sbGJhY2s6IHtcbiAgICAgICAgICBmYWlsZWREZXBsb3ltZW50OiBmYWxzZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIEFwcGxpY2F0aW9uTmFtZToge1xuICAgICAgICAgIFJlZjogJ015QXBwM0NFMzFDMjYnLFxuICAgICAgICB9LFxuICAgICAgICBTZXJ2aWNlUm9sZUFybjoge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgJ015REdTZXJ2aWNlUm9sZTVFOTRGRDg4JyxcbiAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIERlcGxveW1lbnRDb25maWdOYW1lOiAnQ29kZURlcGxveURlZmF1bHQuTGFtYmRhQWxsQXRPbmNlJyxcbiAgICAgICAgRGVwbG95bWVudFN0eWxlOiB7XG4gICAgICAgICAgRGVwbG95bWVudE9wdGlvbjogJ1dJVEhfVFJBRkZJQ19DT05UUk9MJyxcbiAgICAgICAgICBEZXBsb3ltZW50VHlwZTogJ0JMVUVfR1JFRU4nLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdjYW4gZW5hYmxlIHJvbGxiYWNrIHdoZW4gZGVwbG95bWVudCBzdG9wcycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgcG9zdEhvb2s6IG1vY2tGdW5jdGlvbihzdGFjaywgJ1Bvc3RIb29rJyksXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgICAgYXV0b1JvbGxiYWNrOiB7XG4gICAgICAgICAgc3RvcHBlZERlcGxveW1lbnQ6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgQXV0b1JvbGxiYWNrQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgRXZlbnRzOiBbXG4gICAgICAgICAgICAnREVQTE9ZTUVOVF9GQUlMVVJFJyxcbiAgICAgICAgICAgICdERVBMT1lNRU5UX1NUT1BfT05fUkVRVUVTVCcsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnY2FuIGRpc2FibGUgcm9sbGJhY2sgd2hlbiBhbGFybSBpbiBmYWlsdXJlIHN0YXRlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGFwcGxpY2F0aW9uID0gbmV3IGNvZGVkZXBsb3kuTGFtYmRhQXBwbGljYXRpb24oc3RhY2ssICdNeUFwcCcpO1xuICAgICAgY29uc3QgYWxpYXMgPSBtb2NrQWxpYXMoc3RhY2spO1xuICAgICAgbmV3IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnTXlERycsIHtcbiAgICAgICAgYXBwbGljYXRpb24sXG4gICAgICAgIGFsaWFzLFxuICAgICAgICBwb3N0SG9vazogbW9ja0Z1bmN0aW9uKHN0YWNrLCAnUG9zdEhvb2snKSxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50Q29uZmlnLkFMTF9BVF9PTkNFLFxuICAgICAgICBhdXRvUm9sbGJhY2s6IHtcbiAgICAgICAgICBkZXBsb3ltZW50SW5BbGFybTogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICAgIGFsYXJtczogW25ldyBjbG91ZHdhdGNoLkFsYXJtKHN0YWNrLCAnRmFpbHVyZXMnLCB7XG4gICAgICAgICAgbWV0cmljOiBhbGlhcy5tZXRyaWNFcnJvcnMoKSxcbiAgICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9USFJFU0hPTEQsXG4gICAgICAgICAgdGhyZXNob2xkOiAxLFxuICAgICAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgICAgICB9KV0sXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgQXV0b1JvbGxiYWNrQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgRXZlbnRzOiBbXG4gICAgICAgICAgICAnREVQTE9ZTUVOVF9GQUlMVVJFJyxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2ltcG9ydGVkIHdpdGggZnJvbUxhbWJkYURlcGxveW1lbnRHcm91cEF0dHJpYnV0ZXMnOiB7XG4gICAgICAnZGVmYXVsdHMgdGhlIERlcGxveW1lbnQgQ29uZmlnIHRvIENhbmFyeTEwUGVyY2VudDVNaW51dGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgICAgIGNvbnN0IGxhbWJkYUFwcCA9IGNvZGVkZXBsb3kuTGFtYmRhQXBwbGljYXRpb24uZnJvbUxhbWJkYUFwcGxpY2F0aW9uTmFtZShzdGFjaywgJ0xBJywgJ0xhbWJkYUFwcGxpY2F0aW9uJyk7XG4gICAgICAgIGNvbnN0IGltcG9ydGVkR3JvdXAgPSBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cC5mcm9tTGFtYmRhRGVwbG95bWVudEdyb3VwQXR0cmlidXRlcyhzdGFjaywgJ0xERycsIHtcbiAgICAgICAgICBhcHBsaWNhdGlvbjogbGFtYmRhQXBwLFxuICAgICAgICAgIGRlcGxveW1lbnRHcm91cE5hbWU6ICdMYW1iZGFEZXBsb3ltZW50R3JvdXAnLFxuICAgICAgICB9KTtcblxuICAgICAgICB0ZXN0LmVxdWFsKGltcG9ydGVkR3JvdXAuZGVwbG95bWVudENvbmZpZywgY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50Q29uZmlnLkNBTkFSWV8xMFBFUkNFTlRfNU1JTlVURVMpO1xuXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcbiAgICB9LFxuICB9LFxufTtcbiJdfQ==