# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['autofff']

package_data = \
{'': ['*']}

install_requires = \
['configobj>=5.0.6,<6.0.0',
 'overrides>=6.1.0,<7.0.0',
 'pycparser>=2.21,<3.0',
 'validator>=0.7.1,<0.8.0']

entry_points = \
{'console_scripts': ['autofff = autofff.__main__:main']}

setup_kwargs = {
    'name': 'autofff',
    'version': '0.4',
    'description': 'Auto-generate FFF fake definitions for C API header files',
    'long_description': '# AutoFFF\n\n[![build](https://github.com/ChiefGokhlayeh/autofff/actions/workflows/build.yml/badge.svg)](https://github.com/ChiefGokhlayeh/autofff/actions/workflows/build.yml)\n[![PyPI version](https://badge.fury.io/py/autofff.svg)](https://badge.fury.io/py/autofff)\n[![License: MIT](https://img.shields.io/badge/License-MIT-blue.svg)](https://opensource.org/licenses/MIT)\n\nAuto-generate [FFF](https://github.com/meekrosoft/fff) fake definitions for C API header files.\n\nIncorporate this script into your normal build environment (like _make_) and automatically generate test-headers with faked function definitions ready-to-use with [FFF](https://github.com/meekrosoft/fff)\'s own _gtest_ or some other unit-testing-framework.\n\n## The Idea Behind Faking\n\nEspecially in the embedded world, compiling your (unit-)tests against the actual target platform isn\'t feasible, as the architecture you\'re executing the test on and your target platform you\'re writing the code for are generally not the same.\n\nThis is where faking a specific platform API might help you. Instead of calling the actual target platform API, you link your code-under-test (CuT) against a faked version of said API. Now, whenever your CuT tries to access the platform API,it is instead calling the fake-implementation which you can easily configure in your test-cases\' setup phase.\n\n[FFF](https://github.com/meekrosoft/fff) (fake-functions-framework) is a framework designed to easily create faked definitions of your API function-declarations, allowing you to configure return values and inspect call and argument histories that were called during the tests\' runtime. Check out their awesome project on [Github](https://github.com/meekrosoft/fff). AutoFFF utilizes the ability to easily create fake definitions provided by FFF, but could also be adapted to other mocking frameworks.\n\nThe problem with faking an API in embedded C is usually the infeasibility of using dynamic linking and C\'s lack of techniques like \'reflection\' to manipulate your CuT during runtime. This makes the process of writing fake definitions a tedious, labor intensive and error prone matter.\n\nIntroducing [_AutoFFF_](https://github.com/ChiefGokhlayeh/autofff), an attempt at automating the process of writing so called test-headers (headers which include the faked definitions).\n\n### Two Philosophies of Faking\n\nWhen writing fakes you will notice that there are two approaches of laying out your fake.\n\n1. **Banning** the original API header\\\n   This strategy _bans_ the original header by defining the API headers include guard, making it impossible to include the original function, variable and type declarations. This gives you ultimate freedom in the test-header, but also means that you will have to manually declare any types, functions and variables the API-user might expect. It also allows you to control the include hierarchy and maybe skip some headers which aren\'t compatible with your test-runner\'s architecture. In general this approach usually involves a lot of copy&pasting and is therefore more prone to _"code rot"_. You also need to deep-dive any header you want to fake, understand its structure and inspect all the declarations and defines very closely. Not the optimal strategy if you\'re looking for an easy-to-maintain way of managing test-headers.\n1. **Wrapping** the original API header\\\n   Conversely to the banning method, the _wrapping_ strategy directly includes the original API header, and thereby imports any type, variable and function declarations. Also the include hierarchy is taken over from the original. The only thing to add into the test-header are the fake definitions. This method evidently grants you less freedom in the test-header, but is usually much shorter and slightly less prone to _"rot"_ over time.\n\nIt should become obvious which method is better suited for automation. _AutoFFF_ follows the _wrapping_ approach of writing test-headers, which for most cases should be good enough.\n\nFinally it must be stated, that these two philosophies seldomly mix well!\n\n## Installation\n\nUse `pip` to download and install _AutoFFF_ from the [PyPi](https://pypi.org/project/autofff/) repositories:\n\n```shell\npy -3.6 -m pip install autofff\n```\n\nOr install from source:\n\n```shell\npy -3.6 -m pip install .\n```\n\nNote that you\'ll most likely require the pycparser `fake_libc_include`s header collection for AutoFFF to work. The `pip` package does **not** ship with this external code. You may download the faked libc headers from [`pycparser`s Github](https://github.com/eliben/pycparser), or check out the project as a submodule (when installing from source run `git submodule update --init`).\n\n## Usage\n\n### As a Module\n\n```shell\npy -3.6 -m autofff \\\n    ./examples/driver.h \\\n    -O ./output/driver_th.h \\\n    -I ./examples \\\n    -F ./dependencies/pycparser/utils/fake_libc_include\n```\n\n### Using the provided Makefile\n\nTo run build and run the tests, simply execute:\n\n```shell\nmake run_tests\n```\n\nYou can also let the makefile do the installation of _AutoFFF_ for you.\n\n```shell\nmake install_autofff\n```\n\n### Running the \'Generate Fakes\' Example\n\n```shell\nmake -f examples/generate-via-makefile/generate_fakes.mk CRAWL_PATHS=examples\n```\n\n### As a Python Package\n\n```python\nimport autofff\n\nimport os.path\n\ntargetHeader = input("Enter the path of the header you would like to scan: ")\noutputHeader = input("Enter the path of the target header file which shall be generated: ")\nfakes = \'./autofff/dependencies/pycparser/utils/fake_libc_include\'\n\nscnr = autofff.GCCScanner(targetHeader, fakes) # Create GCC code scanner\nresult = scnr.scan() # Scan for function declarations and definitions\n\ngen = autofff.SimpleFakeGenerator(os.path.splitext(os.path.basename(outputHeader))[0], targetHeader) # Create new generator with name output-header and path to target-header\n\nif not os.path.exists(os.path.dirname(outputHeader)):\n    dirname = os.path.dirname(outputHeader)\n    os.makedirs(dirname)\n\nwith open(outputHeader, "w") as fs:\n    gen.generate(result, fs) # Generate fff-fakes from scanner-result\n```\n\n## How Fakes Are Generated\n\nThe format of the generated test-header obviously depends on the specifics of the `FakeGenerator` being used.\n\n1. The `BareFakeGenerator` will only generate the `FAKE_VALUE_`- and `FAKE_VOID_FUNC` macros without any decorations, like include guards or header includes. Use this generator if you want to add your own (shell-based-)processing on top.\n2. The `SimpleFakeGenerator` will generate a "minimum viable test header", meaning the result should be compilable without too much effort.\n\n### In-Header Defined Functions\n\nIn some API headers functions may be defined within the header. This will cause issues when trying to fake this function, because by including the header the function definition is copied into each translation unit. If we try to apply a fake definition the usual way, we will end up with a _"redefinition of function *x*"_ error.\n\n_AutoFFF_ implements a workaround to avoid this redefinition error and allowing to fake the original function. This workaround simply consists of some defines which will re-route any call to the original in-header definition to our faked one. For this to work it is required that the test-header is included (and thereby pre-processed) _before_ any function call to the function under consideration is instructed, i.e. the test-header must be included _before_ the CuT. Any function call that is processed before the workaround is being pre-processed will leave this function call targeted towards the original in-header definition.\n\nIn practice the workaround looks like this:\n\n```c\n/* api.h */\n#ifndef API_HEADER_H_\n#define API_HEADER_H_\n\nconst char* foo(void)\n{\n    return "Definitions inside headers are great!";\n}\n\n#endif\n```\n\n```c\n/* api_th.h */\n#ifndef TEST_HEADER_H_\n#define TEST_HEADER_H_\n\n#include "fff.h"\n#include "api.h"\n\n/* Re-route any call to \'foo\' to \'foo_fff\' (our fake definition). */\n#define foo foo_fff\n/* By re-routing the \'_fake\' and \'_reset\' type and function the workaround becomes invisible in the test-case. */\n#define foo_fake Foo_fff_fake\n#define foo_reset Foo_fff_reset\n/* Create the fake definition using the now re-routed \'foo\'-symbol. */\nFAKE_VALUE_FUNC(const char *, foo);\n\n#endif\n```\n\n```c\n/* cut.c - code-under-test */\n#include "api.h"\n#include <stdio.h>\n\nconst char* bar(void)\n{\n    const char* str = foo();\n    return str;\n}\n```\n\n```c\n/* test.c */\n#include "fff.h"\n#include "api_th.h" /* fakes + workaround */\n#include "cut.c"\n\nsetup(void)\n{\n    RESET_FAKE(foo);\n}\n\nTEST_F(foo, ReturnBar_Success)\n{\n    const char* expected_retval = "Definitions inside headers make faking difficult!";\n    foo_fake.return_val = expected_retval\n\n    const char* str = bar();\n\n    ASSERT_STREQ(expected_retval, str);\n}\n```\n\n### Working with _obscure_ include policies\n\nSome libraries like FreeRTOS or CMSIS require you to include their API headers in a very specific way. AutoFFF can\'t guess these policies (yet! ;P) from source-code alone. For cases where the include policy of your vendor lib does not allow each header to be preprocessed individually check out the `-D` (`--define`) and `-i` (`--includefile`) command line options. They may allow you to fix/trick the broken include chain.\nAs an example, for FreeRTOS\' `list.h` run:\n\n```shell\npy -3.6 -m autofff\n    [...]/include/list.h\n    -o [...]\n    -i [...]/include/FreeRTOS.h <<< inject FreeRTOS.h before preprocessing list.h\n    -F [...]\n```\n',
    'author': 'Andreas Baulig',
    'author_email': 'free.geronimo@hotmail.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/ChiefGokhlayeh/autofff',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
