"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabContainerRunner = exports.InstanceInterruptionBehavior = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const ssm = require("aws-cdk-lib/aws-ssm");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * InstanceInterruptionBehavior enum.
 */
var InstanceInterruptionBehavior;
(function (InstanceInterruptionBehavior) {
    /**
     * hibernate
     */
    InstanceInterruptionBehavior["HIBERNATE"] = "hibernate";
    /**
     * stop
     */
    InstanceInterruptionBehavior["STOP"] = "stop";
    /**
     * terminate
     */
    InstanceInterruptionBehavior["TERMINATE"] = "terminate";
})(InstanceInterruptionBehavior = exports.InstanceInterruptionBehavior || (exports.InstanceInterruptionBehavior = {}));
/**
 * GitlabContainerRunner Construct for create a Gitlab Runner.
 */
class GitlabContainerRunner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const spotFleetId = id;
        const defaultProps = {
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:latest',
            gitlaburl: 'https://gitlab.com/',
            ec2type: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            concurrentJobs: 1,
            runnerDescription: 'Docker Runner',
        };
        const runnerProps = { ...defaultProps, ...props };
        const tokenParameterStore = new ssm.StringParameter(this, 'GitlabTokenParameter', {
            stringValue: 'GITLAB_TOKEN',
        });
        const shell = aws_ec2_1.UserData.forLinux();
        shell.addCommands(...this.createUserData(runnerProps, tokenParameterStore.parameterName));
        this.runnerRole =
            runnerProps.ec2iamrole ??
                new aws_iam_1.Role(this, 'runner-role', {
                    assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                    description: 'For Gitlab EC2 Runner Role',
                });
        this.validUntil = runnerProps.validUntil;
        const instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.runnerRole.roleName],
        });
        tokenParameterStore.grantWrite(this.runnerRole);
        tokenParameterStore.grantRead(this.runnerRole);
        this.vpc =
            runnerProps.selfvpc ??
                new aws_ec2_1.Vpc(this, 'VPC', {
                    ipAddresses: aws_ec2_1.IpAddresses.cidr('10.0.0.0/16'),
                    maxAzs: 2,
                    subnetConfiguration: [
                        {
                            cidrMask: 26,
                            name: 'RunnerVPC',
                            subnetType: aws_ec2_1.SubnetType.PUBLIC,
                        },
                    ],
                    natGateways: 0,
                });
        this.defaultRunnerSG = new aws_ec2_1.SecurityGroup(this, 'SpotFleetSg', {
            vpc: this.vpc,
        });
        this.defaultRunnerSG.connections.allowFromAnyIpv4(aws_ec2_1.Port.tcp(22));
        const spotOrOnDemand = runnerProps.spotFleet ?? false;
        if (spotOrOnDemand) {
            //throw new Error('yes new spotfleet');
            const imageId = aws_ec2_1.MachineImage.latestAmazonLinux2().getImage(this).imageId;
            const lt = new aws_ec2_1.CfnLaunchTemplate(this, 'LaunchTemplate', {
                launchTemplateData: {
                    imageId,
                    instanceType: runnerProps.ec2type,
                    blockDeviceMappings: [
                        {
                            deviceName: '/dev/xvda',
                            ebs: {
                                volumeSize: runnerProps.ebsSize ?? 60,
                            },
                        },
                    ],
                    userData: aws_cdk_lib_1.Fn.base64(shell.render()),
                    keyName: runnerProps.keyName,
                    tagSpecifications: [
                        {
                            resourceType: 'instance',
                            tags: [
                                {
                                    key: 'Name',
                                    value: `${aws_cdk_lib_1.Stack.of(this).stackName}/spotFleetGitlabRunner/${spotFleetId}`,
                                },
                            ],
                        },
                    ],
                    instanceMarketOptions: {
                        marketType: 'spot',
                        spotOptions: {
                            instanceInterruptionBehavior: runnerProps.instanceInterruptionBehavior ??
                                InstanceInterruptionBehavior.TERMINATE,
                        },
                    },
                    securityGroupIds: this.defaultRunnerSG.connections.securityGroups.map((m) => m.securityGroupId),
                    iamInstanceProfile: {
                        arn: instanceProfile.attrArn,
                    },
                },
            });
            const spotFleetRole = new aws_iam_1.Role(this, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            const vpcSubnetSelection = runnerProps.vpcSubnet ?? {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            };
            const subnetConfig = this.vpc
                .selectSubnets(vpcSubnetSelection)
                .subnets.map((s) => ({
                subnetId: s.subnetId,
            }));
            this.cfnSpotFleet = new aws_ec2_1.CfnSpotFleet(this, id, {
                spotFleetRequestConfigData: {
                    launchTemplateConfigs: [
                        {
                            launchTemplateSpecification: {
                                launchTemplateId: lt.ref,
                                version: lt.attrLatestVersionNumber,
                            },
                            overrides: subnetConfig,
                        },
                    ],
                    iamFleetRole: spotFleetRole.roleArn,
                    targetCapacity: 1,
                    validUntil: aws_cdk_lib_1.Lazy.string({ produce: () => this.validUntil }),
                    terminateInstancesWithExpiration: true,
                },
            });
            const onEvent = new lambda.Function(this, 'OnEvent', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.on_event',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: aws_cdk_lib_1.Duration.seconds(60),
            });
            const isComplete = new lambda.Function(this, 'IsComplete', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.is_complete',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: aws_cdk_lib_1.Duration.seconds(60),
                role: onEvent.role,
            });
            const myProvider = new cr.Provider(this, 'MyProvider', {
                onEventHandler: onEvent,
                isCompleteHandler: isComplete,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            onEvent.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['ec2:DescribeSpotFleetInstances'],
                resources: ['*'],
            }));
            const fleetInstancesId = new aws_cdk_lib_1.CustomResource(this, 'GetInstanceId', {
                serviceToken: myProvider.serviceToken,
                properties: {
                    SpotFleetRequestId: this.cfnSpotFleet.ref,
                },
            });
            fleetInstancesId.node.addDependency(this.cfnSpotFleet);
            this.spotFleetInstanceId = aws_cdk_lib_1.Token.asString(fleetInstancesId.getAtt('InstanceId'));
            this.spotFleetRequestId = aws_cdk_lib_1.Token.asString(fleetInstancesId.getAtt('SpotInstanceRequestId'));
            new aws_cdk_lib_1.CfnOutput(this, 'InstanceId', { value: this.spotFleetInstanceId });
            new aws_cdk_lib_1.CfnOutput(this, 'SpotFleetId', { value: this.cfnSpotFleet.ref });
        }
        else {
            this.runnerEc2 = new aws_ec2_1.Instance(this, 'GitlabRunner', {
                instanceType: new aws_ec2_1.InstanceType(runnerProps.ec2type),
                instanceName: 'Gitlab-Runner',
                vpc: this.vpc,
                vpcSubnets: runnerProps.vpcSubnet ?? {
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                },
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux2(),
                role: this.runnerRole,
                userData: shell,
                securityGroup: this.defaultRunnerSG,
                blockDevices: [
                    {
                        deviceName: '/dev/xvda',
                        volume: aws_ec2_1.BlockDeviceVolume.ebs(runnerProps.ebsSize ?? 60),
                    },
                ],
            });
            new aws_cdk_lib_1.CfnOutput(this, 'Runner-Instance-ID', {
                value: this.runnerEc2.instanceId,
            });
        }
        const unregisterRunnerOnEvent = new lambda.Function(this, 'unregisterRunnerOnEvent', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'unregister_runner.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: aws_cdk_lib_1.Duration.seconds(60),
        });
        const unregisterRunnerProvider = new cr.Provider(this, 'unregisterRunnerProvider', {
            onEventHandler: unregisterRunnerOnEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const unregisterRunnerCR = new aws_cdk_lib_1.CustomResource(this, 'unregisterRunnerCR', {
            resourceType: 'Custom::unregisterRunnerProvider',
            serviceToken: unregisterRunnerProvider.serviceToken,
            properties: {
                TokenParameterStoreName: tokenParameterStore.parameterName,
                GitlabUrl: runnerProps.gitlaburl,
            },
        });
        tokenParameterStore.grantRead(unregisterRunnerOnEvent);
        unregisterRunnerCR.node.addDependency(tokenParameterStore);
        this.runnerRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        new aws_cdk_lib_1.CfnOutput(this, 'Runner-Role-Arn', {
            value: this.runnerRole.roleArn,
        });
    }
    /**
     * Add expire time function for spotfleet runner !!! .
     *
     * @param duration - Block duration.
     */
    expireAfter(duration) {
        const date = new Date();
        date.setSeconds(date.getSeconds() + duration.toSeconds());
        this.validUntil = date.toISOString();
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @param tokenParameterStoreName - the tokenParameterStoreName to put gitlab runner token.
     * @returns Array.
     */
    createUserData(props, tokenParameterStoreName) {
        return [
            'yum update -y ',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlaburl} --registration-token ${props.gitlabtoken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props?.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "${props.runnerDescription}" \
      --tag-list "${props.tags?.join(',')}" --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
            `sed -i 's/concurrent = .*/concurrent = ${props.concurrentJobs}/g' /home/ec2-user/.gitlab-runner/config.toml`,
            'TOKEN=$(cat /home/ec2-user/.gitlab-runner/config.toml | grep token | awk \'{print $3}\'| tr -d \'"\')',
            `aws ssm put-parameter --name ${tokenParameterStoreName} --value $TOKEN --overwrite --region ${aws_cdk_lib_1.Stack.of(this).region}`,
        ];
    }
}
_a = JSII_RTTI_SYMBOL_1;
GitlabContainerRunner[_a] = { fqn: "cdk-gitlab-runner.GitlabContainerRunner", version: "2.1.332" };
exports.GitlabContainerRunner = GitlabContainerRunner;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2l0bGFiLXJ1bm5lci1pbnN0YW5jZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9naXRsYWItcnVubmVyLWluc3RhbmNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkJBQTZCO0FBQzdCLDZDQVFxQjtBQUNyQixpREFpQjZCO0FBQzdCLGlEQU82QjtBQUM3QixpREFBaUQ7QUFDakQsNkNBQTZDO0FBQzdDLDJDQUEyQztBQUMzQyxtREFBbUQ7QUFDbkQsMkNBQXVDO0FBZ1B2Qzs7R0FFRztBQUNILElBQVksNEJBYVg7QUFiRCxXQUFZLDRCQUE0QjtJQUN0Qzs7T0FFRztJQUNILHVEQUF1QixDQUFBO0lBQ3ZCOztPQUVHO0lBQ0gsNkNBQWEsQ0FBQTtJQUNiOztPQUVHO0lBQ0gsdURBQXVCLENBQUE7QUFDekIsQ0FBQyxFQWJXLDRCQUE0QixHQUE1QixvQ0FBNEIsS0FBNUIsb0NBQTRCLFFBYXZDO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLHFCQUFzQixTQUFRLHNCQUFTO0lBcUNsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWlDO1FBQ3pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsTUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDO1FBRXZCLE1BQU0sWUFBWSxHQUFHO1lBQ25CLGlCQUFpQixFQUFFLDRDQUE0QztZQUMvRCxTQUFTLEVBQUUscUJBQXFCO1lBQ2hDLE9BQU8sRUFBRSxVQUFVO1lBQ25CLElBQUksRUFBRSxDQUFDLFFBQVEsRUFBRSxRQUFRLEVBQUUsUUFBUSxDQUFDO1lBQ3BDLGNBQWMsRUFBRSxDQUFDO1lBQ2pCLGlCQUFpQixFQUFFLGVBQWU7U0FDbkMsQ0FBQztRQUNGLE1BQU0sV0FBVyxHQUFHLEVBQUUsR0FBRyxZQUFZLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQztRQUVsRCxNQUFNLG1CQUFtQixHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7WUFDaEYsV0FBVyxFQUFFLGNBQWM7U0FDNUIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxLQUFLLEdBQUcsa0JBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUNsQyxLQUFLLENBQUMsV0FBVyxDQUFDLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLEVBQUUsbUJBQW1CLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUUxRixJQUFJLENBQUMsVUFBVTtZQUNiLFdBQVcsQ0FBQyxVQUFVO2dCQUN0QixJQUFJLGNBQUksQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO29CQUM1QixTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztvQkFDcEQsV0FBVyxFQUFFLDRCQUE0QjtpQkFDMUMsQ0FBQyxDQUFDO1FBQ0wsSUFBSSxDQUFDLFVBQVUsR0FBRyxXQUFXLENBQUMsVUFBVSxDQUFDO1FBQ3pDLE1BQU0sZUFBZSxHQUFHLElBQUksNEJBQWtCLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3RFLEtBQUssRUFBRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDO1NBQ2xDLENBQUMsQ0FBQztRQUNILG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDaEQsbUJBQW1CLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsR0FBRztZQUNOLFdBQVcsQ0FBQyxPQUFPO2dCQUNuQixJQUFJLGFBQUcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO29CQUNuQixXQUFXLEVBQUUscUJBQVcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO29CQUM1QyxNQUFNLEVBQUUsQ0FBQztvQkFDVCxtQkFBbUIsRUFBRTt3QkFDbkI7NEJBQ0UsUUFBUSxFQUFFLEVBQUU7NEJBQ1osSUFBSSxFQUFFLFdBQVc7NEJBQ2pCLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE1BQU07eUJBQzlCO3FCQUNGO29CQUNELFdBQVcsRUFBRSxDQUFDO2lCQUNmLENBQUMsQ0FBQztRQUNMLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSx1QkFBYSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDNUQsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1NBQ2QsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGVBQWUsQ0FBQyxXQUFXLENBQUMsZ0JBQWdCLENBQUMsY0FBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2hFLE1BQU0sY0FBYyxHQUFHLFdBQVcsQ0FBQyxTQUFTLElBQUksS0FBSyxDQUFDO1FBQ3RELElBQUksY0FBYyxFQUFFO1lBQ2xCLHVDQUF1QztZQUV2QyxNQUFNLE9BQU8sR0FBRyxzQkFBWSxDQUFDLGtCQUFrQixFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUN6RSxNQUFNLEVBQUUsR0FBRyxJQUFJLDJCQUFpQixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtnQkFDdkQsa0JBQWtCLEVBQUU7b0JBQ2xCLE9BQU87b0JBQ1AsWUFBWSxFQUFFLFdBQVcsQ0FBQyxPQUFPO29CQUNqQyxtQkFBbUIsRUFBRTt3QkFDbkI7NEJBQ0UsVUFBVSxFQUFFLFdBQVc7NEJBQ3ZCLEdBQUcsRUFBRTtnQ0FDSCxVQUFVLEVBQUUsV0FBVyxDQUFDLE9BQU8sSUFBSSxFQUFFOzZCQUN0Qzt5QkFDRjtxQkFDRjtvQkFDRCxRQUFRLEVBQUUsZ0JBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO29CQUNuQyxPQUFPLEVBQUUsV0FBVyxDQUFDLE9BQU87b0JBQzVCLGlCQUFpQixFQUFFO3dCQUNqQjs0QkFDRSxZQUFZLEVBQUUsVUFBVTs0QkFDeEIsSUFBSSxFQUFFO2dDQUNKO29DQUNFLEdBQUcsRUFBRSxNQUFNO29DQUNYLEtBQUssRUFBRSxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQ3pCLDBCQUEwQixXQUFXLEVBQUU7aUNBQ3hDOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNELHFCQUFxQixFQUFFO3dCQUNyQixVQUFVLEVBQUUsTUFBTTt3QkFDbEIsV0FBVyxFQUFFOzRCQUNYLDRCQUE0QixFQUMxQixXQUFXLENBQUMsNEJBQTRCO2dDQUN4Qyw0QkFBNEIsQ0FBQyxTQUFTO3lCQUN6QztxQkFDRjtvQkFDRCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUNuRSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FDekI7b0JBQ0Qsa0JBQWtCLEVBQUU7d0JBQ2xCLEdBQUcsRUFBRSxlQUFlLENBQUMsT0FBTztxQkFDN0I7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxNQUFNLGFBQWEsR0FBRyxJQUFJLGNBQUksQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFO2dCQUNoRCxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyx5QkFBeUIsQ0FBQztnQkFDMUQsZUFBZSxFQUFFO29CQUNmLHVCQUFhLENBQUMsd0JBQXdCLENBQ3BDLDRDQUE0QyxDQUM3QztpQkFDRjthQUNGLENBQUMsQ0FBQztZQUVILE1BQU0sa0JBQWtCLEdBQUcsV0FBVyxDQUFDLFNBQVMsSUFBSTtnQkFDbEQsVUFBVSxFQUFFLG9CQUFVLENBQUMsTUFBTTthQUM5QixDQUFDO1lBQ0YsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLEdBQUc7aUJBQzFCLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQztpQkFDakMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztnQkFDbkIsUUFBUSxFQUFFLENBQUMsQ0FBQyxRQUFRO2FBQ3JCLENBQUMsQ0FBQyxDQUFDO1lBRU4sSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLHNCQUFZLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtnQkFDN0MsMEJBQTBCLEVBQUU7b0JBQzFCLHFCQUFxQixFQUFFO3dCQUNyQjs0QkFDRSwyQkFBMkIsRUFBRTtnQ0FDM0IsZ0JBQWdCLEVBQUUsRUFBRSxDQUFDLEdBQUc7Z0NBQ3hCLE9BQU8sRUFBRSxFQUFFLENBQUMsdUJBQXVCOzZCQUNwQzs0QkFDRCxTQUFTLEVBQUUsWUFBWTt5QkFDeEI7cUJBQ0Y7b0JBQ0QsWUFBWSxFQUFFLGFBQWEsQ0FBQyxPQUFPO29CQUNuQyxjQUFjLEVBQUUsQ0FBQztvQkFDakIsVUFBVSxFQUFFLGtCQUFJLENBQUMsTUFBTSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztvQkFDM0QsZ0NBQWdDLEVBQUUsSUFBSTtpQkFDdkM7YUFDRixDQUFDLENBQUM7WUFDSCxNQUFNLE9BQU8sR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtnQkFDbkQsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHFCQUFxQixDQUFDLENBQUM7Z0JBQ3hFLE9BQU8sRUFBRSxnQkFBZ0I7Z0JBQ3pCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVU7Z0JBQ2xDLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7YUFDOUIsQ0FBQyxDQUFDO1lBRUgsTUFBTSxVQUFVLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7Z0JBQ3pELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO2dCQUN4RSxPQUFPLEVBQUUsbUJBQW1CO2dCQUM1QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO2dCQUNsQyxPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2dCQUM3QixJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUk7YUFDbkIsQ0FBQyxDQUFDO1lBRUgsTUFBTSxVQUFVLEdBQUcsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7Z0JBQ3JELGNBQWMsRUFBRSxPQUFPO2dCQUN2QixpQkFBaUIsRUFBRSxVQUFVO2dCQUM3QixZQUFZLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxPQUFPO2FBQ3pDLENBQUMsQ0FBQztZQUVILE9BQU8sQ0FBQyxlQUFlLENBQ3JCLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsT0FBTyxFQUFFLENBQUMsZ0NBQWdDLENBQUM7Z0JBQzNDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUNqQixDQUFDLENBQ0gsQ0FBQztZQUVGLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSw0QkFBYyxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7Z0JBQ2pFLFlBQVksRUFBRSxVQUFVLENBQUMsWUFBWTtnQkFDckMsVUFBVSxFQUFFO29CQUNWLGtCQUFrQixFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRztpQkFDMUM7YUFDRixDQUFDLENBQUM7WUFFSCxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUN2RCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsbUJBQUssQ0FBQyxRQUFRLENBQ3ZDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsQ0FDdEMsQ0FBQztZQUNGLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxtQkFBSyxDQUFDLFFBQVEsQ0FDdEMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLHVCQUF1QixDQUFDLENBQ2pELENBQUM7WUFDRixJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1lBQ3ZFLElBQUksdUJBQVMsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztTQUN0RTthQUFNO1lBQ0wsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLGtCQUFRLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtnQkFDbEQsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDO2dCQUNuRCxZQUFZLEVBQUUsZUFBZTtnQkFDN0IsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLFVBQVUsRUFBRSxXQUFXLENBQUMsU0FBUyxJQUFJO29CQUNuQyxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxNQUFNO2lCQUM5QjtnQkFDRCxZQUFZLEVBQUUsc0JBQVksQ0FBQyxrQkFBa0IsRUFBRTtnQkFDL0MsSUFBSSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUNyQixRQUFRLEVBQUUsS0FBSztnQkFDZixhQUFhLEVBQUUsSUFBSSxDQUFDLGVBQWU7Z0JBQ25DLFlBQVksRUFBRTtvQkFDWjt3QkFDRSxVQUFVLEVBQUUsV0FBVzt3QkFDdkIsTUFBTSxFQUFFLDJCQUFpQixDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQztxQkFDekQ7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFDSCxJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO2dCQUN4QyxLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVO2FBQ2pDLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSx1QkFBdUIsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUFFO1lBQ25GLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBQ3hFLE9BQU8sRUFBRSw0QkFBNEI7WUFDckMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1NBQzlCLENBQUMsQ0FBQztRQUVILE1BQU0sd0JBQXdCLEdBQUcsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSwwQkFBMEIsRUFBRTtZQUNqRixjQUFjLEVBQUUsdUJBQXVCO1lBQ3ZDLFlBQVksRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLE9BQU87U0FDekMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLDRCQUFjLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO1lBQ3hFLFlBQVksRUFBRSxrQ0FBa0M7WUFDaEQsWUFBWSxFQUFFLHdCQUF3QixDQUFDLFlBQVk7WUFDbkQsVUFBVSxFQUFFO2dCQUNWLHVCQUF1QixFQUFFLG1CQUFtQixDQUFDLGFBQWE7Z0JBQzFELFNBQVMsRUFBRSxXQUFXLENBQUMsU0FBUzthQUNqQztTQUNGLENBQUMsQ0FBQztRQUVILG1CQUFtQixDQUFDLFNBQVMsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ3ZELGtCQUFrQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUMzRCxJQUFJLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUM5Qix1QkFBYSxDQUFDLHdCQUF3QixDQUFDLDhCQUE4QixDQUFDLENBQ3ZFLENBQUM7UUFFRixJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3JDLEtBQUssRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU87U0FDL0IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsUUFBa0I7UUFDbkMsTUFBTSxJQUFJLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztRQUN4QixJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsR0FBRyxRQUFRLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUMxRCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUN2QyxDQUFDO0lBRU8saUJBQWlCLENBQUMsWUFBeUM7UUFDakUsSUFBSSxVQUFVLEdBQVcsOERBQThELENBQUM7UUFDeEYsSUFBSSxZQUFZLEVBQUU7WUFDaEIsSUFBSSxRQUFRLEdBQWEsRUFBRSxDQUFDO1lBQzVCLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ3ZCLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxJQUFJLENBQUMsQ0FBQyxhQUFhLEdBQUcsQ0FBQyxDQUFDO1lBQ3RELENBQUMsQ0FBQyxDQUFDO1lBQ0gsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDbkIsVUFBVSxHQUFHLEdBQUcsVUFBVSxxQkFBcUIsQ0FBQyxFQUFFLENBQUM7WUFDckQsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUNELE9BQU8sVUFBVSxDQUFDO0lBQ3BCLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksY0FBYyxDQUFDLEtBQWlDLEVBQUUsdUJBQStCO1FBQ3RGLE9BQU87WUFDTCxnQkFBZ0I7WUFDaEIscUxBQXFMO1lBQ3JMLHFEQUFxRDtZQUNyRDtzQ0FDZ0MsS0FBSyxDQUFDLGlCQUFpQixxQ0FBcUMsS0FBSyxDQUFDLFNBQVMseUJBQXlCLEtBQUssQ0FBQyxXQUFXOzRDQUMvRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLGFBQWEsQ0FBQzt3RUFDaEIsS0FBSyxDQUFDLGlCQUFpQjtvQkFDM0UsS0FBSyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QjtZQUMxRCxtS0FBbUssS0FBSyxDQUFDLGlCQUFpQixFQUFFO1lBQzVMLDBDQUEwQyxLQUFLLENBQUMsY0FBYywrQ0FBK0M7WUFDN0csdUdBQXVHO1lBQ3ZHLGdDQUFnQyx1QkFBdUIsd0NBQXdDLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sRUFBRTtTQUN2SCxDQUFDO0lBQ0osQ0FBQzs7OztBQTNUVSxzREFBcUIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHtcbiAgQ2ZuT3V0cHV0LFxuICBGbixcbiAgU3RhY2ssXG4gIER1cmF0aW9uLFxuICBMYXp5LFxuICBDdXN0b21SZXNvdXJjZSxcbiAgVG9rZW4sXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7XG4gIEluc3RhbmNlLFxuICBJbnN0YW5jZVR5cGUsXG4gIE1hY2hpbmVJbWFnZSxcbiAgVXNlckRhdGEsXG4gIEJsb2NrRGV2aWNlVm9sdW1lLFxuICBTdWJuZXRUeXBlLFxuICBWcGMsXG4gIElWcGMsXG4gIElJbnN0YW5jZSxcbiAgU2VjdXJpdHlHcm91cCxcbiAgUG9ydCxcbiAgQ2ZuTGF1bmNoVGVtcGxhdGUsXG4gIENmblNwb3RGbGVldCxcbiAgSVNlY3VyaXR5R3JvdXAsXG4gIFN1Ym5ldFNlbGVjdGlvbixcbiAgSXBBZGRyZXNzZXMsXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lYzInO1xuaW1wb3J0IHtcbiAgSVJvbGUsXG4gIFJvbGUsXG4gIFNlcnZpY2VQcmluY2lwYWwsXG4gIE1hbmFnZWRQb2xpY3ksXG4gIENmbkluc3RhbmNlUHJvZmlsZSxcbiAgUG9saWN5U3RhdGVtZW50LFxufSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgc3NtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zc20nO1xuaW1wb3J0ICogYXMgY3IgZnJvbSAnYXdzLWNkay1saWIvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuaW1wb3J0IHsgRG9ja2VyVm9sdW1lcyB9IGZyb20gJy4vZ2l0bGFiLXJ1bm5lci1pbnRlcmZhY2VzJztcbi8qKlxuICogR2l0bGFiQ29udGFpbmVyUnVubmVyIFByb3BzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEdpdGxhYkNvbnRhaW5lclJ1bm5lclByb3BzIHtcbiAgLyoqXG4gICAqIEdpdGxhYiB0b2tlbiBmb3IgdGhlIFJlZ2lzdGVyIFJ1bm5lciAuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIG5ldyBHaXRsYWJDb250YWluZXJSdW5uZXIoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJyB9KTtcbiAgICpcbiAgICogQGRlZmF1bHQgLSBZb3UgbXVzdCB0byBnaXZlIHRoZSB0b2tlbiAhISFcbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGdpdGxhYnRva2VuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEltYWdlIFVSTCBvZiBHaXRsYWIgUnVubmVyLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBuZXcgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmcoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYlRva2VuOiAnR0lUTEFCX1RPS0VOJywgZ2l0bGFiUnVubmVySW1hZ2U6ICdnaXRsYWIvZ2l0bGFiLXJ1bm5lcjphbHBpbmUnIH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCBwdWJsaWMuZWNyLmF3cy9naXRsYWIvZ2l0bGFiLXJ1bm5lcjpsYXRlc3RcbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGdpdGxhYlJ1bm5lckltYWdlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSdW5uZXIgZGVmYXVsdCBFQzIgaW5zdGFuY2UgdHlwZS5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogbmV3IEdpdGxhYkNvbnRhaW5lclJ1bm5lcihzdGFjaywgJ3J1bm5lcicsIHsgZ2l0bGFidG9rZW46ICdHSVRMQUJfVE9LRU4nLCBlYzJ0eXBlOiAndDMuc21hbGwnIH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIHQzLm1pY3JvXG4gICAqXG4gICAqL1xuICByZWFkb25seSBlYzJ0eXBlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBWUEMgZm9yIHRoZSBHaXRsYWIgUnVubmVyIC5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogY29uc3QgbmV3dnBjID0gbmV3IFZwYyhzdGFjaywgJ05FV1ZQQycsIHtcbiAgICogICBpcEFkZHJlc3NlczogSXBBZGRyZXNzZXMuY2lkcignMTAuMC4wLjAvMTYnKSxcbiAgICogICBtYXhBenM6IDIsXG4gICAqICAgc3VibmV0Q29uZmlndXJhdGlvbjogW3tcbiAgICogICAgIGNpZHJNYXNrOiAyNixcbiAgICogICAgIG5hbWU6ICdSdW5uZXJWUEMnLFxuICAgKiAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAqICAgfV0sXG4gICAqICAgbmF0R2F0ZXdheXM6IDAsXG4gICAqIH0pO1xuICAgKlxuICAgKiBuZXcgR2l0bGFiQ29udGFpbmVyUnVubmVyKHN0YWNrLCAncnVubmVyJywgeyBnaXRsYWJ0b2tlbjogJ0dJVExBQl9UT0tFTicsIHNlbGZ2cGM6IG5ld3ZwYyB9KTtcbiAgICpcbiAgICogQGRlZmF1bHQgLSBuZXcgVlBDIHdpbGwgYmUgY3JlYXRlZCAsIDEgVnBjICwgMiBQdWJsaWMgU3VibmV0IC5cbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IHNlbGZ2cGM/OiBJVnBjO1xuXG4gIC8qKlxuICAgKiBJQU0gcm9sZSBmb3IgdGhlIEdpdGxhYiBSdW5uZXIgSW5zdGFuY2UgLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBjb25zdCByb2xlID0gbmV3IFJvbGUoc3RhY2ssICdydW5uZXItcm9sZScsIHtcbiAgICogICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdlYzIuYW1hem9uYXdzLmNvbScpLFxuICAgKiAgIGRlc2NyaXB0aW9uOiAnRm9yIEdpdGxhYiBFQzIgUnVubmVyIFRlc3QgUm9sZScsXG4gICAqICAgcm9sZU5hbWU6ICdNeXNlbGYtUnVubmVyLVJvbGUnLFxuICAgKiB9KTtcbiAgICpcbiAgICogbmV3IEdpdGxhYkNvbnRhaW5lclJ1bm5lcihzdGFjaywgJ3J1bm5lcicsIHsgZ2l0bGFidG9rZW46ICdHSVRMQUJfVE9LRU4nLCBlYzJpYW1yb2xlOiByb2xlIH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5ldyBSb2xlIGZvciBHaXRsYWIgUnVubmVyIEluc3RhbmNlICwgYXR0YWNoIEFtYXpvblNTTU1hbmFnZWRJbnN0YW5jZUNvcmUgUG9saWN5IC5cbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGVjMmlhbXJvbGU/OiBJUm9sZTtcblxuICAvKipcbiAgICogR2l0bGFiIFJ1bm5lciByZWdpc3RlciB0YWcxICAuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIG5ldyBHaXRsYWJDb250YWluZXJSdW5uZXIoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJywgdGFnMTogJ2FhJyB9KTtcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgLSB1c2UgdGFncyBbJ3J1bm5lcicsICdnaXRsYWInLCAnYXdzY2RrJ11cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0YWcxOiBnaXRsYWIgLlxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgdGFnMT86IHN0cmluZztcblxuICAvKipcbiAgICogR2l0bGFiIFJ1bm5lciByZWdpc3RlciB0YWcyICAuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIG5ldyBHaXRsYWJDb250YWluZXJSdW5uZXIoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJywgdGFnMjogJ2JiJyB9KTtcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgLSB1c2UgdGFncyBbJ3J1bm5lcicsICdnaXRsYWInLCAnYXdzY2RrJ11cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0YWcyOiBhd3NjZGsgLlxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgdGFnMj86IHN0cmluZztcblxuICAvKipcbiAgICogR2l0bGFiIFJ1bm5lciByZWdpc3RlciB0YWczICAuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIG5ldyBHaXRsYWJDb250YWluZXJSdW5uZXIoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJywgdGFnMzogJ2NjJyB9KTtcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgLSB1c2UgdGFncyBbJ3J1bm5lcicsICdnaXRsYWInLCAnYXdzY2RrJ11cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0YWczOiBydW5uZXIgLlxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgdGFnMz86IHN0cmluZztcblxuICAvKipcbiAgICogdGFncyBmb3IgdGhlIHJ1bm5lclxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFsncnVubmVyJywgJ2dpdGxhYicsICdhd3NjZGsnXVxuICAgKi9cbiAgcmVhZG9ubHkgdGFncz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBHaXRsYWIgUnVubmVyIHJlZ2lzdGVyIHVybCAuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIGNvbnN0IHJ1bm5lciA9IG5ldyBHaXRsYWJDb250YWluZXJSdW5uZXIoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJyxnaXRsYWJ1cmw6ICdodHRwczovL2dpdGxhYi5jb20vJ30pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIGdpdGxhYnVybD0naHR0cHM6Ly9naXRsYWIuY29tLycgLCBwbGVhc2UgdXNlIGh0dHBzOi8veW91cmdpdGxhYi5jb20vIGRvIG5vdCB1c2UgaHR0cHM6Ly95b3VyZ2l0bGFiLmNvbVxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgZ2l0bGFidXJsPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBHaXRsYWIgUnVubmVyIGluc3RhbmNlIEVCUyBzaXplIC5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogY29uc3QgcnVubmVyID0gbmV3IEdpdGxhYkNvbnRhaW5lclJ1bm5lcihzdGFjaywgJ3J1bm5lcicsIHsgZ2l0bGFidG9rZW46ICdHSVRMQUJfVE9LRU4nLGVic1NpemU6IDEwMH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIGVic1NpemU9NjBcbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGVic1NpemU/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEdpdGxhYiBSdW5uZXIgaW5zdGFuY2UgVXNlIFNwb3QgRmxlZXQgb3Igbm90ID8hLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBjb25zdCBydW5uZXIgPSBuZXcgR2l0bGFiQ29udGFpbmVyUnVubmVyKHN0YWNrLCAncnVubmVyJywgeyBnaXRsYWJ0b2tlbjogJ0dJVExBQl9UT0tFTicsc3BvdEZsZWV0OiB0cnVlfSk7XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gc3BvdEZsZWV0PWZhbHNlXG4gICAqXG4gICAqL1xuICByZWFkb25seSBzcG90RmxlZXQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBHaXRsYWIgUnVubmVyIGNvbmN1cnJlbnQgam9iIGNvbmZpZ3VyYXRpb24uXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIGNvbnN0IHJ1bm5lciA9IG5ldyBHaXRsYWJDb250YWluZXJSdW5uZXIoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJyxjb25jdXJyZW50Sm9iczogM30pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIGNvbmN1cnJlbnRKb2JzPTFcbiAgICovXG4gIHJlYWRvbmx5IGNvbmN1cnJlbnRKb2JzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBHaXRsYWIgUnVubmVyIGRlc2NyaXB0aW9uLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBjb25zdCBydW5uZXIgPSBuZXcgR2l0bGFiQ29udGFpbmVyUnVubmVyKHN0YWNrLCAncnVubmVyJywgeyBnaXRsYWJ0b2tlbjogJ0dJVExBQl9UT0tFTicscnVubmVyRGVzY3JpcHRpb246ICdTaW1wbGUgR2l0TGFiIFJ1bm5lcid9KTtcbiAgICpcbiAgICogQGRlZmF1bHQgLSBydW5uZXJEZXNjcmlwdGlvbj0nRG9ja2VyIFJ1bm5lcidcbiAgICovXG4gIHJlYWRvbmx5IHJ1bm5lckRlc2NyaXB0aW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTU0gga2V5IG5hbWVcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBzc2gga2V5IHdpbGwgYmUgYXNzaWduZWQgLCAhISEgb25seSBzdXBwb3J0IHNwb3RmbGVldCBydW5uZXIgISEhIC5cbiAgICovXG4gIHJlYWRvbmx5IGtleU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBiZWhhdmlvciB3aGVuIGEgU3BvdCBSdW5uZXIgSW5zdGFuY2UgaXMgaW50ZXJydXB0ZWRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBJbnN0YW5jZUludGVycnVwdGlvbkJlaGF2aW9yLlRFUk1JTkFURSAsICEhISBvbmx5IHN1cHBvcnQgc3BvdGZsZWV0IHJ1bm5lciAhISEgLlxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VJbnRlcnJ1cHRpb25CZWhhdmlvcj86IEluc3RhbmNlSW50ZXJydXB0aW9uQmVoYXZpb3I7XG5cbiAgLyoqXG4gICAqIHRoZSB0aW1lIHdoZW4gdGhlIHNwb3QgZmxlZXQgYWxsb2NhdGlvbiBleHBpcmVzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gZXhwaXJhdGlvbiAsICEhISBvbmx5IHN1cHBvcnQgc3BvdGZsZWV0IHJ1bm5lciAhISEgLlxuICAgKi9cbiAgcmVhZG9ubHkgdmFsaWRVbnRpbD86IHN0cmluZztcblxuICAvKipcbiAgICogVlBDIHN1Ym5ldCBmb3IgdGhlIHNwb3QgZmxlZXRcbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ25hdCcsIHtcbiAgICogbmF0R2F0ZXdheXM6IDEsXG4gICAqIG1heEF6czogMixcbiAgICogfSk7XG4gICAqIGNvbnN0IHJ1bm5lciA9IG5ldyBHaXRsYWJDb250YWluZXJSdW5uZXIoc3RhY2ssICd0ZXN0aW5nJywge1xuICAgKiAgIGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJyxcbiAgICogICBlYzJ0eXBlOiAndDMubGFyZ2UnLFxuICAgKiAgIGVjMmlhbXJvbGU6IHJvbGUsXG4gICAqICAgZWJzU2l6ZTogMTAwLFxuICAgKiAgIHNlbGZ2cGM6IHZwYyxcbiAgICogICB2cGNTdWJuZXQ6IHtcbiAgICogICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDLFxuICAgKiAgIH0sXG4gICAqIH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIHB1YmxpYyBzdWJuZXRcbiAgICovXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldD86IFN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogYWRkIGFub3RoZXIgR2l0bGFiIENvbnRhaW5lciBSdW5uZXIgRG9ja2VyIFZvbHVtZXMgUGF0aCBhdCBqb2IgcnVubmVyIHJ1bnRpbWUuXG4gICAqXG4gICAqIG1vcmUgZGV0YWlsIHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9ydW5uZXIvY29uZmlndXJhdGlvbi9hZHZhbmNlZC1jb25maWd1cmF0aW9uLmh0bWwjdGhlLXJ1bm5lcnNkb2NrZXItc2VjdGlvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGFscmVhZHkgbW91bnQgXCIvdmFyL3J1bi9kb2NrZXIuc29jazovdmFyL3J1bi9kb2NrZXIuc29ja1wiXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIGRvY2tlclZvbHVtZXM6IFtcbiAgICogICB7XG4gICAqICAgICBob3N0UGF0aDogJy90bXAvY2FjaGUnLFxuICAgKiAgICAgY29udGFpbmVyUGF0aDogJy90bXAvY2FjaGUnLFxuICAgKiAgIH0sXG4gICAqIF0sXG4gICAqL1xuICByZWFkb25seSBkb2NrZXJWb2x1bWVzPzogRG9ja2VyVm9sdW1lc1tdO1xufVxuXG4vKipcbiAqIEluc3RhbmNlSW50ZXJydXB0aW9uQmVoYXZpb3IgZW51bS5cbiAqL1xuZXhwb3J0IGVudW0gSW5zdGFuY2VJbnRlcnJ1cHRpb25CZWhhdmlvciB7XG4gIC8qKlxuICAgKiBoaWJlcm5hdGVcbiAgICovXG4gIEhJQkVSTkFURSA9ICdoaWJlcm5hdGUnLFxuICAvKipcbiAgICogc3RvcFxuICAgKi9cbiAgU1RPUCA9ICdzdG9wJyxcbiAgLyoqXG4gICAqIHRlcm1pbmF0ZVxuICAgKi9cbiAgVEVSTUlOQVRFID0gJ3Rlcm1pbmF0ZScsXG59XG5cbi8qKlxuICogR2l0bGFiQ29udGFpbmVyUnVubmVyIENvbnN0cnVjdCBmb3IgY3JlYXRlIGEgR2l0bGFiIFJ1bm5lci5cbiAqL1xuZXhwb3J0IGNsYXNzIEdpdGxhYkNvbnRhaW5lclJ1bm5lciBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgYXNzdW1lZCBieSB0aGUgUnVubmVyIGluc3RhbmNlIC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBydW5uZXJSb2xlOiBJUm9sZTtcblxuICAvKipcbiAgICogVGhpcyByZXByZXNlbnRzIGEgUnVubmVyIEVDMiBpbnN0YW5jZSAsICEhISBvbmx5IHN1cHBvcnQgT24tZGVtYW5kIHJ1bm5lciBpbnN0YW5jZSAhISFcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBydW5uZXJFYzIhOiBJSW5zdGFuY2U7XG5cbiAgLyoqXG4gICAqIFRoZSBFQzIgcnVubmVyJ3MgdnBjLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZwYyE6IElWcGM7XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lIHdoZW4gdGhlIHRoZSBmbGVldCBhbGxvY2F0aW9uIHdpbGwgZXhwaXJlICwgISEhIG9ubHkgc3VwcG9ydCBzcG90ZmxlZXQgcnVubmVyICEhIVxuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZFVudGlsPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgRUMyIHJ1bm5lcidzIGRlZmF1bHQgU2VjdXJpdHlHcm91cC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0UnVubmVyU0chOiBJU2VjdXJpdHlHcm91cDtcblxuICAvKipcbiAgICogU3BvdEZsZWV0UmVxdWVzdElkIGZvciB0aGlzIHNwb3QgZmxlZXQgLCAhISEgb25seSBzdXBwb3J0IHNwb3RmbGVldCBydW5uZXIgISEhXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3BvdEZsZWV0UmVxdWVzdElkITogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiB0aGUgZmlyc3QgaW5zdGFuY2UgaWQgaW4gdGhpcyBmbGVldCAsICEhISBvbmx5IHN1cHBvcnQgc3BvdGZsZWV0IHJ1bm5lciAhISFcbiAgICovXG4gIHJlYWRvbmx5IHNwb3RGbGVldEluc3RhbmNlSWQhOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBjZm5TcG90RmxlZXQ/OiBDZm5TcG90RmxlZXQ7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBHaXRsYWJDb250YWluZXJSdW5uZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgY29uc3Qgc3BvdEZsZWV0SWQgPSBpZDtcblxuICAgIGNvbnN0IGRlZmF1bHRQcm9wcyA9IHtcbiAgICAgIGdpdGxhYlJ1bm5lckltYWdlOiAncHVibGljLmVjci5hd3MvZ2l0bGFiL2dpdGxhYi1ydW5uZXI6bGF0ZXN0JyxcbiAgICAgIGdpdGxhYnVybDogJ2h0dHBzOi8vZ2l0bGFiLmNvbS8nLFxuICAgICAgZWMydHlwZTogJ3QzLm1pY3JvJyxcbiAgICAgIHRhZ3M6IFsnZ2l0bGFiJywgJ2F3c2NkaycsICdydW5uZXInXSxcbiAgICAgIGNvbmN1cnJlbnRKb2JzOiAxLFxuICAgICAgcnVubmVyRGVzY3JpcHRpb246ICdEb2NrZXIgUnVubmVyJyxcbiAgICB9O1xuICAgIGNvbnN0IHJ1bm5lclByb3BzID0geyAuLi5kZWZhdWx0UHJvcHMsIC4uLnByb3BzIH07XG5cbiAgICBjb25zdCB0b2tlblBhcmFtZXRlclN0b3JlID0gbmV3IHNzbS5TdHJpbmdQYXJhbWV0ZXIodGhpcywgJ0dpdGxhYlRva2VuUGFyYW1ldGVyJywge1xuICAgICAgc3RyaW5nVmFsdWU6ICdHSVRMQUJfVE9LRU4nLFxuICAgIH0pO1xuXG4gICAgY29uc3Qgc2hlbGwgPSBVc2VyRGF0YS5mb3JMaW51eCgpO1xuICAgIHNoZWxsLmFkZENvbW1hbmRzKC4uLnRoaXMuY3JlYXRlVXNlckRhdGEocnVubmVyUHJvcHMsIHRva2VuUGFyYW1ldGVyU3RvcmUucGFyYW1ldGVyTmFtZSkpO1xuXG4gICAgdGhpcy5ydW5uZXJSb2xlID1cbiAgICAgIHJ1bm5lclByb3BzLmVjMmlhbXJvbGUgPz9cbiAgICAgIG5ldyBSb2xlKHRoaXMsICdydW5uZXItcm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgZGVzY3JpcHRpb246ICdGb3IgR2l0bGFiIEVDMiBSdW5uZXIgUm9sZScsXG4gICAgICB9KTtcbiAgICB0aGlzLnZhbGlkVW50aWwgPSBydW5uZXJQcm9wcy52YWxpZFVudGlsO1xuICAgIGNvbnN0IGluc3RhbmNlUHJvZmlsZSA9IG5ldyBDZm5JbnN0YW5jZVByb2ZpbGUodGhpcywgJ0luc3RhbmNlUHJvZmlsZScsIHtcbiAgICAgIHJvbGVzOiBbdGhpcy5ydW5uZXJSb2xlLnJvbGVOYW1lXSxcbiAgICB9KTtcbiAgICB0b2tlblBhcmFtZXRlclN0b3JlLmdyYW50V3JpdGUodGhpcy5ydW5uZXJSb2xlKTtcbiAgICB0b2tlblBhcmFtZXRlclN0b3JlLmdyYW50UmVhZCh0aGlzLnJ1bm5lclJvbGUpO1xuICAgIHRoaXMudnBjID1cbiAgICAgIHJ1bm5lclByb3BzLnNlbGZ2cGMgPz9cbiAgICAgIG5ldyBWcGModGhpcywgJ1ZQQycsIHtcbiAgICAgICAgaXBBZGRyZXNzZXM6IElwQWRkcmVzc2VzLmNpZHIoJzEwLjAuMC4wLzE2JyksXG4gICAgICAgIG1heEF6czogMixcbiAgICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNpZHJNYXNrOiAyNixcbiAgICAgICAgICAgIG5hbWU6ICdSdW5uZXJWUEMnLFxuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgbmF0R2F0ZXdheXM6IDAsXG4gICAgICB9KTtcbiAgICB0aGlzLmRlZmF1bHRSdW5uZXJTRyA9IG5ldyBTZWN1cml0eUdyb3VwKHRoaXMsICdTcG90RmxlZXRTZycsIHtcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgfSk7XG4gICAgdGhpcy5kZWZhdWx0UnVubmVyU0cuY29ubmVjdGlvbnMuYWxsb3dGcm9tQW55SXB2NChQb3J0LnRjcCgyMikpO1xuICAgIGNvbnN0IHNwb3RPck9uRGVtYW5kID0gcnVubmVyUHJvcHMuc3BvdEZsZWV0ID8/IGZhbHNlO1xuICAgIGlmIChzcG90T3JPbkRlbWFuZCkge1xuICAgICAgLy90aHJvdyBuZXcgRXJyb3IoJ3llcyBuZXcgc3BvdGZsZWV0Jyk7XG5cbiAgICAgIGNvbnN0IGltYWdlSWQgPSBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgyKCkuZ2V0SW1hZ2UodGhpcykuaW1hZ2VJZDtcbiAgICAgIGNvbnN0IGx0ID0gbmV3IENmbkxhdW5jaFRlbXBsYXRlKHRoaXMsICdMYXVuY2hUZW1wbGF0ZScsIHtcbiAgICAgICAgbGF1bmNoVGVtcGxhdGVEYXRhOiB7XG4gICAgICAgICAgaW1hZ2VJZCxcbiAgICAgICAgICBpbnN0YW5jZVR5cGU6IHJ1bm5lclByb3BzLmVjMnR5cGUsXG4gICAgICAgICAgYmxvY2tEZXZpY2VNYXBwaW5nczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBkZXZpY2VOYW1lOiAnL2Rldi94dmRhJyxcbiAgICAgICAgICAgICAgZWJzOiB7XG4gICAgICAgICAgICAgICAgdm9sdW1lU2l6ZTogcnVubmVyUHJvcHMuZWJzU2l6ZSA/PyA2MCxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgICB1c2VyRGF0YTogRm4uYmFzZTY0KHNoZWxsLnJlbmRlcigpKSxcbiAgICAgICAgICBrZXlOYW1lOiBydW5uZXJQcm9wcy5rZXlOYW1lLFxuICAgICAgICAgIHRhZ1NwZWNpZmljYXRpb25zOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHJlc291cmNlVHlwZTogJ2luc3RhbmNlJyxcbiAgICAgICAgICAgICAgdGFnczogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIGtleTogJ05hbWUnLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6IGAke1N0YWNrLm9mKHRoaXMpLnN0YWNrTmFtZVxuICAgICAgICAgICAgICAgICAgfS9zcG90RmxlZXRHaXRsYWJSdW5uZXIvJHtzcG90RmxlZXRJZH1gLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgaW5zdGFuY2VNYXJrZXRPcHRpb25zOiB7XG4gICAgICAgICAgICBtYXJrZXRUeXBlOiAnc3BvdCcsXG4gICAgICAgICAgICBzcG90T3B0aW9uczoge1xuICAgICAgICAgICAgICBpbnN0YW5jZUludGVycnVwdGlvbkJlaGF2aW9yOlxuICAgICAgICAgICAgICAgIHJ1bm5lclByb3BzLmluc3RhbmNlSW50ZXJydXB0aW9uQmVoYXZpb3IgPz9cbiAgICAgICAgICAgICAgICBJbnN0YW5jZUludGVycnVwdGlvbkJlaGF2aW9yLlRFUk1JTkFURSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzZWN1cml0eUdyb3VwSWRzOiB0aGlzLmRlZmF1bHRSdW5uZXJTRy5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwcy5tYXAoXG4gICAgICAgICAgICAobSkgPT4gbS5zZWN1cml0eUdyb3VwSWQsXG4gICAgICAgICAgKSxcbiAgICAgICAgICBpYW1JbnN0YW5jZVByb2ZpbGU6IHtcbiAgICAgICAgICAgIGFybjogaW5zdGFuY2VQcm9maWxlLmF0dHJBcm4sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBzcG90RmxlZXRSb2xlID0gbmV3IFJvbGUodGhpcywgJ0ZsZWV0Um9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnc3BvdGZsZWV0LmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgICAgTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoXG4gICAgICAgICAgICAnc2VydmljZS1yb2xlL0FtYXpvbkVDMlNwb3RGbGVldFRhZ2dpbmdSb2xlJyxcbiAgICAgICAgICApLFxuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IHZwY1N1Ym5ldFNlbGVjdGlvbiA9IHJ1bm5lclByb3BzLnZwY1N1Ym5ldCA/PyB7XG4gICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDLFxuICAgICAgfTtcbiAgICAgIGNvbnN0IHN1Ym5ldENvbmZpZyA9IHRoaXMudnBjXG4gICAgICAgIC5zZWxlY3RTdWJuZXRzKHZwY1N1Ym5ldFNlbGVjdGlvbilcbiAgICAgICAgLnN1Ym5ldHMubWFwKChzKSA9PiAoe1xuICAgICAgICAgIHN1Ym5ldElkOiBzLnN1Ym5ldElkLFxuICAgICAgICB9KSk7XG5cbiAgICAgIHRoaXMuY2ZuU3BvdEZsZWV0ID0gbmV3IENmblNwb3RGbGVldCh0aGlzLCBpZCwge1xuICAgICAgICBzcG90RmxlZXRSZXF1ZXN0Q29uZmlnRGF0YToge1xuICAgICAgICAgIGxhdW5jaFRlbXBsYXRlQ29uZmlnczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBsYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb246IHtcbiAgICAgICAgICAgICAgICBsYXVuY2hUZW1wbGF0ZUlkOiBsdC5yZWYsXG4gICAgICAgICAgICAgICAgdmVyc2lvbjogbHQuYXR0ckxhdGVzdFZlcnNpb25OdW1iZXIsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIG92ZXJyaWRlczogc3VibmV0Q29uZmlnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIGlhbUZsZWV0Um9sZTogc3BvdEZsZWV0Um9sZS5yb2xlQXJuLFxuICAgICAgICAgIHRhcmdldENhcGFjaXR5OiAxLFxuICAgICAgICAgIHZhbGlkVW50aWw6IExhenkuc3RyaW5nKHsgcHJvZHVjZTogKCkgPT4gdGhpcy52YWxpZFVudGlsIH0pLFxuICAgICAgICAgIHRlcm1pbmF0ZUluc3RhbmNlc1dpdGhFeHBpcmF0aW9uOiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgICBjb25zdCBvbkV2ZW50ID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnT25FdmVudCcsIHtcbiAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvZnVuY3Rpb25zJykpLFxuICAgICAgICBoYW5kbGVyOiAnaW5kZXgub25fZXZlbnQnLFxuICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM184LFxuICAgICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKDYwKSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBpc0NvbXBsZXRlID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnSXNDb21wbGV0ZScsIHtcbiAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvZnVuY3Rpb25zJykpLFxuICAgICAgICBoYW5kbGVyOiAnaW5kZXguaXNfY29tcGxldGUnLFxuICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM184LFxuICAgICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKDYwKSxcbiAgICAgICAgcm9sZTogb25FdmVudC5yb2xlLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IG15UHJvdmlkZXIgPSBuZXcgY3IuUHJvdmlkZXIodGhpcywgJ015UHJvdmlkZXInLCB7XG4gICAgICAgIG9uRXZlbnRIYW5kbGVyOiBvbkV2ZW50LFxuICAgICAgICBpc0NvbXBsZXRlSGFuZGxlcjogaXNDb21wbGV0ZSxcbiAgICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICAgIH0pO1xuXG4gICAgICBvbkV2ZW50LmFkZFRvUm9sZVBvbGljeShcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogWydlYzI6RGVzY3JpYmVTcG90RmxlZXRJbnN0YW5jZXMnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICB9KSxcbiAgICAgICk7XG5cbiAgICAgIGNvbnN0IGZsZWV0SW5zdGFuY2VzSWQgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0dldEluc3RhbmNlSWQnLCB7XG4gICAgICAgIHNlcnZpY2VUb2tlbjogbXlQcm92aWRlci5zZXJ2aWNlVG9rZW4sXG4gICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICBTcG90RmxlZXRSZXF1ZXN0SWQ6IHRoaXMuY2ZuU3BvdEZsZWV0LnJlZixcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICBmbGVldEluc3RhbmNlc0lkLm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLmNmblNwb3RGbGVldCk7XG4gICAgICB0aGlzLnNwb3RGbGVldEluc3RhbmNlSWQgPSBUb2tlbi5hc1N0cmluZyhcbiAgICAgICAgZmxlZXRJbnN0YW5jZXNJZC5nZXRBdHQoJ0luc3RhbmNlSWQnKSxcbiAgICAgICk7XG4gICAgICB0aGlzLnNwb3RGbGVldFJlcXVlc3RJZCA9IFRva2VuLmFzU3RyaW5nKFxuICAgICAgICBmbGVldEluc3RhbmNlc0lkLmdldEF0dCgnU3BvdEluc3RhbmNlUmVxdWVzdElkJyksXG4gICAgICApO1xuICAgICAgbmV3IENmbk91dHB1dCh0aGlzLCAnSW5zdGFuY2VJZCcsIHsgdmFsdWU6IHRoaXMuc3BvdEZsZWV0SW5zdGFuY2VJZCB9KTtcbiAgICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ1Nwb3RGbGVldElkJywgeyB2YWx1ZTogdGhpcy5jZm5TcG90RmxlZXQucmVmIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnJ1bm5lckVjMiA9IG5ldyBJbnN0YW5jZSh0aGlzLCAnR2l0bGFiUnVubmVyJywge1xuICAgICAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUocnVubmVyUHJvcHMuZWMydHlwZSksXG4gICAgICAgIGluc3RhbmNlTmFtZTogJ0dpdGxhYi1SdW5uZXInLFxuICAgICAgICB2cGM6IHRoaXMudnBjLFxuICAgICAgICB2cGNTdWJuZXRzOiBydW5uZXJQcm9wcy52cGNTdWJuZXQgPz8ge1xuICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDLFxuICAgICAgICB9LFxuICAgICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eDIoKSxcbiAgICAgICAgcm9sZTogdGhpcy5ydW5uZXJSb2xlLFxuICAgICAgICB1c2VyRGF0YTogc2hlbGwsXG4gICAgICAgIHNlY3VyaXR5R3JvdXA6IHRoaXMuZGVmYXVsdFJ1bm5lclNHLFxuICAgICAgICBibG9ja0RldmljZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBkZXZpY2VOYW1lOiAnL2Rldi94dmRhJyxcbiAgICAgICAgICAgIHZvbHVtZTogQmxvY2tEZXZpY2VWb2x1bWUuZWJzKHJ1bm5lclByb3BzLmVic1NpemUgPz8gNjApLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KTtcbiAgICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ1J1bm5lci1JbnN0YW5jZS1JRCcsIHtcbiAgICAgICAgdmFsdWU6IHRoaXMucnVubmVyRWMyLmluc3RhbmNlSWQsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCB1bnJlZ2lzdGVyUnVubmVyT25FdmVudCA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ3VucmVnaXN0ZXJSdW5uZXJPbkV2ZW50Jywge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvZnVuY3Rpb25zJykpLFxuICAgICAgaGFuZGxlcjogJ3VucmVnaXN0ZXJfcnVubmVyLm9uX2V2ZW50JyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzgsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKDYwKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHVucmVnaXN0ZXJSdW5uZXJQcm92aWRlciA9IG5ldyBjci5Qcm92aWRlcih0aGlzLCAndW5yZWdpc3RlclJ1bm5lclByb3ZpZGVyJywge1xuICAgICAgb25FdmVudEhhbmRsZXI6IHVucmVnaXN0ZXJSdW5uZXJPbkV2ZW50LFxuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHVucmVnaXN0ZXJSdW5uZXJDUiA9IG5ldyBDdXN0b21SZXNvdXJjZSh0aGlzLCAndW5yZWdpc3RlclJ1bm5lckNSJywge1xuICAgICAgcmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjp1bnJlZ2lzdGVyUnVubmVyUHJvdmlkZXInLFxuICAgICAgc2VydmljZVRva2VuOiB1bnJlZ2lzdGVyUnVubmVyUHJvdmlkZXIuc2VydmljZVRva2VuLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBUb2tlblBhcmFtZXRlclN0b3JlTmFtZTogdG9rZW5QYXJhbWV0ZXJTdG9yZS5wYXJhbWV0ZXJOYW1lLFxuICAgICAgICBHaXRsYWJVcmw6IHJ1bm5lclByb3BzLmdpdGxhYnVybCxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0b2tlblBhcmFtZXRlclN0b3JlLmdyYW50UmVhZCh1bnJlZ2lzdGVyUnVubmVyT25FdmVudCk7XG4gICAgdW5yZWdpc3RlclJ1bm5lckNSLm5vZGUuYWRkRGVwZW5kZW5jeSh0b2tlblBhcmFtZXRlclN0b3JlKTtcbiAgICB0aGlzLnJ1bm5lclJvbGUuYWRkTWFuYWdlZFBvbGljeShcbiAgICAgIE1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TU01NYW5hZ2VkSW5zdGFuY2VDb3JlJyksXG4gICAgKTtcblxuICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ1J1bm5lci1Sb2xlLUFybicsIHtcbiAgICAgIHZhbHVlOiB0aGlzLnJ1bm5lclJvbGUucm9sZUFybixcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgZXhwaXJlIHRpbWUgZnVuY3Rpb24gZm9yIHNwb3RmbGVldCBydW5uZXIgISEhIC5cbiAgICpcbiAgICogQHBhcmFtIGR1cmF0aW9uIC0gQmxvY2sgZHVyYXRpb24uXG4gICAqL1xuICBwdWJsaWMgZXhwaXJlQWZ0ZXIoZHVyYXRpb246IER1cmF0aW9uKSB7XG4gICAgY29uc3QgZGF0ZSA9IG5ldyBEYXRlKCk7XG4gICAgZGF0ZS5zZXRTZWNvbmRzKGRhdGUuZ2V0U2Vjb25kcygpICsgZHVyYXRpb24udG9TZWNvbmRzKCkpO1xuICAgIHRoaXMudmFsaWRVbnRpbCA9IGRhdGUudG9JU09TdHJpbmcoKTtcbiAgfVxuXG4gIHByaXZhdGUgZG9ja2VyVm9sdW1lc0xpc3QoZG9ja2VyVm9sdW1lOiBEb2NrZXJWb2x1bWVzW10gfCB1bmRlZmluZWQpOiBzdHJpbmcge1xuICAgIGxldCB0ZW1wU3RyaW5nOiBzdHJpbmcgPSAnLS1kb2NrZXItdm9sdW1lcyBcIi92YXIvcnVuL2RvY2tlci5zb2NrOi92YXIvcnVuL2RvY2tlci5zb2NrXCInO1xuICAgIGlmIChkb2NrZXJWb2x1bWUpIHtcbiAgICAgIGxldCB0ZW1wTGlzdDogc3RyaW5nW10gPSBbXTtcbiAgICAgIGRvY2tlclZvbHVtZS5mb3JFYWNoKGUgPT4ge1xuICAgICAgICB0ZW1wTGlzdC5wdXNoKGBcIiR7ZS5ob3N0UGF0aH06JHtlLmNvbnRhaW5lclBhdGh9XCJgKTtcbiAgICAgIH0pO1xuICAgICAgdGVtcExpc3QuZm9yRWFjaChlID0+IHtcbiAgICAgICAgdGVtcFN0cmluZyA9IGAke3RlbXBTdHJpbmd9IC0tZG9ja2VyLXZvbHVtZXMgJHtlfWA7XG4gICAgICB9KTtcbiAgICB9XG4gICAgcmV0dXJuIHRlbXBTdHJpbmc7XG4gIH1cbiAgLyoqXG4gICAqIEBwYXJhbSBwcm9wc1xuICAgKiBAcGFyYW0gdG9rZW5QYXJhbWV0ZXJTdG9yZU5hbWUgLSB0aGUgdG9rZW5QYXJhbWV0ZXJTdG9yZU5hbWUgdG8gcHV0IGdpdGxhYiBydW5uZXIgdG9rZW4uXG4gICAqIEByZXR1cm5zIEFycmF5LlxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVVzZXJEYXRhKHByb3BzOiBHaXRsYWJDb250YWluZXJSdW5uZXJQcm9wcywgdG9rZW5QYXJhbWV0ZXJTdG9yZU5hbWU6IHN0cmluZyk6IHN0cmluZ1tdIHtcbiAgICByZXR1cm4gW1xuICAgICAgJ3l1bSB1cGRhdGUgLXkgJyxcbiAgICAgICdzbGVlcCAxNSAmJiBhbWF6b24tbGludXgtZXh0cmFzIGluc3RhbGwgZG9ja2VyICYmIHl1bSBpbnN0YWxsIC15IGFtYXpvbi1jbG91ZHdhdGNoLWFnZW50ICYmIHN5c3RlbWN0bCBzdGFydCBkb2NrZXIgJiYgdXNlcm1vZCAtYUcgZG9ja2VyIGVjMi11c2VyICYmIGNobW9kIDc3NyAvdmFyL3J1bi9kb2NrZXIuc29jaycsXG4gICAgICAnc3lzdGVtY3RsIHJlc3RhcnQgZG9ja2VyICYmIHN5c3RlbWN0bCBlbmFibGUgZG9ja2VyJyxcbiAgICAgIGBkb2NrZXIgcnVuIC1kIC12IC9ob21lL2VjMi11c2VyLy5naXRsYWItcnVubmVyOi9ldGMvZ2l0bGFiLXJ1bm5lciAtdiAvdmFyL3J1bi9kb2NrZXIuc29jazovdmFyL3J1bi9kb2NrZXIuc29jayBcXFxuICAgICAgLS1uYW1lIGdpdGxhYi1ydW5uZXItcmVnaXN0ZXIgJHtwcm9wcy5naXRsYWJSdW5uZXJJbWFnZX0gcmVnaXN0ZXIgLS1ub24taW50ZXJhY3RpdmUgLS11cmwgJHtwcm9wcy5naXRsYWJ1cmx9IC0tcmVnaXN0cmF0aW9uLXRva2VuICR7cHJvcHMuZ2l0bGFidG9rZW59IFxcXG4gICAgICAtLWRvY2tlci1wdWxsLXBvbGljeSBpZi1ub3QtcHJlc2VudCAke3RoaXMuZG9ja2VyVm9sdW1lc0xpc3QocHJvcHM/LmRvY2tlclZvbHVtZXMpfSBcXFxuICAgICAgLS1leGVjdXRvciBkb2NrZXIgLS1kb2NrZXItaW1hZ2UgXCJhbHBpbmU6bGF0ZXN0XCIgLS1kZXNjcmlwdGlvbiBcIiR7cHJvcHMucnVubmVyRGVzY3JpcHRpb259XCIgXFxcbiAgICAgIC0tdGFnLWxpc3QgXCIke3Byb3BzLnRhZ3M/LmpvaW4oJywnKX1cIiAtLWRvY2tlci1wcml2aWxlZ2VkYCxcbiAgICAgIGBzbGVlcCAyICYmIGRvY2tlciBydW4gLS1yZXN0YXJ0IGFsd2F5cyAtZCAtdiAvaG9tZS9lYzItdXNlci8uZ2l0bGFiLXJ1bm5lcjovZXRjL2dpdGxhYi1ydW5uZXIgLXYgL3Zhci9ydW4vZG9ja2VyLnNvY2s6L3Zhci9ydW4vZG9ja2VyLnNvY2sgLS1uYW1lIGdpdGxhYi1ydW5uZXIgJHtwcm9wcy5naXRsYWJSdW5uZXJJbWFnZX1gLFxuICAgICAgYHNlZCAtaSAncy9jb25jdXJyZW50ID0gLiovY29uY3VycmVudCA9ICR7cHJvcHMuY29uY3VycmVudEpvYnN9L2cnIC9ob21lL2VjMi11c2VyLy5naXRsYWItcnVubmVyL2NvbmZpZy50b21sYCxcbiAgICAgICdUT0tFTj0kKGNhdCAvaG9tZS9lYzItdXNlci8uZ2l0bGFiLXJ1bm5lci9jb25maWcudG9tbCB8IGdyZXAgdG9rZW4gfCBhd2sgXFwne3ByaW50ICQzfVxcJ3wgdHIgLWQgXFwnXCJcXCcpJyxcbiAgICAgIGBhd3Mgc3NtIHB1dC1wYXJhbWV0ZXIgLS1uYW1lICR7dG9rZW5QYXJhbWV0ZXJTdG9yZU5hbWV9IC0tdmFsdWUgJFRPS0VOIC0tb3ZlcndyaXRlIC0tcmVnaW9uICR7U3RhY2sub2YodGhpcykucmVnaW9ufWAsXG4gICAgXTtcbiAgfVxufVxuIl19