# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/00_outlier_detection.ipynb (unless otherwise specified).

__all__ = ['__load_df', '__scipy_zscore__', 'zscore', '__median', '__left_median', '__right_median', '__first_quartile',
           '__third_quartile', '__iqr', 'iqr', 'dbscan', 'ZSCORE', 'IQR']

# Cell
import numpy as np
import pandas as pd
import databricks.koalas as ks
from dataprep.eda import plot, plot_missing
import scipy
import sklearn
from sklearn.cluster import DBSCAN
from numba import jit

# Cell


def __load_df(filename='../data/sensor.csv'):
    """ Reads  censor data

    -----
    Parameter
    ------

    filename : str: csv filename


    ----
    Returns
    -----

    Pandas' Dataframe
    """

    df = pd.read_csv(filename,)
    df.timestamp = pd.to_datetime(df.timestamp)
    df.drop(
        columns=['Unnamed: 0', 'sensor_15'],
        inplace=True,
    )
    df.set_index('timestamp', inplace=True)
    return df

# Cell
def __scipy_zscore__(df,
                     ignored_cols=(
                         'timestamp',
                         'machine_status',
                     ),
                     index_col='timestamp'):
    """
    Apply scipy z-score over a dataframe
    -------
    Paramaters
    -------
     - df: pandas Dataframe
     - ignore_cols: list of names to ignore
     - index: column name to be considered as index
    -------
    Returns

    -------
      - dataframe
    """

    def zscore_f(x):
        return scipy.stats.zscore(x.to_numpy(), nan_policy='omit')

    columns = [x for x in df.columns if x not in ignored_cols]

    dff = pd.DataFrame(df[columns].apply(
        zscore_f,
        axis=0,
    ))
    dff.index = df[index_col]
    return dff

# Cell
def zscore(
    df,
    ignored_cols=(
        'timestamp',
        'machine_status',
    ),
    index_col='timestamp',
    threshold=np.inf,
    engine=pd,
):
    """
    Apply scipy z-score over a dataframe with a threshold.

    -------
    Paramaters
    -------

     - df: pandas Dataframe
     - ignore_cols: list of names to ignore
     - index: column name to be considered as index
     - threshold: Values greater than the threshold
                  are converted to np.nan
     - engine(=pd): Use pandas(pd) or another, i.e., koalas

    -------
    Returns
    -------

      - Dataframe
    """

    def zscore_f(x):

        zs = scipy.stats.zscore(
            x.to_numpy(dtype=float,),
            nan_policy='omit',
        )

        zs[np.abs(zs) >= threshold] = np.nan
        return zs

    columns = [x for x in df.columns if x not in ignored_cols]

    dff = engine.DataFrame(df[columns].apply(
        zscore_f,
        axis=0,
    ))
    dff.index = df[index_col]
    return dff

# Cell
def __median(s):
    ss = np.sort(s[~np.isnan(s)])
    m = len(ss) // 2
    if len(ss) % 2 == 0:
        median = (ss[m - 1] + ss[m]) / 2
    else:
        median = ss[m]
    return median


def __left_median(s, q):

    if q > 1 or q < 0:
        raise Exception("q should be on the interval (0,1)")
    ns = s[~np.isnan(s)]
    ss = np.sort(ns)[:int(len(ns) * q)]

    return __median(ss)


def __right_median(s, q):

    if q > 1 or q < 0:
        raise Exception("q should be on the interval (0,1)")
    ns = s[~np.isnan(s)]
    ss = np.sort(ns)[int(len(ns) * q):]

    return __median(ss)


def __first_quartile(s):
    return __left_median(s, q=1 / 2)


def __third_quartile(s):
    return __right_median(s, q=1 / 2)


def __iqr(s):
    return __third_quartile(s) - __first_quartile(s)

# Cell
def iqr(
    df,
    ignored_cols=(
        'timestamp',
        'machine_status',
    ),
    index_col='timestamp',
    k=1.5,
    engine=pd,
):
    """
    Apply scipy iqr over a dataframe with a threshold.

    -------
    Paramaters
    -------

     - df: pandas Dataframe
     - ignore_cols: list of names to ignore
     - index: column name to be considered as index
     - threshold: Values greater than the threshold
                  are converted to np.nan
     - engine(=pd): Use pandas(pd) or another, i.e., koalas

    -------
    Returns
    -------

      - Dataframe
    """

    def selection_by_iqr(x):

        X = x.to_numpy(dtype=float,)

        ll = (1 + k) * __first_quartile(X,) - k * __third_quartile(X,)
        hl = (1 + k) * __third_quartile(X,) - k * __first_quartile(X,)
        X[(X < ll) | (X > hl)] = np.nan

        return X

    columns = [x for x in df.columns if x not in ignored_cols]

    dff = engine.DataFrame(df[columns].apply(
        selection_by_iqr,
        axis=0,
    ))
    dff.index = df[index_col]
    return dff

# Cell
def dbscan(
    df,
    ignored_cols=(
        'timestamp',
        'machine_status',
    ),
    index_col='timestamp',
    engine=pd,
    **args,
):
    """
    Apply sklearn DBSCAN over a dataframe. First
    data is normalized with the zscore function.

    -------
    Paramaters
    -------

     - df: pandas Dataframe
     - ignore_cols: list of names to ignore
     - index: column name to be considered as index
     - args: arguments of sklearn.cluster.DBSCAN
     - engine(=pd): Use pandas(pd) or another, i.e., koalas

    -------
    Returns
    -------

      - Dataframe
    """

    def dbscan_f(x):
        X = x.to_numpy().reshape(-1, 1)
        return sklearn.cluster.DBSCAN(
       #     force_all_finite='allow-nan',
            **args,
        ).fit(X,y='validation')

    columns = [x for x in df.columns if x not in ignored_cols]

    df_z = zscore(
        df,
        ignored_cols=ignored_cols,
        index_col=index_col,
        engine=engine,
    )

    dff = engine.DataFrame(df_z[columns].apply(
        dbscan_f,
        axis=0,
    ))
    dff.index = df[index_col]
    return dff

# Cell
def ZSCORE(**kwargs):
    """
    Filters a Dataframe by a zscore.

    -----
    Parameters
    -----

    args: list of parameters

    -------
    Returns
    -------
    tuple: First element is the Koalas Dataframe and the second
           is ploty's fig object
    """
    keys = ('freq', 'threshold', 'filename')
    values = ('12H', 3, '../data/sensor.csv')
    for k, v in zip(keys, values):
        if k not in kwargs.keys():
            kwargs[k] = v

    df = __load_df(filename=kwargs['filename'])
    dff = df.reset_index().set_index(['machine_status', 'timestamp'])
    level_values = dff.index.get_level_values

    dfz = dff.groupby(
        [level_values(0),
         pd.Grouper(
             freq=kwargs['freq'],
             level=-1,
         )]).mean().reset_index().groupby('machine_status').apply(
             lambda x: zscore(
                 x,
                 threshold=kwargs['threshold'],
             ),)

    kdfz = ks.from_pandas(dfz,)

    machine_status = ['NORMAL', 'BROKEN', 'RECOVERING']

    figs = {}
    for ms in machine_status:

        figs[ms] = kdfz.loc[ms].plot(kind='scatter',
                                     title=f'Z-SCORE. Machine Status -> {ms}',
                                     backend='plotly')

        figs[ms].update_layout(template="plotly_dark",)
        figs[ms].update_traces(marker=dict(size=1.5))

    return kdfz, figs

# Cell
def IQR(**kwargs):
    """
    Filters a Dataframe by a IQR.

    -----
    Parameters
    -----

    args: list of parameters

    -------
    Returns
    -------
    tuple: First element is the Koalas Dataframe and the second
           is ploty's fig object
    """
    keys = ('freq', 'filename')
    values = ('12H', '../data/sensor.csv')
    for k, v in zip(keys, values):
        if k not in kwargs.keys():
            kwargs[k] = v

    df = __load_df(filename=kwargs['filename'])
    dff = df.reset_index().set_index(['machine_status', 'timestamp'])
    level_values = dff.index.get_level_values

    dfz = dff.groupby(
        [level_values(0),
         pd.Grouper(
             freq=kwargs['freq'],
             level=-1,
         )]).mean().reset_index().groupby('machine_status').apply(iqr)

    kdfz = ks.from_pandas(dfz,)

    machine_status = ['NORMAL', 'BROKEN', 'RECOVERING']

    figs = {}
    for ms in machine_status:

        figs[ms] = kdfz.loc[ms].plot(kind='scatter',
                                     title=f'Z-SCORE. Machine Status -> {ms}',
                                     backend='plotly')

        figs[ms].update_layout(template="plotly_dark",)
        figs[ms].update_traces(marker=dict(size=1.5))

    return kdfz, figs