"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName]
            }
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type, identifier) {
        this.scopeTo({
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        });
    }
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types) {
        this.scopeTo({
            complianceResourceTypes: types
        });
    }
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key, value) {
        this.scopeTo({
            tagKey: key,
            tagValue: value
        });
    }
    scopeTo(scope) {
        if (!this.isManaged && !this.isCustomWithChanges) {
            throw new Error('Cannot scope rule when `configurationChanges` is set to false.');
        }
        this.scope = scope;
    }
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier
            }
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification'
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification'
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification'
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com')
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn
            }
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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