import * as s3 from '@aws-cdk/aws-s3';
import * as s3_assets from '@aws-cdk/aws-s3-assets';
import { Construct } from '@aws-cdk/core';
/**
 * Represents the Application Code.
 *
 * @stability stable
 */
export declare abstract class Code {
    /**
     * Application code as an S3 object.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @stability stable
     */
    static fromBucket(bucket: s3.IBucket, key: string, objectVersion?: string): S3Code;
    /**
     * Loads the application code from a local disk path.
     *
     * @param path Either a directory with the application code bundle or a .zip file.
     * @stability stable
     */
    static fromAsset(path: string, options?: s3_assets.AssetOptions): AssetCode;
    /**
     * Called when the deployment object is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @param scope The binding scope.
     * @stability stable
     */
    abstract bind(scope: Construct): CodeConfig;
}
/**
 * Result of binding `Code` into a `Ec2Deployer`.
 *
 * @stability stable
 */
export interface CodeConfig {
    /**
     * The location of the code in S3.
     *
     * @default - code is an s3 location
     * @stability stable
     */
    readonly s3Location: s3.Location;
}
/**
 * Application code from an S3 archive.
 *
 * @stability stable
 */
export declare class S3Code extends Code {
    private key;
    private objectVersion?;
    /**
     * @stability stable
     */
    readonly isInline = false;
    private bucketName;
    /**
     * @stability stable
     */
    constructor(bucket: s3.IBucket, key: string, objectVersion?: string | undefined);
    /**
     * Called when the deployment object is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope: Construct): CodeConfig;
}
/**
 * Application code from a local directory.
 *
 * @stability stable
 */
export declare class AssetCode extends Code {
    readonly path: string;
    private readonly options;
    private asset?;
    /**
     * @param path The path to the asset file or directory.
     * @stability stable
     */
    constructor(path: string, options?: s3_assets.AssetOptions);
    /**
     * Called when the deployment object is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope: Construct): CodeConfig;
}
/**
 * @stability stable
 */
export interface ResourceBindOptions {
    /**
     * The name of the CloudFormation property to annotate with asset metadata.
     *
     * @default Code
     * @see https://github.com/aws/aws-cdk/issues/1432
     * @stability stable
     */
    readonly resourceProperty?: string;
}
