from .expression import evaluate
from .data import Unsigned
from .base import NoSize


def g1_message_size(handle, totalLength, section4Length, section4Offset):

    totalLength = evaluate(totalLength, handle, 'r')
    section4Length = evaluate(section4Length, handle, 'r')

    if section4Length < 120 and (totalLength & 0x800000):

        totalLength &= 0x7fffff
        totalLength *= 120
        totalLength -= section4Length
        totalLength += 4

        section4Length = totalLength - section4Offset - 4

    return (totalLength, section4Length)


class G1_message_length(Unsigned):

    def __init__(self, name, length, section4Length):
        super().__init__(name, length)
        self.section4Length = section4Length

    def get_r(self, handle):  # 'r' is for raw
        return super().get(handle)

    def get(self, handle):
        (totalLength, section4Length) = g1_message_size(handle,
                                                        super().get(handle),
                                                        self.section4Length,
                                                        0)
        return totalLength


class G1_section4_length(Unsigned):

    def __init__(self, name, size, totalLength):
        super().__init__(name, size)
        self.size = size
        self.totalLength = totalLength

    def get_r(self, handle):  # 'r' is for raw
        return super().get(handle)

    def get(self, handle):
        (totalLength, section4Length) = g1_message_size(handle,
                                                        self.totalLength,
                                                        super().get(handle),
                                                        self.offset)
        return section4Length


class G1_half_byte_codeflag(Unsigned):

    def __init__(self, name):
        super().__init__(name, 1)

    def get(self, handle):
        return super().get(handle) & 0xf

    @property
    def length(self):
        return 0


class G1date(NoSize):

    def __init__(self, name, century, year, month, day):
        super().__init__(name)
        self.century = century
        self.year = year
        self.month = month
        self.day = day

    def get(self, handle):
        century = evaluate(self.century, handle)
        year = evaluate(self.year, handle)
        month = evaluate(self.month, handle)
        day = evaluate(self.day, handle)
        return ((century - 1) * 100 + year) * 10000 + month * 100 + day


class Mars_param(NoSize):

    def __init__(self, name, paramId, table, param):
        super().__init__(name)
        self.paramId = paramId
        self.table = table
        self.param = param

    def get(self, handle):
        return "%s.%s" % (evaluate(self.param, handle), evaluate(self.table, handle))
