"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_secret_encryptor_1 = require("./github-secret-encryptor");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    const smClient = new client_secrets_manager_1.SecretsManagerClient({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const physicalId = event.PhysicalResourceId;
    console.log('Create new resource with props ' + JSON.stringify(props));
    await createOrUpdateRepoSecret(event, octokit, smClient);
    return { PhysicalResourceId: physicalId };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const physicalId = event.PhysicalResourceId;
    console.log(`Update resource ${physicalId} with props ${JSON.stringify(props)}`);
    await createOrUpdateRepoSecret(event, octokit, smClient);
    return { PhysicalResourceId: physicalId };
};
const onDelete = async (event, octokit) => {
    const physicalId = event.PhysicalResourceId;
    console.log('Delete resource ' + physicalId);
    await deleteRepoSecret(event, octokit);
    return { PhysicalResourceId: physicalId };
};
const createOrUpdateRepoSecret = async (event, octokit, smClient) => {
    const { repositoryOwner: owner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const secretId = event.ResourceProperties.sourceSecretArn;
    const secretToEncrypt = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: secretId }));
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const secretResponse = await octokit.request('PUT /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteRepoSecret = async (event, octokit) => {
    const { repositoryOwner: owner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const deleteSecretResponse = await octokit.request('DELETE /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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