"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_secret_encryptor_1 = require("../action-secrets/github-secret-encryptor");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    const smClient = new client_secrets_manager_1.SecretsManagerClient({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const physicalId = event.PhysicalResourceId;
    console.log('Create new resource with props ' + JSON.stringify(props));
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    return { PhysicalResourceId: physicalId };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const physicalId = event.PhysicalResourceId;
    console.log(`Update resource ${physicalId} with props ${JSON.stringify(props)}`);
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    return { PhysicalResourceId: physicalId };
};
const onDelete = async (event, octokit) => {
    const physicalId = event.PhysicalResourceId;
    console.log('Delete resource ' + physicalId);
    await deleteEnvironmentSecret(event, octokit);
    return { PhysicalResourceId: physicalId };
};
const createOrUpdateEnvironmentSecret = async (event, octokit, smClient) => {
    const { repositoryOwner: owner, repositoryName: repo, repositorySecretName: secret_name, environment: environment_name, sourceSecretArn: secretId, } = event.ResourceProperties;
    const secretToEncrypt = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: secretId }));
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const repository_id = await getRepositoryId(event, octokit);
    const secretResponse = await octokit.request('PUT /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteEnvironmentSecret = async (event, octokit) => {
    const { environment: environment_name, repositorySecretName: secret_name } = event.ResourceProperties;
    const repository_id = await getRepositoryId(event, octokit);
    const deleteSecretResponse = await octokit.request('DELETE /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const getRepositoryId = async (event, octokit) => {
    const { repositoryOwner: owner, repositoryName: repo } = event.ResourceProperties;
    const { data } = await octokit.request('GET /repos/{owner}/{repo}', {
        owner,
        repo,
    });
    return data.id;
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWN0aW9uLWVudmlyb25tZW50LXNlY3JldC1oYW5kbGVyLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9oYW5kbGVyL2FjdGlvbi1lbnZpcm9ubWVudC1zZWNyZXRzL2FjdGlvbi1lbnZpcm9ubWVudC1zZWNyZXQtaGFuZGxlci5sYW1iZGEudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNEVBQThGO0FBQzlGLHdDQUF3QztBQUd4Qyx1RkFBeUU7QUFFekUsTUFBTSxPQUFPLEdBQUcsS0FBSyxFQUFFLEtBQXdELEVBQUUsRUFBRTtJQUNqRixPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDL0MsTUFBTSxRQUFRLEdBQUcsSUFBSSw2Q0FBb0IsQ0FBQyxFQUFFLE1BQU0sRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztJQUMxRixNQUFNLGlCQUFpQixHQUFHLE1BQU0sUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLDhDQUFxQixDQUFDLEVBQUUsUUFBUSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNuSSxNQUFNLE9BQU8sR0FBRyxJQUFJLGNBQU8sQ0FBQyxFQUFFLElBQUksRUFBRSxpQkFBaUIsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDO0lBRXRFLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7SUFDdEMsUUFBUSxXQUFXLEVBQUU7UUFDbkIsS0FBSyxRQUFRO1lBQ1gsT0FBTyxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUM1QyxLQUFLLFFBQVE7WUFDWCxPQUFPLFFBQVEsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzVDLEtBQUssUUFBUTtZQUNYLE9BQU8sUUFBUSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNsQztZQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsNkJBQTZCLFdBQVcsR0FBRyxDQUFDLENBQUM7S0FDaEU7QUFDSCxDQUFDLENBQUM7QUFFRixNQUFNLFFBQVEsR0FBRyxLQUFLLEVBQ3BCLEtBQXdELEVBQ3hELE9BQWdCLEVBQ2hCLFFBQThCLEVBQzlCLEVBQUU7SUFDRixNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7SUFDdkMsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBQzVDLE9BQU8sQ0FBQyxHQUFHLENBQUMsaUNBQWlDLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBRXZFLE1BQU0sK0JBQStCLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNoRSxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsVUFBVSxFQUFFLENBQUM7QUFDNUMsQ0FBQyxDQUFDO0FBRUYsTUFBTSxRQUFRLEdBQUcsS0FBSyxFQUNwQixLQUF3RCxFQUN4RCxPQUFnQixFQUNoQixRQUE4QixFQUM5QixFQUFFO0lBQ0YsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBQ3ZDLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztJQUM1QyxPQUFPLENBQUMsR0FBRyxDQUFDLG1CQUFtQixVQUFVLGVBQWUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFakYsTUFBTSwrQkFBK0IsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ2hFLE9BQU8sRUFBRSxrQkFBa0IsRUFBRSxVQUFVLEVBQUUsQ0FBQztBQUM1QyxDQUFDLENBQUM7QUFFRixNQUFNLFFBQVEsR0FBRyxLQUFLLEVBQ3BCLEtBQXdELEVBQ3hELE9BQWdCLEVBQ2hCLEVBQUU7SUFDRixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7SUFDNUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsR0FBRyxVQUFVLENBQUMsQ0FBQztJQUM3QyxNQUFNLHVCQUF1QixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM5QyxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsVUFBVSxFQUFFLENBQUM7QUFDNUMsQ0FBQyxDQUFDO0FBRUYsTUFBTSwrQkFBK0IsR0FBRyxLQUFLLEVBQzNDLEtBQXdELEVBQ3hELE9BQWdCLEVBQ2hCLFFBQThCLEVBQzlCLEVBQUU7SUFDRixNQUFNLEVBQ0osZUFBZSxFQUFFLEtBQUssRUFDdEIsY0FBYyxFQUFFLElBQUksRUFDcEIsb0JBQW9CLEVBQUUsV0FBVyxFQUNqQyxXQUFXLEVBQUUsZ0JBQWdCLEVBQzdCLGVBQWUsRUFBRSxRQUFRLEdBQzFCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBRTdCLE1BQU0sZUFBZSxHQUFHLE1BQU0sUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLDhDQUFxQixDQUFDLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUMvRixPQUFPLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0lBRTVELE1BQU0sWUFBWSxHQUFHLGVBQWUsQ0FBQyxZQUFZLENBQUM7SUFDbEQsSUFBSSxDQUFDLFlBQVksRUFBRTtRQUNqQixNQUFNLElBQUksS0FBSyxDQUFDLDZDQUE2QyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0tBQzNFO0lBRUQsTUFBTSxFQUFFLElBQUksRUFBRSxHQUFHLE1BQU0sT0FBTyxDQUFDLE9BQU8sQ0FBQyxzREFBc0QsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBRWhILE1BQU0sZUFBZSxHQUFHLE1BQU0sc0NBQVksQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ25FLE9BQU8sQ0FBQyxHQUFHLENBQUMsNkRBQTZELENBQUMsQ0FBQztJQUUzRSxNQUFNLGFBQWEsR0FBRyxNQUFNLGVBQWUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDNUQsTUFBTSxjQUFjLEdBQUcsTUFBTSxPQUFPLENBQUMsT0FBTyxDQUFDLHlGQUF5RixFQUFFO1FBQ3RJLGFBQWE7UUFDYixnQkFBZ0I7UUFDaEIsV0FBVztRQUNYLGVBQWUsRUFBRSxlQUFlO1FBQ2hDLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtLQUNwQixDQUFDLENBQUM7SUFFSCxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztJQUM1QyxPQUFPLGNBQWMsQ0FBQztBQUN4QixDQUFDLENBQUM7QUFFRixNQUFNLHVCQUF1QixHQUFHLEtBQUssRUFDbkMsS0FBd0QsRUFDeEQsT0FBZ0IsRUFDaEIsRUFBRTtJQUNGLE1BQU0sRUFBRSxXQUFXLEVBQUUsZ0JBQWdCLEVBQUUsb0JBQW9CLEVBQUUsV0FBVyxFQUFFLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBQ3RHLE1BQU0sYUFBYSxHQUFHLE1BQU0sZUFBZSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM1RCxNQUFNLG9CQUFvQixHQUFHLE1BQU0sT0FBTyxDQUFDLE9BQU8sQ0FBQyw0RkFBNEYsRUFBRTtRQUMvSSxhQUFhO1FBQ2IsZ0JBQWdCO1FBQ2hCLFdBQVc7S0FDWixDQUFDLENBQUM7SUFDSCxPQUFPLENBQUMsR0FBRyxDQUFDLFdBQVcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUMvRCxPQUFPLG9CQUFvQixDQUFDO0FBQzlCLENBQUMsQ0FBQztBQUVGLE1BQU0sZUFBZSxHQUFHLEtBQUssRUFDM0IsS0FBd0QsRUFDeEQsT0FBZ0IsRUFDaEIsRUFBRTtJQUNGLE1BQU0sRUFBRSxlQUFlLEVBQUUsS0FBSyxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7SUFDbEYsTUFBTSxFQUFFLElBQUksRUFBRSxHQUFHLE1BQU0sT0FBTyxDQUFDLE9BQU8sQ0FBQywyQkFBMkIsRUFBRTtRQUNsRSxLQUFLO1FBQ0wsSUFBSTtLQUNMLENBQUMsQ0FBQztJQUNILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQztBQUNqQixDQUFDLENBQUM7QUFHVyxRQUFBLE9BQU8sR0FBRyxPQUFPLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBTZWNyZXRzTWFuYWdlckNsaWVudCwgR2V0U2VjcmV0VmFsdWVDb21tYW5kIH0gZnJvbSAnQGF3cy1zZGsvY2xpZW50LXNlY3JldHMtbWFuYWdlcic7XG5pbXBvcnQgeyBPY3Rva2l0IH0gZnJvbSAnQG9jdG9raXQvY29yZSc7XG5pbXBvcnQgdHlwZSB7IE9uRXZlbnRSZXF1ZXN0LCBBY3Rpb25FbnZpcm9ubWVudFNlY3JldEV2ZW50UHJvcHMgfSBmcm9tICcuLi8uLi90eXBlcyc7XG5cbmltcG9ydCB7IGVuY3J5cHRWYWx1ZSB9IGZyb20gJy4uL2FjdGlvbi1zZWNyZXRzL2dpdGh1Yi1zZWNyZXQtZW5jcnlwdG9yJztcblxuY29uc3Qgb25FdmVudCA9IGFzeW5jIChldmVudDogT25FdmVudFJlcXVlc3Q8QWN0aW9uRW52aXJvbm1lbnRTZWNyZXRFdmVudFByb3BzPikgPT4ge1xuICBjb25zb2xlLmxvZyhgRXZlbnQ6ICR7SlNPTi5zdHJpbmdpZnkoZXZlbnQpfWApO1xuICBjb25zdCBzbUNsaWVudCA9IG5ldyBTZWNyZXRzTWFuYWdlckNsaWVudCh7IHJlZ2lvbjogZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzLmF3c1JlZ2lvbiB9KTtcbiAgY29uc3QgZ2l0aHViVG9rZW5TZWNyZXQgPSBhd2FpdCBzbUNsaWVudC5zZW5kKG5ldyBHZXRTZWNyZXRWYWx1ZUNvbW1hbmQoeyBTZWNyZXRJZDogZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzLmdpdGh1YlRva2VuU2VjcmV0IH0pKTtcbiAgY29uc3Qgb2N0b2tpdCA9IG5ldyBPY3Rva2l0KHsgYXV0aDogZ2l0aHViVG9rZW5TZWNyZXQuU2VjcmV0U3RyaW5nIH0pO1xuXG4gIGNvbnN0IHJlcXVlc3RUeXBlID0gZXZlbnQuUmVxdWVzdFR5cGU7XG4gIHN3aXRjaCAocmVxdWVzdFR5cGUpIHtcbiAgICBjYXNlICdDcmVhdGUnOlxuICAgICAgcmV0dXJuIG9uQ3JlYXRlKGV2ZW50LCBvY3Rva2l0LCBzbUNsaWVudCk7XG4gICAgY2FzZSAnVXBkYXRlJzpcbiAgICAgIHJldHVybiBvblVwZGF0ZShldmVudCwgb2N0b2tpdCwgc21DbGllbnQpO1xuICAgIGNhc2UgJ0RlbGV0ZSc6XG4gICAgICByZXR1cm4gb25EZWxldGUoZXZlbnQsIG9jdG9raXQpO1xuICAgIGRlZmF1bHQ6XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuZXhwZWN0ZWQgcmVxdWVzdCB0eXBlOiAnJHtyZXF1ZXN0VHlwZX0nYCk7XG4gIH1cbn07XG5cbmNvbnN0IG9uQ3JlYXRlID0gYXN5bmMgKFxuICBldmVudDogT25FdmVudFJlcXVlc3Q8QWN0aW9uRW52aXJvbm1lbnRTZWNyZXRFdmVudFByb3BzPixcbiAgb2N0b2tpdDogT2N0b2tpdCxcbiAgc21DbGllbnQ6IFNlY3JldHNNYW5hZ2VyQ2xpZW50LFxuKSA9PiB7XG4gIGNvbnN0IHByb3BzID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzO1xuICBjb25zdCBwaHlzaWNhbElkID0gZXZlbnQuUGh5c2ljYWxSZXNvdXJjZUlkO1xuICBjb25zb2xlLmxvZygnQ3JlYXRlIG5ldyByZXNvdXJjZSB3aXRoIHByb3BzICcgKyBKU09OLnN0cmluZ2lmeShwcm9wcykpO1xuXG4gIGF3YWl0IGNyZWF0ZU9yVXBkYXRlRW52aXJvbm1lbnRTZWNyZXQoZXZlbnQsIG9jdG9raXQsIHNtQ2xpZW50KTtcbiAgcmV0dXJuIHsgUGh5c2ljYWxSZXNvdXJjZUlkOiBwaHlzaWNhbElkIH07XG59O1xuXG5jb25zdCBvblVwZGF0ZSA9IGFzeW5jIChcbiAgZXZlbnQ6IE9uRXZlbnRSZXF1ZXN0PEFjdGlvbkVudmlyb25tZW50U2VjcmV0RXZlbnRQcm9wcz4sXG4gIG9jdG9raXQ6IE9jdG9raXQsXG4gIHNtQ2xpZW50OiBTZWNyZXRzTWFuYWdlckNsaWVudCxcbikgPT4ge1xuICBjb25zdCBwcm9wcyA9IGV2ZW50LlJlc291cmNlUHJvcGVydGllcztcbiAgY29uc3QgcGh5c2ljYWxJZCA9IGV2ZW50LlBoeXNpY2FsUmVzb3VyY2VJZDtcbiAgY29uc29sZS5sb2coYFVwZGF0ZSByZXNvdXJjZSAke3BoeXNpY2FsSWR9IHdpdGggcHJvcHMgJHtKU09OLnN0cmluZ2lmeShwcm9wcyl9YCk7XG5cbiAgYXdhaXQgY3JlYXRlT3JVcGRhdGVFbnZpcm9ubWVudFNlY3JldChldmVudCwgb2N0b2tpdCwgc21DbGllbnQpO1xuICByZXR1cm4geyBQaHlzaWNhbFJlc291cmNlSWQ6IHBoeXNpY2FsSWQgfTtcbn07XG5cbmNvbnN0IG9uRGVsZXRlID0gYXN5bmMgKFxuICBldmVudDogT25FdmVudFJlcXVlc3Q8QWN0aW9uRW52aXJvbm1lbnRTZWNyZXRFdmVudFByb3BzPixcbiAgb2N0b2tpdDogT2N0b2tpdCxcbikgPT4ge1xuICBjb25zdCBwaHlzaWNhbElkID0gZXZlbnQuUGh5c2ljYWxSZXNvdXJjZUlkO1xuICBjb25zb2xlLmxvZygnRGVsZXRlIHJlc291cmNlICcgKyBwaHlzaWNhbElkKTtcbiAgYXdhaXQgZGVsZXRlRW52aXJvbm1lbnRTZWNyZXQoZXZlbnQsIG9jdG9raXQpO1xuICByZXR1cm4geyBQaHlzaWNhbFJlc291cmNlSWQ6IHBoeXNpY2FsSWQgfTtcbn07XG5cbmNvbnN0IGNyZWF0ZU9yVXBkYXRlRW52aXJvbm1lbnRTZWNyZXQgPSBhc3luYyAoXG4gIGV2ZW50OiBPbkV2ZW50UmVxdWVzdDxBY3Rpb25FbnZpcm9ubWVudFNlY3JldEV2ZW50UHJvcHM+LFxuICBvY3Rva2l0OiBPY3Rva2l0LFxuICBzbUNsaWVudDogU2VjcmV0c01hbmFnZXJDbGllbnQsXG4pID0+IHtcbiAgY29uc3Qge1xuICAgIHJlcG9zaXRvcnlPd25lcjogb3duZXIsXG4gICAgcmVwb3NpdG9yeU5hbWU6IHJlcG8sXG4gICAgcmVwb3NpdG9yeVNlY3JldE5hbWU6IHNlY3JldF9uYW1lLFxuICAgIGVudmlyb25tZW50OiBlbnZpcm9ubWVudF9uYW1lLFxuICAgIHNvdXJjZVNlY3JldEFybjogc2VjcmV0SWQsXG4gIH0gPSBldmVudC5SZXNvdXJjZVByb3BlcnRpZXM7XG5cbiAgY29uc3Qgc2VjcmV0VG9FbmNyeXB0ID0gYXdhaXQgc21DbGllbnQuc2VuZChuZXcgR2V0U2VjcmV0VmFsdWVDb21tYW5kKHsgU2VjcmV0SWQ6IHNlY3JldElkIH0pKTtcbiAgY29uc29sZS5sb2coYEVuY3J5cHQgdmFsdWUgb2Ygc2VjcmV0IHdpdGggaWQ6ICR7c2VjcmV0SWR9YCk7XG5cbiAgY29uc3Qgc2VjcmV0U3RyaW5nID0gc2VjcmV0VG9FbmNyeXB0LlNlY3JldFN0cmluZztcbiAgaWYgKCFzZWNyZXRTdHJpbmcpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ1NlY3JldFN0cmluZyBpcyBlbXB0eSBmcm9tIHNlY3JldCB3aXRoIGlkOiAnICsgc2VjcmV0SWQpO1xuICB9XG5cbiAgY29uc3QgeyBkYXRhIH0gPSBhd2FpdCBvY3Rva2l0LnJlcXVlc3QoJ0dFVCAvcmVwb3Mve293bmVyfS97cmVwb30vYWN0aW9ucy9zZWNyZXRzL3B1YmxpYy1rZXknLCB7IG93bmVyLCByZXBvIH0pO1xuXG4gIGNvbnN0IGVuY3J5cHRlZFNlY3JldCA9IGF3YWl0IGVuY3J5cHRWYWx1ZShzZWNyZXRTdHJpbmcsIGRhdGEua2V5KTtcbiAgY29uc29sZS5sb2coJ0VuY3J5cHRlZCBzZWNyZXQsIGF0dGVtcHRpbmcgdG8gY3JlYXRlL3VwZGF0ZSBnaXRodWIgc2VjcmV0Jyk7XG5cbiAgY29uc3QgcmVwb3NpdG9yeV9pZCA9IGF3YWl0IGdldFJlcG9zaXRvcnlJZChldmVudCwgb2N0b2tpdCk7XG4gIGNvbnN0IHNlY3JldFJlc3BvbnNlID0gYXdhaXQgb2N0b2tpdC5yZXF1ZXN0KCdQVVQgL3JlcG9zaXRvcmllcy97cmVwb3NpdG9yeV9pZH0vZW52aXJvbm1lbnRzL3tlbnZpcm9ubWVudF9uYW1lfS9zZWNyZXRzL3tzZWNyZXRfbmFtZX0nLCB7XG4gICAgcmVwb3NpdG9yeV9pZCxcbiAgICBlbnZpcm9ubWVudF9uYW1lLFxuICAgIHNlY3JldF9uYW1lLFxuICAgIGVuY3J5cHRlZF92YWx1ZTogZW5jcnlwdGVkU2VjcmV0LFxuICAgIGtleV9pZDogZGF0YS5rZXlfaWQsXG4gIH0pO1xuXG4gIGNvbnNvbGUubG9nKEpTT04uc3RyaW5naWZ5KHNlY3JldFJlc3BvbnNlKSk7XG4gIHJldHVybiBzZWNyZXRSZXNwb25zZTtcbn07XG5cbmNvbnN0IGRlbGV0ZUVudmlyb25tZW50U2VjcmV0ID0gYXN5bmMgKFxuICBldmVudDogT25FdmVudFJlcXVlc3Q8QWN0aW9uRW52aXJvbm1lbnRTZWNyZXRFdmVudFByb3BzPixcbiAgb2N0b2tpdDogT2N0b2tpdCxcbikgPT4ge1xuICBjb25zdCB7IGVudmlyb25tZW50OiBlbnZpcm9ubWVudF9uYW1lLCByZXBvc2l0b3J5U2VjcmV0TmFtZTogc2VjcmV0X25hbWUgfSA9IGV2ZW50LlJlc291cmNlUHJvcGVydGllcztcbiAgY29uc3QgcmVwb3NpdG9yeV9pZCA9IGF3YWl0IGdldFJlcG9zaXRvcnlJZChldmVudCwgb2N0b2tpdCk7XG4gIGNvbnN0IGRlbGV0ZVNlY3JldFJlc3BvbnNlID0gYXdhaXQgb2N0b2tpdC5yZXF1ZXN0KCdERUxFVEUgL3JlcG9zaXRvcmllcy97cmVwb3NpdG9yeV9pZH0vZW52aXJvbm1lbnRzL3tlbnZpcm9ubWVudF9uYW1lfS9zZWNyZXRzL3tzZWNyZXRfbmFtZX0nLCB7XG4gICAgcmVwb3NpdG9yeV9pZCxcbiAgICBlbnZpcm9ubWVudF9uYW1lLFxuICAgIHNlY3JldF9uYW1lLFxuICB9KTtcbiAgY29uc29sZS5sb2coYERlbGV0ZTogJHtKU09OLnN0cmluZ2lmeShkZWxldGVTZWNyZXRSZXNwb25zZSl9YCk7XG4gIHJldHVybiBkZWxldGVTZWNyZXRSZXNwb25zZTtcbn07XG5cbmNvbnN0IGdldFJlcG9zaXRvcnlJZCA9IGFzeW5jIChcbiAgZXZlbnQ6IE9uRXZlbnRSZXF1ZXN0PEFjdGlvbkVudmlyb25tZW50U2VjcmV0RXZlbnRQcm9wcz4sXG4gIG9jdG9raXQ6IE9jdG9raXQsXG4pID0+IHtcbiAgY29uc3QgeyByZXBvc2l0b3J5T3duZXI6IG93bmVyLCByZXBvc2l0b3J5TmFtZTogcmVwbyB9ID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzO1xuICBjb25zdCB7IGRhdGEgfSA9IGF3YWl0IG9jdG9raXQucmVxdWVzdCgnR0VUIC9yZXBvcy97b3duZXJ9L3tyZXBvfScsIHtcbiAgICBvd25lcixcbiAgICByZXBvLFxuICB9KTtcbiAgcmV0dXJuIGRhdGEuaWQ7XG59O1xuXG5cbmV4cG9ydCBjb25zdCBoYW5kbGVyID0gb25FdmVudDtcbiJdfQ==