"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeadlineClient = void 0;
const http = require("http");
const https = require("https");
/**
 * Implements a simple client that supports HTTP/HTTPS GET and POST requests.
 * It is intended to be used within Custom Resources that need to send the requests to the Render Queue.
 */
class DeadlineClient {
    constructor(props) {
        var _a, _b, _c;
        this.requestOptions = {
            host: props.host,
            port: props.port,
        };
        if (props.protocol === 'HTTPS') {
            this.protocol = https;
            this.requestOptions.agent = new https.Agent({
                pfx: (_a = props.tls) === null || _a === void 0 ? void 0 : _a.pfx,
                passphrase: (_b = props.tls) === null || _b === void 0 ? void 0 : _b.passphrase,
                ca: (_c = props.tls) === null || _c === void 0 ? void 0 : _c.ca,
            });
        }
        else {
            this.protocol = http;
        }
    }
    /**
     * Perform an HTTP GET request.
     *
     * @param path The resource to request for.
     * @param requestOptions Other request options, including headers, timeout, etc.
     */
    async GetRequest(path, requestOptions) {
        const options = this.FillRequestOptions(path, 'GET', requestOptions);
        return this.performRequest(options);
    }
    /**
     * Perform an HTTP POST request.
     *
     * @param path The resource to request for.
     * @param data The data (body) of the request that contains the information to be sent.
     * @param requestOptions Other request options, including headers, timeout, etc.
     */
    async PostRequest(path, data, requestOptions) {
        const options = this.FillRequestOptions(path, 'POST', requestOptions);
        return this.performRequest(options, data ? JSON.stringify(data) : undefined);
    }
    FillRequestOptions(path, method, requestOptions) {
        const options = {
            ...requestOptions,
            port: this.requestOptions.port,
            host: this.requestOptions.host,
            agent: this.requestOptions.agent,
            path: path,
            method: method,
        };
        return options;
    }
    async performRequest(options, data) {
        return new Promise((resolve, reject) => {
            try {
                const req = this.protocol.request(options, response => {
                    const { statusCode } = response;
                    if (!statusCode || statusCode >= 300) {
                        return reject(response.statusMessage);
                    }
                    else {
                        const chunks = [];
                        response.on('data', (chunk) => {
                            chunks.push(chunk);
                        });
                        response.on('end', () => {
                            const stringData = Buffer.concat(chunks).toString();
                            const result = {
                                data: JSON.parse(stringData),
                                fullResponse: response,
                            };
                            return resolve(result);
                        });
                    }
                });
                req.on('error', reject);
                if (data) {
                    req.write(data);
                }
                req.end();
            }
            catch (e) {
                reject(e);
            }
        });
    }
}
exports.DeadlineClient = DeadlineClient;
//# sourceMappingURL=data:application/json;base64,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