"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginClient = void 0;
/**
 * Provides a simple interface to send requests to the Render Queue API related to the Deadline Spot Event Plugin.
 */
class SpotEventPluginClient {
    constructor(client) {
        this.deadlineClient = client;
    }
    async saveServerData(config) {
        console.log('Saving server data configuration:');
        console.log(config);
        try {
            // Get the concurrency token required to save server data
            const concurrencyToken = await this.concurrencyToken();
            await this.deadlineClient.PostRequest('/rcs/v1/putServerData', {
                ServerData: [
                    {
                        ID: SpotEventPluginClient.EVENT_PLUGIN_ID,
                        ServerDataDictionary: {
                            Config: config,
                        },
                        ConcurrencyToken: concurrencyToken,
                    },
                ],
            }, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            return true;
        }
        catch (e) {
            console.error(`Failed to save server data. Reason: ${e}`);
            return false;
        }
    }
    async configureSpotEventPlugin(configs) {
        console.log('Saving plugin configuration:');
        console.log(configs);
        try {
            await this.deadlineClient.PostRequest('/db/plugins/event/config/save', {
                ID: 'spot',
                DebugLogging: false,
                DlInit: configs,
                Icon: null,
                Limits: [],
                Meta: [],
                Name: 'Spot',
                PluginEnabled: 1,
            }, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            return true;
        }
        catch (e) {
            console.error(`Failed to save plugin configuration. Reason: ${e}`);
            return false;
        }
    }
    async describeServerData() {
        return await this.deadlineClient.PostRequest('/rcs/v1/describeServerData', {
            ServerDataIds: [
                SpotEventPluginClient.EVENT_PLUGIN_ID,
            ],
        }, {
            headers: {
                'Content-Type': 'application/json; charset=utf-8',
            },
        });
    }
    /**
     * Requests a concurrencyToken required to save spot fleet request configuration.
     * If data already exists under the ID, an existing ConcurrencyToken has to be used.
     * First obtain the token and then save the data with the same ConcurrencyToken.
     * If there is no data under the ID, then real token is not required,
     * but the ConcurrencyToken property still has to be set.
     * NOTE:
     * saveServerData() will have a ConcurrencyToken in its response but we do not use it,
     * instead we always call this function to get a latest token.
     */
    async concurrencyToken() {
        var _a;
        const response = await this.describeServerData();
        const describedData = response.data;
        if (!describedData.ServerData || !Array.isArray(describedData.ServerData)) {
            throw new Error(`Failed to receive a ConcurrencyToken. Invalid response: ${describedData}.`);
        }
        const found = describedData.ServerData.find(element => element.ID === SpotEventPluginClient.EVENT_PLUGIN_ID);
        return (_a = found === null || found === void 0 ? void 0 : found.ConcurrencyToken) !== null && _a !== void 0 ? _a : '';
    }
}
exports.SpotEventPluginClient = SpotEventPluginClient;
SpotEventPluginClient.EVENT_PLUGIN_ID = 'event.plugin.spot';
//# sourceMappingURL=data:application/json;base64,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