"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const asset_constants_1 = require("../../deadline/test/asset-constants");
const lib_1 = require("../lib");
const asset_constants_2 = require("./asset-constants");
let app;
let stack;
let vpc;
let version;
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: ., $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack');
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    class MockVersion extends lib_1.Version {
        constructor() {
            super(...arguments);
            this.linuxInstallers = {
                patchVersion: 0,
                repository: {
                    objectKey: 'testInstaller',
                    s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxInstallerBucket'),
                },
            };
        }
        linuxFullVersionString() {
            return this.toString();
        }
    }
    version = new MockVersion([10, 1, 9, 2]);
});
test('can create two repositories', () => {
    // GIVEN
    new lib_1.Repository(stack, 'Repo1', {
        vpc,
        version,
    });
    // THEN
    expect(() => {
        new lib_1.Repository(stack, 'Repo2', {
            vpc,
            version,
        });
    }).not.toThrow();
});
test('repository installer instance is created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        Properties: {
            MaxSize: '1',
            MinSize: '1',
            VPCZoneIdentifier: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
        },
        CreationPolicy: {
            AutoScalingCreationPolicy: {
                MinSuccessfulInstancesPercent: 100,
            },
            ResourceSignal: {
                Timeout: 'PT15M',
            },
        },
        UpdatePolicy: {
            AutoScalingReplacingUpdate: {
                WillReplace: true,
            },
            AutoScalingScheduledAction: {
                IgnoreUnmodifiedGroupSizeProperties: true,
            },
        },
        DependsOn: [
            'repositoryInstallerDocumentDatabaseInstance11A6F8C8E',
        ],
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't3.large',
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
test('repository installer honors vpcSubnet', () => {
    // Note: Default is private subnets, so it's sufficient to test something other than
    // private subnets.
    // WHEN
    const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
    const attrVpc = aws_ec2_1.Vpc.fromVpcAttributes(stack, 'TestVpc', {
        availabilityZones: ['us-east-1a', 'us-east-1b'],
        vpcId: 'vpcid',
        publicSubnetIds,
    });
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc: attrVpc,
        version,
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: publicSubnetIds,
    }));
});
test('repository installer security groups created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        ToPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseSecurityGroupBEFDC58F',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
/*
IAM Policy document tests. The policy for the installer instance is:
  expectCDK(stack).to(haveResourceLike('AWS::IAM::Policy', {
    PolicyDocument: {
      Statement: [
        {}, // CloudWatch log group put
        {}, // cloudwatch agent install script
        {}, // cloudwatch agent string parameters
        {}, // cloudwatch agent get installer permissions
        {}, // gpg get installer permissions
        {}, // DocDB secret get
        {}, // filesystem mount script get
        {}, // installer get
        {}, // repository installation script asset get
        {}, // update autoscaling policy
        {}, // describe tags
      ],
    },
  }));

  We only explicitly test for the permissions we explicitly add:
    - docDB secret get
    - installer get
    - autoscaling policy (handled by: 'alidate instance self-termination' test)
    - describe tags (handled by: 'alidate instance self-termination' test)
  The others are side-effects of adding the cloudwatch agent, and mounting the filesystem.
  We make sure that we do those things in other ways, and trust to their constructs to be
  verifying proper permissions.
 */
test('repository installer iam permissions: db secret access', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {},
                {},
                {},
                {},
                {},
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'repositoryInstallerDocumentDatabaseSecretAttachment29753B7C',
                    },
                },
                {},
                {},
                {},
                {},
                {},
            ],
        },
    }));
});
test('repository installer iam permissions: installer get', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Effect: 'Allow',
                Resource: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                            '/*',
                        ],
                    ],
                }),
            }),
        },
    }));
});
test('default repository installer log group created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/repositoryInstaller',
    }));
});
test('repository installer logs all required files', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    // The CloudWatchAgent stores the agent configuration in an SSM Parameter. Check it for the required setup.
    // Note: This would be better implemented using the tools in: https://github.com/aws/aws-cdk/pull/8444
    // but that is not yet available.
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
        Type: 'String',
        Value: {
            'Fn::Join': [
                '',
                [
                    '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"deadlineRepositoryInstallationLogs-{instance_id}\",\"file_path\":\"/tmp/bitrock_installer.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                ],
            ],
        },
    }));
});
test('repository mounts repository filesystem', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // WHEN
    const userData = repo.node.defaultChild.userData.render();
    // THEN
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 rw')));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors database removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            database: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors filesystem removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EFS::FileSystem', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository warns if removal policy for filesystem when filesystem provided', () => {
    // GIVEN
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct',
        }),
    ]));
});
test('repository warns if removal policy for database when database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceProps: {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        removalPolicy: {
            database: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates deadlineDatabase if none provided', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBInstance'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    }));
});
test('disabling Audit logging does not enable Cloudwatch audit logs', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        databaseAuditLogging: false,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
});
test('repository warns if databaseAuditLogging defined and database is specified', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceProps: {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        databaseAuditLogging: true,
    });
    const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
        'Please ensure that the Database provided is configured correctly.';
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: warningMsg,
        }),
    ]));
});
test('honors subnet specification', () => {
    // GIVEN
    const dependencyStack = new core_1.Stack(app, 'DepStack');
    const dependencyVpc = new aws_ec2_1.Vpc(dependencyStack, 'DepVpc');
    const subnets = [
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
            subnetId: 'SubnetID1',
            availabilityZone: 'us-west-2a',
        }),
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
            subnetId: 'SubnetID2',
            availabilityZone: 'us-west-2b',
        }),
    ];
    const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
    // WHEN
    new lib_1.Repository(isolatedStack, 'repositoryInstaller', {
        vpc: dependencyVpc,
        version,
        vpcSubnets: {
            subnets,
        },
    });
    // THEN
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::DocDB::DBSubnetGroup', {
        SubnetIds: [
            'SubnetID1',
            'SubnetID2',
        ],
    }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID1' }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID2' }));
});
test('repository honors database instance count', () => {
    // GIVEN
    const instanceCount = 2;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        documentDbInstanceCount: instanceCount,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::DocDB::DBInstance', instanceCount, {
        AutoMinorVersionUpgrade: true,
    }));
});
test('repository honors database retention period', () => {
    // GIVEN
    const period = 20;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(period),
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        BackupRetentionPeriod: period,
    }));
});
test('warns if both retention period and database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceProps: {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(20),
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'Backup retention for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates filesystem if none provided', () => {
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceProps: {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        },
        backup: {
            retention: core_1.Duration.days(15),
        },
    });
    if (!fsDatabase.secret) {
        throw new Error('secret cannot be null');
    }
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::FileSystem'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::MountTarget'));
});
test('default repository instance is created using user defined installation path prefix', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: 'xyz',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('default repository instance is created using user defined installation path prefix with extra slashes in path', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: '/xyz//',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('repository instance is created with user defined timeout', () => {
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            AutoScalingCreationPolicy: {
                MinSuccessfulInstancesPercent: 100,
            },
            ResourceSignal: {
                Timeout: 'PT30M',
            },
        },
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository instance is created with correct installer path version', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/10\.1\.9\.2/);
});
test.each([
    'test-prefix/',
    '',
])('repository instance is created with correct LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'repositoryInstaller';
    // WHEN
    new lib_1.Repository(stack, id, {
        vpc,
        version,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        LogGroupName: testPrefix + id,
    }));
});
test('validate instance self-termination', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const asgLogicalId = stack.getLogicalId(repo.node.defaultChild.node.defaultChild);
    // THEN
    const regionToken = escapeTokenRegex('${Token[AWS.Region.\\d+]}');
    const expectedString = `function exitTrap\\(\\)\\{\nexitCode=\\$\\?\nsleep 1m\nINSTANCE="\\$\\(curl http:\\/\\/169\\.254\\.169\\.254\\/latest\\/meta-data\\/instance-id\\)"\nASG="\\$\\(aws --region ${regionToken} ec2 describe-tags --filters "Name=resource-id,Values=\\$\\{INSTANCE\\}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags\\[0\\]\\.Value" --output text\\)"\naws --region ${regionToken} autoscaling update-auto-scaling-group --auto-scaling-group-name \\$\\{ASG\\} --min-size 0 --max-size 0 --desired-capacity 0\n\\/opt\\/aws\\/bin\\/cfn-signal --stack ${stack.stackName} --resource ${asgLogicalId} --region ${regionToken} -e \\$exitCode \\|\\| echo 'Failed to send Cloudformation Signal'\n\\}`;
    expect(repo.node.defaultChild.userData.render()).toMatch(new RegExp(expectedString));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: 'autoscaling:UpdateAutoScalingGroup',
                Condition: {
                    StringEquals: {
                        'autoscaling:ResourceTag/resourceLogicalId': repo.node.uniqueId,
                    },
                },
                Effect: 'Allow',
                Resource: '*',
            }, {
                Action: 'ec2:DescribeTags',
                Effect: 'Allow',
                Resource: '*',
            }),
        },
    }));
});
test('repository configure client instance', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance,
        mountPoint: '/mnt/repository',
    });
    const userData = instance.userData.render();
    // THEN
    // white-box testing. If we mount the filesystem, then we've called: setupDirectConnect()
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/repository rw')));
    // Make sure we added the DB connection args
    expect(userData).toMatch(/.*export -f configure_deadline_database.*/);
    // Make sure we call the configureRepositoryDirectConnect script with appropriate argument.
    const regex = new RegExp(escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \\"/mnt/repository/DeadlineRepository\\"'));
    expect(userData).toMatch(regex);
});
test('configureClientInstance uses singleton for repo config script', () => {
    // Note: If this test fails, then check the asset hash for REPO_DC_ASSET -- it may have changed.
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance1 = new aws_ec2_1.Instance(stack, 'Instance1', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instance2 = new aws_ec2_1.Instance(stack, 'Instance2', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance1,
        mountPoint: '/mnt/repository',
    });
    repo.configureClientInstance({
        host: instance2,
        mountPoint: '/mnt/repository',
    });
    // THEN
    // Make sure that both instances have access to the same Asset for the configureRepositoryDirectConnect script
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Policy', 2, {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Effect: 'Allow',
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Resource: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':s3:::',
                                {
                                    Ref: asset_constants_2.REPO_DC_ASSET.Bucket,
                                },
                            ],
                        ],
                    },
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':s3:::',
                                {
                                    Ref: asset_constants_2.REPO_DC_ASSET.Bucket,
                                },
                                '/*',
                            ],
                        ],
                    },
                ],
            }),
        },
    }));
});
test('windows client cannot direct connect to repository', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
    });
    // THEN
    expect(() => {
        repo.configureClientInstance({
            host: instance,
            mountPoint: 'd:\\',
        });
    }).toThrowError('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'Repository',
        createConstruct: () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.Repository(isolatedStack, 'Repository', {
                vpc,
                version,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 3,
            'AWS::DocDB::DBClusterParameterGroup': 1,
            'AWS::DocDB::DBSubnetGroup': 1,
            'AWS::SecretsManager::Secret': 1,
            'AWS::DocDB::DBCluster': 1,
            'AWS::DocDB::DBInstance': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
describe('Security Groups', () => {
    let repositorySecurityGroup;
    beforeEach(() => {
        repositorySecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'AdditionalSecurityGroup', { vpc });
    });
    describe('DocDB', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    database: repositorySecurityGroup,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
                VpcSecurityGroupIds: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
    describe('EFS', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    fileSystem: repositorySecurityGroup,
                },
            });
            // THEN
            // The EFS construct adds the security group to each mount target, and one mount target is generated per subnet.
            const numMountTargets = vpc.selectSubnets().subnets.length;
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::EFS::MountTarget', numMountTargets, {
                SecurityGroups: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
    describe('Installer', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    installer: repositorySecurityGroup,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
});
test('validates VersionQuery is not in a different stack', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    version = new lib_1.VersionQuery(stack, 'Version');
    new lib_1.Repository(newStack, 'Repository', {
        vpc,
        version,
    });
    // WHEN
    function synth() {
        assert_1.SynthUtils.synthesize(newStack);
    }
    // THEN
    expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
});
test('creates an EFS AccessPoint when no filesystem is supplied', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
    });
    // THEN
    const efsResource = repo.node.findChild('FileSystem').node.defaultChild;
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::AccessPoint', {
        FileSystemId: stack.resolve(efsResource.ref),
        PosixUser: {
            Gid: '0',
            Uid: '0',
        },
        RootDirectory: {},
    }));
});
test('throws an error if supplied a MountableEfs with no Access Point', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    const fs = new aws_efs_1.FileSystem(newStack, 'FileSystem', {
        vpc,
    });
    const mountableFs = new core_2.MountableEfs(newStack, {
        filesystem: fs,
    });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version,
            vpc,
            fileSystem: mountableFs,
        });
    }
    // THEN
    expect(when).toThrow('When using EFS with the Repository, you must provide an EFS Access Point');
});
//# sourceMappingURL=data:application/json;base64,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