"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const __1 = require("../..");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('RenderQueue', () => {
    let app;
    let dependencyStack;
    let stack;
    let vpc;
    let rcsImage;
    let images;
    let repository;
    let version;
    let renderQueueVersion;
    let renderQueueCommon;
    // GIVEN
    beforeEach(() => {
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DepStack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc');
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        stack = new core_1.Stack(app, 'Stack');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        images = {
            remoteConnectionServer: rcsImage,
        };
        renderQueueVersion = new lib_1.VersionQuery(stack, 'Version');
        renderQueueCommon = new lib_1.RenderQueue(stack, 'RenderQueueCommon', {
            images,
            repository,
            version: renderQueueVersion,
            vpc,
        });
    });
    afterEach(() => {
        jest.resetAllMocks();
    });
    test('creates cluster', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
    });
    test('creates service', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service'));
    });
    test('creates task definition', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition'));
    });
    test('closed ingress by default', () => {
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            // The openListener=true option would create an ingress rule in the listener's SG.
            // make sure that we don't have that.
            SecurityGroupIngress: assert_1.arrayWith(assert_1.objectLike({})),
        }));
    });
    test('creates load balancer with default values', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
    });
    test('creates a log group with default prefix of "/renderfarm/"', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/RenderQueueCommon',
            RetentionInDays: 3,
        }));
    });
    test('configure the container log driver', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                assert_1.objectLike({
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                'Fn::GetAtt': [
                                    'RenderQueueCommonLogGroupWrapperA0EF7057',
                                    'LogGroupName',
                                ],
                            },
                            'awslogs-stream-prefix': 'RCS',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                }),
            ],
        }));
    });
    test('child dependencies added', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Host', {
            vpc,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        renderQueueCommon.addChildDependency(host);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: assert_1.arrayWith('RenderQueueCommonLBPublicListener935F5635', 'RenderQueueCommonRCSTask2A4D5EA5', 'RenderQueueCommonAlbEc2ServicePatternService42BEFF4C', 'RenderQueueCommonWaitForStableServiceDB53E266'),
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    describe('renderQueueSize.min', () => {
        describe('defaults to 1', () => {
            function assertSpecifiesMinSize(stackToAssert) {
                assert_1.expect(stackToAssert).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: '1',
                }));
            }
            test('renderQueueSize unspecified', () => {
                // THEN
                assertSpecifiesMinSize(stack);
            });
            test('renderQueueSize.min unspecified', () => {
                // GIVEN
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {},
                });
                // THEN
                assertSpecifiesMinSize(isolatedStack);
            });
        });
        // Asserts that only one RCS container and ASG instance can be created.
        // Deadline currently requires that successive API requests are serviced by a single RCS.
        test.each([
            [0],
            [2],
        ])('clamps to 1 - using %d', (min) => {
            // GIVEN
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow('renderQueueSize.min');
        });
        // Asserts that when the renderQueueSize.min prop is specified, the underlying ASG's min property is set accordingly.
        test('configures minimum number of ASG instances', () => {
            // GIVEN
            const min = 1;
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: min.toString(),
            }));
        });
    });
    describe('renderQueueSize.desired', () => {
        describe('defaults', () => {
            test('unset ASG desired', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: assert_1.ABSENT,
                }));
            });
        });
        test('caps at 1', () => {
            // GIVEN
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                renderQueueSize: {
                    desired: 2,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow('renderQueueSize.desired cannot be greater than 1');
        });
        describe('is specified', () => {
            const desired = 1;
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {
                        desired,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('configures desired number of ASG instances', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: desired.toString(),
                }));
            });
            // THEN
            test('configures desired number of ECS tasks in the service', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desired,
                }));
            });
        });
    });
    describe('trafficEncryption', () => {
        describe('defaults', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {},
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('to HTTPS internally between ALB and RCS', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('to HTTP externally between clients and ALB', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('when interalProtocol is HTTPS', () => {
            let isolatedStack;
            let renderQueue;
            let caCertPemLogicalId;
            let caCertPkcsLogicalId;
            let caCertPkcsPassphraseLogicalId;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                caCertPemLogicalId = isolatedStack.getLogicalId(renderQueue.node.findChild('TlsCaCertPem').node.defaultChild);
                const caCertPkcs = renderQueue.node.findChild('TlsRcsCertBundle');
                const caCertPkcsPassphrase = caCertPkcs.node.findChild('Passphrase');
                caCertPkcsLogicalId = isolatedStack.getLogicalId(caCertPkcs.node.defaultChild);
                caCertPkcsPassphraseLogicalId = isolatedStack.getLogicalId(caCertPkcsPassphrase.node.defaultChild);
            });
            // THEN
            test('ALB connects with HTTPS to port 4433', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('creates RCS cert', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
                    ServiceToken: {
                        'Fn::GetAtt': assert_1.arrayWith('Arn'),
                    },
                    DistinguishedName: { CN: 'renderfarm.local' },
                    Secret: {
                        NamePrefix: 'IsolatedStack/RenderQueue/TlsCaCertPem',
                    },
                }));
            });
            test('grants read access to secrets containing the certs and passphrase', () => {
                const taskDef = renderQueue.node.findChild('RCSTask');
                const taskRoleLogicalId = isolatedStack.getLogicalId(taskDef.taskRole.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                        Version: '2012-10-17',
                    },
                    Roles: assert_1.arrayWith({ Ref: taskRoleLogicalId }),
                }));
            });
            test('configures environment variables for cert secret URIs', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.deepObjectLike({
                        Environment: assert_1.arrayWith({
                            Name: 'RCS_TLS_CA_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_PASSPHRASE_URI',
                            Value: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                    })),
                }));
            });
        });
        describe('when internal protocol is HTTP', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('no certs are created', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('Custom::RFDK_X509Generator'));
            });
            test('ALB connects with HTTP to port 8080', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('externalProtocol is HTTPS', () => {
            let isolatedStack;
            const CERT_ARN = 'certarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('configures the ALB listener to use the specified ACM certificate', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                    Certificates: assert_1.arrayWith({
                        CertificateArn: CERT_ARN,
                    }),
                }));
            });
            test('raises an error when a cert is specified without a hostname', () => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'certArn'),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert2', CA_ARN),
                        },
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueue', props);
                })
                    // THEN
                    .toThrow(/A hostname must be provided when the external protocol is HTTPS/);
            });
        });
        describe('externalProtocol is HTTPS importing cert', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                    subject: {
                        cn: `ca.${ZONE_NAME}`,
                    },
                });
                const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                    subject: {
                        cn: `server.${ZONE_NAME}`,
                    },
                    signingCertificate: caCert,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            rfdkCertificate: serverCert,
                        },
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                    },
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('Imports Cert to ACM', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
                    X509CertificatePem: {
                        Cert: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Cert',
                            ],
                        },
                        Key: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Key',
                            ],
                        },
                        Passphrase: {
                            Ref: 'ServerCertPassphraseE4C3CB38',
                        },
                        CertChain: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                }));
            });
        });
        test('Throws if given ACM cert and RFDK Cert', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                subject: {
                    cn: `ca.${ZONE_NAME}`,
                },
            });
            const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                subject: {
                    cn: `server.${ZONE_NAME}`,
                },
                signingCertificate: caCert,
            });
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                        acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(isolatedStack, 'CA_Cert', CA_ARN),
                        rfdkCertificate: serverCert,
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if no Cert given', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {},
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if ACM Cert is given without a cert chain', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate./);
        });
    });
    describe('Client Connection', () => {
        describe('externalProtocol is http', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            let rq;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/http:\/\/.*:8080$/);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "http://renderqueue.${ZONE_NAME}:8080" \n` +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "http://renderqueue.${ZONE_NAME}:8080"  2>&1\n` +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
        });
        describe('externalProtocol is https', () => {
            let isolatedStack;
            let zone;
            let rq;
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'arn:a:b:c:dcertarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        zone,
                    },
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/https:\/\/.*:4433$/);
                expect(env).toHaveProperty('RENDER_QUEUE_TLS_CA_CERT_URI', CA_ARN);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "https://renderqueue.${ZONE_NAME}:4433" --tls-ca "${CA_ARN}"\n` +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "https://renderqueue.${ZONE_NAME}:4433" --tls-ca \"${CA_ARN}\" 2>&1\n` +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
        });
    });
    test('can specify subnets', () => {
        // GIVEN
        const subnets = [
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
                subnetId: 'SubnetID1',
                availabilityZone: 'us-west-2a',
            }),
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
                subnetId: 'SubnetID2',
                availabilityZone: 'us-west-2b',
            }),
        ];
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
            vpcSubnets: {
                subnets,
            },
            vpcSubnetsAlb: {
                subnets,
            },
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: assert_1.arrayWith('SubnetID1', 'SubnetID2'),
        }));
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                'SubnetID1',
                'SubnetID2',
            ],
        }));
    });
    test('can specify instance type', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'c5.large',
        }));
    });
    test('no deletion protection', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
            deletionProtection: false,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    describe('hostname', () => {
        // GIVEN
        const zoneName = 'mydomain.local';
        describe('not specified', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('does not create a record set', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('AWS::Route53::RecordSet'));
            });
        });
        describe('specified with zone but no hostname', () => {
            let zone;
            let isolatedStack;
            let renderQueue;
            beforeEach(() => {
                // GIVEN
                zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                    vpc,
                    zoneName,
                });
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('creates a record set using default hostname', () => {
                const loadBalancerLogicalId = dependencyStack.getLogicalId(renderQueue.loadBalancer.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
                    Name: `renderqueue.${zoneName}.`,
                    Type: 'A',
                    AliasTarget: assert_1.objectLike({
                        HostedZoneId: {
                            'Fn::GetAtt': [
                                loadBalancerLogicalId,
                                'CanonicalHostedZoneID',
                            ],
                        },
                    }),
                }));
            });
        });
        test.each([
            ['rq.somedomain.local'],
            ['1startswithnumber'],
            ['-startswithhyphen'],
            ['endswith-'],
            ['contains$symbol'],
            ['thisstringisexactlysixtyfourcharacterswhichisonelargerthanthemax'],
        ])('.hostname validation - %s', (hostname) => {
            // GIVEN
            const zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                zoneName: 'somedomain.local',
                vpc,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                hostname: {
                    hostname,
                    zone,
                },
            };
            // WHEN
            function when() {
                new lib_1.RenderQueue(stack, 'NewRenderQueue', props);
            }
            // THEN
            expect(when).toThrow(/Invalid RenderQueue hostname/);
        });
    });
    describe('Access Logs', () => {
        let isolatedStack;
        let isolatedVpc;
        let isolatedRepository;
        let isolatedVersion;
        let isolatedimages;
        let accessBucket;
        beforeEach(() => {
            // GIVEN
            isolatedStack = new core_1.Stack(app, 'IsolatedStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            isolatedVpc = new aws_ec2_1.Vpc(isolatedStack, 'Vpc');
            isolatedVersion = new lib_1.VersionQuery(isolatedStack, 'Version');
            isolatedRepository = new lib_1.Repository(isolatedStack, 'Repo', {
                version: isolatedVersion,
                vpc: isolatedVpc,
            });
            isolatedimages = {
                remoteConnectionServer: rcsImage,
            };
            accessBucket = new aws_s3_1.Bucket(isolatedStack, 'AccessBucket');
        });
        test('enabling access logs sets attributes and policies', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
        test('enabling access logs works with prefix', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                    prefix: 'PREFIX_STRING',
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }, {
                    Key: 'access_logs.s3.prefix',
                    Value: 'PREFIX_STRING',
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/PREFIX_STRING/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'RenderQueue',
            createConstruct: () => {
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 2,
                'AWS::IAM::Role': 8,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 4,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::DynamoDB::Table': 2,
                'AWS::SecretsManager::Secret': 2,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
    describe('SEP Policies', () => {
        test('with resource tracker', () => {
            // WHEN
            renderQueueCommon.addSEPPolicies();
            // THEN
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                        ],
                    ],
                }),
            }));
        });
        test('no resource tracker', () => {
            // WHEN
            renderQueueCommon.addSEPPolicies(false);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                        ],
                    ],
                }),
            }));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                        ],
                    ],
                }),
            }));
        });
    });
    test('creates WaitForStableService by default', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_WaitForStableService', {
            cluster: stack.resolve(renderQueueCommon.cluster.clusterArn),
            // eslint-disable-next-line dot-notation
            services: [stack.resolve(renderQueueCommon['pattern'].service.serviceArn)],
        }));
    });
    describe('Security Groups', () => {
        let backendSecurityGroup;
        let frontendSecurityGroup;
        beforeEach(() => {
            backendSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'ASGSecurityGroup', { vpc });
            frontendSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'LBSecurityGroup', { vpc });
        });
        test('adds security groups on construction', () => {
            // GIVEN
            const securityGroups = {
                backend: backendSecurityGroup,
                frontend: frontendSecurityGroup,
            };
            // WHEN
            new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                securityGroups,
            });
            // THEN
            assertSecurityGroupsWereAdded(securityGroups);
        });
        test('adds backend security groups post-construction', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            // WHEN
            renderQueue.addBackendSecurityGroups(backendSecurityGroup);
            // THEN
            assertSecurityGroupsWereAdded({
                backend: backendSecurityGroup,
            });
        });
        test('adds frontend security groups post-construction', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            // WHEN
            renderQueue.addFrontendSecurityGroups(frontendSecurityGroup);
            // THEN
            assertSecurityGroupsWereAdded({
                frontend: frontendSecurityGroup,
            });
        });
        test('security groups added post-construction are not attached to Connections object', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            renderQueue.addBackendSecurityGroups(backendSecurityGroup);
            renderQueue.addFrontendSecurityGroups(frontendSecurityGroup);
            const peerSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'PeerSecurityGroup', { vpc });
            // WHEN
            renderQueue.connections.allowFrom(peerSecurityGroup, aws_ec2_1.Port.tcp(22));
            // THEN
            // Existing LoadBalancer security groups shouldn't have the ingress rule added
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 22,
                ToPort: 22,
                GroupId: stack.resolve(frontendSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(peerSecurityGroup.securityGroupId),
            }));
            // Existing AutoScalingGroup security groups shouldn't have the ingress rule added
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 22,
                ToPort: 22,
                GroupId: stack.resolve(backendSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(peerSecurityGroup.securityGroupId),
            }));
        });
        function assertSecurityGroupsWereAdded(securityGroups) {
            if (securityGroups.backend !== undefined) {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroups.backend.securityGroupId)),
                }));
            }
            if (securityGroups.frontend !== undefined) {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                    SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroups.frontend.securityGroupId)),
                }));
            }
        }
    });
    test('validates VersionQuery is not in a different stack', () => {
        // GIVEN
        const newStack = new core_1.Stack(app, 'NewStack');
        // WHEN
        new lib_1.RenderQueue(newStack, 'RenderQueueNew', {
            images,
            repository,
            version,
            vpc,
        });
        // WHEN
        function synth() {
            assert_1.SynthUtils.synthesize(newStack);
        }
        // THEN
        expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
    });
});
//# sourceMappingURL=data:application/json;base64,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